(*---------------------------------------------------------------------------

  File: rip.sml (for Hol98.Taupo.1)

  Description: Proofs for the stability of the Routing Information
               Protocol (RIP, IETF RFC1058) and Abstraction proofs
               for Protocol convergence.

  Authors: Karthikeyan Bhargavan and Davor Obradovic
           University of Pennsylvania

  Date: July 1999

  Port to Hol98 : Konrad Slind, Sept. 26 1999

  Notes:  This file was used in conjunction with code for the
          SPIN model checker to prove correctness properties 
          for the RIP standard specification.

  Reference: 

        "Formal Verification of Standards for Distance Vector Routing",
        Karthikeyan Bhargavan, Davor Obradovic, and Carl Gunter,
        Proceedings of INFOCOM 2000

        http://www.cis.upenn.edu/~gunter/hol/papers/rip.ps

 ---------------------------------------------------------------------------*)

val _ = new_theory "rip";

app load ["bossLib", "pred_setTheory", "numLib", "Q"];

open bossLib numLib
     pairTheory pred_setTheory
     arithmeticTheory prim_recTheory numTheory;

infix 8 by;

(*--------------------------------------------------------------------------
          Undischarges the first assumption satisfying the predicate p. 
 ---------------------------------------------------------------------------*)

val _ = Rewrite.add_implicit_rewrites pairTheory.pair_rws;

fun FIRST_UNDISCH_TAC p = FIRST_ASSUM 
(fn t =>	let val ct = concl t in
		if (p ct) then (UNDISCH_TAC ct) else NO_TAC end);

fun FIRST_DROP_TAC p = FIRST_UNDISCH_TAC p THEN DISCH_THEN (fn t => ALL_TAC);

fun DROP_TAC a = FIRST_DROP_TAC (fn t => (t=a));

fun SUPPOSE_TAC x = 
    (ASM_CASES_TAC x) THENL
    [
     ALL_TAC,
     (FIRST_UNDISCH_TAC (fn x => true)) THEN
     (CONV_TAC CONTRAPOS_CONV) THEN
     (DISCH_THEN (fn x => ALL_TAC)) THEN
     (REWRITE_TAC [])

     ];


(*---------------------------------------------------------------------------
      The type for (one entry of) a routing table: 

            (next_network, next_router, distance)
 ---------------------------------------------------------------------------*)

Hol_datatype `rtable = <| next_net : 'N ; 
                          next_rtr : 'R ; 
                          hops     : num |>`;

(*---------------------------------------------------------------------------
      The type for the system state.
 ---------------------------------------------------------------------------*)

val state_ty = Type `:'R -> ('N,'R)rtable`;
val network  = Term `network :'N -> bool`;
val router   = Term `router  :'R -> bool`;

val state   = ty_antiq state_ty;

(*---------------------------------------------------------------------------
      Describes valid relations between 'R and 'N that can represent
      connections between routers and networks.
 ---------------------------------------------------------------------------*)

val connections_def = 
 Define
     `!^network ^router (c:'R->'N->bool). 
          connections network router c 
             = 
          !r. router r 
              ==> ?n1 n2. network n1 /\ network n2 /\ 
                          c r n1 /\ c r n2 /\ ~(n1=n2)`;


(*---------------------------------------------------------------------------
   rnr (r1,n,r2) is true if r1 and r2 are routers attached to the network n.
 ---------------------------------------------------------------------------*)

val rnr_def = 
 Define
     `!^network ^router c r1 n r2.
          rnr network router c (r1,n,r2) 
            = 
          router r1 /\ router r2 /\ network n /\ 
          c r1 n /\ c r2 n`;


(*---------------------------------------------------------------------------
     rnr (r1,_,r2) implies that r1 and r2 are routers (obviously).
 ---------------------------------------------------------------------------*)

val rnr_routers = Q.store_thm("rnr_routers",
`!^network ^router c r1 n r2.
     rnr network router c (r1,n,r2) ==> router r1 /\ router r2`,
RW_TAC std_ss [rnr_def]);
	

(*---------------------------------------------------------------------------
        rnr is symmetric 
 ---------------------------------------------------------------------------*)

val rnr_sym = Q.store_thm("rnr_sym",
`!^network ^router c r1 n r2.
      rnr network router c (r1,n,r2) ==> rnr network router c (r2,n,r1)`,
RW_TAC std_ss [rnr_def]);


(*---------------------------------------------------------------------------
      rnr is transitive 
 ---------------------------------------------------------------------------*)

val rnr_trans = Q.store_thm("rnr_trans",
`!^network ^router c r1 n r2 r3.
     rnr network router c (r1,n,r2) /\ rnr network router c (r1,n,r3) 
       ==> 
     rnr network router c (r2,n,r3)`,
RW_TAC std_ss [rnr_def]);


(*---------------------------------------------------------------------------
    Two routers are "neighbors" if they are different and share a network.
 ---------------------------------------------------------------------------*)

val neighbor_def = 
 Define
     `!^network ^router c r1 r2. 
         neighbor network router c r1 r2 
           = 
         ~(r1=r2) /\ ?n. rnr network router c (r1,n,r2)`;


(*---------------------------------------------------------------------------
     "neighbor" is a symmetric relation.
 ---------------------------------------------------------------------------*)

val neighbor_symmetric = Q.store_thm("neighbor_symmetric",
`!^network ^router c r1 r2.
     neighbor network router c r1 r2 ==> neighbor network router c r2 r1`,
RW_TAC std_ss [neighbor_def, rnr_def]
  THEN PROVE_TAC[]);


(*---------------------------------------------------------------------------
    neighbor r1 r2 implies that r1 and r2 are routers (obviously). 
 ---------------------------------------------------------------------------*)

val neighbor_routers = Q.store_thm("neighbor_routers",
`!^network ^router c r1 r2.
     neighbor network router c r1 r2 ==> router r1 /\ router r2`,
RW_TAC std_ss [neighbor_def]
  THEN PROVE_TAC [rnr_routers]);
	
	
(*---------------------------------------------------------------------------
      Successor in the arithmetic where 16 is infinity
 ---------------------------------------------------------------------------*)

val suc_def = Define `!n. suc n = if n<16 then SUC n else 16`;

val suc16 = Q.prove (`suc 16 = 16`, RW_TAC arith_ss [suc_def]);


(*---------------------------------------------------------------------------
     single_update rt (s,n,d2) 

       returns a new routing table that results when the old
       routing table 'rt' receives from the router 's' over the 
       network 'n' advertisement for the route of size 'd2'. 
       Makes sure that the receiver DOES an update no matter what, 
       when it receives a message from its "next router".
 ---------------------------------------------------------------------------*)

val single_update_def = 
 Define
     `single_update (rt:('N,'R)rtable) (s,n,d2) 
          = if (rt.next_rtr = s) /\ (rt.next_net = n) 
             then rt with <| hops := suc d2 |>
             else if suc d2 < rt.hops 
                   then <| next_net := n; next_rtr := s; hops := suc d2 |>
                   else rt`;


(*---------------------------------------------------------------------------
     update (sen, n, rec) 

        returns the state transformer (a state->state function)
        that corresponds to sending an advertisment from the router
        'sen' through the network 'n' to the router 'rec'. It takes 
        care of the poison-reverse, i.e. when the message is sent over
        the interface (network) through which the current best route had
        been acquired, an infinite route is advertised.
 ---------------------------------------------------------------------------*)

val update_def = 
 Define
     `!(sen:'R) (n:'N) (rec:'R) (s:^state) (r:'R). 
        update (sen, n, rec) s r 
          = if r = rec 
            then 
              (if (s sen).next_net = n 
                 then single_update (s r) (sen, n, 16)
                 else single_update (s r) (sen, n, (s sen).hops)) 
            else s r`;


(*---------------------------------------------------------------------------
   A sequence of messages given by the triple 

      ((send: num->'R), (through: num->'N), (receive: num->'R)) 

   is fair iff:
   
      * for every i, (send i, through i, receive i) is an rnr-triple 
        with sender different from the receiver

      * for every pair of neighbors (s,r), where r is not attached to
        the destination network, an advertisment message from s to r
        occurs infinitely often
 ---------------------------------------------------------------------------*)

val fair_def = 
 Define
     `!^network ^router c send through receive.
         fair network router c (send, through, receive) 
          =
         (!i. ~(send i = receive i) /\ 
              rnr network router c (send i, through i, receive i)) /\
         (!s r. neighbor network router c s r 
                 ==>
                !i. ?j. j>i /\ (send j = s) /\ (receive j = r))`;


(*---------------------------------------------------------------------------
      Defining a state sequence generated by a message sequence
 ---------------------------------------------------------------------------*)

val state_seq_def = 
 Define
     `(state_seq (s0:^state) send through receive 0 = s0) /\
      (state_seq s0 send through receive (SUC n)
        = update (send n, through n, receive n)
                 (state_seq s0 send through receive n))`;


(*---------------------------------------------------------------------------
    A state s is sound if all the next_network and next_router pointers 
    are meaningful. I.e. for each router r with state (nn,nr,d), it must
    be the case that (r,nn,nr) is an rnr-triple.
 ---------------------------------------------------------------------------*)

val sound_state_def = 
 Define
     `!^network ^router c s.
	 sound_state network router c dest s 
           = 
         !r. router(r) 
              ==>
             rnr network router c (r, (s r).next_net, (s r).next_rtr) /\
             (if c r dest
              then ((s r).next_net = dest) /\ 
                   ((s r).next_rtr = r)    /\
                   ((s r).hops = 1)
              else ~(r = (s r).next_rtr) /\ (s r).hops > 1)`;


(*---------------------------------------------------------------------------
   State soundness is preserved through updates triggered by advertisments. 
 ---------------------------------------------------------------------------*)

val soundness_preserved = Q.store_thm("soundness_preserved",
`!^network ^router c dest s sen n rec.
    sound_state network router c dest s 
   /\   rnr network router c (sen,n,rec)    
   /\  ~(sen = rec) 
      ==>
     sound_state network router c dest (update (sen,n,rec) s)`,
e (RW_TAC std_ss [sound_state_def,rnr_def,update_def,single_update_def] 
    THEN ZAP_TAC std_ss [suc16,DECIDE`16>1`,DECIDE`16<x ==> (x=1) ==> F`]);
val saved_gstack = p();
(*1*)
e (Q.PAT_ASSUM `$! M` (MP_TAC o Q.ID_SPEC)
     THEN RW_TAC std_ss [GSYM IMP_DISJ_THM,suc_def]);
e (PROVE_TAC [suc_def,DECIDE`x>y ==> SUC x > y`,DECIDE`16>1`]);
(*2*)
e (Q.PAT_ASSUM `$! M` (MP_TAC o Q.ID_SPEC)
     THEN ZAP_TAC std_ss
             [suc_def, DECIDE`~(SUC n < 1)`,DECIDE`16<x ==> (x=1) ==> F`]);
(*3*)
e (Q.PAT_ASSUM `$! M` (MP_TAC o Q.SPEC`sen`)
     THEN RW_TAC arith_ss [suc_def]);


(*---------------------------------------------------------------------------
      A trivial consequence of "sound_forever": if the initial state
      is sound, THEN all states (generated by a fair sequence) are 
      sound. 

      Note (KS) : in this version, the proof uses standard induction 
                  - much simpler! Also, "sound_always" doesn't seem to
                  be used in the sequel.
 ---------------------------------------------------------------------------*)

val sound_always = Q.store_thm("sound_always",
`!^network ^router c dest s0 send through receive.
       fair network router c (send,through,receive) 
       ==> sound_state network router c dest s0 
       ==> !i. sound_state network router c dest 
                   (state_seq s0 send through receive i)`,
RW_TAC std_ss [fair_def]
 THEN Induct_on `i`
 THEN RW_TAC std_ss [state_seq_def]
 THEN MATCH_MP_TAC soundness_preserved
 THEN RW_TAC arith_ss []);
	


(*---------------------------------------------------------------------------
      path network router c dest r n 
        = 
      there is a path of length n from r to dest. 
 ---------------------------------------------------------------------------*)

val path_def = 
 Define
   `(path ^network ^router c dest r 0 = F) /\
    (path network router c dest r (SUC 0) = c r dest) /\
    (path network router c dest r (SUC (SUC n)) 
       = ?s. router s /\ 
             path network router c dest s (SUC n) /\
             neighbor network router c s r)`;

(*---------------------------------------------------------------------------
     There are no paths of the length 0
 ---------------------------------------------------------------------------*)

val no_path_0 = Q.store_thm ("no_path_0",
 `!^network ^router c dest r d. path network router c dest r d ==> ~(d=0)`,
PROVE_TAC [NOT_SUC,path_def]);


(*---------------------------------------------------------------------------
    Paths of size 1 exists exactly between the destination and 
    routers directly connected to it.
 ---------------------------------------------------------------------------*)

val path_1 = Q.store_thm ("path_1",
`!^network ^router c dest r. path network router c dest r (SUC 0) = c r dest`,
PROVE_TAC [path_def]);


(*---------------------------------------------------------------------------
   If there is a path of length n to r, THEN there is a path of 
   length (n+1) to every neighbor of r. 
 ---------------------------------------------------------------------------*)

val path_neighbor = Q.store_thm ("path_neighbor",
`!^network ^router c dest r s n.
     neighbor network router c r s /\ path network router c dest r n
       ==>
     path network router c dest s (SUC n)`,
Cases_on`n` 
  THEN RW_TAC arith_ss [path_def,neighbor_def, rnr_def]
  THEN PROVE_TAC []);
	

(*---------------------------------------------------------------------------
    Describes when is a "universe" considered to be connected. 
     At the same time, gives a useful induction principle. 
 ---------------------------------------------------------------------------*)

val connected_def = 
 Define
     `!^network ^router c dest.
	connected network router c dest 
           = 
        !P. (!r. router r /\ c r dest ==> P r) /\
            (!r. (!s. P s /\ neighbor network router c s r ==> P r))
             ==>
	       !r. router r ==> P r`;

(*---------------------------------------------------------------------------
    If a "universe" is connected, THEN there are indeed finite paths 
    from every router to the destination. 
 ---------------------------------------------------------------------------*)

val connected_paths = Q.store_thm ("connected_paths",
`!^network ^router c dest.
    connected network router c dest 
      ==> 
	!r. router r ==> ?d. path network router c dest r d`,
REWRITE_TAC [connected_def]
  THEN REPEAT GEN_TAC THEN DISCH_THEN Ho_resolve.MATCH_MP_TAC
  THEN RW_TAC std_ss [] THENL [ALL_TAC, Cases_on `d`]
  THEN PROVE_TAC [path_def,neighbor_def,rnr_def]);


(*---------------------------------------------------------------------------
    Assuming connectedness, this gives the existance of the *minimum* 
    paths from every router to the destination.
 ---------------------------------------------------------------------------*)

val minimum_paths =
 let val t1 = Q.ISPECL [`P:num->bool`, `\x:num. x`] SET_MINIMUM
     val t2 = BETA_RULE (REWRITE_RULE [SPECIFICATION] t1)
     val t = GEN (--`P:num->bool`--) t2
 in
    ONCE_REWRITE_RULE [t] connected_paths
 end;
	

(*---------------------------------------------------------------------------
     Existence of the distance function
 ---------------------------------------------------------------------------*)

val distance_exists = Q.store_thm("distance_exists",
`?D. !^network ^router c dest. 
       connected network router c dest 
       ==> 
	(!r. router r ==> 
		(path network router c dest r (D network router c dest r)) /\
             (!d. path network router c dest r d 
                    ==> D network router c dest r <= d))`,
let val D = Term
      `\^network ^router c dest r.
	   @x. router r 
                ==> path network router c dest r x /\
                    !d. path network router c dest r d ==> x <= d`
in 
  EXISTS_TAC D THEN BETA_TAC 
   THEN REPEAT (GEN_TAC ORELSE DISCH_TAC) 
   THEN ASM_REWRITE_TAC [] 
   THEN CONV_TAC SELECT_CONV
   THEN PROVE_TAC [minimum_paths]
end);


(*---------------------------------------------------------------------------
     Definition of the distance function 
 ---------------------------------------------------------------------------*)

val distance_def = 
  Rsyntax.new_specification
	{consts  = [{const_name="D", fixity=Prefix}],
	 name    = "distance",
	 sat_thm = distance_exists};


(*---------------------------------------------------------------------------
     Distance is never equal to 0 
 ---------------------------------------------------------------------------*)

val distance_nonzero = Q.store_thm ("distance_nonzero",
`!^network ^router c dest r.
     connected network router c dest /\ router r
        ==>
     ~(D network router c dest r = 0)`,
PROVE_TAC [distance_def,path_def]);

(*---------------------------------------------------------------------------
    Distance is always > 0 (equivalent to "distance_nonzero", only nicer) 
 ---------------------------------------------------------------------------*)

val distance_positive = Q.store_thm("distance_positive",
`!^network ^router c dest r.
     connected network router c dest /\ router r
         ==>
     0 < D network router c dest r`,
PROVE_TAC [distance_nonzero,NOT_ZERO_LT_ZERO]);


(*---------------------------------------------------------------------------
     Distance = 1 is equivalent to connectedness to the destination.
 ---------------------------------------------------------------------------*)

val distance1_connected = 
let val dist1_path1 = 
     Q.prove(
       `!^network ^router c dest r.
	  connected network router c dest /\ router r 
            ==> 
          ((D network router c dest r = SUC 0) 
             = 
           path network router c dest r (SUC 0))`,
     RW_TAC std_ss [] THEN EQ_TAC 
       THENL [PROVE_TAC[distance_def],
              PROVE_TAC [distance_def,distance_nonzero,
                   DECIDE`x<=SUC 0 /\ ~(x=0) ==> (x=SUC 0)`]])
in
Q.store_thm("distance1_connected",
`!^network ^router c dest r.
    connected network router c dest /\ router r 
      ==> 
    ((D network router c dest r = SUC 0) = c r dest)`,
PROVE_TAC [dist1_path1,path_def])
end;
	

(*---------------------------------------------------------------------------
    Every router at distance n+1 has a neighbor at distance n (n>0). 
 ---------------------------------------------------------------------------*)
TRICKY
val predecessor_exists = Q.store_thm ("predecessor_exists",
 `!^network ^router c dest r n.
     ~(n=0) /\ connected network router c dest /\ router r /\ 
     (D network router c dest r = SUC n)
      ==>
      ?s. neighbor network router c s r /\ 
          (D network router c dest s = n)`,
RW_TAC std_ss [] 
  THEN IMP_RES_TAC distance_def 
  THEN Q.PAT_ASSUM `x <= x ` (K ALL_TAC)
  THEN Q.PAT_ASSUM `x = y` SUBST_ALL_TAC
  THEN `?k. n = SUC k` by PROVE_TAC [num_CASES] THEN RW_TAC std_ss []
  THEN Q.PAT_ASSUM `path x y z a b c` (fn th => 
         ASSUME_TAC th THEN STRIP_ASSUME_TAC(REWRITE_RULE [path_def] th))
  THEN Q.EXISTS_TAC `s` THEN RW_TAC std_ss [EQ_LESS_EQ] 
  THENL
   [PROVE_TAC [distance_def],
    IMP_RES_TAC distance_def
      THEN `path (network:'N->bool) (router:'R->bool) c dest r 
	      (SUC (D network router c dest s))`
        by RW_TAC arith_ss []
    PROVE_TAC [distance_def,LESS_EQ_TRANS
  THEN Q.PAT_ASSUM `path x y z a b c` 
         (STRIP_ASSUME_TAC o REWRITE_RULE[path_def])

(* Quite tricky! *)
REPEAT STRIP_TAC THEN IMP_RES_TAC distance_def
  THEN Q.UNDISCH_TAC `path (network:'N->bool) (router:'R->bool) c dest r 
	                 (D network router c dest r)` 
  THEN ASM_REWRITE_TAC [] THEN STRIP_TAC THEN IMP_RES_TAC path_DEF 
  THEN FIRST_ASSUM (UNDISCH_TAC o concl) THEN ASM_REWRITE_TAC [] 
  THEN STRIP_TAC THEN EXISTS_TAC (--`s:'R`--) 
  THEN ASM_REWRITE_TAC [EQ_LESS_EQ] THEN STRIP_TAC 
  THENL
    [IMP_RES_TAC distance_def,
     ONCE_REWRITE_TAC [GSYM LESS_EQ_MONO] 
       THEN SUPPOSE_TAC 
         (--`path (network:'N->bool) (router:'R->bool) c dest r 
	         (SUC (D network router c dest s))`--) 
       THENL
        [IMP_RES_TAC distance_def   (* use supposition *)
           THEN UNDISCH_TAC 
                (--`D (network:'N->bool) (router:'R->bool) c dest r 
                     <= SUC (D network router c dest s)`--) 
           THEN ASM_REWRITE_TAC [],
         ASM_REWRITE_TAC [path_DEF] 
           THEN IMP_RES_TAC distance_nonzero 
           THEN ASM_REWRITE_TAC [] THEN EXISTS_TAC (--`s:'R`--) 
           THEN ASM_REWRITE_TAC [] THEN IMP_RES_TAC distance_def]];

(*---------------------------------------------------------------------------
     No router at distance n has a neighbor at distance < n-1. 
 ---------------------------------------------------------------------------*)

val neighbor_distance = Q.store_thm("neighbor_distance",
`!^network ^router c dest r s.
    connected network router c dest /\ 
    router r /\ router s /\ 
    neighbor network router c s r 
      ==> 
    ~(SUC(D network router c dest s) < D network router c dest r)`,
PROVE_TAC 
 [distance_def,path_neighbor, LESS_EQ_LESS_TRANS,LESS_REFL]);

(*---------------------------------------------------------------------------
   The same as the neighbor_distance theorem, but for rnr-triples, 
   rather than only neighbors.
 ---------------------------------------------------------------------------*)

val rnr_distance = Q.store_thm("rnr_distance",
`!^network ^router c dest r s  n.
    connected network router c dest /\ 
    router r /\ router s /\ 
    rnr network router c (s,n,r) 
      ==> 
    ~(SUC(D network router c dest s) < D network router c dest r)`,
PROVE_TAC 
  [SUC_LESS,LESS_REFL,neighbor_distance,neighbor_def]);


(*---------------------------------------------------------------------------
     measure_induction was proved here, but is already available in Hol98
 ---------------------------------------------------------------------------*)


(*---------------------------------------------------------------------------
    `coreStable s`

        means that in the state s, all the routers directly
        connected to the destination have hops = 1.
 ---------------------------------------------------------------------------*)

val coreStable_def = 
 Define
     `!^network ^router c (dest:'N) (s:^state).
	  coreStable network router c dest s 
             = 
	  !r. router r 
                ==> (D network router c dest r = SUC 0)
                  ==> (hops s r = SUC 0)`;


(*---------------------------------------------------------------------------
    `midStable s k`

        means that in the state s, every router from the k-circle
        not directly connected to the destination has its
        hops = D (the actual distance) and is pointing at a router
        at the distance D-1.
 ---------------------------------------------------------------------------*)

val midStable_def = 
 Define
     `!^network ^router c (dest:'N) (s:^state) k.
	midStable network router c dest s k 
           = 
	!r. router r 
             ==> 
	    SUC 0 < D network router c dest r /\ 
            D network router c dest r <= k
             ==>
            (hops s r = D network router c dest r) /\ 
            (suc (D network router c dest (next_rtr s r)) 
              =
             D network router c dest r)`;

(*---------------------------------------------------------------------------
     `farStable s k`

          means that in the state s, every router outside of
          the k-circle has its' hops > k.
 ---------------------------------------------------------------------------*)

val farStable_def = 
 Define
     `!^network ^router c (dest:'N) (s:^state) k.
	farStable network router c dest s k 
           = 
	!r. router r 
              ==> k < D network router c dest r
              ==> k < hops s r`;

(*---------------------------------------------------------------------------
    `stable s k`

         means that in the state s, k-circle around the destination 
         stabilized (all the routers from inside the circle have 
         hops = the actual distance D, hops o next_rtr = D-1) and
         all the routers outside the circle have their hops > k.
 ---------------------------------------------------------------------------*)

val stable_def =
 Define
     `!^network ^router c (dest:'N) (s:^state) k.
	 stable network router c dest s k 
           = 
	coreStable network router c dest s   /\ 
	midStable network router c dest s k  /\ 
	farStable network router c dest s k`;


(*---------------------------------------------------------------------------
      Stable state preserves coreStability through updates. 
 ---------------------------------------------------------------------------*)

val coreStabilityPreserved = Q.store_thm ("coreStabilityPreserved",
`!^network ^router c (dest:'N) (s:^state) k sen n rec. 
      sound_state network router c dest s /\ 
      connected network router c dest     /\
      stable network router c dest s k    /\
      rnr network router c (sen,n,rec)    /\ 
      ~(sen=rec) 
        ==>
	  coreStable network router c dest (update (sen,n,rec) s)`,
RW_TAC std_ss [stable_def,coreStable_def,hops_def,update_def,sound_state_def]
  THEN RW_TAC arith_ss [state_expansion, single_update_def, suc_def] 
  THEN PROVE_TAC [distance1_connected, hops_def, 
                  DECIDE `~(16 < SUC 0)`, DECIDE`SUC x < SUC 0 ==> F`]);



val midStabilityPreserved = Q.store_thm("midStabilityPreserved",
`!^network ^router c (dest:'N) (s:^state) k sen n rec. 
     k < 16 /\
     sound_state network router c dest s /\ 
     connected network router c dest     /\
     stable network router c dest s k    /\
     rnr network router c (sen,n,rec)    /\ 
     ~(sen=rec) 
        ==>
         midStable network router c dest (update (sen,n,rec) s) k`,
REPEAT GEN_TAC THEN REWRITE_TAC [stable_def, midStable_def] THEN 
	REPEAT STRIP_TAC THEN ASM_REWRITE_TAC [next_rtr_def, hops_def, 
	update_def] THEN ASM_REWRITE_TAC [state_expansion, single_update_def,
	suc_def, LESS_REFL] THEN REPEAT COND_CASES_TAC THEN 
	ASM_REWRITE_TAC [FST, SND] THENL

	[(RES_TAC THEN IMP_RES_TAC (rnr_routers) THEN SUPPOSE_TAC 
	(--`rnr (network:'N->bool) router c ((next_rtr(s:^state) sen),n,r)`--) 
	THENL
		[(IMP_RES_TAC (rnr_routers) THEN 
		MP_TAC (rnr_distance) THEN CONV_TAC (CONTRAPOS_CONV) THEN
		STRIP_TAC THEN CONV_TAC (NOT_FORALL_CONV) THEN 
		EXISTS_TAC (--`network:'N->bool`--) THEN 
		CONV_TAC (NOT_FORALL_CONV) THEN 
		EXISTS_TAC (--`router:'R->bool`--) THEN
		CONV_TAC (NOT_FORALL_CONV) THEN
		EXISTS_TAC (--`c:'R -> 'N -> bool`--) THEN 
		CONV_TAC (NOT_FORALL_CONV) THEN EXISTS_TAC (--`dest:'N`--) THEN
		CONV_TAC (NOT_FORALL_CONV) THEN EXISTS_TAC (--`r:'R`--) THEN
		CONV_TAC (NOT_FORALL_CONV) THEN 
		EXISTS_TAC (--`next_rtr (s:^state) sen`--) THEN 
		UNDISCH_TAC (--`r=(rec:'R)`--) THEN 
		FIRST_UNDISCH_TAC (is_eq) THEN ASM_REWRITE_TAC [suc_def] THEN
		STRIP_TAC THEN STRIP_TAC THEN 
		SUPPOSE_TAC (--`D network router c (dest:'N) (sen:'R) < 16`--)
		THEN (FIRST_UNDISCH_TAC (is_cond o rand o rator)) THENL
		[(ASM_REWRITE_TAC [] THEN STRIP_TAC THEN
		FIRST_UNDISCH_TAC (is_forall) THEN CONV_TAC (CONTRAPOS_CONV)
		THEN STRIP_TAC THEN CONV_TAC (NOT_FORALL_CONV) THEN 
		EXISTS_TAC (--`sen:'R`--) THEN ASM_REWRITE_TAC [NOT_IMP] THEN
		REPEAT STRIP_TAC THENL
			[(MATCH_MP_TAC (LESS_SUC_EQ_COR) THEN 
			IMP_RES_TAC (distance_positive) THEN 
			ASM_REWRITE_TAC [] THEN 
			let val t = ISPECL [--`SUC 0`--,
			--`D network router c (dest:'N) (sen:'R)`--] EQ_SYM
			in MATCH_MP_TAC (CONTRAPOS t) end THEN
			REPEAT (FIRST_DROP_TAC is_forall) THEN 
			IMP_RES_TAC (distance1_connected) THEN 
			REPEAT (FIRST_DROP_TAC is_forall) THEN 
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN 
			UNDISCH_TAC (--`sound_state network router c (dest:'N)
			(s:^state)`--) THEN REWRITE_TAC [sound_state_def] THEN
			CONV_TAC (NOT_FORALL_CONV) THEN 
			EXISTS_TAC (--`sen:'R`--) THEN ASM_REWRITE_TAC [] THEN
			STRIP_TAC THEN UNDISCH_TAC 
			(--`rnr network router c ((sen:'R),(n:'N),rec)`--) THEN
			ASM_REWRITE_TAC [rnr_def] THEN STRIP_TAC THEN 
			RES_TAC THEN UNDISCH_TAC
			(--`SUC 0 < D network router c (dest:'N) (r:'R)`--)
			THEN ASM_REWRITE_TAC [LESS_REFL]),

			(let val d=(--`D network router c (dest:'N)(sen:'R)`--)
			in MP_TAC (ISPEC d LESS_SUC_REFL) end THEN
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN UNDISCH_TAC
			(--`D network router c (dest:'N) (r:'R) <= k`--) THEN
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN  
			IMP_RES_TAC (LESS_LESS_EQ_TRANS) THEN 
			ASM_REWRITE_TAC [LESS_OR_EQ]),

			(FIRST_ASSUM (UNDISCH_TAC o concl) THEN 
			ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THENL
			[(STRIP_TAC THEN (FIRST_UNDISCH_TAC is_neg) THEN 
			ASM_REWRITE_TAC [] THEN 
			CONV_TAC (NOT_FORALL_CONV) THEN 
			EXISTS_TAC (--`n:'N`--) THEN REWRITE_TAC [NOT_IMP] THEN
			UNDISCH_TAC (--`rnr network router c 
			(next_rtr (s:^state) sen,(n:'N),r)`--) THEN
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN 
			ASM_REWRITE_TAC [] THEN MATCH_MP_TAC (EQ_LESS) THEN
			ASM_REWRITE_TAC []),

			(DISCH_THEN (ASSUME_TAC o GSYM) THEN UNDISCH_TAC
			(--`D network router c (dest:'N)(sen:'R) < 16`--) THEN
			ASM_REWRITE_TAC [LESS_REFL])])
			]),

		(COND_CASES_TAC THEN STRIP_TAC THEN ASM_REWRITE_TAC [] THEN
		UNDISCH_TAC (--`16 = D network router c (dest:'N) (r:'R)`--)
		THEN ASM_REWRITE_TAC [])]),
		
		(UNDISCH_TAC (--`rnr network router c ((sen:'R),(n:'N),rec)`--)
		THEN ASM_REWRITE_TAC [rnr_def] THEN STRIP_TAC THEN 
		ASM_REWRITE_TAC [] THEN IMP_RES_TAC (sound_state_def) THEN
		IMP_RES_TAC (rnr_routers) THEN ASM_REWRITE_TAC [] THEN
		UNDISCH_TAC (--`rnr network router c 
		(sen,next_net (s:^state) sen,next_rtr s sen)`--) THEN
		ASM_REWRITE_TAC [rnr_def])]),

	(RES_TAC THEN MATCH_MP_TAC (EQ_SYM) THEN 
	UNDISCH_TAC (--`16 < hops (s:^state) r`--) THEN ASM_REWRITE_TAC [] THEN
	CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN 
	REWRITE_TAC [NOT_LESS] THEN MATCH_MP_TAC (LESS_EQ_TRANS) THEN 
	EXISTS_TAC (--`k:num`--) THEN 
	UNDISCH_TAC (--`D network router c (dest:'N) (r:'R) <= k`--) THEN
	ASM_REWRITE_TAC [] THEN STRIP_TAC THEN ASM_REWRITE_TAC [] THEN
	ASM_REWRITE_TAC [LESS_OR_EQ]),

	(RES_TAC THEN UNDISCH_TAC 
	(--`hops (s:^state) (r:'R) = D network router c (dest:'N) r`--) THEN
	ASM_REWRITE_TAC []),

	(RES_TAC THEN FIRST_ASSUM (UNDISCH_TAC o concl) THEN 
	ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THEN 
	IMP_RES_TAC (rnr_routers) THENL
		[(STRIP_TAC THEN ASM_CASES_TAC(--`(c:'R->'N->bool)sen dest`--)
		 THENL
			[(IMP_RES_TAC (sound_state_def) THEN FIRST_UNDISCH_TAC
			(fn t => rand(rator(rator t))=
			  (--`(c:'R->'N->bool) sen dest`--)) THEN 
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN UNDISCH_TAC
			(--`SUC (D network router c dest sen) = 
			D network router c (dest:'N) (rec:'R)`--) THEN
			IMP_RES_TAC (distance1_connected) THEN 
			REPEAT (FIRST_DROP_TAC is_forall) THEN 
			ASM_REWRITE_TAC [] THEN CONV_TAC (DEPTH_CONV num_CONV)
			THEN REWRITE_TAC []),

			(FIRST_UNDISCH_TAC (is_forall) THEN 
			CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN
			CONV_TAC (NOT_FORALL_CONV) THEN 
			EXISTS_TAC (--`sen:'R`--) THEN ASM_REWRITE_TAC [] THEN
			let val tm = SPEC 
				(--`D network router c (dest:'N) (sen:'R)`--)
				LESS_SUC_REFL
			in MP_TAC tm end THEN ASM_REWRITE_TAC [] THEN
			STRIP_TAC THEN IMP_RES_TAC (LESS_LESS_EQ_TRANS) THEN 
			ASM_REWRITE_TAC [NOT_IMP, DE_MORGAN_THM] THEN 
			REPEAT STRIP_TAC THENL
				[(MATCH_MP_TAC (LESS_CASES_IMP) THEN 
				STRIP_TAC THENL
					[(ASM_REWRITE_TAC [LESS_EQ, 
					LESS_EQ_MONO, LESS_EQ_0] THEN
					IMP_RES_TAC (distance_nonzero)),

					(MP_TAC (distance1_connected) THEN
					CONV_TAC (CONTRAPOS_CONV) THEN 
					REWRITE_TAC [] THEN STRIP_TAC THEN
					CONV_TAC (NOT_FORALL_CONV) THEN
					EXISTS_TAC (--`network:'N->bool`--)THEN
					CONV_TAC (NOT_FORALL_CONV) THEN
					EXISTS_TAC (--`router:'R->bool`--)THEN
					CONV_TAC (NOT_FORALL_CONV) THEN
					EXISTS_TAC (--`c:'R->'N->bool`--)THEN
					CONV_TAC (NOT_FORALL_CONV) THEN
					EXISTS_TAC (--`dest:'N`--) THEN
					CONV_TAC (NOT_FORALL_CONV) THEN
					EXISTS_TAC (--`sen:'R`--) THEN
					ASM_REWRITE_TAC [])]
				),

				(REWRITE_TAC [LESS_OR_EQ] THEN DISJ1_TAC THEN
				MATCH_MP_TAC (LESS_LESS_EQ_TRANS) THEN
				EXISTS_TAC (--`D network router c (dest:'N)
				  (r:'R)`--) THEN ASM_REWRITE_TAC []),

				(DISJ1_TAC THEN 
				ONCE_REWRITE_TAC [GSYM INV_SUC_EQ] THEN
				ASM_REWRITE_TAC [])
				]
			)]
		),

		
		(DISCH_THEN (ASSUME_TAC o GSYM) THEN 
		UNDISCH_TAC (--`~(D network router c (dest:'N) (sen:'R)<16)`--)
		THEN REWRITE_TAC [NOT_LESS, LESS_OR_EQ] THEN STRIP_TAC THENL
			[(IMP_RES_TAC (farStable_def) THEN 
			IMP_RES_TAC (LESS_EQ_LESS_TRANS) THEN UNDISCH_TAC
			   (--`D network router c (dest:'N)(r:'R)<16`--) THEN
			ASM_REWRITE_TAC [LESS_REFL]),

			(ASM_REWRITE_TAC [] THEN FIRST_UNDISCH_TAC (is_forall)
			THEN CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN
			CONV_TAC (NOT_FORALL_CONV) THEN 
			EXISTS_TAC (--`sen:'R`--) THEN
			ASM_REWRITE_TAC [NOT_IMP, DE_MORGAN_THM] THEN 
			REPEAT STRIP_TAC THENL
				[(MATCH_MP_TAC (LESS_LESS_EQ_TRANS) THEN 
				EXISTS_TAC (--`16`--) THEN
				REWRITE_TAC [LESS_OR_EQ] THEN STRIP_TAC THENL
					[(CONV_TAC (DEPTH_CONV num_CONV) THEN
					CONV_TAC (DEPTH_CONV num_CONV) THEN
					MATCH_MP_TAC (LESS_MONO) THEN
					REWRITE_TAC [LESS_0]),
					(ASM_REWRITE_TAC [])]
				),

				(UNDISCH_TAC (--`16 = D network router c 
				  (dest:'N) (sen:'R)`--) THEN 
				DISCH_THEN (ASSUME_TAC o GSYM) THEN
				ASM_REWRITE_TAC [] THEN UNDISCH_TAC 
				(--`D network router c (dest:'N)(r:'R) <= k`--)
				THEN ASM_REWRITE_TAC []),

				(DISJ1_TAC THEN UNDISCH_TAC  
				 (--`hops (s:^state) sen < 16`--) THEN 
				ASM_REWRITE_TAC [] THEN STRIP_TAC THEN
				MATCH_MP_TAC (LESS_NOT_EQ) THEN 
				ASM_REWRITE_TAC [])
				]
			
			)]

		)]
	),

	(RES_TAC THEN IMP_RES_TAC (rnr_routers) THEN 
	FIRST_UNDISCH_TAC (is_forall) THEN CONV_TAC (CONTRAPOS_CONV) THEN
	DISCH_THEN (ASSUME_TAC o GSYM) THEN CONV_TAC (NOT_FORALL_CONV) THEN
	EXISTS_TAC (--`sen:'R`--) THEN 
	ASM_REWRITE_TAC [NOT_IMP, DE_MORGAN_THM] THEN REPEAT STRIP_TAC THENL
		[(MATCH_MP_TAC (LESS_SUC_EQ_COR) THEN STRIP_TAC THENL
			[(let val t = SPEC 
			  (--`D network router c (dest:'N) (sen:'R)`--)
			  LESS_0_CASES
			  in MP_TAC t end THEN STRIP_TAC THEN 
			  ASM_REWRITE_TAC [] THEN 
			  IMP_RES_TAC (distance_nonzero) THEN 
			  REPEAT (FIRST_DROP_TAC is_forall) THEN UNDISCH_TAC
			  (--`~(D network router c (dest:'N) (sen:'R) = 0)`--)
			  THEN CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN
			  ASM_REWRITE_TAC []),
			(DISCH_THEN (MP_TAC o GSYM) THEN 
			MP_TAC (distance1_connected) THEN 
			CONV_TAC (CONTRAPOS_CONV) THEN REWRITE_TAC [] THEN 
			STRIP_TAC THEN CONV_TAC (NOT_FORALL_CONV) THEN
			EXISTS_TAC (--`network:'N->bool`--) THEN
			CONV_TAC (NOT_FORALL_CONV) THEN
			EXISTS_TAC (--`router:'R->bool`--) THEN
			CONV_TAC (NOT_FORALL_CONV) THEN
			EXISTS_TAC (--`c:'R->'N->bool`--) THEN
			CONV_TAC (NOT_FORALL_CONV) THEN
			EXISTS_TAC (--`dest:'N`--) THEN
			CONV_TAC (NOT_FORALL_CONV) THEN
			EXISTS_TAC (--`sen:'R`--) THEN ASM_REWRITE_TAC [] THEN
			STRIP_TAC THEN IMP_RES_TAC (sound_state_def) THEN 
			FIRST_DROP_TAC (fn t=> true) THEN 
			FIRST_ASSUM (UNDISCH_TAC o concl) THEN 
			ASM_REWRITE_TAC [DE_MORGAN_THM] THEN DISJ2_TAC THEN
			DISJ2_TAC THEN STRIP_TAC THEN 
			UNDISCH_TAC (--`~(hops (s:^state) sen < 16)`--) THEN
			ASM_REWRITE_TAC [] THEN 
			CONV_TAC (DEPTH_CONV num_CONV) THEN
			CONV_TAC (DEPTH_CONV num_CONV) THEN 
			MATCH_MP_TAC (LESS_MONO) THEN REWRITE_TAC [LESS_0])
			]

		),
		(MATCH_MP_TAC (LESS_EQ_TRANS) THEN EXISTS_TAC
		(--`D network router c (dest:'N) (r:'R)`--) THEN 
		ASM_REWRITE_TAC [] THEN FIRST_UNDISCH_TAC (is_eq) THEN 
		ASM_REWRITE_TAC [] THEN DISCH_THEN (ASSUME_TAC o GSYM) THEN
		ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THENL
			[(REWRITE_TAC [LESS_OR_EQ, LESS_SUC_REFL]),
			(FIRST_UNDISCH_TAC (is_eq) THEN 
			DISCH_THEN (MP_TAC o GSYM) THEN 
			ASM_REWRITE_TAC [suc_def] THEN 
			DISCH_THEN (MP_TAC o GSYM) THEN ASM_REWRITE_TAC [])
			]
		),

		(DISJ1_TAC THEN STRIP_TAC THEN UNDISCH_TAC 
		  (--`suc (D network router c dest (next_rtr (s:^state) r)) =
		           D network router (c:'R->'N->bool) dest r`--) THEN
		ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THENL
			[(STRIP_TAC THEN UNDISCH_TAC  
				(--`~(hops (s:^state) sen < 16)`--) THEN
			ASM_REWRITE_TAC []),

			(DISCH_THEN (MP_TAC o GSYM) THEN ASM_REWRITE_TAC [])
			]
		)]
	),

	(FIRST_UNDISCH_TAC(is_cond o rand o rator) THEN ASM_REWRITE_TAC[] THEN 
	STRIP_TAC THEN RES_TAC THEN IMP_RES_TAC (rnr_routers) THEN
	ASM_CASES_TAC(--`SUC 0 < D network router c (dest:'N)(sen:'R)`--) THENL
		[(ASM_CASES_TAC 
			(--`D network router c (dest:'N)(sen:'R) <=k`--) THENL
			[(FIRST_UNDISCH_TAC (is_forall) THEN 
			CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN
			CONV_TAC (NOT_FORALL_CONV) THEN 
			EXISTS_TAC (--`sen:'R`--) THEN 
			ASM_REWRITE_TAC [DE_MORGAN_THM] THEN DISJ1_TAC THEN
			STRIP_TAC THEN UNDISCH_TAC (--`hops (s:^state) r = 
			  D network router c (dest:'N)(r:'R)`--) THEN
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN UNDISCH_TAC 
			  (--`SUC (hops (s:^state) (sen:'R)) < hops s rec`--)
			THEN ASM_REWRITE_TAC [] THEN IMP_RES_TAC rnr_distance),

			(FIRST_UNDISCH_TAC (is_neg) THEN DISCH_THEN
			(ASSUME_TAC o (REWRITE_RULE [GSYM NOT_LESS]))
			THEN IMP_RES_TAC (farStable_def) THEN 
			UNDISCH_TAC (--`D network router c 
			  (dest:'N) (r:'R) <= k`--) THEN 
			CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN
			ASM_REWRITE_TAC [GSYM NOT_LESS] THEN
			MATCH_MP_TAC (LESS_TRANS) THEN 
			EXISTS_TAC (--`hops (s:^state)(sen:'R)`--) THEN
			ASM_REWRITE_TAC [] THEN 
			MATCH_MP_TAC (LESS_TRANS) THEN EXISTS_TAC 
			  (--`SUC (hops (s:^state)(sen:'R))`--) THEN
			REWRITE_TAC [LESS_SUC_REFL] THEN 
			MATCH_MP_TAC (LESS_LESS_EQ_TRANS) THEN 
			EXISTS_TAC (--`hops (s:^state)(rec:'R)`--) THEN
			ASM_REWRITE_TAC [] THEN 
			UNDISCH_TAC (--`hops (s:^state) (r:'R) = 
			  D network router c (dest:'N) r`--) THEN 
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN
			ASM_REWRITE_TAC [LESS_EQ_REFL])
			]
		),

		(FIRST_UNDISCH_TAC (is_neg) THEN CONV_TAC (CONTRAPOS_CONV) THEN
		STRIP_TAC THEN REWRITE_TAC [] THEN 
		MATCH_MP_TAC (LESS_SUC_EQ_COR) THEN STRIP_TAC THENL
			[(IMP_RES_TAC (distance_positive)),

			(DISCH_THEN (MP_TAC o GSYM) THEN STRIP_TAC THEN
			IMP_RES_TAC (distance1_connected) THEN 
			REPEAT (FIRST_DROP_TAC is_forall) THEN
			REPEAT (FIRST_DROP_TAC is_imp) THEN
			IMP_RES_TAC sound_state_def THEN 
			FIRST_DROP_TAC  is_cond THEN 
			FIRST_UNDISCH_TAC is_cond THEN ASM_REWRITE_TAC [] THEN
			FIRST_DROP_TAC is_cond THEN STRIP_TAC THEN UNDISCH_TAC
		 	(--`hops (s:^state) (r:'R) = 
			    D network router c (dest:'N) r`--) THEN 
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN UNDISCH_TAC
			(--`SUC (hops (s:^state) (sen:'R)) < hops s rec`--)THEN
			ASM_REWRITE_TAC [] THEN UNDISCH_TAC 
			(--`D network router c (dest:'N)(sen:'R)=SUC 0`--) THEN
			DISCH_THEN (ASSUME_TAC o GSYM) THEN 
			CONV_TAC (DEPTH_CONV num_CONV) THEN 
			ASM_REWRITE_TAC [] THEN MATCH_MP_TAC(rnr_distance) THEN
			EXISTS_TAC (--`n:'N`--) THEN ASM_REWRITE_TAC [])
			]
		)]
	),

	(RES_TAC THEN FIRST_UNDISCH_TAC (is_cond o rand o rator) THEN 
	ASM_REWRITE_TAC [] THEN CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN
	REWRITE_TAC [NOT_LESS] THEN MATCH_MP_TAC (LESS_IMP_LESS_OR_EQ) THEN
	MATCH_MP_TAC (LESS_EQ_LESS_TRANS) THEN EXISTS_TAC (--`k:num`--) THEN 
	ASM_REWRITE_TAC [] THEN 
	UNDISCH_TAC (--`D network router c (dest:'N) (r:'R) <= k`--) THEN
	ASM_REWRITE_TAC []),

	(RES_TAC THEN UNDISCH_TAC 
	  (--`hops (s:^state) (r:'R) = D network router c (dest:'N) r`--) THEN
	ASM_REWRITE_TAC []),


	(RES_TAC),


	(RES_TAC THEN FIRST_ASSUM (UNDISCH_TAC o concl) THEN 
	ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THEN 
	DISCH_THEN (ASSUME_TAC o GSYM) THEN ASM_REWRITE_TAC []  THEN
	FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
	ASM_REWRITE_TAC []),


	(FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN 
	FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
	ASM_REWRITE_TAC [] THEN CONV_TAC (CONTRAPOS_CONV) THEN 
	REWRITE_TAC [NOT_IMP] THEN STRIP_TAC THEN ASM_REWRITE_TAC [] THEN 
	STRIP_TAC THEN FIRST_UNDISCH_TAC (is_forall) THEN REWRITE_TAC [] THEN
	CONV_TAC (NOT_FORALL_CONV) THEN EXISTS_TAC (--`r:'R`--) THEN 
	ASM_REWRITE_TAC [DE_MORGAN_THM] THEN DISJ1_TAC THEN 
	DISCH_THEN (MP_TAC o GSYM) THEN REWRITE_TAC [] THEN 
	MATCH_MP_TAC (LESS_NOT_EQ) THEN MATCH_MP_TAC (LESS_TRANS) THEN
	EXISTS_TAC (--`16`--) THEN ASM_REWRITE_TAC [] THEN 
	MATCH_MP_TAC (LESS_EQ_LESS_TRANS) THEN EXISTS_TAC (--`k:num`--) THEN
	UNDISCH_TAC (--`D network router c (dest:'N) (r:'R) <= k`--) THEN 
	ASM_REWRITE_TAC []),


	(FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
	ASM_REWRITE_TAC [] THEN RES_TAC THEN STRIP_TAC THEN
	FIRST_UNDISCH_TAC (is_eq) THEN ASM_REWRITE_TAC [suc_def]),


	(FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
	FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
	ASM_REWRITE_TAC [] THEN RES_TAC THEN STRIP_TAC THEN STRIP_TAC THEN 
	FIRST_UNDISCH_TAC (is_eq) THEN ASM_REWRITE_TAC [suc_def]),


	(FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
	ASM_REWRITE_TAC [] THEN RES_TAC THEN STRIP_TAC THEN 
	FIRST_UNDISCH_TAC (is_eq) THEN ASM_REWRITE_TAC [suc_def]),

	(FIRST_UNDISCH_TAC (is_cond o rand o rator) THEN ASM_REWRITE_TAC[] THEN
	STRIP_TAC THEN FIRST_UNDISCH_TAC (fn t => (rand t = (--`16`--))) THEN
	FIRST_UNDISCH_TAC (fn t => (rand t = (--`16`--))) THEN 
	ASM_REWRITE_TAC [] THEN CONV_TAC (CONTRAPOS_CONV) THEN 
	REWRITE_TAC [NOT_IMP] THEN STRIP_TAC THEN ASM_REWRITE_TAC [] THEN
	STRIP_TAC THEN FIRST_UNDISCH_TAC (is_neg) THEN REWRITE_TAC [] THEN
	RES_TAC THEN IMP_RES_TAC (rnr_routers) THEN
	let val t = ISPECL [--`SUC (D network router c (dest:'N) (sen:'R))`--,
		--`D network router c (dest:'N) (rec:'R)`--] LESS_LESS_CASES
	in MP_TAC t end THEN STRIP_TAC THEN ASM_REWRITE_TAC [] THENL
		[(IMP_RES_TAC rnr_distance),
		(FIRST_ASSUM (UNDISCH_TAC o concl) THEN 
		REWRITE_TAC [LESS_THM] THEN ONCE_REWRITE_TAC [DISJ_SYM] THEN
		REWRITE_TAC [GSYM LESS_OR_EQ] THEN STRIP_TAC THEN 
		FIRST_UNDISCH_TAC (is_forall) THEN 
		CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN 
		CONV_TAC (NOT_FORALL_CONV) THEN EXISTS_TAC (--`sen:'R`--) THEN
		ASM_REWRITE_TAC [NOT_IMP,DE_MORGAN_THM] THEN 
		REPEAT STRIP_TAC THENL
			[(MATCH_MP_TAC (LESS_LESS_EQ_TRANS) THEN EXISTS_TAC
			(--`D network router c (dest:'N) (r:'R)`--) THEN 
			ASM_REWRITE_TAC []),

			(REWRITE_TAC [GSYM NOT_LESS] THEN STRIP_TAC THEN
			IMP_RES_TAC (farStable_def) THEN UNDISCH_TAC
			  (--`hops (s:^state) (r:'R) = 
			      D network router c (dest:'N) r`--) THEN 
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN 
			MP_TAC (ISPEC (--`k:num`--) LESS_REFL) THEN 
			REWRITE_TAC [] THEN MATCH_MP_TAC (LESS_TRANS) THEN 
			EXISTS_TAC (--`hops (s:^state) (sen:'R)`--) THEN 
			ASM_REWRITE_TAC [] THEN MATCH_MP_TAC (LESS_TRANS) THEN
			EXISTS_TAC (--`SUC(hops (s:^state) (sen:'R))`--) THEN 
			REWRITE_TAC [LESS_SUC_REFL] THEN 
			MATCH_MP_TAC (LESS_LESS_EQ_TRANS) THEN 
			EXISTS_TAC (--`hops (s:^state) (rec:'R)`--) THEN 
			ASM_REWRITE_TAC [] THEN UNDISCH_TAC
			  (--`D network router c (dest:'N) (r:'R) <= k`--) THEN
			ASM_REWRITE_TAC []),

			(DISJ1_TAC THEN STRIP_TAC THEN UNDISCH_TAC
			  (--`SUC (hops (s:^state) (sen:'R)) < hops s rec`--)
			THEN UNDISCH_TAC
			   (--`hops (s:^state) (r:'R) = 
			       D network router c (dest:'N) r`--) THEN
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN
			ASM_REWRITE_TAC [] THEN IMP_RES_TAC (rnr_distance)
			)]
		)]

	),

	(FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
	FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
	ASM_REWRITE_TAC [] THEN CONV_TAC (CONTRAPOS_CONV) THEN 
	REWRITE_TAC [NOT_IMP] THEN STRIP_TAC THEN ASM_REWRITE_TAC [] THEN 
	RES_TAC THEN STRIP_TAC THEN UNDISCH_TAC
	  (--`hops (s:^state) (r:'R) = D network router c (dest:'N) r`--) THEN
	ASM_REWRITE_TAC [] THEN STRIP_TAC THEN 
	UNDISCH_TAC (--`16 < hops (s:^state) (rec:'R)`--) THEN 
	ASM_REWRITE_TAC [NOT_LESS] THEN MATCH_MP_TAC (LESS_IMP_LESS_OR_EQ) THEN
	MATCH_MP_TAC (LESS_EQ_LESS_TRANS) THEN EXISTS_TAC (--`k:num`--) THEN 
	UNDISCH_TAC  (--`D network router c (dest:'N) (r:'R) <= k`--) THEN
	ASM_REWRITE_TAC []),

	(FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
	ASM_REWRITE_TAC [] THEN STRIP_TAC THEN RES_TAC THEN 
	FIRST_ASSUM (UNDISCH_TAC o concl) THEN ASM_REWRITE_TAC [suc_def]),

	(FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN 
	ASM_REWRITE_TAC [] THEN STRIP_TAC THEN RES_TAC THEN 
	FIRST_ASSUM (UNDISCH_TAC o concl) THEN ASM_REWRITE_TAC [suc_def]),

	(FIRST_UNDISCH_TAC is_neg THEN CONV_TAC (CONTRAPOS_CONV) THEN 
	STRIP_TAC THEN REWRITE_TAC [] THEN REPEAT COND_CASES_TAC THEN 
	ASM_REWRITE_TAC [FST,SND] THEN 
	UNDISCH_TAC  (--`~(16 = D network router c (dest:'N) (r:'R))`--) THEN
	ASM_REWRITE_TAC [] THEN STRIP_TAC THENL 
		[(RES_TAC THEN FIRST_ASSUM (UNDISCH_TAC o concl) THEN
		ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THEN
		ASM_REWRITE_TAC []), (* 1 *)

		(RES_TAC THEN UNDISCH_TAC 
		  (--`hops (s:^state)(r:'R)=D network router c (dest:'N) r`--)
		THEN CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN 
		DISCH_THEN (MP_TAC o GSYM) THEN REWRITE_TAC [] THEN
		MATCH_MP_TAC (LESS_NOT_EQ) THEN MATCH_MP_TAC (LESS_TRANS) THEN 
		EXISTS_TAC (--`16`--) THEN ASM_REWRITE_TAC [] THEN
		MATCH_MP_TAC (LESS_EQ_LESS_TRANS) THEN 
		EXISTS_TAC (--`k:num`--) THEN UNDISCH_TAC 
		  (--`D network router c (dest:'N) (r:'R) <= k`--) THEN
		ASM_REWRITE_TAC []),

		(RES_TAC THEN FIRST_ASSUM (UNDISCH_TAC o concl) THEN 
		REWRITE_TAC [suc_def] THEN COND_CASES_TAC THEN 
		ASM_REWRITE_TAC [] THEN STRIP_TAC THEN
		FIRST_UNDISCH_TAC (fn t => (rator (rator t)) = (--`<`--)) THEN
		ASM_REWRITE_TAC []),
 
		(RES_TAC THEN FIRST_ASSUM (UNDISCH_TAC o concl) THEN
		ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THEN
		ASM_REWRITE_TAC []), (* 1 *)

		(RES_TAC THEN FIRST_ASSUM (UNDISCH_TAC o concl) THEN
		ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THEN
		ASM_REWRITE_TAC []), (* 1 *)

		(FIRST_UNDISCH_TAC (is_cond o rand o rator) THEN 
		ASM_REWRITE_TAC [] THEN STRIP_TAC THEN RES_TAC THEN
		IMP_RES_TAC (rnr_routers) THEN ASM_CASES_TAC 
		  (--`D network router c (dest:'N) (sen:'R) <= k`--) THENL
			[(IMP_RES_TAC (LESS_EQ_LESS_TRANS)),

			(FIRST_ASSUM (UNDISCH_TAC o concl) THEN 
			REWRITE_TAC [GSYM NOT_LESS] THEN STRIP_TAC THEN
			UNDISCH_TAC 
			  (--`D network router c (dest:'N)(r:'R) <= k`--) THEN 
			CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN
			REWRITE_TAC [GSYM NOT_LESS] THEN 
			MATCH_MP_TAC (LESS_TRANS) THEN 
			EXISTS_TAC(--`hops (s:^state)(sen:'R)`--) THEN
			IMP_RES_TAC(farStable_def) THEN ASM_REWRITE_TAC[] THEN 
			MATCH_MP_TAC (LESS_TRANS) THEN EXISTS_TAC
			  (--`SUC(hops (s:^state)(sen:'R))`--) THEN 
			ASM_REWRITE_TAC [LESS_SUC_REFL] THEN 
			MATCH_MP_TAC (LESS_LESS_EQ_TRANS) THEN
			EXISTS_TAC (--`hops (s:^state)(rec:'R)`--)THEN
			ASM_REWRITE_TAC [] THEN UNDISCH_TAC 
			  (--`hops (s:^state) (r:'R) = 
			      D network router c (dest:'N) r`--) THEN 
			ASM_REWRITE_TAC [] THEN STRIP_TAC THEN
			ASM_REWRITE_TAC [LESS_OR_EQ])
			]
		),

		(FIRST_UNDISCH_TAC (is_cond o rand o rator) THEN 
		ASM_REWRITE_TAC [] THEN STRIP_TAC THEN RES_TAC THEN
		IMP_RES_TAC (rnr_routers) THEN UNDISCH_TAC 
		  (--`hops (s:^state)(r:'R) = D network router c (dest:'N)r`--)
		THEN CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN
		DISCH_THEN (MP_TAC o GSYM) THEN REWRITE_TAC [] THEN
		MATCH_MP_TAC (LESS_NOT_EQ) THEN MATCH_MP_TAC(LESS_TRANS) THEN 
		EXISTS_TAC(--`16`--) THEN ASM_REWRITE_TAC [] THEN
		MATCH_MP_TAC (LESS_EQ_LESS_TRANS) THEN 
		EXISTS_TAC (--`k:num`--) THEN 
		UNDISCH_TAC (--`D network router c (dest:'N)(r:'R)<=k`--) THEN
		ASM_REWRITE_TAC []),

		(RES_TAC THEN FIRST_ASSUM (UNDISCH_TAC o concl) THEN
		ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THEN
		ASM_REWRITE_TAC []),  (* 1 *)

		(RES_TAC THEN FIRST_ASSUM (UNDISCH_TAC o concl) THEN
		ASM_REWRITE_TAC [suc_def] THEN COND_CASES_TAC THEN
		ASM_REWRITE_TAC [])
		]
	)]);


val farStabilityPreserved = Q.store_thm("farStabilityPreserved",
`!^network ^router c (dest:'N) (s:^state) k sen n rec. 
     k < 16 /\ 
     sound_state network router c dest s /\ 
     connected network router c dest     /\
     stable network router c dest s k    /\
     rnr network router c (sen,n,rec)    /\ 
     ~(sen=rec) 
        ==>
          farStable network router c dest (update (sen,n,rec) s) k`,
	
let 
		val tac1 =
		(IMP_RES_TAC (rnr_routers) THEN
		let val t = ISPECL [--`network:'N->bool`--,
 		  --`router:'R->bool`--, --`c:'R->'N->bool`--, --`dest:'N`--,
		  --`r:'R`--, --`sen:'R`--, --`n:'N`--] rnr_distance
		in MP_TAC t end THEN ASM_REWRITE_TAC [NOT_LESS] THEN 
		UNDISCH_TAC(--`k < D network router c (dest:'N)(r:'R)`--) THEN 
		ASM_REWRITE_TAC [] THEN STRIP_TAC THEN STRIP_TAC THEN 
		IMP_RES_TAC (LESS_LESS_EQ_TRANS) THEN UNDISCH_TAC
		  (--`k < SUC (D network router c (dest:'N)(sen:'R))`--) THEN
		DISCH_THEN (MP_TAC o (REWRITE_RULE [LESS_EQ])) THEN 
		REWRITE_TAC [LESS_EQ_MONO] THEN REWRITE_TAC [LESS_OR_EQ] THEN
		STRIP_TAC THENL
			[(RES_TAC THEN REWRITE_TAC [suc_def] THEN 
			COND_CASES_TAC THEN ASM_REWRITE_TAC [] THEN 
			MATCH_MP_TAC (LESS_TRANS) THEN 
			EXISTS_TAC (--`hops (s:^state) (sen:'R)`--) THEN 
			ASM_REWRITE_TAC [LESS_SUC_REFL]),

			(FIRST_ASSUM (UNDISCH_TAC o concl) THEN 
			DISCH_THEN (ASSUME_TAC o GSYM) THEN 
			REWRITE_TAC [suc_def] THEN COND_CASES_TAC THEN
			ASM_REWRITE_TAC [] THEN ASM_CASES_TAC (--`SUC 0 < k`--)
			THENL	[(UNDISCH_TAC (--`midStable network router c 
				  (dest:'N) (s:^state) k`--) THEN 
				REWRITE_TAC [midStable_def] THEN 
				CONV_TAC (CONTRAPOS_CONV) THEN STRIP_TAC THEN
				CONV_TAC (NOT_FORALL_CONV) THEN 
				EXISTS_TAC (--`sen:'R`--) THEN ASM_REWRITE_TAC
				  [LESS_EQ_REFL,DE_MORGAN_THM] THEN 
				DISJ1_TAC THEN STRIP_TAC THEN
				FIRST_UNDISCH_TAC (is_neg) THEN 
				ASM_REWRITE_TAC [LESS_SUC_REFL]),

				(SUPPOSE_TAC (--`k=SUC 0`--) THENL
					[(UNDISCH_TAC (--`D network router c 
					  (dest:'N) (sen:'R) = k`--) THEN 
					ASM_REWRITE_TAC [LESS_MONO_EQ] THEN 
					STRIP_TAC THEN 
					IMP_RES_TAC (distance1_connected) THEN
					REPEAT (FIRST_DROP_TAC is_forall) THEN
					UNDISCH_TAC (--`sound_state network 
					  router c (dest:'N) (s:^state)`--)THEN
					REWRITE_TAC [sound_state_def] THEN
					CONV_TAC (CONTRAPOS_CONV) THEN 
					STRIP_TAC THEN 
					CONV_TAC (NOT_FORALL_CONV) THEN 
					EXISTS_TAC (--`sen:'R`--) THEN 
					ASM_REWRITE_TAC [] THEN STRIP_TAC THEN
					FIRST_UNDISCH_TAC (is_neg) THEN
					ASM_REWRITE_TAC [] THEN CONV_TAC
					  (DEPTH_CONV num_CONV) THEN 
					REWRITE_TAC [LESS_0]),

					(MATCH_MP_TAC (LESS_EQUAL_ANTISYM) THEN
					STRIP_TAC THENL
					[(ASM_REWRITE_TAC [GSYM NOT_LESS]),
					(FIRST_UNDISCH_TAC (is_eq) THEN
					DISCH_THEN (ASSUME_TAC o GSYM) THEN
					ASM_REWRITE_TAC [GSYM LESS_EQ] THEN 
					IMP_RES_TAC (distance_positive))]
					)]
				)]
			)]
		
		)

		val tac2 =
		(UNDISCH_TAC (--`r=(rec:'R)`--) THEN 
		 DISCH_THEN (ASSUME_TAC o GSYM) THEN ASM_REWRITE_TAC [] THEN
		 RES_TAC)

	in
	REPEAT GEN_TAC THEN REWRITE_TAC [stable_def, farStable_def] THEN
	REPEAT STRIP_TAC THEN REWRITE_TAC [hops_def, update_def] THEN
	REWRITE_TAC [state_expansion, single_update_def] THEN 
	REPEAT COND_CASES_TAC THEN ASM_REWRITE_TAC [FST, SND, suc16] THENL
		[(tac2),

		(tac1),

		(tac1),

		(tac2),

		(RES_TAC)
		]
 
	end
	);


val stabilityPreserved = Q.store_thm("stabilityPreserved",
`!^network ^router c (dest:'N) (s:^state) k sen n rec. 
     k < 16 /\ 
     sound_state network router c dest s /\ 
     connected network router c dest     /\
     stable network router c dest s k    /\
     rnr network router c (sen,n,rec)    /\ 
     ~(sen=rec) 
        ==>
         stable network router c dest (update (sen,n,rec) s) k`,

	REPEAT STRIP_TAC THEN REWRITE_TAC [stable_def] THEN 
	REPEAT STRIP_TAC THENL
	[(IMP_RES_TAC coreStabilityPreserved),

	(IMP_RES_TAC midStabilityPreserved),

	(IMP_RES_TAC farStabilityPreserved)
	]);
