(*---------------------------------------------------------------------------*
 * This file generates a database containing the HOL ".doc" files, and also  *
 * a database that indexes everything that is accessible in the system.      *
 *---------------------------------------------------------------------------*)

(* app load ["Mosml","Database","Parsspec"]; *)

fun mkentry s = let
  val content = let
    val (pfx,sfx) = Substring.position "-lc.adoc" (Substring.all s)
  in
    if Substring.size sfx = 0 orelse Substring.size sfx <> 8 then
      String.substring(s,0,size s - 5)
    else
      String.substring(s,0,size s - 8)
  end
in
  {comp=Database.Term(content, SOME"HOL"), file=s, line=0}
end

local fun is_adocfile s =
         size s>5 andalso String.extract(s, size s - 5, NONE) = ".adoc"
in
fun docdir_to_entries path (endpath, entries) =
  let val dir = Path.concat (path, endpath)
      val L0 = Mosml.listDir dir
      val L1 = List.filter is_adocfile L0
  in
    List.foldl (fn (e,l) => (mkentry e::l)) entries L1
  end
end;

val docdirs = let
  val instr = TextIO.openIn "../../tools/documentation-directories"
    handle _ => (print "Couldn't open documentation directories file";
                 raise Fail "File not found")
  val wholefile = TextIO.inputAll instr
  val _ = TextIO.closeIn instr
in
  String.tokens Char.isSpace wholefile
end

fun quote s = "\"" ^ s ^ "\""


fun print_component out c = let
  fun print s = TextIO.output(out, s)
  open Database
in
  case c of
    Str => print "Str"
  | Exc s => print ("Exc "^quote s)
  | Typ s => print ("Typ "^quote s)
  | Val s => print ("Val "^quote s)
  | Con s => print ("Con "^quote s)
  | Term(s, NONE) => print ("Term "^quote s)
  | Term(s, SOME s2) => print ("Term "^quote s^" "^quote s2)
end


fun print_entry out {comp, file,line} = let
  fun print s = TextIO.output(out,s)
in
  print (Int.toString line); print " ";
  print (quote file);        print " ";
  print_component out comp
end


fun print_entrylist out [] = ()
  | print_entrylist out [x] = (print_entry out x; TextIO.output(out, "\n"))
  | print_entrylist out (x::xs) = 
      (print_entry out x; TextIO.output(out, ";\n");
       print_entrylist out xs)


(*---------------------------------------------------------------------------
               The database of all hol ".doc" files.
 ---------------------------------------------------------------------------*)

fun buildDb holpath = let
  val doc_indices = List.foldl (docdir_to_entries holpath) [] docdirs
  fun foldfn (arg as (s, indices)) =
    Parsspec.processfile [] (Path.concat(holpath, "sigobj")) arg
    handle _ =>
      (print ("\n  *** Exception raised attempting to process file "^s^
              " ***\n");
       indices)
  val all_indices =
    List.foldr foldfn doc_indices
    [(* portableML *)
     "Arbnum.sig", "Arbint.sig", "Portable.sig",

     (*0*)
      "Type-sig.sml", "Term-sig.sml", "Thm-sig.sml", 
      "Theory-sig.sml", "Net-sig.sml", "Count.sig", 
      "Feedback.sig", "Lexis.sig", "Lib.sig", "Globals.sig", 

     (* parse *)
     "Parse.sig",  "Hol_pp.sig", "Absyn.sig", "Preterm.sig", 

     (* boolLib *)
     "Abbrev.sig", "DB.sig", "boolSyntax.sig", "boolTheory.sig",
     "Drule.sig", "Tactical.sig", "Tactic.sig", "Thm_cont.sig",
     "Conv.sig", "QConv.sig", "Ho_Net.sig", "Ho_Rewrite.sig", 
     "Rewrite.sig", "Rsyntax.sig", "Psyntax.sig",
     "TypeBase.sig", "DefnBase.sig", "Prim_rec.sig",

     (* jrh ind_defs *)
     "IndDefLib.sig", "InductiveDefinition.sig", "IndDefRules.sig",

     (* HolBdd *)
     "HolBdd.sig", "HolBddTheory.sig", "StateEnum.sig",

     (* multisets *)
     "bagTheory.sig", "bagLib.sig",

     (* basic automated proof *)
     "BasicProvers.sig",

     (* boss *)
     "bossLib.sig", "SingleStep.sig",

     (* combin theory *)
     "combinTheory.sig",

     (* computeLib *)
     "computeLib.sig", "compute_rules.sig",

     (* datatype *)
     "Datatype.sig", "ind_typeTheory.sig", "ind_types.sig",
     "RecordType.sig", "EquivType.sig",

     (* finite maps *)
     "finite_mapTheory.sig", 

     (* goalstackLib *)
     "goalstackLib.sig",

     (* hol88 *)
     "hol88Lib.sig",

     (* Integer *)
     "integerTheory.sig", "Cooper.sig", "intLib.sig",
     "gcdTheory.sig", "dividesTheory.sig", "intSyntax.sig",

     (* list *)
     "rich_listTheory.sig","listTheory.sig", "listLib.sig",
     "operatorTheory.sig",

     (* lite *)
     "liteLib.sig",

     (* lazy list *)
     "llistTheory.sig",

     (* meson *)
     "Canon_Port.sig","jrhTactics.sig","mesonLib.sig",

     (* muddy *)
     "bdd.sig","bvec.sig","fdd.sig", "muddyLib.sig",

     (* num *)
     "numTheory.sig", "prim_recTheory.sig", "arithmeticTheory.sig",
     "numeralTheory.sig", "numLib.sig", "arithSimps.sig",
     "reduceLib.sig",  "Arithconv.sig","Boolconv.sig", "arithLib.sig",

     (* one *)
     "oneTheory.sig",

     (* option *)
     "optionLib.sig","optionTheory.sig",

     (* pair *)
     "pairLib.sig", "pairTheory.sig", "pairSyntax.sig",
     "PairedLambda.sig", "pairSimps.sig", "pairTools.sig",

     (* pred_set *)
     "PFset_conv.sig", "PSet_ind.sig", "PGspec.sig","pred_setLib.sig",
     "pred_setTheory.sig", "setlistTheory.sig",

     (* probability *)
     "probLib.sig", "probTheory.sig",
     "booleanSequenceTheory.sig", "booleanSequenceTools.sig",
     "probAlgebraTheory.sig",     "probIndepTheory.sig", 
     "probCanonTheory.sig", "probCanonTools.sig", 
     "probExtraTheory.sig", "probExtraTools.sig", 
     "probPseudoTheory.sig", "probPseudoTools.sig", 
     "probUniformTheory.sig", "probUniformTools.sig", 
     "stateTransformerTheory.sig",

     (* Quotations *)
     "Q.sig",

     (* real numbers *)
     "Diff.sig", "limTheory.sig", "realTheory.sig",
     "RealArith.sig", "netsTheory.sig", "realaxTheory.sig",
     "RealSS.sig", "polyTheory.sig", "seqTheory.sig",
     "hratTheory.sig", "powserTheory.sig", "topologyTheory.sig",
     "hrealTheory.sig", "realLib.sig", "transcTheory.sig",

     (* refute *)
     "AC.sig","Canon.sig",

     (* relation *)
     "relationTheory.sig",

     (* res_quan *)
     "Res_quan.sig","res_quanLib.sig", "res_quanTheory.sig",

     (* Rings *)

     "abstraction.sig",        "prelimTheory.sig",
     "canonicalTheory.sig",    "quoteTheory.sig",
     "integerRingLib.sig",     "ringLib.sig",
     "integerRingTheory.sig",  "ringNormTheory.sig",
     "numRingLib.sig",         "ringTheory.sig",
     "numRingTheory.sig",      "semi_ringTheory.sig",

     (* simpLib *)
     "Cache.sig", "SatisfySimps.sig","Unify.sig",
     "Cond_rewr.sig", "Unwind.sig", "pureSimps.sig",
     "simpLib.sig", "boolSimps.sig", "Satisfy.sig", "combinSimps.sig",

     (* string *)
     "stringLib.sig", "stringTheory.sig", "stringSyntax.sig", 
     "stringSimps.sig",

     (* disjoint union *)
     "sumTheory.sig", "sumSimps.sig",

     (* tautLib *)
     "tautLib.sig",

     (* temporalLib *)
     "Omega_AutomataTheory.sig", "Past_Temporal_LogicTheory.sig",
     "Temporal_LogicTheory.sig", "temporalLib.sig",

     (* TFL *)
     "Defn.sig", "TotalDefn.sig",

     (* unwind *)
     "unwindLib.sig",

     (* word *)
     "wordLib.sig",
     "bword_arithTheory.sig","wordTheory.sig","word_numTheory.sig",
     "bword_bitopTheory.sig","word_baseTheory.sig",
     "bword_numTheory.sig","word_bitopTheory.sig"]

  val destination = Path.concat(holpath, Path.concat ("help", "HOLdbase"))
  val outs = TextIO.openOut destination
    handle Io{cause,name,function} =>
      ((raise cause)
         handle OS.SysErr(s, SOME error) =>
           (print ("Got error: "^OS.errorMsg error^
                   "\nfrom call to openOut on "^ name^"\n");
            Process.exit Process.failure)
              | _ =>
           (print ("I/O error opening "^name^"\n");
            Process.exit Process.failure))
  val _ = print_entrylist outs all_indices
in
  TextIO.closeOut outs
end;


fun errmsg s = TextIO.output(TextIO.stdErr, s ^ "\n");

val _ =
    case Mosml.argv () of
	[_,path]  => buildDb path
      |    _      => errmsg "Usage: mk_dbs <hol98-directory>/"
