(* Invoke with

     hol98 -I kls_list
*)

app load ["bossLib", "tflLib", "QLib",
          "permTheory", "sortingTheory", "kls_listTheory" ];

open tflLib kls_listTheory sortingTheory permTheory bossLib;
infix 8 by;

val o_DEF = combinTheory.o_DEF;

(*---------------------------------------------------------------------------*
 * Relation properties.                                                      *
 *---------------------------------------------------------------------------*)
val transitive_def = TCTheory.transitive_def;

val total_def      = Define `total R = !x y. R x y \/ R y x`;


(*---------------------------------------------------------------------------*
 * The quicksort algorithm.                                                  *
 *---------------------------------------------------------------------------*)

val qsort_def = 
Rfunction "qsort_def" `measure (LENGTH o SND)` 
   `(qsort ord [] = []) 
 /\ (qsort ord (CONS x rst) = 
       APPEND
          (qsort ord (filter ($~ o ord x) rst))
          (APPEND [x] (qsort ord (filter (ord x) rst))))`;


(*---------------------------------------------------------------------------*
 *  Termination of quicksort.                                                *
 *---------------------------------------------------------------------------*)
val qsort_terminates = save_thm("qsort_terminates",
tflLib.prove_termination qsort_def
(PROVE_TAC[listTheory.LENGTH, length_filter,
       arithmeticTheory.LESS_EQ_IMP_LESS_SUC]));


val qsort_induction = save_thm("qsort_induction",
  REWRITE_RULE [qsort_terminates](DISCH_ALL (#induction qsort_def)));

val qsort_eqns = save_thm("qsort_eqns", 
  REWRITE_RULE[qsort_terminates](#rules qsort_def));


val QSORT_TAC = PROGRAM_TAC{induction = qsort_induction,
                                rules = qsort_eqns};

(*----------------------------------------------------------------------------*
 *           PROPERTIES OF QSORT                                              *
 *----------------------------------------------------------------------------*)

val qsort_mem_stable = Q.store_thm
("qsort_mem_stable",
`!x R L. mem x (qsort R L) = mem x L`,
GEN_TAC THEN QSORT_TAC THENL
 [PROVE_TAC[],
  RW_TAC list_ss [o_DEF,mem_of_append,mem_def,mem_filter] 
    THEN PROVE_TAC[]]);


val qsort_permutation = 
Q.store_thm
("qsort_permutation",
`!R L. permutation L (qsort R L)`,
QSORT_TAC THENL
 [PROVE_TAC [permutation_refl],
  REWRITE_TAC [APPEND] 
    THEN MATCH_MP_TAC cons_permutation 
    THEN MATCH_MP_TAC trans_permute 
    THEN Q.EXISTS_TAC`APPEND(filter(~ o ord x) rst) (filter(ord (x:'a)) rst)`
    THEN PROVE_TAC [permutation_split, permutation_cong,
                ONCE_REWRITE_RULE[permutation_sym] append_permutation_sym]]);


val qsort_sorts = 
Q.store_thm
("qsort_sorts",
`!R L. transitive R /\ total R ==> sorted R (qsort R L)`,
QSORT_TAC THENL
 [RW_TAC list_ss [sorted_eqns],
  MATCH_MP_TAC sorted_append
   THEN RW_TAC list_ss [sorted_eq,qsort_mem_stable,mem_filter,o_DEF] 
   THEN Q.PAT_ASSUM `(x ==> y) /\ (x ==> z)` (K ALL_TAC)
   THEN Q.PAT_ASSUM `transitive ord /\ total ord` MP_TAC 
   THEN RW_TAC bool_ss [total_def,transitive_def] THEN
   `(y=x) \/ ord x y` by (Q.PAT_ASSUM `mem y (CONS x M)` MP_TAC THEN 
                       RW_TAC list_ss [mem_def,mem_filter,qsort_mem_stable]
                       THEN PROVE_TAC[]) 
   THEN PROVE_TAC []]);

(*---------------------------------------------------------------------------*
 * Bring everything together.                                                *
 *---------------------------------------------------------------------------*)
val qsort_correct = Q.store_thm
("qsort_correct", 
`!R. transitive R /\ total R ==> performs_sorting qsort R`,
PROVE_TAC[performs_sorting_def, 
          qsort_permutation, 
          qsort_sorts]);



(*---------------------------------------------------------------------------*
 * For fun, can try something like the following (needs "reduceLib" to       *
 * be loaded, and it is):                                                    *
 *                                                                           *
  open reduceLib;
  (REPEATC (CHANGED_CONV
    (REWRITE_CONV[qsort_eqns, filter_def, o_DEF, APPEND] THENC 
     REDUCE_CONV THENC DEPTH_CONV BETA_CONV)))
  (Term `qsort $<= [0;3;5;2;2;1;5]`);

 *                                                                           *
 *---------------------------------------------------------------------------*)
