structure permScript =
struct


open HolKernel basicHol90Lib listTheory kls_listTheory bossLib;
infix THEN THENL |-> ;
infix 8 by;

val _ = new_theory"perm";


(*---------------------------------------------------------------------------*
 * What's a permutation? This definition uses universal quantification to    *
 * define it. There are other ways, which could be compared to this, e.g.    *
 * as an inductive definition, or as a particular kind of function.          *
 *---------------------------------------------------------------------------*)

val permutation_def = Define
  `permutation L1 L2 = !x. filter ($= x) L1 = filter ($= x) L2`;



val permutation_refl = Q.store_thm
("permutation_refl",
    `!L. permutation L L`,
    PROVE_TAC[permutation_def]);


val permutation_intro = Q.store_thm
("permutation_intro",
    `!x y. (x=y) ==> permutation x y`,
    PROVE_TAC[permutation_refl]);


val permutation_trans =  
Q.store_thm
("permutation_trans",
  `transitive permutation`,
 RW_TAC list_ss [TCTheory.transitive_def] 
  THEN PROVE_TAC[permutation_def]);


val trans_permute = save_thm
("trans_permute",
 REWRITE_RULE [TCTheory.transitive_def] permutation_trans);


val permutation_sym = 
Q.store_thm
("permutation_sym",
  `!l1 l2. permutation l1 l2 = permutation l2 l1`,
PROVE_TAC [permutation_def]);


val permutation_cong = 
Q.store_thm
("permutation_cong",
`!(L1:'a list) L2 L3 L4. 
     permutation L1 L3 /\ 
     permutation L2 L4
     ==> permutation (APPEND L1 L2) (APPEND L3 L4)`,
PROVE_TAC [permutation_def,filter_append_distrib]);

val cons_append = PROVE [APPEND] `!L h. CONS h L = APPEND [h] L`;

val permutation_mono = 
Q.store_thm
("permutation_mono",
`!l1 l2 x. permutation l1 l2 ==> permutation (CONS x l1) (CONS x l2)`,
PROVE_TAC [cons_append,permutation_cong, permutation_refl]);


val permutation_cons_iff = 
let val lem = 
Q.prove`permutation (CONS x l1) (CONS x l2) ==> permutation l1 l2`
(RW_TAC list_ss [permutation_def,filter_def]
   THEN POP_ASSUM (MP_TAC o Q.SPEC`x'`)
   THEN RW_TAC list_ss [])
in 
  save_thm ("permutation_cons_iff",
            GEN_ALL(IMP_ANTISYM_RULE lem (SPEC_ALL permutation_mono)))
end;

val permutation_nil = 
Q.store_thm
("permutation_nil",
 `!L. (permutation L [] = (L=[])) /\ 
      (permutation [] L = (L=[]))`,
Cases THEN RW_TAC list_ss [permutation_def,filter_def]
 THEN Q.EXISTS_TAC `h`
 THEN RW_TAC list_ss []);


val lem = Q.prove
 `!h l1 l2. APPEND (filter ($=h) l1) (CONS h l2)
            = CONS h (APPEND (filter ($=h) l1) l2)`
(GEN_TAC THEN Induct 
   THEN RW_TAC list_ss [filter_def]
   THEN PROVE_TAC []);


val permutation_append = 
Q.store_thm
("permutation_append",
 `!l1 l2. permutation (APPEND l1 l2) (APPEND l2 l1)`,
RW_TAC list_ss [permutation_def,filter_append_distrib]
  THEN Induct_on `l1` 
  THEN RW_TAC list_ss [filter_def,lem]
  THEN PROVE_TAC[]);;


val cons_permutation = 
Q.store_thm
("cons_permutation",
`!x L M N. permutation L (APPEND M N) 
            ==> 
          permutation (CONS x L) (APPEND M (CONS x N))`,
REPEAT GEN_TAC THEN DISCH_TAC 
 THEN MATCH_MP_TAC trans_permute
 THEN PROVE_TAC [permutation_mono, permutation_append, APPEND, trans_permute]);


val append_permutation_sym = 
Q.store_thm
("append_permutation_sym",
`!A B C. permutation (APPEND A B) C ==> permutation (APPEND B A) C`,
PROVE_TAC [trans_permute, permutation_append]);

val permutation_split = 
Q.store_thm
("permutation_split",
`!P l. permutation l (APPEND (filter P l) (filter (~ o P) l))`,
GEN_TAC THEN REWRITE_TAC[combinTheory.o_DEF] THEN BETA_TAC 
 THEN Induct
 THEN RW_TAC list_ss [filter_def,permutation_refl] 
 THEN PROVE_TAC [APPEND,permutation_mono,cons_permutation]);


(*---------------------------------------------------------------------------
 * Directly performs one "sorting step" between 2 non-empty permutations.
 *---------------------------------------------------------------------------*)
val perm_sort_step = Q.prove
`!L h t. permutation (CONS h t) L ==> ?rst. CONS h rst = filter ($=h) L`
(RW_TAC list_ss [permutation_def,filter_def] 
  THEN POP_ASSUM (MP_TAC o Q.SPEC`h`) THEN REWRITE_TAC[]
  THEN PROVE_TAC[]);


val perm_step = Q.prove
`!l h t. permutation (CONS h t) l 
          ==> ?u. permutation l (CONS h u) /\ 
                  (LENGTH l = LENGTH (CONS h u))`
(RW_TAC bool_ss []
  THEN IMP_RES_TAC perm_sort_step
  THEN Q.EXISTS_TAC `APPEND rst (filter ($~ o $= h) l)`
  THEN REPEAT CONJ_TAC THENL 
  [PROVE_TAC [APPEND,permutation_split],
   ONCE_REWRITE_TAC [GSYM (el 3 (CONJUNCTS APPEND))]
     THEN ASM_REWRITE_TAC[]
     THEN PROVE_TAC [length_append_filter]]);


val permutation_length = Q.store_thm("permutation_length",
`!l1 l2. permutation l1 l2 ==> (LENGTH l1 = LENGTH l2)`,
Induct THENL
 [RW_TAC list_ss [permutation_nil],
  RW_TAC bool_ss [] 
   THEN IMP_RES_TAC perm_step
   THEN `permutation l1 u` by PROVE_TAC [trans_permute,permutation_cons_iff]
   THEN RW_TAC list_ss []]);


val _ = export_theory();

end;
