/*************************************************************************
  $Header: /home/jl/phd/bdd/RCS/pairs.c,v 1.2 1998/03/03 08:34:43 jl Exp jl $
  FILE:  pairs.c
  DESCR: Pair management for BDD package.
  AUTH:  Jorn Lind
  DATE:  february 1997
*************************************************************************/
#include <stdlib.h>
#include <limits.h>
#include "kernel.h"

/*======================================================================*/

static int      pairsid;            /* Pair identifier */
static BddPair* pairs;              /* List of all replacement pairs in use */


/*************************************************************************
*************************************************************************/

void bdd_pairs_init(void)
{
   pairsid = 0;
   pairs = NULL;
}


void bdd_pairs_done(void)
{
   BddPair *p = pairs;

   while (p != NULL)
   {
      BddPair *next = p->next;
      free(p->result);
      free(p);
      p = next;
   }
}


static int update_pairsid(void)
{
   pairsid++;
   
   if (pairsid == INT_MAX)
   {
      BddPair *p;
      pairsid = 0;
      for (p=pairs ; p!=NULL ; p=p->next)
	 p->id = pairsid++;
      bdd_operator_reset();
   }

   return pairsid;
}


void bdd_register_pair(BddPair *p)
{
   p->next = pairs;
   pairs = p;
}


void bdd_pairs_vardown(int level)
{
   BddPair *p;
   
   for (p=pairs ; p!=NULL ; p=p->next)
   {
      int tmp, n;

      for (n=0 ; n<bddvarnum ; n++)
      {
	 if (p->result[n] == level)
	    p->result[n] = level+1;
	 else
	 if (p->result[n] == level+1)
	    p->result[n] = level;
      }

      tmp = p->result[level];
      p->result[level] = p->result[level+1];
      p->result[level+1] = tmp;
      
      if (p->last == level)
	 p->last++;
   }
}


int bdd_pairs_resize(void)
{
   BddPair *p;

   for (p=pairs ; p!=NULL ; p=p->next)
   {
      if ((p->result=(int*)realloc(p->result,sizeof(int)*bddvarnum)) == NULL)
	 return bdd_error(BDD_MEMORY);
   }

   return 0;
}


/*
NAME    {* bdd\_newpair *}
SECTION {* kernel *}
SHORT   {* creates an empty variable pair table *}
PROTO   {* BddPair *bdd_newpair(void) *}
DESCR   {* Variable pairs of the type {\tt BddPair} are used in
           {\tt bdd\_replace} to define which variables to replace with
	   other variables. This function allocates such an empty table. *}
RETURN  {* Returns a new table of pairs. *}
ALSO    {* bdd\_replace, bdd\_setpair, bdd\_setpairs *}
*/
BddPair *bdd_newpair(void)
{
   int n;
   BddPair *p;
   
   if ((p=(BddPair*)malloc(sizeof(BddPair))) == NULL)
   {
      bdd_error(BDD_MEMORY);
      return NULL;
   }

   if ((p->result=(int*)malloc(sizeof(int)*bddvarnum)) == NULL)
   {
      free(p);
      bdd_error(BDD_MEMORY);
      return NULL;
   }

   for (n=0 ; n<bddvarnum ; n++)
      p->result[n] = n;
   p->id = update_pairsid();
   p->last = -1;
   
   bdd_register_pair(p);
   return p;
}


/*
NAME    {* bdd\_setpair *}
SECTION {* kernel *}
SHORT   {* sets one variable pair *}
PROTO   {* int bdd_setpair(BddPair *pair, int oldvar, int newvar) *}
DESCR   {* Adds the pair {\tt (oldvar,newvar)} to the table of pairs
           {\tt pair}. This results in {\tt oldvar} being substituted
	   with {\tt newvar} in a call to {\tt bdd\_replace}. *}
RETURN  {* Zero on success, otherwise a negative error code. *}
ALSO    {* bdd\_replace, bdd\_newpair, bdd\_setpairs *}
*/
int bdd_setpair(BddPair *pair, int oldvar, int newvar)
{
   if (pair == NULL)
      return 0;
   
   if (oldvar < 0  ||  oldvar > bddvarnum-1)
      return bdd_error(BDD_VAR);
   if (oldvar < 0  ||  newvar > bddvarnum-1)
      return bdd_error(BDD_VAR);

   pair->result[bddvar2level[oldvar]] = bddvar2level[newvar];
   pair->id = update_pairsid();
   
   if (bddvar2level[oldvar] > pair->last)
      pair->last = bddvar2level[oldvar];
      
   return 0;
}


/*
NAME    {* bdd\_setpairs *}
SECTION {* kernel *}
SHORT   {* defines a whole set of pairs *}
PROTO   {* int bdd_setpairs(BddPair *pair, int *oldvar, int *newvar, int size) *}
DESCR   {* As for {\tt bdd\_setpair} but with {\tt oldvar} and {\tt newvar}
           being arrays of variables of size {\tt size}. *}
RETURN  {* Zero on success, otherwise a negative error code. *}
ALSO    {* bdd\_replace, bdd\_newpair, bdd\_setpair *}
*/
int bdd_setpairs(BddPair *pair, int *oldvar, int *newvar, int size)
{
   int n,e;
   if (pair == NULL)
      return 0;
   
   for (n=0 ; n<size ; n++)
      if ((e=bdd_setpair(pair, oldvar[n], newvar[n])) < 0)
	 return e;
   
   return 0;
}


/*
NAME    {* bdd\_freepair *}
SECTION {* kernel *}
SHORT   {* frees a table of pairs *}
PROTO   {* void bdd_freepair(BddPair *pair) *}
DESCR   {* Frees the table of pairs {\tt pair} that has been allocated
           by a call to {\tt bdd\_newpair}. *}
ALSO    {* bdd\_replace, bdd\_newpair, bdd\_setpair, bdd\_setpairs *}
*/
void bdd_freepair(BddPair *p)
{
   if (p == NULL)
      return;
   
   if (pairs != p)
   {
      BddPair *bp = pairs;
      while (bp != NULL  &&  bp->next != p)
	 bp = bp->next;

      if (bp != NULL)
	 bp->next = p->next;
   }
   else
      pairs = p->next;
   
   free(p->result);
   free(p);
}


/* EOF */
