/*************************************************************************
  $Header: /home/jl/phd/bdd/src/RCS/kernel.h,v 1.11 1998/03/06 14:22:38 jl Exp jl $
  FILE:  kernel.h
  DESCR: Kernel specific definitions for BDD package
  AUTH:  Jorn Lind
  DATE:  (C) june 1997
*************************************************************************/

#ifndef _KERNEL_H
#define _KERNEL_H

#include <limits.h>
#include "bdd.h"

/*=== SANITY CHECKS ====================================================*/

#define CHECK(r)\
   if (!bddrunning) return bdd_error(BDD_RUNNING);\
   else if ((r) < 0  ||  (r) >= bddnodesize) return bdd_error(BDD_ILLBDD);\
   else if (r >= 2 && bddnodes[r].low == -1) return bdd_error(BDD_ILLBDD)\

   /* Make sure we use at least 32 bit integers */
#if (INT_MAX < 0x7FFFFFFF)
#error The compiler does not support 4 byte integers!
#endif

/* #define USECOND */       /* Define to allow for error condition checks */


/*=== SEMI-INTERNAL TYPES ==============================================*/

typedef struct s_BddNode /* Node table entry */
{
   unsigned int refcou : 10;
   unsigned int level  : 22;
   int low;
   int high;
   int hash;
   int next;
} BddNode;


/*=== KERNEL VARIABLES =================================================*/

extern int          bddrunning;        /* Flag - package initialized */
extern int          bdderrorcond;      /* Some error condition was met */
extern unsigned int bddnodesize;       /* Number of allocated nodes */
extern int          bddmaxnodesize;    /* Maximum allowed number of nodes */
extern BddNode*     bddnodes;          /* All of the bdd nodes */
extern int          bddvarnum;         /* Number of defined BDD variables */
extern int*         bddrefstack;       /* Internal node reference stack */
extern int*         bddrefstacktop;    /* Internal node reference stack top */
extern int*         bddvar2level;
extern int*         bddlevel2var;


/*=== KERNEL DEFINITIONS ===============================================*/

#define VERSION 11

#define MARKON   0x200000    /* Bit used to mark a node (1) */
#define MARKOFF  0x1FFFFF    /* - unmark */
#define MARKHIDE 0x1FFFFF

#define MAXVAR 0x1FFFFF
#define MAXREF 0x3FF

#define DECREF(n) if (bddnodes[n].refcou<MAXREF) bddnodes[n].refcou--
#define INCREF(n) if (bddnodes[n].refcou<MAXREF) bddnodes[n].refcou++

#define SETMARK(n) ((n)->level) |= MARKON
#define UNMARK(n)  ((n)->level) &= MARKOFF
#define MARKED(n)  (((n)->level) & MARKON)

   /* Hashfunctions */
#define PAIR(a,b)      ((unsigned int)(((a)+(b))*((a)+(b)+1)/2+(a)))
#define TRIPLE(a,b,c)  ((unsigned int)(PAIR(PAIR(a,b),c)))


/*=== KERNEL PROTOTYPES ================================================*/

#ifdef CPLUSPLUS
extern "C" {
#endif
   
extern int    bdd_error(int);
extern int    bdd_makenode(int, int, int);
extern int    bdd_noderesize(void(*)(void));
extern void   bdd_mark(int);
extern void   bdd_mark_upto(int, int);
extern void   bdd_markcount(int, int*);
extern void   bdd_unmark(int);
extern void   bdd_unmark_upto(int, int);
extern void   bdd_register_pair(BddPair*);
extern int    bdd_correctify(int, int, int);

extern int    bdd_operator_init(int);
extern void   bdd_operator_done(void);
extern void   bdd_operator_reset(void);

extern void   bdd_pairs_init(void);
extern void   bdd_pairs_done(void);
extern int    bdd_pairs_resize(void);
extern void   bdd_pairs_vardown(int);

extern void   bdd_fddinit(void);
extern void   bdd_fdddone(void);

extern int    bdd_reorder_vardown(int);
extern int    bdd_reorder_varup(int);

#ifdef CPLUSPLUS
}
#endif

#endif /* _KERNEL_H */


/* EOF */
