/*************************************************************************
  $Header: /home/jl/phd/bdd/src/RCS/bddop.c,v 1.10 1998/03/08 20:26:27 jl Exp jl $
  FILE:  bddop.c
  DESCR: BDD operators
  AUTH:  Jorn Lind
  DATE:  (C) nov 1997
*************************************************************************/
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "kernel.h"
#include "cache.h"
#include "bdd.h"

   /* Hash value modifiers to distinguish between entries in misccache */
#define CACHEID_RESTRICT  (0x1)
#define CACHEID_SATCOU    (0x2)
#define CACHEID_SATCOULN  (0x3)

   /* Number of boolean operators */
#define OPERATOR_NUM    11

   /* Operator results - entry = left<<1 | right  (left,right in {0,1}) */
static int oprres[OPERATOR_NUM][4] = 
{ {0,0,0,1},  /* and                       ( & )         */
  {0,1,1,0},  /* xor                       ( ^ )         */
  {0,1,1,1},  /* or                        ( | )         */
  {1,1,1,0},  /* nand                                    */
  {1,0,0,0},  /* nor                                     */
  {1,1,0,1},  /* implication               ( >> )        */
  {1,0,0,1},  /* bi-implication                          */
  {0,0,1,0},  /* difference /greater than  ( - ) ( > )   */
  {0,1,0,0},  /* less than                 ( < )         */
  {1,0,1,1},  /* inverse implication       ( << )        */
  {1,1,0,0}   /* not                       ( ! )         */
};


   /* Variables needed for the operators */
static int applyop;                 /* Current operator for apply */
static int appexop;                 /* Current operator for appex */
static int appexid;                 /* Current cache id for appex */
static int quantid;                 /* Current cache id for quantifications */
static int replaceid;               /* Current cache id for replace */
static int *replacepair;            /* Current replace pair */
static int replacelast;             /* Current last var. level to replace */
static int miscid;                  /* Current cache id for other results */
static int *varprofile;             /* Current variable profile */
static BddCache applycache;         /* Cache for apply results */
static BddCache quantcache;         /* Cache for exist/forall results */
static BddCache appexcache;         /* Cache for appex/appall results */
static BddCache replacecache;       /* Cache for replace results */
static BddCache misccache;          /* Cache for other results */


   /* Internal prototypes */
static int    bdd_not_rec(int);
static int    bdd_apply_rec(int, int);
static int    bdd_simplify_rec(int, int);
static int    bdd_quant_rec(int, int*, int);
static int    bdd_appquant_rec(int, int, int *, int);
static int    bdd_restrict_rec(int, int*, int);
static int    bdd_replace_rec(int);
static void   bdd_support_rec(int, char *);
static int    bdd_satone_rec(int);
static int    bdd_fullsatone_rec(int);
static double bdd_satcount_rec(int);
static double bdd_satcountln_rec(int);
static void   bdd_varprofile_rec(int);
static int    bdd_set2intp(BDD, int**, int*);
static int    bdd_set2sintp(BDD, int**, int*);


   /* Hashvalues */
#define NOTHASH(r,op)        (r)
#define APPLYHASH(l,r,op)    (TRIPLE(l,r,op))
#define RESTRHASH(r,var)     (PAIR(r,var))
#define QUANTHASH(r)         (r)
#define REPLACEHASH(r)       (r)
#define SATCOUHASH(r)        (r)
/*#define APPEXHASH(l,r,op)    (TRIPLE(l,r,op))*/
#define APPEXHASH(l,r,op)    (PAIR(l,r))

#ifndef M_LN2
#define M_LN2 0.69314718055994530942
#endif

#define log1p(a) (log(1.0+a))


/*************************************************************************
  Setup and shutdown
*************************************************************************/

int bdd_operator_init(int cachesize)
{
   if (BddCache_init(&applycache,cachesize) < 0)
      return bdd_error(BDD_MEMORY);
   
   if (BddCache_init(&quantcache,cachesize) < 0)
      return bdd_error(BDD_MEMORY);

   if (BddCache_init(&appexcache,cachesize) < 0)
      return bdd_error(BDD_MEMORY);

   if (BddCache_init(&replacecache,cachesize) < 0)
      return bdd_error(BDD_MEMORY);

   if (BddCache_init(&misccache,cachesize) < 0)
      return bdd_error(BDD_MEMORY);
   
   return 0;
}


void bdd_operator_done(void)
{
   BddCache_done(&applycache);
   BddCache_done(&quantcache);
   BddCache_done(&appexcache);
   BddCache_done(&replacecache);
   BddCache_done(&misccache);
}


void bdd_operator_reset(void)
{
   BddCache_reset(&applycache);
   BddCache_reset(&quantcache);
   BddCache_reset(&appexcache);
   BddCache_reset(&replacecache);
   BddCache_reset(&misccache);
}


/*************************************************************************
  Operators
*************************************************************************/

/*=== NOT ==============================================================*/

/*
NAME    {* bdd\_not *}
SECTION {* kernel *}
SHORT   {* Negates a bdd *}
PROTO   {* BDD bdd_not(BDD r) *}
DESCR   {* Negates the BDD {\tt r} by exchanging
           all references to the zero-terminal with references to the
	   one-terminal and vice versa. *}
RETURN  {* The negated bdd. *}
*/
BDD bdd_not(BDD r)
{
   int res;
   
   bddrefstacktop = bddrefstack;
   applyop = bddop_not;

   res = bdd_not_rec(r);
   bdderrorcond = 0;
   return res;
}


static int bdd_not_rec(int r)
{
   BddCacheData *entry;
   int res;

   if (r == 0)
      return 1;
   if (r == 1)
      return 0;
   
   entry = BddCache_lookup(&applycache, NOTHASH(r,bddop_not));
      
   if (entry->a == r  &&  entry->c == bddop_not)
      return entry->r.res;

   *(bddrefstacktop++) = bdd_not_rec(bddnodes[r].low);
   *(bddrefstacktop++) = bdd_not_rec(bddnodes[r].high);
   res = bdd_makenode(bddnodes[r].level,
		      *(bddrefstacktop-2), *(bddrefstacktop-1));
   bddrefstacktop -= 2;
   
   entry->a = r;
   entry->c = bddop_not;
   entry->r.res = res;

   return res;
}


/*=== APPLY ============================================================*/

/*
NAME    {* bdd\_apply *}
SECTION {* kernel *}
SHORT   {* Basic bdd operations *}
PROTO   {* BDD bdd_apply(BDD left, BDD right, int opr) *}
DESCR   {* The {\tt bdd\_apply} function performs all of the basic
           bdd operations with two operands, such as AND, OR etc.
	   The {\tt left} argument is the left bdd operand and {\tt right}
	   is the right operand. The {\tt opr} argument is the requested
	   operation and must be one of the following\\
	   
   \begin{tabular}{lllc}
     {\bf Identifier}    & {\bf Description} & {\bf Truth table}
        & {\bf C++ operator} \\
     {\tt bddop\_and}    & logical and    ($A \wedge B$)         & [0,0,0,1]
        & \verb%&% \\
     {\tt bddop\_xor}    & logical xor    ($A \oplus B$)         & [0,1,1,0]
        & \verb%^% \\
     {\tt bddop\_or}     & logical or     ($A \vee B$)           & [0,1,1,1]
        & \verb%|% \\
     {\tt bddop\_nand}   & logical not-and                       & [1,1,1,0] \\
     {\tt bddop\_nor}    & logical not-or                        & [1,0,0,0] \\
     {\tt bddop\_imp}    & implication    ($A \Rightarrow B$)    & [1,1,0,1]
        & \verb%>>% \\
     {\tt bddop\_biimp}  & bi-implication ($A \Leftrightarrow B$)& [1,0,0,1] \\
     {\tt bddop\_diff}   & set difference ($A \setminus B$)      & [0,0,1,0]
        & \verb%-% \\
     {\tt bddop\_less}   & less than      ($A < B$)              & [0,1,0,0]
        & \verb%<% \\
     {\tt bddop\_invimp} & reverse implication ($A \Leftarrow B$)& [1,0,1,1]
        & \verb%<<% \\
   \end{tabular}
   *}
   RETURN  {* The result of the operation. *}
*/
BDD bdd_apply(BDD l, BDD r, int op)
{
   int res;
   
   if (op<0 || op>bddop_invimp)
   {
      bdd_error(BDD_OP);
      return bddfalse;
   }
   
   bddrefstacktop = bddrefstack;
   applyop = op;

   res = bdd_apply_rec(l, r);
   bdderrorcond = 0;
   return res;
}


static int bdd_apply_rec(int l, int r)
{
   BddCacheData *entry;
   int res;
   
   if (bdderrorcond)
      return 0;

   switch (applyop)
   {
    case bddop_and:
       if (l == r)
	  return l;
       if (l == 0  ||  r == 0)
	  return 0;
       if (l == 1)
	  return r;
       if (r == 1)
	  return l;
       break;
    case bddop_or:
       if (l == r)
	  return l;
       if (l == 1  ||  r == 1)
	  return 1;
       if (l == 0)
	  return r;
       if (r == 0)
	  return l;
       break;
    case bddop_xor:
       if (l == r)
	  return 0;
       if (l == 0)
	  return r;
       if (r == 0)
	  return l;
       break;
    case bddop_nand:
       if (l == 0  ||  r == 0)
	  return 1;
       break;
    case bddop_nor:
       if (l == 1  ||  r == 1)
	  return 0;
       break;
   case bddop_imp:
      if (l == 0)
	 return 1;
      if (l == 1)
	 return r;
      if (r == 1)
	 return 1;
      break;
   }

   if (l < 2  &&  r < 2)
      res = oprres[applyop][l<<1 | r];
   else
   {
      entry = BddCache_lookup(&applycache, APPLYHASH(l,r,applyop));
      
      if (entry->a == l  &&  entry->b == r  &&  entry->c == applyop)
	 return entry->r.res;
      
      if (bddnodes[l].level == bddnodes[r].level)
      {
	 *(bddrefstacktop++) = bdd_apply_rec(bddnodes[l].low, bddnodes[r].low);
	 *(bddrefstacktop++) = bdd_apply_rec(bddnodes[l].high, bddnodes[r].high);
	 res = bdd_makenode(bddnodes[l].level, *(bddrefstacktop-2), *(bddrefstacktop-1));
      }
      else
      if (bddnodes[l].level < bddnodes[r].level)
      {
	 *(bddrefstacktop++) = bdd_apply_rec(bddnodes[l].low, r);
	 *(bddrefstacktop++) = bdd_apply_rec(bddnodes[l].high, r);
	 res = bdd_makenode(bddnodes[l].level, *(bddrefstacktop-2), *(bddrefstacktop-1));
      }
      else
      {
	 *(bddrefstacktop++) = bdd_apply_rec(l, bddnodes[r].low);
	 *(bddrefstacktop++) = bdd_apply_rec(l, bddnodes[r].high);
	 res = bdd_makenode(bddnodes[r].level, *(bddrefstacktop-2), *(bddrefstacktop-1));
      }

      bddrefstacktop -= 2;

#ifdef USECOND      
      if (res < 0 || bdderrorcond)
      {
	 bdderrorcond = BDD_MEMORY;
	 return 0;
      }
#endif
      
      entry->a = l;
      entry->b = r;
      entry->c = applyop;
      entry->r.res = res;
   }

   return res;
}


/*=== RESTRICT =========================================================*/

/*
NAME    {* bdd\_restrict *}
SECTION {* kernel *}
SHORT   {* Restric a set of variables to constant values *}
PROTO   {* BDD bdd_restrict(BDD r, BDD var) *}
DESCR   {* This function restricts the variables in {\tt r} to constant
           true or false. How this is done
	   depends on how the variables are included in the variable set
	   {\tt var}. If they
	   are included in their positive form then they are restricted to
	   true and vice versa. Unfortunately it is not possible to
	   insert variables in their negated form using {\tt bdd\_makeset},
	   so the variable set has to be build manually as a
	   conjunction of the variables. *}
RETURN  {* The restricted bdd. *}
ALSO    {* bdd\_makeset, bdd\_exist, bdd\_forall *}
*/
BDD bdd_restrict(BDD r, BDD var)
{
   int size;
   int *varset;
   BDD res;

   if (var < 2)  /* Empty set */
      return r;
   
   if (bdd_set2sintp(var, &varset, &size) < 0)
      return bddfalse;

   bddrefstacktop = bddrefstack;
   miscid = (var << 2) | CACHEID_RESTRICT;
   
   res = bdd_restrict_rec(r, varset, size);

   free(varset);
   bdderrorcond = 0;
   return res;
}


static int bdd_restrict_rec(int r, int *lvl, int num)
{
   BddCacheData *entry;
   int res, level;
   
#if USECOND
   if (bdderrorcond)
      return 0;
#endif
   
   if (r < 2  ||  num == 0)
      return r;

   entry = BddCache_lookup(&misccache, RESTRHASH(r,miscid));
   if (entry->a == r  &&  entry->c == miscid)
      return entry->r.res;
   
   level = abs(*lvl)-1;
   
   if (bddnodes[r].level == level)
   {
      if (*lvl > 0)
	 res = bdd_restrict_rec(bddnodes[r].high, lvl+1, num-1);
      else
	 res = bdd_restrict_rec(bddnodes[r].low, lvl+1, num-1);
   }
   else
   if (bddnodes[r].level < level)
   {
      *(bddrefstacktop++) = bdd_restrict_rec(bddnodes[r].low, lvl, num);
      *(bddrefstacktop++) = bdd_restrict_rec(bddnodes[r].high, lvl, num);
      res = bdd_makenode(bddnodes[r].level, *(bddrefstacktop-2),
			 *(bddrefstacktop-1));
      bddrefstacktop -= 2;
   }
   else
   {
      res = bdd_restrict_rec(r, lvl+1, num-1);
   }

#if USECOND
   if (res < 0  ||  bdderrorcond)
   {
      bdderrorcond = BDD_MEMORY;
      return 0;
   }
#endif

   entry->a = r;
   entry->c = miscid;
   entry->r.res = res;

   return res;
}


/*=== REPLACE ==========================================================*/

/*
NAME    {* bdd\_replace *}
SECTION {* kernel *}
SHORT   {* Replaces variables with other variables *}
PROTO   {* BDD bdd_replace(BDD r, BddPair *pair) *}
DESCR   {* Replaces all variables in the BDD {\tt r} with the variables
           defined by {\tt pair}. Each entry in {\tt pair} consists of a
	   old and a new variable. Whenever the old variable is found in
	   {\tt r} then a new node with the new variable is inserted instead.
	*}
ALSO   {* bdd\_newpair, bdd\_setpair, bdd\_setpairs *}
RETURN {* The result of the operation. *}
*/
BDD bdd_replace(BDD r, BddPair *pair)
{
   int res;
   
   bddrefstacktop = bddrefstack;
   replacepair = pair->result;
   replacelast = pair->last;
   replaceid = pair->id;

   res = bdd_replace_rec(r);
   bdderrorcond = 0;
   return res;
}


static int bdd_replace_rec(int r)
{
   BddCacheData *entry;
   int res;
   
#ifdef USECOND   
   if (bdderrorcond)
      return 0;
#endif
   
   if (r < 2  ||  bddnodes[r].level > replacelast)
      return r;

   entry = BddCache_lookup(&replacecache, REPLACEHASH(r));
   if (entry->a == r  &&  entry->c == replaceid)
      return entry->r.res;

   *(bddrefstacktop++) = bdd_replace_rec(bddnodes[r].low);
   *(bddrefstacktop++) = bdd_replace_rec(bddnodes[r].high);

   res = bdd_correctify(replacepair[bddnodes[r].level],
		     *(bddrefstacktop-2), *(bddrefstacktop-1));
   bddrefstacktop -= 2;

#ifdef USECOND   
   if (res < 0  ||  bdderrorcond)
   {
      bdderrorcond = BDD_MEMORY;
      return 0;
   }
#endif

   entry->a = r;
   entry->c = replaceid;
   entry->r.res = res;

   return res;
}


int bdd_correctify(int level, int l, int r)
{
   int res;
   
   if (level < bddnodes[l].level  &&  level < bddnodes[r].level)
      return bdd_makenode(level, l, r);

   if (level == bddnodes[l].level  ||  level == bddnodes[r].level)
   {
      bdd_error(BDD_REPLACE);
      return 0;
   }

   if (bddnodes[l].level == bddnodes[r].level)
   {
      *(bddrefstacktop++) = bdd_correctify(level,
					bddnodes[l].low, bddnodes[r].low);
      *(bddrefstacktop++) = bdd_correctify(level,
					bddnodes[l].high, bddnodes[r].high);
      res = bdd_makenode(bddnodes[l].level,
			 *(bddrefstacktop-2), *(bddrefstacktop-1));
      bddrefstacktop -= 2;
   }
   else
   if (bddnodes[l].level < bddnodes[r].level)
   {
      *(bddrefstacktop++) = bdd_correctify(level, bddnodes[l].low, r);
      *(bddrefstacktop++) = bdd_correctify(level, bddnodes[l].high, r);
      res = bdd_makenode(bddnodes[l].level,
			 *(bddrefstacktop-2), *(bddrefstacktop-1));
      bddrefstacktop -= 2;
   }
   else
   {
      *(bddrefstacktop++) = bdd_correctify(level, l, bddnodes[r].low);
      *(bddrefstacktop++) = bdd_correctify(level, l, bddnodes[r].high);
      res = bdd_makenode(bddnodes[r].level,
			 *(bddrefstacktop-2), *(bddrefstacktop-1));
      bddrefstacktop -= 2;
   }
   
   return res;
}


/*=== SIMPLIFY =========================================================*/

/*
NAME    {* bdd\_simplify *}
SECTION {* kernel *}
SHORT   {* Coudert and Madre's restrict function *}
PROTO   {* BDD bdd_simplify(BDD f, BDD d) *}
DESCR   {* Tries to simplify the BDD {\tt f} by restricting it to the
           domaine covered by {\tt d}. *}
ALSO    {* bdd\_restrict *}
RETURN  {* Returns the simplified BDD *}
*/
BDD bdd_simplify(BDD f, BDD d)
{
   int res;
   
   bddrefstacktop = bddrefstack;
   
   res = bdd_simplify_rec(f, d);
   bdderrorcond = 0;
   return res;
}


static int bdd_simplify_rec(int f, int d)
{
   BddCacheData *entry;
   int res;

#ifdef USECOND
   if (bdderrorcond)
      return 0;
#endif
   
   if (d == 0)
      return 0;
   if (d == 1)
      return f;
   if (f < 2)
      return f;

   entry = BddCache_lookup(&applycache, APPLYHASH(f,d,bddop_simplify));
   
   if (entry->a == f  &&  entry->b == d  &&
       entry->c == bddop_simplify)
      return entry->r.res;
   
   if (bddnodes[f].level == bddnodes[d].level)
   {
      if (bddnodes[d].low == 0)
	 res = bdd_simplify_rec(bddnodes[f].high, bddnodes[d].high);
      else
      if (bddnodes[d].high == 0)
	 res = bdd_simplify_rec(bddnodes[f].low, bddnodes[d].low);
      else
      {
	 *(bddrefstacktop++) = bdd_simplify_rec(bddnodes[f].low,
						bddnodes[d].low);
	 *(bddrefstacktop++) = bdd_simplify_rec(bddnodes[f].high,
						bddnodes[d].high);
	 res = bdd_makenode(bddnodes[f].level,
			    *(bddrefstacktop-2), *(bddrefstacktop-1));
	 bddrefstacktop -= 2;
      }
   }
   else
   if (bddnodes[f].level < bddnodes[d].level)
   {
      *(bddrefstacktop++) = bdd_simplify_rec(bddnodes[f].low, d);
      *(bddrefstacktop++) = bdd_simplify_rec(bddnodes[f].high, d);
      res = bdd_makenode(bddnodes[f].level,
			 *(bddrefstacktop-2), *(bddrefstacktop-1));
      bddrefstacktop -= 2;
   }
   else
   {
      *(bddrefstacktop++) = bdd_simplify_rec(f, bddnodes[d].low);
      *(bddrefstacktop++) = bdd_simplify_rec(f, bddnodes[d].high);
      res = bdd_makenode(bddnodes[d].level,
			 *(bddrefstacktop-2), *(bddrefstacktop-1));
      bddrefstacktop -= 2;
   }

#ifdef USECOND   
   if (res < 0 || bdderrorcond)
   {
      bdderrorcond = BDD_MEMORY;
      return 0;
   }
#endif
   
   entry->a = f;
   entry->b = d;
   entry->c = bddop_simplify;
   entry->r.res = res;

   return res;
}


/*=== QUANTIFICATION ===================================================*/

/*
NAME    {* bdd\_exist *}
SECTION {* kernel *}
SHORT   {* Existential quantification of variables *}
PROTO   {* BDD bdd_exist(BDD r, BDD var) *}
DESCR   {* Removes all occurences in {\tt r} of variables in the set
           {\tt var} by existential quantification. *}
ALSO    {* bdd\_forall, bdd\_makeset *}
RETURN  {* The quantified BDD. *}
*/
BDD bdd_exist(BDD r, BDD var)
{
   int size;
   int *varset;
   BDD res;

   if (var < 2)  /* Empty set */
      return r;
   
   if (bdd_set2intp(var, &varset, &size) < 0)
      return bddfalse;

   bddrefstacktop = bddrefstack;
   quantid = var << 2;
   applyop = bddop_or;

   res = bdd_quant_rec(r, varset, size);

   free(varset);
   bdderrorcond = 0;
   return res;
}


/*
NAME    {* bdd\_forall *}
SECTION {* kernel *}
SHORT   {* Universal quantification of variables *}
PROTO   {* BDD bdd_forall(BDD r, BDD var) *}
DESCR   {* Removes all occurences in {\tt r} of variables in the set
           {\tt var} by universal quantification. *}
ALSO    {* bdd\_exist, bdd\_makeset *}
RETURN  {* The quantified BDD. *}
*/
BDD bdd_forall(BDD r, BDD var)
{
   int size;
   int *varset;
   BDD res;

   if (var < 2)  /* Empty set */
      return r;
   
   if (bdd_set2intp(var, &varset, &size) < 0)
      return bddfalse;

   bddrefstacktop = bddrefstack;
   quantid = (var << 2) | 1;
   applyop = bddop_and;
   
   res =  bdd_quant_rec(r, varset, size);

   free(varset);
   bdderrorcond = 0;
   return res;
}


static int bdd_quant_rec(int r, int* var, int num)
{
   BddCacheData *entry;
   int res;
   
#if USECOND
   if (bdderrorcond)
      return 0;
#endif
   
   if (r < 2  ||  num == 0)
      return r;

   entry = BddCache_lookup(&quantcache, QUANTHASH(r));
   if (entry->a == r  &&  entry->c == quantid)
      return entry->r.res;

   if (bddnodes[r].level == *var)
   {
      *(bddrefstacktop++) = bdd_quant_rec(bddnodes[r].low, var+1, num-1);
      *(bddrefstacktop++) = bdd_quant_rec(bddnodes[r].high, var+1, num-1);
      res = bdd_apply_rec(*(bddrefstacktop-2), *(bddrefstacktop-1));
      bddrefstacktop -= 2;
   }
   else
   if (bddnodes[r].level < *var)
   {
      *(bddrefstacktop++) = bdd_quant_rec(bddnodes[r].low, var, num);
      *(bddrefstacktop++) = bdd_quant_rec(bddnodes[r].high, var, num);
      res = bdd_makenode(bddnodes[r].level, *(bddrefstacktop-2),
			 *(bddrefstacktop-1));
      bddrefstacktop -= 2;
   }
   else
   {
      res = bdd_quant_rec(r, var+1, num-1);
   }

#if USECOND
   if (res < 0  ||  bdderrorcond)
   {
      bdderrorcond = BDD_MEMORY;
      return 0;
   }
#endif

   entry->a = r;
   entry->c = quantid;
   entry->r.res = res;

   return res;
}


/*=== APPLY & QUANTIFY =================================================*/

/*
NAME    {* bdd\_appex *}
SECTION {* kernel *}
SHORT   {* Apply operation and existential quantification *}
PROTO   {* BDD bdd_appex(BDD left, BDD right, int opr, BDD var) *}
DESCR   {* Applies the binary operator {\tt opr} to the arguments
           {\tt left} and {\tt right} and then performs an existential
	   quantification of the variables from the variable set
	   {\tt var}. This is done in a bottom up manner such that both the
	   apply and quantification is done on the lower nodes before
	   stepping up to the higher nodes. This makes the {\tt bdd\_appex}
	   function much more efficient than an apply operation followed
	   by a quantification. *}
ALSO    {* bdd\_appall, bdd\_apply, bdd\_exist, bdd\_forall, bdd\_makeset *}
RETURN  {* The result of the operation. *}
*/
BDD bdd_appex(BDD l, BDD r, int opr, BDD var)
{
   static int size;
   static int *varset;
   BDD res;
   
   if (opr<0 || opr>bddop_invimp)
   {
      bdd_error(BDD_OP);
      return bddfalse;
   }
   
   if (var < 2)  /* Empty set */
      return bdd_apply(l,r,opr);
   
   if (bdd_set2intp(var, &varset, &size) < 0)
      return bddfalse;
   
   bddrefstacktop = bddrefstack;
   applyop = bddop_or;
   appexop = opr;
   appexid = (var << 5) | (appexop << 1);
   quantid = (appexid << 2) | 2;

   res = bdd_appquant_rec(l, r, varset, size);
   free(varset);
   bdderrorcond = 0;
   return res;
}


/*
NAME    {* bdd\_appall *}
SECTION {* kernel *}
SHORT   {* Apply operation and universal quantification *}
PROTO   {* BDD bdd_appall(BDD left, BDD right, int opr, BDD var) *}
DESCR   {* Applies the binary operator {\tt opr} to the arguments
           {\tt left} and {\tt right} and then performs an universal
	   quantification of the variables from the variable set
	   {\tt var}. This is done in a bottom up manner such that both the
	   apply and quantification is done on the lower nodes before
	   stepping up to the higher nodes. This makes the {\tt bdd\_appex}
	   function much more efficient than an apply operation followed
	   by a quantification. *}
ALSO    {* bdd\_appex, bdd\_apply, bdd\_exist, bdd\_forall, bdd\_makeset *}
RETURN  {* The result of the operation. *}
*/
BDD bdd_appall(BDD l, BDD r, int opr, BDD var)
{
   int size;
   int *varset;
   BDD res;

   if (opr<0 || opr>bddop_invimp)
   {
      bdd_error(BDD_OP);
      return bddfalse;
   }
   
   if (var < 2)  /* Empty set */
      return bdd_apply(l,r,opr);
   
   if (bdd_set2intp(var, &varset, &size) < 0)
      return bddfalse;
   
   bddrefstacktop = bddrefstack;
   applyop = bddop_and;
   appexop = opr;
   appexid = (var << 5) | (appexop << 1) | 1;
   quantid = (appexid << 2) | 3;

   res = bdd_appquant_rec(l, r, varset, size);
   free(varset);
   bdderrorcond = 0;
   return res;
}


static int bdd_appquant_rec(int l, int r, int *var, int num)
{
   BddCacheData *entry;
   int res;
   
#ifdef USECOND   
   if (bdderrorcond)
      return 0;
#endif

   switch (appexop)
   {
    case bddop_and:
       if (l == 0  ||  r == 0)
	  return 0;
       if (l == r)
	  return bdd_quant_rec(l, var, num);
       if (l == 1)
	  return bdd_quant_rec(r, var, num);
       if (r == 1)
	  return bdd_quant_rec(l, var, num);
       break;
    case bddop_or:
       if (l == 1  ||  r == 1)
	  return 1;
       if (l == r)
	  return bdd_quant_rec(l, var, num);
       if (l == 0)
	  return bdd_quant_rec(r, var, num);
       if (r == 0)
	  return bdd_quant_rec(l, var, num);
       break;
    case bddop_xor:
       if (l == r)
	  return 0;
       if (l == 0)
	  return bdd_quant_rec(r, var, num);
       if (r == 0)
	  return bdd_quant_rec(l, var, num);
       break;
    case bddop_nand:
       if (l == 0  ||  r == 0)
	  return 1;
       break;
    case bddop_nor:
       if (l == 1  ||  r == 1)
	  return 0;
       break;
   }

   if (l < 2  &&  r < 2)
      res = oprres[appexop][(l<<1) | r];
   else
   if (num == 0)
   {
      int oldop = applyop;
      applyop = appexop;
      res = bdd_apply_rec(l,r);
      applyop = oldop;
   }
   else
   {
      entry = BddCache_lookup(&appexcache, APPEXHASH(l,r,appexop));
      if (entry->a == l  &&  entry->b == r  &&  entry->c == appexid)
	 return entry->r.res;

      if (bddnodes[l].level == bddnodes[r].level)
      {
	 if (bddnodes[l].level == *var)
	 {
	    *(bddrefstacktop++) = bdd_appquant_rec(bddnodes[l].low,
						   bddnodes[r].low,
						   var+1,num-1);
	    *(bddrefstacktop++) = bdd_appquant_rec(bddnodes[l].high,
						   bddnodes[r].high,
						   var+1,num-1);
	    res = bdd_apply_rec(*(bddrefstacktop-2), *(bddrefstacktop-1));
	    bddrefstacktop -= 2;
	 }
	 else
	 if (bddnodes[l].level < *var)
	 {
	    *(bddrefstacktop++) = bdd_appquant_rec(bddnodes[l].low,
						   bddnodes[r].low, var, num);
	    *(bddrefstacktop++) = bdd_appquant_rec(bddnodes[l].high,
						   bddnodes[r].high, var, num);
	    res = bdd_makenode(bddnodes[l].level,
			       *(bddrefstacktop-2), *(bddrefstacktop-1));
	    bddrefstacktop -= 2;
	 }
	 else
	 {
	    res = bdd_appquant_rec(l, r, var+1, num-1);
	 }
      }
      else
      if (bddnodes[l].level < bddnodes[r].level)
      {
	 if (bddnodes[l].level == *var)
	 {
	    *(bddrefstacktop++) = bdd_appquant_rec(bddnodes[l].low, r,
						   var+1, num-1);
	    *(bddrefstacktop++) = bdd_appquant_rec(bddnodes[l].high, r,
						   var+1, num-1);
	    res = bdd_apply_rec(*(bddrefstacktop-2), *(bddrefstacktop-1));
	    bddrefstacktop -= 2;
	 }
	 else
	 if (bddnodes[l].level < *var)
	 {
	    *(bddrefstacktop++) = bdd_appquant_rec(bddnodes[l].low, r,
						   var, num);
	    *(bddrefstacktop++) = bdd_appquant_rec(bddnodes[l].high, r,
						   var, num);
	    res = bdd_makenode(bddnodes[l].level,
			       *(bddrefstacktop-2), *(bddrefstacktop-1));
	    bddrefstacktop -= 2;
	 }
	 else
	 {
	    res = bdd_appquant_rec(l, r, var+1, num-1);
	 }
      }
      else
      {
	 if (bddnodes[r].level == *var)
	 {
	    *(bddrefstacktop++) = bdd_appquant_rec(l, bddnodes[r].low,
						   var+1, num-1);
	    *(bddrefstacktop++) = bdd_appquant_rec(l, bddnodes[r].high,
						   var+1, num-1);
	    res = bdd_apply_rec(*(bddrefstacktop-2), *(bddrefstacktop-1));
	    bddrefstacktop -= 2;
	 }
	 else
	 if (bddnodes[r].level < *var)
	 {
	    *(bddrefstacktop++) = bdd_appquant_rec(l, bddnodes[r].low,
						   var, num);
	    *(bddrefstacktop++) = bdd_appquant_rec(l, bddnodes[r].high,
						   var, num);
	    res = bdd_makenode(bddnodes[r].level,
			       *(bddrefstacktop-2), *(bddrefstacktop-1));
	    bddrefstacktop -= 2;
	 }
	 else
	 {
	    res = bdd_appquant_rec(l, r, var+1, num-1);
	 }
      }

#ifdef USECOND      
      if (res < 0 || bdderrorcond)
      {
	 bdderrorcond = BDD_MEMORY;
	 return 0;
      }
#endif
      
      entry->a = l;
      entry->b = r;
      entry->c = appexid;
      entry->r.res = res;
   }

   return res;
}


/*************************************************************************
  Informational functions
*************************************************************************/

/*=== SUPPORT ==========================================================*/

/*
NAME    {* bdd\_support *}
SECTION {* kernel *}
SHORT   {* Returns the variable support of a BDD *}
PROTO   {* BDD bdd_support(BDD r) *}
DESCR   {* Finds all the variables that {\tt r} depends on. That is
           the support of {\tt r}. *}
ALSO    {* bdd\_makeset *}
RETURN  {* A BDD variable set. *}
*/
BDD bdd_support(BDD r)
{
   char *support;
   int n;
   int res=1;
   
   if (r < 2)
      return bddfalse;

   if ((support=(char *)malloc(bddvarnum)) == NULL)
   {
      bdd_error(BDD_MEMORY);
      return bddfalse;
   }
   memset(support, 0, bddvarnum);

   bdd_support_rec(r, support);
   bdd_unmark(r);
   
   for (n=bddvarnum-1 ; n>=0 ; n--)
      if (support[n])
      {
	 register BDD tmp;
	 bdd_addref(res);
	 tmp = bdd_makenode(n, 0, res);
	 bdd_delref(res);
	 res = tmp;
      }
   
   free(support);
   bdderrorcond = 0;
   return res;
}


static void bdd_support_rec(int r, char *support)
{
   BddNode *node;
   
   if (r < 2)
      return;

   node = &bddnodes[r];
   if (node->level & MARKON  ||  node->low == -1)
      return;

   support[node->level] = 1;
   node->level |= MARKON;
   
   bdd_support_rec(node->low, support);
   bdd_support_rec(node->high, support);
}


/*=== ONE SATISFYING VARIABLE ASSIGNMENT ===============================*/

/*
NAME    {* bdd\_satone *}
SECTION {* kernel *}
SHORT   {* Finds one satisfying variable assignment *}
PROTO   {* BDD bdd_satone(BDD r) *}
DESCR   {* Finds a BDD, with at most one variable at each level, that
           implies {\tt r} and which is not false unless {\tt r} is
	   false. *}
ALSO    {* bdd\_fullsatone, bdd\_satcount, bdd\_satcountln *}
RETURN  {* The result of the operation. *}
*/
BDD bdd_satone(BDD r)
{
   int res;
   
   if (r < 2)
      return r;

   bddrefstacktop = bddrefstack;
   
   res = bdd_satone_rec(r);
   bdderrorcond = 0;
   return res;
}


static int bdd_satone_rec(int r)
{
   if (r < 2)
      return r;

   if (bddnodes[r].low == 0)
   {
      int res = bdd_satone_rec(bddnodes[r].high);
      return *(bddrefstacktop++) = bdd_makenode(bddnodes[r].level, 0, res);
   }
   else
   {
      int res = bdd_satone_rec(bddnodes[r].low);
      return *(bddrefstacktop++) = bdd_makenode(bddnodes[r].level, res, 0);
   }
}


/*=== EXACTLY ONE SATISFYING VARIABLE ASSIGNMENT =======================*/

/*
NAME    {* bdd\_fullsatone *}
SECTION {* kernel *}
SHORT   {* Finds one satisfying variable assignment *}
PROTO   {* BDD bdd_fullsatone(BDD r) *}
DESCR   {* Finds a BDD, with exactly one variable at all levels, that
           implies {\tt r} and which is not false unless {\tt r} is
	   false. *}
ALSO    {* bdd\_satone, bdd\_satcount, bdd\_satcountln *}
RETURN  {* The result of the operation. *}
*/
BDD bdd_fullsatone(BDD r)
{
   int res, v;

   if (r == 0)
      return 0;

   bddrefstacktop = bddrefstack;

   res = bdd_fullsatone_rec(r);
   
   for (v=bddnodes[r].level-1 ; v>=0 ; v--)
   {
      res = *(bddrefstacktop++) = bdd_makenode(v, res, 0);
   }
   
   bdderrorcond = 0;
   return res;
}


static int bdd_fullsatone_rec(int r)
{
   if (r < 2)
      return r;
   
   if (bddnodes[r].low != 0)
   {
      int res = bdd_fullsatone_rec(bddnodes[r].low);
      int v;
      
      for (v=bddnodes[bddnodes[r].low].level-1 ; v>bddnodes[r].level ; v--)
      {
	 res = *(bddrefstacktop++) = bdd_makenode(v, res, 0);
      }

      return *(bddrefstacktop++) = bdd_makenode(bddnodes[r].level, res, 0);
   }
   else
   {
      int res = bdd_fullsatone_rec(bddnodes[r].high);
      int v;
      
      for (v=bddnodes[bddnodes[r].high].level-1 ; v>bddnodes[r].level ; v--)
      {
	 res = *(bddrefstacktop++) = bdd_makenode(v, res, 0);
      }

      return *(bddrefstacktop++) = bdd_makenode(bddnodes[r].level, 0, res);
   }
}


/*=== COUNT NUMBER OF SATISFYING ASSIGNMENT ============================*/

/*
NAME    {* bdd\_satcount *}
SECTION {* kernel *}
SHORT   {* Calculates the number of satisfying variable assignments *}
PROTO   {* double bdd_satcount(BDD r) *}
DESCR   {* Calculates how many possible variable assignments there exists
           such that {\tt r} is satisfied, taking all defined variables
	   into account. *}
ALSO    {* bdd\_satone, bdd\_fullsatone, bdd\_satcountln *}
RETURN  {* The number of possible assignments. *}
*/
double bdd_satcount(BDD r)
{
   double size=1;
   int n;

   miscid = CACHEID_SATCOU;
   for (n=0 ; n<bddnodes[r].level ; n++)
      size *= 2;
   
   return size * bdd_satcount_rec(r);
}


static double bdd_satcount_rec(int root)
{
   BddCacheData *entry;
   BddNode *node;
   double size, s;
   int m;
   
   if (root < 2)
      return root;

   entry = BddCache_lookup(&misccache, SATCOUHASH(root));
   if (entry->a == root  &&  entry->c == miscid)
      return entry->r.dres;

   node = &bddnodes[root];
   size = 0;
   s = 1;
   
   for (m=node->level+1 ; m<bddnodes[node->low].level ; m++)
      s *= 2;
   size += s * bdd_satcount_rec(node->low);

   s = 1;
   for (m=node->level+1 ; m<bddnodes[node->high].level ; m++)
      s *= 2;
   size += s * bdd_satcount_rec(node->high);

   entry->a = root;
   entry->c = miscid;
   entry->r.dres = size;
   
   return size;
}


/*
NAME    {* bdd\_satcountln *}
SECTION {* kernel *}
SHORT   {* Calculates the log. number of satisfying variable assignments *}
PROTO   {* double bdd_satcountln(BDD r) *}
DESCR   {* Calculates how many possible variable assignments there exists
           such that {\tt r} is satisfied, taking all defined variables
	   into account, and returns the logarithm of this. The result
	   is calculated in such a manner that it is practically impossible
	   to get an overflow, which is very possible for {\tt bdd\_satcount}
	   if the number of defined variables is too large. *}
ALSO    {* bdd\_satone, bdd\_fullsatone, bdd\_satcount *}
RETURN  {* The logarithm of the number of possible assignments. *}
*/
double bdd_satcountln(BDD r)
{
   double size;

   miscid = CACHEID_SATCOULN;
   size = bdd_satcountln_rec(r);

   if (size >= 0.0)
      size += bddnodes[r].level;

   return size;
}


static double bdd_satcountln_rec(int root)
{
   BddCacheData *entry;
   BddNode *node;
   double size, s1,s2;
   
   if (root == 0)
      return -1.0;
   if (root == 1)
      return 0.0;

   entry = BddCache_lookup(&misccache, SATCOUHASH(root));
   if (entry->a == root  &&  entry->c == miscid)
      return entry->r.dres;

   node = &bddnodes[root];

   s1 = bdd_satcountln_rec(node->low);
   if (s1 >= 0.0)
      s1 += (bddnodes[node->low].level - node->level - 1);

   s2 = bdd_satcountln_rec(node->high);
   if (s2 >= 0.0)
      s2 += (bddnodes[node->high].level - node->level - 1);

   if (s1 < 0.0)
      size = s2;
   else
   if (s2 < 0.0)
      size = s1;
   else
   if (s1 < s2)
      size = s2 + log1p(pow(2.0,s1-s2)) / M_LN2;
   else
      size = s1 + log1p(pow(2.0,s2-s1)) / M_LN2;
   
   entry->a = root;
   entry->c = miscid;
   entry->r.dres = size;
   
   return size;
}


/*=== COUNT NUMBER OF ALLOCATED NODES ==================================*/

/*
NAME    {* bdd\_nodecount *}
SECTION {* operator *}
SHORT   {* Finds the number of nodes used for a BDD *}
PROTO   {* int bdd_nodecount(BDD r) *}
DESCR   {* Traverses the BDD and counts all distinct nodes that are used
           for the BDD. *}
RETURN  {* The number of nodes. *}
*/
int bdd_nodecount(BDD r)
{
   int num=0;
   
   if (r < 0)
      bdd_error(BDD_ILLBDD);
   
   bdd_markcount(r, &num);
   bdd_unmark(r);

   return num;
}


/*=== NODE PROFILE =====================================================*/

/*
NAME    {* bdd\_varprofile *}
SECTION {* kernel *}
SHORT   {* returns a variable profile *}
PROTO   {* int *bdd_varprofile(BDD r) *}
DESCR   {* Counts the number of times each variable occurs in the
           bdd {\tt r}. The result is stored and returned in an integer array
	   where the i'th position stores the number of times the i'th
	   variable occured in the BDD. It is the users responsibility to
	   free the array again using a call to {\tt free}. *}
RETURN  {* A pointer to an integer array with the profile *}
*/
int *bdd_varprofile(BDD r)
{
   if (r < 0)
   {
      bdd_error(BDD_ILLBDD);
      return NULL;
   }
   
   if ((varprofile=(int*)malloc(sizeof(int)*bddvarnum)) == NULL)
   {
      bdd_error(BDD_MEMORY);
      return NULL;
   }

   memset(varprofile, 0, sizeof(int)*bddvarnum);
   bdd_varprofile_rec(r);
   bdd_unmark(r);
   return varprofile;
}


static void bdd_varprofile_rec(int r)
{
   BddNode *node;
   
   if (r < 2)
      return;

   node = &bddnodes[r];
   if (node->level & MARKON)
      return;

   varprofile[bddlevel2var[node->level]]++;
   node->level |= MARKON;
   
   bdd_varprofile_rec(node->low);
   bdd_varprofile_rec(node->high);
}


/*************************************************************************
  Other internal functions
*************************************************************************/
   
static int bdd_set2intp(BDD r, int **varset, int *varnum)
{
   int n, num;

   if (r < 2)
      return bdd_error(BDD_VARSET);
   
   for (n=r, num=0 ; n > 1 ; n=bddnodes[n].high)
      num++;

   if (((*varset) = (int *)malloc(sizeof(int)*num)) == NULL)
      return bdd_error(BDD_MEMORY);
   
   for (n=r, num=0 ; n > 1 ; n=bddnodes[n].high)
      (*varset)[num++] = bddnodes[n].level;

   *varnum = num;

   return 0;
}


static int bdd_set2sintp(BDD r, int **varset, int *varnum)
{
   int n, num;

   if (r < 2)
      return bdd_error(BDD_VARSET);

   n=r;
   num=0;
   while (n > 1)
   {
      if (bddnodes[n].high >= 1)
	 n = bddnodes[n].high;
      else
	 n = bddnodes[n].low;
      num++;
   }

   if (((*varset) = (int *)malloc(sizeof(int)*num)) == NULL)
      return bdd_error(BDD_MEMORY);
   
   n=r;
   num=0;
   while (n > 1)
   {
      if (bddnodes[n].high >= 1)
      {
	 (*varset)[num++] = bddnodes[n].level+1;
	 n = bddnodes[n].high;
      }
      else
      {
	 (*varset)[num++] = -(bddnodes[n].level+1);
	 n = bddnodes[n].low;
      }
   }

   *varnum = num;

   return 0;
}



/* EOF */
