(* ===================================================================== *)
(* FILE          : Portable.sml                                          *)
(* DESCRIPTION   : Structure for SML System dependent functions.         *)
(*                 (Please look at the structures Portable_* as well)    *)
(* AUTHOR        : Ken Larsen, University of Cambridge (or DTU)          *)
(*                 based on code by                                      *)
(*                 Elsa L. Gunter, AT&T Bell Laboratories                *)
(* DATE          : 19 September, 1997                                    *)
(* ===================================================================== *)

(* Copyright 1993 by AT&T Bell Laboratories *)
(* Copyright 1997 by University of Cambridge *)

(* Share and Enjoy *)

structure Portable = 
struct

(* Set system variables relevant to the interactive mode of hol90 *)
(*
val _ = quotation := true

val _ = (printLength := 1000;
	 printDepth := 350)
*)


val string_to_int  = Int.fromString;
val real_to_string = Real.toString
val implode        = Portable_String.concat;
val explode        = map Portable_String.str o Portable_String.explode;

val say       = TextIO.print
val linewidth = ref 72 (* ### POTENTIAL ### *) 

val getEnv = Process.getEnv
val cd     = FileSys.chDir
val pwd    = FileSys.getDir


val listDir = Mosml.listDir (* ### POTENTIAL ### *)

fun system s = if Process.system s = Process.success then 0
	       else 1
fun ls()     = system "ls"
val getArgs  = CommandLine.arguments
val argv     = getArgs

fun file_exists_for_reading s = FileSys.access(s,[FileSys.A_READ])

fun exit() = Process.exit Process.success

(* (* ### POTENTIAL ### *) *)
val version    = "1.42"
val sml_banner = "Moscow ML"


(*--------------------------------------------------------------------
 * Misc. - interactions which manipulate the ML environment
 *-------------------------------------------------------------------*)
(*  (* ### POTENTIAL ### *)
val execute = Unix.streamsOf o Unix.execute

fun reap proc = (Unix.reap proc; ()) handle OS.SysErr _ => ();

val unfiltered_use_stream = Compiler.Interact.useStream;
val unfiltered_use_file = Compiler.Interact.useFile;

fun use_and_exit print_err filename =
   (Compiler.Interact.useFile filename; exit ())
   handle e => (print_err e; OS.Process.exit OS.Process.failure);
*)

val interp = ref false

(*  (* ### POTENTIAL ### *)
fun load files = app Compiler.Interact.useFile files; 
*)
(*--------------------------------------------------------------------
 * Exporting the executable.
 *-------------------------------------------------------------------*)
(*
fun exportML str = (SMLofNJ.exportML str; ());
*)

exception Io of string;

type instream      = TextIO.instream 
type outstream     = TextIO.outstream 
val std_out        = TextIO.stdOut
val stdin          = TextIO.stdIn
fun open_in file   = TextIO.openIn file 
                     handle General.Io{cause=SysErr(s,_),...} => raise (Io s)
                                   (* handle OS.SysErr (s,_) => raise Io s; *)
fun open_out file  = TextIO.openOut file 
                     handle General.Io{cause=SysErr(s,_),...} => raise (Io s)
                                   (* handle OS.SysErr (s,_) => raise Io s; *)
val output         = TextIO.output
fun outputc strm s = output(strm,s)
val close_in       = TextIO.closeIn
val close_out      = TextIO.closeOut
val flush_out      = TextIO.flushOut
val input_line     = TextIO.inputLine

(* (* ### POTENTIAL ### *)
val open_string    = TextIO.openString
*)

val end_of_stream  = TextIO.endOfStream


fun pointer_eq(x,y) = x=y
(* ((Unsafe.cast x:int) = (Unsafe.cast y:int)) *)

  
(*---------------------------------------------------------------------------*
 * The contents of the following declarations are                            *
 *                                                                           *
 *      Copyright 1996 by University of Cambridge                            *
 *---------------------------------------------------------------------------*)
(*

local

val unix_pid = 0 (* Should be the Unix process number *)

val pipe = SysParams.HOL_base_directory ^ "bin/pipe"

val quote_filter = SysParams.HOL_base_directory 
                   ^ "bin/quote-filter." 
                   ^ SysParams.arch

fun connect_streams (is,os) =
   if (end_of_stream is)
   then flush_out os
   else (output (os,TextIO.inputN (is,1)); connect_streams (is,os))

fun restore_filename filename s =
   if (String.substring (s,0,10) = "<instream>"
       handle General.Subscript => false)
   then filename ^ String.substring (s,10,String.size s - 10)
   else s

in

(* Resorted to using a tmp file! Sigh ... *)
fun use_stream str =
   let val tmp_file = "/tmp/hol90_use_file." ^ Int.toString unix_pid ^ "." ^
                      Time.toString (Time.now ())
       val tmp = open_out tmp_file
       val _ = (connect_streams (str,tmp); close_out tmp)
       val proc = Unix.execute (pipe,[tmp_file,quote_filter])
       val (is,os) = Unix.streamsOf proc
       fun tidy_up () = (close_in is; reap proc)
   in  close_out os;
       (Compiler.Interact.useStream is handle e => (tidy_up (); raise e));
       tidy_up ();
       OS.Process.system ("/bin/rm " ^ tmp_file);
       ()
   end

fun use filename =
   let val out = !Compiler.Control.Print.out
       val is = (#say out ("[opening " ^ filename ^ "]\n");
                 open_in filename handle e as OS.SysErr (s,_) =>
                 (#say out ("[use failed: " ^ s ^ "]\n"); raise e))
       val _ = close_in is
       val proc = Unix.execute (pipe,[filename,quote_filter])
       val (is,os) = Unix.streamsOf proc
       fun tidy_up () =
          (Compiler.Control.Print.out := out; close_in is; reap proc)
   in  close_out os;
       Compiler.Control.Print.out :=
          {flush = #flush out,say = #say out o restore_filename filename};
       (Compiler.Interact.useStream is handle e => (tidy_up (); raise e));
       tidy_up ()
   end

end;

(*---------------------------------------------------------------------------*
 *  End of University of Cambridge copyright                                 *
 *---------------------------------------------------------------------------*)

*)

end (* structure Portable *)
