(* ========================================================================= *)
(* Linear decision procedure for the reals, and some theorems.               *)
(* ========================================================================= *)

(* ------------------------------------------------------------------------- *)
(* Additional commutativity properties of the inclusion map.                 *)
(* ------------------------------------------------------------------------- *)

let REAL_OF_NUM_LT = prove
 (`!m n. &m < &n = m < n`,
  REWRITE_TAC[real_lt; GSYM NOT_LE; REAL_OF_NUM_LE]);;

let REAL_OF_NUM_GE = prove
 (`!m n. &m >= &n = m >= n`,
  REWRITE_TAC[GE; real_ge; REAL_OF_NUM_LE]);;

let REAL_OF_NUM_GT = prove
 (`!m n. &m > &n = m > n`,
  REWRITE_TAC[GT; real_gt; REAL_OF_NUM_LT]);;

let REAL_OF_NUM_SUC = prove
 (`!n. &n + &1 = &(SUC n)`,
  REWRITE_TAC[ADD1; REAL_OF_NUM_ADD]);;

let REAL_POS = prove
 (`!n. &0 <= &n`,
  REWRITE_TAC[REAL_OF_NUM_LE; LE_0]);;

(* ------------------------------------------------------------------------- *)
(* A few theorems we need to prove explicitly for later.                     *)
(* ------------------------------------------------------------------------- *)

let REAL_ADD_AC = prove
 (`(m + n = n + m) /\
   ((m + n) + p = m + (n + p)) /\
   (m + (n + p) = n + (m + p))`,
  REWRITE_TAC[REAL_ADD_ASSOC; EQT_INTRO(SPEC_ALL REAL_ADD_SYM)] THEN
  AP_THM_TAC THEN AP_TERM_TAC THEN MATCH_ACCEPT_TAC REAL_ADD_SYM);;

let REAL_MUL_AC = prove
 (`(m * n = n * m) /\
   ((m * n) * p = m * (n * p)) /\
   (m * (n * p) = n * (m * p))`,
  REWRITE_TAC[REAL_MUL_ASSOC; EQT_INTRO(SPEC_ALL REAL_MUL_SYM)] THEN
  AP_THM_TAC THEN AP_TERM_TAC THEN MATCH_ACCEPT_TAC REAL_MUL_SYM);;

let REAL_ADD_RID = prove
 (`!x. x + &0 = x`,
  MESON_TAC[REAL_ADD_SYM; REAL_ADD_LID]);;

let REAL_MUL_RID = prove
 (`!x. x * &1 = x`,
  MESON_TAC[REAL_MUL_SYM; REAL_MUL_LID]);;

let REAL_ADD_RINV = prove
 (`!x. x + --x = &0`,
  MESON_TAC[REAL_ADD_SYM; REAL_ADD_LINV]);;

let REAL_ADD_RDISTRIB = prove
 (`!x y z. (x + y) * z = x * z + y * z`,
  MESON_TAC[REAL_MUL_SYM; REAL_ADD_LDISTRIB]);;

let REAL_EQ_ADD_LCANCEL = prove
 (`!x y z. (x + y = x + z) = (y = z)`,
  REPEAT GEN_TAC THEN EQ_TAC THEN DISCH_TAC THEN ASM_REWRITE_TAC[] THEN
  POP_ASSUM(MP_TAC o AP_TERM `$+ (--x)`) THEN
  REWRITE_TAC[REAL_ADD_ASSOC; REAL_ADD_LINV; REAL_ADD_LID]);;

let REAL_EQ_ADD_RCANCEL = prove
 (`!x y z. (x + z = y + z) = (x = y)`,
  MESON_TAC[REAL_ADD_SYM; REAL_EQ_ADD_LCANCEL]);;

let REAL_MUL_RZERO = prove
 (`!x. x * &0 = &0`,
  GEN_TAC THEN
  MATCH_MP_TAC(GEN_ALL(fst(EQ_IMP_RULE(SPEC_ALL REAL_EQ_ADD_RCANCEL)))) THEN
  EXISTS_TAC `x * &0` THEN REWRITE_TAC[GSYM REAL_ADD_LDISTRIB; REAL_ADD_LID]);;

let REAL_MUL_LZERO = prove
 (`!x. &0 * x = &0`,
  MESON_TAC[REAL_MUL_SYM; REAL_MUL_RZERO]);;

let REAL_NEG_NEG = prove
 (`!x. --(--x) = x`,
  MESON_TAC
   [REAL_EQ_ADD_RCANCEL; REAL_ADD_LINV; REAL_ADD_SYM; REAL_ADD_LINV]);;

let REAL_MUL_RNEG = prove
 (`!x y. x * (--y) = -- (x * y)`,
  REPEAT GEN_TAC THEN
  MATCH_MP_TAC(GEN_ALL(fst(EQ_IMP_RULE(SPEC_ALL REAL_EQ_ADD_RCANCEL)))) THEN
  EXISTS_TAC `x * y` THEN
  REWRITE_TAC[GSYM REAL_ADD_LDISTRIB; REAL_ADD_LINV; REAL_MUL_RZERO]);;

let REAL_MUL_LNEG = prove
 (`!x y. (--x) * y = -- (x * y)`,
  MESON_TAC[REAL_MUL_SYM; REAL_MUL_RNEG]);;

let REAL_NEG_ADD = prove
 (`!x y. --(x + y) = --x + --y`,
  REPEAT GEN_TAC THEN
  MATCH_MP_TAC(GEN_ALL(fst(EQ_IMP_RULE(SPEC_ALL REAL_EQ_ADD_RCANCEL)))) THEN
  EXISTS_TAC `x + y` THEN REWRITE_TAC[REAL_ADD_LINV] THEN
  ONCE_REWRITE_TAC[AC REAL_ADD_AC `(a + b) + (c + d) = (a + c) + (b + d)`] THEN
  REWRITE_TAC[REAL_ADD_LINV; REAL_ADD_LID]);;

let REAL_NEG_0 = prove
 (`--(&0) = &0`,
  MESON_TAC[REAL_ADD_LINV; REAL_ADD_RID]);;

let REAL_LE_LNEG = prove
 (`!x y. --x <= y = &0 <= x + y`,
  REPEAT GEN_TAC THEN EQ_TAC THEN
  DISCH_THEN(MP_TAC o MATCH_MP REAL_LE_LADD_IMP) THENL
   [DISCH_THEN(MP_TAC o SPEC `x:real`) THEN
    REWRITE_TAC[ONCE_REWRITE_RULE[REAL_ADD_SYM] REAL_ADD_LINV];
    DISCH_THEN(MP_TAC o SPEC `--x`) THEN
    REWRITE_TAC[REAL_ADD_LINV; REAL_ADD_ASSOC; REAL_ADD_LID;
        ONCE_REWRITE_RULE[REAL_ADD_SYM] REAL_ADD_LID]]);;

let REAL_LE_NEG2 = prove
 (`!x y. --x <= --y = y <= x`,
  REPEAT GEN_TAC THEN
  GEN_REWRITE_TAC (RAND_CONV o LAND_CONV) [GSYM REAL_NEG_NEG] THEN
  REWRITE_TAC[REAL_LE_LNEG] THEN
  AP_TERM_TAC THEN MATCH_ACCEPT_TAC REAL_ADD_SYM);;

let REAL_LE_RNEG = prove
 (`!x y. x <= --y = x + y <= &0`,
  REPEAT GEN_TAC THEN
  GEN_REWRITE_TAC (LAND_CONV o LAND_CONV) [GSYM REAL_NEG_NEG] THEN
  REWRITE_TAC[REAL_LE_LNEG; GSYM REAL_NEG_ADD] THEN
  GEN_REWRITE_TAC RAND_CONV [GSYM REAL_LE_NEG2] THEN
  AP_THM_TAC THEN AP_TERM_TAC THEN
  REWRITE_TAC[GSYM REAL_ADD_LINV] THEN
  REWRITE_TAC[REAL_NEG_ADD; REAL_NEG_NEG] THEN
  MATCH_ACCEPT_TAC REAL_ADD_SYM);;

let REAL_LT_IMP_LE = prove
 (`!x y. x < y ==> x <= y`,
  MESON_TAC[real_lt; REAL_LE_TOTAL]);;

let REAL_LTE_TRANS = prove
 (`!x y z. x < y /\ y <= z ==> x < z`,
  MESON_TAC[real_lt; REAL_LE_TRANS]);;

let REAL_LET_TRANS = prove
 (`!x y z. x <= y /\ y < z ==> x < z`,
  MESON_TAC[real_lt; REAL_LE_TRANS]);;

let REAL_LT_TRANS = prove
 (`!x y z. x < y /\ y < z ==> x < z`,
  MESON_TAC[real_lt; REAL_LE_TRANS; REAL_LE_TOTAL]);;

do_list add_mizar_transitivity_theorem
  [REAL_LE_TRANS; REAL_LT_TRANS; REAL_LET_TRANS; REAL_LTE_TRANS];;

let REAL_LT_LADD_IMP = prove
 (`!x y z. y < z ==> x + y < x + z`,
  REPEAT GEN_TAC THEN CONV_TAC CONTRAPOS_CONV THEN
  REWRITE_TAC[real_lt] THEN
  DISCH_THEN(MP_TAC o MATCH_MP REAL_LE_LADD_IMP) THEN
  DISCH_THEN(MP_TAC o SPEC `--x`) THEN
  REWRITE_TAC[REAL_ADD_ASSOC; REAL_ADD_LINV; REAL_ADD_LID]);;

let REAL_LT_LE = prove
 (`!x y. x < y = x <= y /\ ~(x = y)`,
  MESON_TAC[real_lt; REAL_LE_TOTAL; REAL_LE_ANTISYM]);;

let REAL_ENTIRE = prove
 (`!x y. (x * y = &0) = (x = &0) \/ (y = &0)`,
  REPEAT GEN_TAC THEN EQ_TAC THEN STRIP_TAC THEN
  ASM_REWRITE_TAC[REAL_MUL_LZERO; REAL_MUL_RZERO] THEN
  ASM_CASES_TAC `x = &0` THEN ASM_REWRITE_TAC[] THEN
  FIRST_ASSUM(MP_TAC o AP_TERM `$* (inv x)`) THEN
  REWRITE_TAC[REAL_MUL_ASSOC] THEN
  FIRST_ASSUM(fun th -> REWRITE_TAC[MATCH_MP REAL_MUL_LINV th]) THEN
  REWRITE_TAC[REAL_MUL_LID; REAL_MUL_RZERO]);;

let REAL_LT_MUL = prove
 (`!x y. &0 < x /\ &0 < y ==> &0 < x * y`,
  REPEAT GEN_TAC THEN REWRITE_TAC[REAL_LT_LE] THEN
  CONV_TAC(ONCE_DEPTH_CONV SYM_CONV) THEN
  STRIP_TAC THEN ASM_REWRITE_TAC[REAL_ENTIRE] THEN
  MATCH_MP_TAC REAL_LE_MUL THEN ASM_REWRITE_TAC[]);;

(* ------------------------------------------------------------------------- *)
(* Arithmetic on integer "constants" in type `:real`.                        *)
(*                                                                           *)
(* These could be done as 1-liners; we try to make them efficient though!    *)
(*                                                                           *)
(* They can leave --(&0); this doesn't matter for the intended use...        *)
(* ------------------------------------------------------------------------- *)

let is_numconst =
  let ptm = `&` in
  fun tm -> try let l,r = dest_comb tm in
                l = ptm & is_numeral r
            with Failure _ -> false;;

let mk_small_numconst,dest_small_numconst =
  let ptm = `&` in
  (fun n -> mk_comb(ptm,mk_small_numeral n)),
  (fun tm -> let l,r = dest_comb tm in
             if l = ptm then dest_small_numeral r
             else failwith "dest_numconst");;

let mk_numconst,dest_numconst =
  let ptm = `&` in
  (fun n -> mk_comb(ptm,mk_numeral n)),
  (fun tm -> let l,r = dest_comb tm in
             if l = ptm then dest_numeral r
             else failwith "dest_numconst");;

let is_intconst =
  let ptm = `--` in
  fun tm ->
    is_numconst tm or
    try let l,r = dest_comb tm in
        l = ptm & is_numconst r
    with Failure _ -> false;;

let mk_small_intconst,dest_small_intconst =
  let ptm = `--` in
  (fun n -> if n < 0 then mk_comb(ptm,mk_small_numconst(-n))
            else mk_small_numconst n),
  (fun tm -> if try rator tm = ptm with Failure _ -> false then
               -(dest_small_numconst(rand tm))
             else dest_small_numconst tm);;

let mk_intconst,dest_intconst =
  let ptm = `--` in
  (fun n -> if n </ Int 0 then mk_comb(ptm,mk_numconst(minus_num n))
            else mk_numconst n),
  (fun tm -> if try rator tm = ptm with Failure _ -> false then
               minus_num (dest_numconst(rand tm))
             else dest_numconst tm);;

(* ------------------------------------------------------------------------- *)
(* First all the comparison operators.                                       *)
(* ------------------------------------------------------------------------- *)

let REAL_INT_LE_CONV,REAL_INT_LT_CONV,
    REAL_INT_GE_CONV,REAL_INT_GT_CONV,REAL_INT_EQ_CONV =
  let tth =
    TAUT `(F /\ F = F) /\ (F /\ T = F) /\ (T /\ F = F) /\ (T /\ T = T)` in
  let nth = TAUT `(~T = F) /\ (~F = T)` in
  let NUM2_EQ_CONV =
    COMB2_CONV (RAND_CONV NUM_EQ_CONV) NUM_EQ_CONV THENC
    GEN_REWRITE_CONV I [tth] in
  let NUM2_NE_CONV =
    RAND_CONV NUM2_EQ_CONV THENC
    GEN_REWRITE_CONV I [nth] in
  let [pth_le1; pth_le2a; pth_le2b; pth_le3] = (CONJUNCTS o prove)
   (`(--(&m) <= &n = T) /\
     (&m <= &n = m <= n) /\
     (--(&m) <= --(&n) = n <= m) /\
     (&m <= --(&n) = (m = 0) /\ (n = 0))`,
    REWRITE_TAC[REAL_LE_NEG2] THEN
    REWRITE_TAC[REAL_LE_LNEG; REAL_LE_RNEG] THEN
    REWRITE_TAC[REAL_OF_NUM_ADD; REAL_OF_NUM_LE; LE_0] THEN
    REWRITE_TAC[LE; ADD_EQ_0]) in
  let REAL_INT_LE_CONV = FIRST_CONV
   [GEN_REWRITE_CONV I [pth_le1];
    GEN_REWRITE_CONV I [pth_le2a; pth_le2b] THENC NUM_LE_CONV;
    GEN_REWRITE_CONV I [pth_le3] THENC NUM2_EQ_CONV] in
  let [pth_lt1; pth_lt2a; pth_lt2b; pth_lt3] = (CONJUNCTS o prove)
   (`(&m < --(&n) = F) /\
     (&m < &n = m < n) /\
     (--(&m) < --(&n) = n < m) /\
     (--(&m) < &n = ~((m = 0) /\ (n = 0)))`,
    REWRITE_TAC[pth_le1; pth_le2a; pth_le2b; pth_le3;
                GSYM NOT_LE; real_lt] THEN
    CONV_TAC TAUT) in
  let REAL_INT_LT_CONV = FIRST_CONV
   [GEN_REWRITE_CONV I [pth_lt1];
    GEN_REWRITE_CONV I [pth_lt2a; pth_lt2b] THENC NUM_LT_CONV;
    GEN_REWRITE_CONV I [pth_lt3] THENC NUM2_NE_CONV] in
  let [pth_ge1; pth_ge2a; pth_ge2b; pth_ge3] = (CONJUNCTS o prove)
   (`(&m >= --(&n) = T) /\
     (&m >= &n = n <= m) /\
     (--(&m) >= --(&n) = m <= n) /\
     (--(&m) >= &n = (m = 0) /\ (n = 0))`,
    REWRITE_TAC[pth_le1; pth_le2a; pth_le2b; pth_le3; real_ge] THEN
    CONV_TAC TAUT) in
  let REAL_INT_GE_CONV = FIRST_CONV
   [GEN_REWRITE_CONV I [pth_ge1];
    GEN_REWRITE_CONV I [pth_ge2a; pth_ge2b] THENC NUM_LE_CONV;
    GEN_REWRITE_CONV I [pth_ge3] THENC NUM2_EQ_CONV] in
  let [pth_gt1; pth_gt2a; pth_gt2b; pth_gt3] = (CONJUNCTS o prove)
   (`(--(&m) > &n = F) /\
     (&m > &n = n < m) /\
     (--(&m) > --(&n) = m < n) /\
     (&m > --(&n) = ~((m = 0) /\ (n = 0)))`,
    REWRITE_TAC[pth_lt1; pth_lt2a; pth_lt2b; pth_lt3; real_gt] THEN
    CONV_TAC TAUT) in
  let REAL_INT_GT_CONV = FIRST_CONV
   [GEN_REWRITE_CONV I [pth_gt1];
    GEN_REWRITE_CONV I [pth_gt2a; pth_gt2b] THENC NUM_LT_CONV;
    GEN_REWRITE_CONV I [pth_gt3] THENC NUM2_NE_CONV] in
  let [pth_eq1a; pth_eq1b; pth_eq2a; pth_eq2b] = (CONJUNCTS o prove)
   (`((&m = &n) = (m = n)) /\
     ((--(&m) = --(&n)) = (m = n)) /\
     ((--(&m) = &n) = (m = 0) /\ (n = 0)) /\
     ((&m = --(&n)) = (m = 0) /\ (n = 0))`,
    REWRITE_TAC[GSYM REAL_LE_ANTISYM; GSYM LE_ANTISYM] THEN
    REWRITE_TAC[pth_le1; pth_le2a; pth_le2b; pth_le3; LE; LE_0] THEN
    CONV_TAC TAUT) in
  let REAL_INT_EQ_CONV = FIRST_CONV
   [GEN_REWRITE_CONV I [pth_eq1a; pth_eq1b] THENC NUM_EQ_CONV;
    GEN_REWRITE_CONV I [pth_eq2a; pth_eq2b] THENC NUM2_EQ_CONV] in
  REAL_INT_LE_CONV,REAL_INT_LT_CONV,
  REAL_INT_GE_CONV,REAL_INT_GT_CONV,REAL_INT_EQ_CONV;;

(* ------------------------------------------------------------------------- *)
(* Negation & multiplication.                                                *)
(* ------------------------------------------------------------------------- *)

let REAL_INT_NEG_CONV =
  let pth = prove
   (`(--(&0) = &0) /\
     (--(--(&x)) = &x)`,
    REWRITE_TAC[REAL_NEG_NEG; REAL_NEG_0]) in
  GEN_REWRITE_CONV I [pth];;

let REAL_INT_MUL_CONV =
  let pth0 = prove
   (`(&0 * &x = &0) /\
     (&0 * --(&x) = &0) /\
     (&x * &0 = &0) /\
     (--(&x) * &0 = &0)`,
    REWRITE_TAC[REAL_MUL_LZERO; REAL_MUL_RZERO])
  and pth1,pth2 = (CONJ_PAIR o prove)
   (`((&m * &n = &(m * n)) /\
      (--(&m) * --(&n) = &(m * n))) /\
     ((--(&m) * &n = --(&(m * n))) /\
      (&m * --(&n) = --(&(m * n))))`,
    REWRITE_TAC[REAL_MUL_LNEG; REAL_MUL_RNEG; REAL_NEG_NEG] THEN
    REWRITE_TAC[REAL_OF_NUM_MUL]) in
  FIRST_CONV
   [GEN_REWRITE_CONV I [pth0];
    GEN_REWRITE_CONV I [pth1] THENC RAND_CONV NUM_MULT_CONV;
    GEN_REWRITE_CONV I [pth2] THENC RAND_CONV(RAND_CONV NUM_MULT_CONV)];;

(* ------------------------------------------------------------------------- *)
(* Addition and subtraction.                                                 *)
(* ------------------------------------------------------------------------- *)

let REAL_INT_ADD_CONV =
  let neg_tm = `--` in
  let amp_tm = `&` in
  let add_tm = `$+` in
  let dest = dest_binop `$+` in
  let m_tm = `m:num` and n_tm = `n:num` in
  let pth0 = prove
   (`(--(&m) + &m = &0) /\
     (&m + --(&m) = &0)`,
    REWRITE_TAC[REAL_ADD_LINV; REAL_ADD_RINV]) in
  let [pth1; pth2; pth3; pth4; pth5; pth6] = (CONJUNCTS o prove)
   (`(--(&m) + --(&n) = --(&(m + n))) /\
     (--(&m) + &(m + n) = &n) /\
     (--(&(m + n)) + &m = --(&n)) /\
     (&(m + n) + --(&m) = &n) /\
     (&m + --(&(m + n)) = --(&n)) /\
     (&m + &n = &(m + n))`,
    REWRITE_TAC[GSYM REAL_OF_NUM_ADD; REAL_NEG_ADD] THEN
    REWRITE_TAC[REAL_ADD_ASSOC; REAL_ADD_LINV; REAL_ADD_LID] THEN
    REWRITE_TAC[REAL_ADD_RINV; REAL_ADD_LID] THEN
    ONCE_REWRITE_TAC[REAL_ADD_SYM] THEN
    REWRITE_TAC[REAL_ADD_ASSOC; REAL_ADD_LINV; REAL_ADD_LID] THEN
    REWRITE_TAC[REAL_ADD_RINV; REAL_ADD_LID]) in
  GEN_REWRITE_CONV I [pth0] ORELSEC
  (fun tm ->
    try let l,r = dest tm in
        if rator l = neg_tm then
          if rator r = neg_tm then
            let th1 = INST [rand(rand l),m_tm; rand(rand r),n_tm] pth1 in
            let tm1 = rand(rand(rand(concl th1))) in
            let th2 = AP_TERM neg_tm (AP_TERM amp_tm (NUM_ADD_CONV tm1)) in
            TRANS th1 th2
          else
            let m = rand(rand l) and n = rand r in
            let m' = dest_numeral m and n' = dest_numeral n in
            if m' <=/ n' then
              let p = mk_numeral (n' -/ m') in
              let th1 = INST [m,m_tm; p,n_tm] pth2 in
              let th2 = NUM_ADD_CONV (rand(rand(lhand(concl th1)))) in
              let th3 = AP_TERM (rator tm) (AP_TERM amp_tm (SYM th2)) in
              TRANS th3 th1
            else
              let p = mk_numeral (m' -/ n') in
              let th1 = INST [n,m_tm; p,n_tm] pth3 in
              let th2 = NUM_ADD_CONV (rand(rand(lhand(lhand(concl th1))))) in
              let th3 = AP_TERM neg_tm (AP_TERM amp_tm (SYM th2)) in
              let th4 = AP_THM (AP_TERM add_tm th3) (rand tm) in
              TRANS th4 th1
        else
          if rator r = neg_tm then
            let m = rand l and n = rand(rand r) in
            let m' = dest_numeral m and n' = dest_numeral n in
            if n' <=/ m' then
              let p = mk_numeral (m' -/ n') in
              let th1 = INST [n,m_tm; p,n_tm] pth4 in
              let th2 = NUM_ADD_CONV (rand(lhand(lhand(concl th1)))) in
              let th3 = AP_TERM add_tm (AP_TERM amp_tm (SYM th2)) in
              let th4 = AP_THM th3 (rand tm) in
              TRANS th4 th1
            else
             let p = mk_numeral (n' -/ m') in
             let th1 = INST [m,m_tm; p,n_tm] pth5 in
             let th2 = NUM_ADD_CONV (rand(rand(rand(lhand(concl th1))))) in
             let th3 = AP_TERM neg_tm (AP_TERM amp_tm (SYM th2)) in
             let th4 = AP_TERM (rator tm) th3 in
             TRANS th4 th1
          else
            let th1 = INST [rand l,m_tm; rand r,n_tm] pth6 in
            let tm1 = rand(rand(concl th1)) in
            let th2 = AP_TERM amp_tm (NUM_ADD_CONV tm1) in
            TRANS th1 th2
    with Failure _ -> failwith "REAL_INT_ADD_CONV");;

let REAL_INT_SUB_CONV =
  GEN_REWRITE_CONV I [real_sub] THENC
  TRY_CONV(RAND_CONV REAL_INT_NEG_CONV) THENC
  REAL_INT_ADD_CONV;;

(* ------------------------------------------------------------------------- *)
(* Encodings of linear inequalities with justifications.                     *)
(* ------------------------------------------------------------------------- *)

type lineq_type = Eq | Le | Lt;;

type lineq = Lineq of int * lineq_type * int list * injust

and injust = Assumed of term
           | Multiplied of int * lineq
           | Added of lineq * lineq;;

(* ------------------------------------------------------------------------- *)
(* Calculate new (in)equality type after addition.                           *)
(* ------------------------------------------------------------------------- *)

let find_add_type =
  fun (Eq,x) -> x
    | (x,Eq) -> x
    | (_,Lt) -> Lt
    | (Lt,_) -> Lt
    | (Le,Le) -> Le;;

(* ------------------------------------------------------------------------- *)
(* Multiply out an (in)equation.                                             *)
(* ------------------------------------------------------------------------- *)

let multiply_ineq n (Lineq(k,ty,l,just) as i) =
  if n = 1 then i
  else if n = 0 & ty = Lt then failwith "multiply_ineq"
  else if n < 0 & (ty = Le or ty = Lt) then failwith "multiply_ineq"
  else Lineq(n * k,ty,map (prefix* n) l,Multiplied(n,i));;

(* ------------------------------------------------------------------------- *)
(* Add together (in)equations.                                               *)
(* ------------------------------------------------------------------------- *)

let add_ineq (Lineq(k1,ty1,l1,just1) as i1) (Lineq(k2,ty2,l2,just2) as i2) =
  let l = map2 prefix+ l1 l2 in
  Lineq(k1+k2,find_add_type(ty1,ty2),l,Added(i1,i2));;

(* ------------------------------------------------------------------------- *)
(* Elimination of variable between a single pair of (in)equations.           *)
(* If they're both inequalities, 1st coefficient must be +ve, 2nd -ve.       *)
(* ------------------------------------------------------------------------- *)

let elim_var v (Lineq(k1,ty1,l1,just1) as i1) (Lineq(k2,ty2,l2,just2) as i2) =
  let c1 = el v l1
  and c2 = el v l2 in
  let m = lcm (abs c1) (abs c2) in
  let m1 = m / (abs c1) and m2 = m / (abs c2) in
  let n1,n2 =
    if sgn(c1) = sgn(c2) then
      if ty1 = Eq then -m1,m2
      else if ty2 = Eq then m1,-m2
      else failwith "elim_var"
    else m1,m2 in
  let p1,p2 =
    if ty1 = Eq & ty2 = Eq & (n1 = -1 or n2 = -1) then -n1,-n2 else n1,n2 in
  add_ineq (multiply_ineq n1 i1) (multiply_ineq n2 i2);;

(* ------------------------------------------------------------------------- *)
(* The main refutation-finding code.                                         *)
(* ------------------------------------------------------------------------- *)

let is_trivial (Lineq(_,_,l,_)) = forall (prefix= 0) l;;

let find_answer (Lineq(k,ty,l,_) as ans) =
  if ty = Eq & not k = 0
  or ty = Le & k > 0
  or ty = Lt & k >= 0 then ans else failwith "find_answer";;

let calc_blowup l =
  let p,n = partition (prefix< 0) (filter (prefix not o prefix= 0) l) in
  (length p) * (length n);;

(* ------------------------------------------------------------------------- *)
(* Main elimination code:                                                    *)
(*                                                                           *)
(* (1) Looks for immediate solutions (false assertions with no variables).   *)
(*                                                                           *)
(* (2) If there are any equations, picks a variable with the lowest absolute *)
(* coefficient in any of them, and uses it to eliminate.                     *)
(*                                                                           *)
(* (3) Otherwise, chooses a variable in the inequality to minimize the       *)
(* blowup (number of consequences generated) and eliminates it.              *)
(* ------------------------------------------------------------------------- *)

let rec elim ineqs =
  let triv,nontriv = partition is_trivial ineqs in
  if not triv = [] then
    try tryfind find_answer triv
    with Failure _ -> elim nontriv else
  if nontriv = [] then failwith "elim" else
  let eqs,noneqs = partition (fun (Lineq(_,ty,_,_)) -> ty = Eq) nontriv in
  if not eqs = [] then
     let clists = map (fun (Lineq(_,_,l,_)) -> l) eqs in
     let sclist = sort (fun (x,y) -> abs(x) <= abs(y))
       (filter (prefix not o prefix= 0) (Union clists)) in
     let c = hd sclist in
     let v,eq = tryfind (fun (Lineq(_,_,l,_) as i) -> index c l,i) eqs in
     let othereqs = filter(prefix not o prefix= eq) eqs in
     let ioth,roth = partition (fun (Lineq(_,_,l,_)) -> el v l = 0)
       (othereqs @ noneqs) in
     let others = map (elim_var v eq) roth @ ioth in
     elim others else
  let lists = map (fun (Lineq(_,_,l,_)) -> l) noneqs in
  let numlist = upto (length(hd lists) - 1) in
  let coeffs = map (fun i -> map (el i) lists) numlist in
  let blows = map calc_blowup coeffs in
  let iblows = zip blows numlist in
  let c,v = hd(sort (fun (x,y) -> fst(x) <= fst(y))
    (filter (prefix not o prefix= 0 o fst) iblows)) in
  let no,yes = partition (fun (Lineq(_,_,l,_)) -> el v l = 0) ineqs in
  let pos,neg = partition(fun (Lineq(_,_,l,_)) -> el v l > 0) yes in
  elim (no @ allpairs (elim_var v) pos neg);;

(* ------------------------------------------------------------------------- *)
(* Multiply standard linear list by a constant.                              *)
(* ------------------------------------------------------------------------- *)

let LINEAR_MULT =
  let mult_tm = `$*` in
  let zero_tm = `&0` in
  let x_tm = `x:real` in
  let add_tm = `$+` in
  let pth = prove
   (`x * &0 = &0`,
    REWRITE_TAC[REAL_MUL_RZERO]) in
  let conv1 = GEN_REWRITE_CONV TOP_SWEEP_CONV [REAL_ADD_LDISTRIB] in
  let conv2 = DEPTH_BINOP_CONV add_tm (REWR_CONV REAL_MUL_ASSOC THENC
                                       LAND_CONV REAL_INT_MUL_CONV) in
  fun n tm ->
    if tm = zero_tm then INST [n,x_tm] pth else
    let ltm = mk_comb(mk_comb(mult_tm,n),tm) in
    (conv1 THENC conv2) ltm;;

(* ------------------------------------------------------------------------- *)
(* Add together canonically ordered standard linear lists.                   *)
(* ------------------------------------------------------------------------- *)

let LINEAR_ADD =
  let pth0a = prove
   (`&0 + x = x`,
    REWRITE_TAC[REAL_ADD_LID])
  and pth0b = prove
   (`x + &0 = x`,
    REWRITE_TAC[REAL_ADD_RID]) in
  let x_tm = `x:real` in
  let [pth1; pth2; pth3; pth4; pth5; pth6] = (CONJUNCTS o prove)
   (`((l1 + r1) + (l2 + r2) = (l1 + l2) + (r1 + r2)) /\
     ((l1 + r1) + tm2 = l1 + (r1 + tm2)) /\
     (tm1 + (l2 + r2) = l2 + (tm1 + r2)) /\
     ((l1 + r1) + tm2 = (l1 + tm2) + r1) /\
     (tm1 + tm2 = tm2 + tm1) /\
     (tm1 + (l2 + r2) = (tm1 + l2) + r2)`,
    REWRITE_TAC[REAL_ADD_AC]) in
  let tm1_tm = `tm1:real` and l1_tm = `l1:real` and r1_tm = `r1:real`
  and tm2_tm = `tm2:real` and l2_tm = `l2:real` and r2_tm = `r2:real` in
  let add_tm = `$+` in
  let dest = dest_binop add_tm
  and mk = mk_binop add_tm in
  let zero_tm = `&0` in
  let COEFF_CONV =
    REWR_CONV (GSYM REAL_ADD_RDISTRIB) THENC
    LAND_CONV REAL_INT_ADD_CONV in
  let rec linear_add tm1 tm2 =
    let ltm = mk tm1 tm2 in
    if tm1 = zero_tm then INST [tm2,x_tm] pth0a
    else if tm2 = zero_tm then INST [tm1,x_tm] pth0b else
    try let l1,r1 = dest tm1 in
        let v1 = rand l1 in
        try let l2,r2 = dest tm2 in
            let v2 = rand l2 in
            if v1 = v2 then
              let th1 = INST [l1,l1_tm; l2,l2_tm; r1,r1_tm; r2,r2_tm] pth1 in
              let th2 = CONV_RULE (RAND_CONV(LAND_CONV COEFF_CONV)) th1 in
              let ctm = rator(rand(concl th2)) in
              TRANS th2 (AP_TERM ctm (linear_add r1 r2))
            else if v1 < v2 then
              let th1 = INST [l1,l1_tm; r1,r1_tm; tm2,tm2_tm] pth2 in
              let ctm = rator(rand(concl th1)) in
              TRANS th1 (AP_TERM ctm (linear_add r1 tm2))
            else
              let th1 = INST [tm1,tm1_tm; l2,l2_tm; r2,r2_tm] pth3 in
              let ctm = rator(rand(concl th1)) in
              TRANS th1 (AP_TERM ctm (linear_add tm1 r2))
        with Failure _ ->
            let v2 = rand tm2 in
            if v1 = v2 then
              let th1 = INST [l1,l1_tm; r1,r1_tm; tm2,tm2_tm] pth4 in
              CONV_RULE (RAND_CONV(LAND_CONV COEFF_CONV)) th1
            else if v1 < v2 then
              let th1 = INST [l1,l1_tm; r1,r1_tm; tm2,tm2_tm] pth2 in
              let ctm = rator(rand(concl th1)) in
              TRANS th1 (AP_TERM ctm (linear_add r1 tm2))
            else
              INST [tm1,tm1_tm; tm2,tm2_tm] pth5
    with Failure _ ->
        let v1 = rand tm1 in
        try let l2,r2 = dest tm2 in
            let v2 = rand l2 in
            if v1 = v2 then
              let th1 = INST [tm1,tm1_tm; l2,l2_tm; r2,r2_tm] pth6 in
              CONV_RULE (RAND_CONV(LAND_CONV COEFF_CONV)) th1
            else if v1 < v2 then
              REFL ltm
            else
              let th1 = INST [tm1,tm1_tm; l2,l2_tm; r2,r2_tm] pth3 in
              let ctm = rator(rand(concl th1)) in
              TRANS th1 (AP_TERM ctm (linear_add tm1 r2))
        with Failure _ ->
            let v2 = rand tm2 in
            if v1 = v2 then
              COEFF_CONV ltm
            else if v1 < v2 then
              REFL ltm
            else
              INST [tm1,tm1_tm; tm2,tm2_tm] pth5 in
  fun tm1 tm2 ->
    let th = linear_add tm1 tm2 in
    let tm = rand(concl th) in
    let zth = GEN_REWRITE_CONV DEPTH_CONV
      [REAL_MUL_LZERO; REAL_ADD_LID; REAL_ADD_RID] tm in
    TRANS th zth;;

(* ------------------------------------------------------------------------- *)
(* Translate back a proof.                                                   *)
(* ------------------------------------------------------------------------- *)

let TRANSLATE_PROOF =
  let TRIVIAL_CONV = DEPTH_CONV
     (CHANGED_CONV REAL_INT_NEG_CONV ORELSEC
      REAL_INT_ADD_CONV ORELSEC
      REAL_INT_MUL_CONV ORELSEC
      REAL_INT_LE_CONV ORELSEC
      REAL_INT_EQ_CONV ORELSEC
      REAL_INT_LT_CONV) THENC
     GEN_REWRITE_CONV TOP_DEPTH_CONV (ARITH::basic_rewrites()) in
  let ADD_INEQS =
    let a_tm = `a:real` and b_tm = `b:real` in
    let pths = (CONJUNCTS o prove)
     (`((&0 = a) /\ (&0 = b) ==> (&0 = a + b)) /\
       ((&0 = a) /\ (&0 <= b) ==> (&0 <= a + b)) /\
       ((&0 = a) /\ (&0 < b) ==> (&0 < a + b)) /\
       ((&0 <= a) /\ (&0 = b) ==> (&0 <= a + b)) /\
       ((&0 <= a) /\ (&0 <= b) ==> (&0 <= a + b)) /\
       ((&0 <= a) /\ (&0 < b) ==> (&0 < a + b)) /\
       ((&0 < a) /\ (&0 = b) ==> (&0 < a + b)) /\
       ((&0 < a) /\ (&0 <= b) ==> (&0 < a + b)) /\
       ((&0 < a) /\ (&0 < b) ==> (&0 < a + b))`,
      CONV_TAC(ONCE_DEPTH_CONV SYM_CONV) THEN
      REPEAT STRIP_TAC THEN
      ASM_REWRITE_TAC[REAL_ADD_LID; REAL_ADD_RID] THENL
       [MATCH_MP_TAC REAL_LE_TRANS;
        MATCH_MP_TAC REAL_LET_TRANS;
        MATCH_MP_TAC REAL_LTE_TRANS;
        MATCH_MP_TAC REAL_LT_TRANS] THEN
      EXISTS_TAC `a:real` THEN ASM_REWRITE_TAC[] THEN
      GEN_REWRITE_TAC LAND_CONV [GSYM REAL_ADD_RID] THEN
      (MATCH_MP_TAC REAL_LE_LADD_IMP ORELSE MATCH_MP_TAC REAL_LT_LADD_IMP) THEN
      ASM_REWRITE_TAC[]) in
    fun th1 th2 ->
      let a = rand(concl th1) and b = rand(concl th2) in
      let xth = tryfind (C MP (CONJ th1 th2) o INST [a,a_tm; b,b_tm]) pths in
      let yth = LINEAR_ADD a b in
      EQ_MP (AP_TERM (rator(concl xth)) yth) xth in
  let MULTIPLY_INEQS =
    let pths = (CONJUNCTS o prove)
     (`((&0 = y) ==> (&0 = x * y)) /\
       (&0 <= y ==> &0 <= x ==> &0 <= x * y) /\
       (&0 < y ==> &0 < x ==> &0 < x * y)`,
      CONV_TAC(ONCE_DEPTH_CONV SYM_CONV) THEN
      REPEAT STRIP_TAC THEN ASM_REWRITE_TAC[REAL_MUL_RZERO] THENL
       [MATCH_MP_TAC REAL_LE_MUL;
        MATCH_MP_TAC REAL_LT_MUL] THEN
      ASM_REWRITE_TAC[]) in
    let x_tm = `x:real` and y_tm = `y:real` in
    fun x th ->
      let y = rand(concl th) in
      let xth = tryfind (C MP th o INST[x,x_tm; y,y_tm]) pths in
      let wth = if is_imp(concl xth) then
        MP (CONV_RULE(LAND_CONV TRIVIAL_CONV) xth) TRUTH else xth in
      let yth = LINEAR_MULT x (rand(rand(concl wth))) in
      EQ_MP (AP_TERM (rator(concl wth)) yth) wth in
  fun refutation ->
    let cache = ref [] in
    let rec translate refut =
      try assoc refut (!cache) with Failure _ ->
      match refut with
        (Lineq(k,ty,l,Assumed(t))) ->
          let th = ASSUME t in
          (cache:= (refut,th)::(!cache); th)
      | (Lineq(k,ty,l,Added(r1,r2))) ->
          let th1 = translate r1
          and th2 = translate r2 in
          let th = ADD_INEQS th1 th2 in
          (cache:= (refut,th)::(!cache); th)
      | (Lineq(k,ty,l,Multiplied(n,r))) ->
          let th1 = translate r in
          let th = MULTIPLY_INEQS (mk_small_intconst n) th1 in
          (cache:= (refut,th)::(!cache); th) in
    CONV_RULE TRIVIAL_CONV (translate refutation);;

(* ------------------------------------------------------------------------- *)
(* Conversion to normalize product terms, i.e:                               *)
(*                                                                           *)
(* Separate out (and multiply out) integer constants; put the term in        *)
(* the form "([-]&n) * x" where either x is a canonically ordered product    *)
(* of the non-integer-constants, or is "&1".                                 *)
(* ------------------------------------------------------------------------- *)

let REAL_PROD_NORM_CONV =
  let REAL_MUL_AC = AC REAL_MUL_AC in
  let x_tm = `x:real` in
  let mul_tm = `$*` in
  let pth1 = SYM(SPEC x_tm REAL_MUL_RID)
  and pth2 = SYM(SPEC x_tm REAL_MUL_LID) in
  let binops_mul = binops mul_tm
  and list_mk_binop_mul = list_mk_binop mul_tm
  and mk_binop_mul = mk_binop mul_tm in
  fun tm ->
    let factors = binops_mul tm in
    let consts,others = partition is_intconst factors in
    if others = [] then
      let th1 = DEPTH_CONV REAL_INT_MUL_CONV tm in
      TRANS th1 (INST [rand(concl th1),x_tm] pth1) else
    let sothers = sort (uncurry prefix<) others in
    if consts = [] then
      let th1 = REAL_MUL_AC (mk_eq(tm,list_mk_binop_mul sothers)) in
      TRANS th1 (INST [rand(concl th1),x_tm] pth2) else
    let th1 = REAL_MUL_AC
     (mk_eq(tm,mk_binop_mul(list_mk_binop_mul consts)
                           (list_mk_binop_mul sothers))) in
    let tm1 = rand(concl th1) in
    let th2 = AP_TERM mul_tm (DEPTH_CONV REAL_INT_MUL_CONV (lhand tm1)) in
    TRANS th1 (AP_THM th2 (rand tm1));;

(* ------------------------------------------------------------------------- *)
(* Collection of like terms.                                                 *)
(* ------------------------------------------------------------------------- *)

let COLLECT_CONV =
  let add_tm = `$+` in
  let dest = dest_binop `$+` in
  let rec collect tm =
    try let l,r = dest tm in
        let lth = collect l
        and rth = collect r in
        let xth = LINEAR_ADD (rand(concl lth)) (rand(concl rth)) in
        TRANS (MK_COMB(AP_TERM add_tm lth,rth)) xth
    with Failure _ -> REFL tm in
  collect;;

(* ------------------------------------------------------------------------- *)
(* Normalize a term in the standard linear form.                             *)
(* ------------------------------------------------------------------------- *)

let REAL_SUM_NORM_CONV =
  let REAL_ADD_AC = AC REAL_ADD_AC in
  let pth1 = prove
   (`--x = --(&1) * x`,
    REWRITE_TAC[REAL_MUL_LNEG; REAL_MUL_LID]) in
  let pth2 = prove
   (`x - y = x + --(&1) * y`,
    REWRITE_TAC[real_sub; GSYM pth1]) in
  let ptm = `--` in
  let stm = `$+` in
  let one_tm = `&1` in
  let binops_add = binops stm in
  let list_mk_binop_add = list_mk_binop stm in
  let prelim_conv =
    DEPTH_CONV((GEN_REWRITE_CONV I [pth1]
                o assert(fun t -> not (rand t = one_tm))) ORELSEC
                GEN_REWRITE_CONV I [pth2]) THENC
    GEN_REWRITE_CONV TOP_DEPTH_CONV
      [REAL_ADD_LDISTRIB; REAL_ADD_RDISTRIB] THENC
    GEN_REWRITE_CONV DEPTH_CONV
      [REAL_MUL_LZERO; REAL_MUL_RZERO;
       REAL_MUL_LID; REAL_MUL_RID;
       REAL_ADD_LID; REAL_ADD_RID] in
  fun tm ->
    let th1 = prelim_conv tm in
    let th2 = DEPTH_BINOP_CONV stm REAL_PROD_NORM_CONV (rand(concl th1)) in
    let tm2 = rand(concl th2) in
    let elements = binops_add tm2 in
    let selements = sort (fun (x,y) -> rand x <= rand y) elements in
    let th3 = REAL_ADD_AC(mk_eq(tm2,list_mk_binop_add selements)) in
    let th4 = COLLECT_CONV (rand(concl th3)) in
    itlist TRANS [th1; th2; th3] th4;;

(* ------------------------------------------------------------------------- *)
(* Produce negated forms of canonicalization theorems.                       *)
(* ------------------------------------------------------------------------- *)

let REAL_NEGATE_CANON =
  let pth1 = prove
   (`((a <= b = &0 <= X) = (b < a = &0 < --X)) /\
     ((a < b = &0 < X) = (b <= a = &0 <= --X))`,
    REWRITE_TAC[real_lt; REAL_LE_LNEG; REAL_LE_RNEG] THEN
    REWRITE_TAC[REAL_ADD_RID; REAL_ADD_LID] THEN
    CONV_TAC TAUT) in
  let pth2 = prove
   (`--((-- a) * x + z) = a * x + --z`,
    REWRITE_TAC[GSYM REAL_MUL_LNEG; REAL_NEG_ADD; REAL_NEG_NEG]) in
  let pth3 = prove
   (`--(a * x + z) = --a * x + --z`,
    REWRITE_TAC[REAL_NEG_ADD; GSYM REAL_MUL_LNEG]) in
  let pth4 = prove
   (`--(--a * x) = a * x`,
    REWRITE_TAC[REAL_MUL_LNEG; REAL_NEG_NEG]) in
  let pth5 = prove
   (`--(a * x) = --a * x`,
    REWRITE_TAC[REAL_MUL_LNEG]) in
  let rewr1_CONV = FIRST_CONV (map REWR_CONV [pth2; pth3])
  and rewr2_CONV = FIRST_CONV (map REWR_CONV [pth4; pth5]) in
  let rec distrib_neg_conv tm =
    try let th = rewr1_CONV tm in
        TRANS th (RAND_CONV distrib_neg_conv (rand(concl th)))
    with Failure _ -> rewr2_CONV tm in
  fun th ->
    let th1 = GEN_REWRITE_RULE I [pth1] th in
    TRANS th1 (RAND_CONV distrib_neg_conv (rand(concl th1)));;

(* ------------------------------------------------------------------------- *)
(* Version for whole atom, with intelligent cacheing.                        *)
(* ------------------------------------------------------------------------- *)

let clear_atom_cache,REAL_ATOM_NORM_CONV =
  let zero_tm = `&0` in
  let pth1 = prove
    (`((x = y) = (&0 = y - x)) /\
      (x <= y = &0 <= y - x) /\
      (x < y = &0 < y - x)`,
     REWRITE_TAC[real_lt; real_sub; GSYM REAL_LE_LNEG; REAL_LE_NEG2] THEN
     REWRITE_TAC[GSYM REAL_LE_RNEG; REAL_NEG_NEG] THEN
     REWRITE_TAC[GSYM REAL_LE_ANTISYM; GSYM REAL_LE_LNEG; GSYM REAL_LE_RNEG;
                 REAL_LE_NEG2; REAL_NEG_NEG]) in
  let zero_CONV = GEN_REWRITE_CONV I [pth1] in
  let right_CONV = RAND_CONV REAL_SUM_NORM_CONV in
  let uncached_CONV tm =
    let l = lhand tm in
    if l = zero_tm then right_CONV tm
    else (zero_CONV THENC right_CONV) tm in
  let atomcache = ref [] in
  let lookup_cache tm = find (fun th -> lhand(concl th) = tm) (!atomcache) in
  let clear_atom_cache() = atomcache := [] in
  let pth2 = prove
   (`(a < b = c < d) = (b <= a = d <= c)`,
    REWRITE_TAC[real_lt] THEN CONV_TAC TAUT) in
  let pth3 = prove
   (`(a <= b = c <= d) = (b < a = d < c)`,
    REWRITE_TAC[real_lt] THEN CONV_TAC TAUT) in
  let negate_CONV = GEN_REWRITE_CONV I [pth2;pth3] in
  let le_tm = `$<= :real->real->bool`
  and lt_tm = `$< :real->real->bool` in
  clear_atom_cache,
  fun tm ->
    try lookup_cache tm with Failure _ ->
    let th = uncached_CONV tm in
    atomcache := th::(!atomcache);
    (try let th' = REAL_NEGATE_CANON th in
         atomcache := th'::(!atomcache)
     with Failure _ -> ());
    th;;

(* ------------------------------------------------------------------------- *)
(* Refute a conjunction of equations and/or inequations.                     *)
(* ------------------------------------------------------------------------- *)

let REAL_SIMPLE_ARITH_CONV =
  let trivthm = prove
   (`&0 < &0 = F`,
    REWRITE_TAC[REAL_LE_REFL; real_lt]) in
  let add_tm = `$+` in
  let one_tm = `&1` in
  let zero_tm = `&0` in
  let less_tm = `$<` in
  let false_tm = `F` in
  let fixup_atom th =
    let th0 = CONV_RULE REAL_ATOM_NORM_CONV th in
    let tm0 = concl th0 in
    if rand tm0 = zero_tm then
      if rator(rator tm0) = less_tm then EQ_MP trivthm th0
      else failwith "trivially true, so useless in refutation"
    else th0 in
  let eq_tm = `$= :real->real->bool` in
  let le_tm = `$<= :real->real->bool` in
  let lt_tm = `$< :real->real->bool` in
  fun tm ->
    let ths = mapfilter fixup_atom (CONJUNCTS (ASSUME tm)) in
    try find (fun th -> concl th = false_tm) ths with Failure _ ->
    let cj1 = map concl ths in
    let allvars = itlist (union o map rand o binops add_tm o rand) cj1 [] in
    let vars = if mem one_tm allvars then one_tm::(subtract allvars [one_tm])
               else one_tm::allvars in
    let untermify t =
      let op = rator(rator t)
      and right = rand t in
      let rights = binops add_tm right in
      let cvps = map (((dest_small_intconst o rand) F_F (C index vars)) o
        dest_comb) rights in
      let k = -(try rev_assoc 0 cvps with Failure _ -> 0) in
      let l = tl(map (fun v -> try rev_assoc v cvps with Failure _ -> 0)
                   (upto (length(vars) - 1))) in
      let ty = if op = eq_tm then Eq else if op = le_tm then Le
               else if op = lt_tm then Lt else failwith "unknown op" in
      Lineq(k,ty,l,Assumed t) in
    let ineqs = map untermify cj1 in
    let proof = elim ineqs in
    let th2 = TRANSLATE_PROOF proof in
    itlist PROVE_HYP ths th2;;

(* ------------------------------------------------------------------------- *)
(* General case: canonicalize and split up, then refute the bits.            *)
(* ------------------------------------------------------------------------- *)

let PURE_REAL_ARITH_TAC =
 let init_CONV =
    GEN_REWRITE_CONV DEPTH_CONV [FORALL_SIMP; EXISTS_SIMP] THENC
    GEN_REWRITE_CONV DEPTH_CONV [real_abs; real_gt; real_ge] THENC
    REPEATC COND_ELIM_CONV THENC PRENEX_CONV in
  let atom_CONV =
    let pth = prove
      (`(~(x <= y) = y < x) /\
        (~(x < y) = y <= x) /\
        (~(x = y) = x < y \/ y < x)`,
       REWRITE_TAC[real_lt] THEN REWRITE_TAC[GSYM DE_MORGAN_THM] THEN
       REWRITE_TAC[REAL_LE_ANTISYM] THEN AP_TERM_TAC THEN
       MATCH_ACCEPT_TAC EQ_SYM_EQ) in
    GEN_REWRITE_CONV I [pth] in
  REPEAT GEN_TAC THEN
  CONV_TAC init_CONV THEN
  REPEAT GEN_TAC THEN
  REFUTE_THEN(MP_TAC o CONV_RULE
    (PRESIMP_CONV THENC NNF_CONV THENC SKOLEM_CONV THENC
     PRENEX_CONV THENC ONCE_DEPTH_CONV atom_CONV THENC PROP_DNF_CONV)) THEN
  DISCH_THEN(REPEAT_TCL
    (CHOOSE_THEN ORELSE_TCL DISJ_CASES_THEN ORELSE_TCL CONJUNCTS_THEN)
    ASSUME_TAC) THEN
  TRY(FIRST_ASSUM CONTR_TAC) THEN
  POP_ASSUM_LIST(ASSUME_TAC o end_itlist CONJ) THEN
  POP_ASSUM(ACCEPT_TAC o REAL_SIMPLE_ARITH_CONV o concl);;

let REAL_ARITH_TAC =
  POP_ASSUM_LIST(K ALL_TAC) THEN PURE_REAL_ARITH_TAC;;

(* ------------------------------------------------------------------------- *)
(* Rule version.                                                             *)
(* ------------------------------------------------------------------------- *)

let REAL_ARITH tm = prove(tm,REAL_ARITH_TAC);;

(* ------------------------------------------------------------------------- *)
(* Prove all the linear theorems we can blow away automatically.             *)
(* ------------------------------------------------------------------------- *)

let REAL_EQ_ADD_LCANCEL_0 = prove
 (`!x y. (x + y = x) = (y = &0)`,
  REAL_ARITH_TAC);;

let REAL_EQ_ADD_RCANCEL_0 = prove
 (`!x y. (x + y = y) = (x = &0)`,
  REAL_ARITH_TAC);;

let REAL_LNEG_UNIQ = prove
 (`!x y. (x + y = &0) = (x = --y)`,
  REAL_ARITH_TAC);;

let REAL_RNEG_UNIQ = prove
 (`!x y. (x + y = &0) = (y = --x)`,
  REAL_ARITH_TAC);;

let REAL_NEG_LMUL = prove
 (`!x y. --(x * y) = (--x) * y`,
  REAL_ARITH_TAC);;

let REAL_NEG_RMUL = prove
 (`!x y. --(x * y) = x * (--y)`,
  REAL_ARITH_TAC);;

let REAL_NEGNEG = prove
 (`!x. --(--x) = x`,
  REAL_ARITH_TAC);;

let REAL_NEG_MUL2 = prove
 (`!x y. (--x) * (--y) = x * y`,
  REAL_ARITH_TAC);;

let REAL_LT_LADD = prove
 (`!x y z. (x + y) < (x + z) = y < z`,
  REAL_ARITH_TAC);;

let REAL_LT_RADD = prove
 (`!x y z. (x + z) < (y + z) = x < y`,
  REAL_ARITH_TAC);;

let REAL_NOT_LT = prove
 (`!x y. ~(x < y) = y <= x`,
  REAL_ARITH_TAC);;

let REAL_LT_ANTISYM = prove
 (`!x y. ~(x < y /\ y < x)`,
  REAL_ARITH_TAC);;

let REAL_LT_GT = prove
 (`!x y. x < y ==> ~(y < x)`,
  REAL_ARITH_TAC);;

let REAL_NOT_EQ = prove
 (`!x y. ~(x = y) = x < y \/ y < x`,
  REAL_ARITH_TAC);;

let REAL_NOT_LE = prove
 (`!x y. ~(x <= y) = y < x`,
  REAL_ARITH_TAC);;

let REAL_LE_TOTAL = prove
 (`!x y. x <= y \/ y <= x`,
  REAL_ARITH_TAC);;

let REAL_LET_TOTAL = prove
 (`!x y. x <= y \/ y < x`,
  REAL_ARITH_TAC);;

let REAL_LTE_TOTAL = prove
 (`!x y. x < y \/ y <= x`,
  REAL_ARITH_TAC);;

let REAL_LE_REFL = prove
 (`!x. x <= x`,
  REAL_ARITH_TAC);;

let REAL_LE_LT = prove
 (`!x y. x <= y = x < y \/ (x = y)`,
  REAL_ARITH_TAC);;

let REAL_LT_LE = prove
 (`!x y. x < y = x <= y /\ ~(x = y)`,
  REAL_ARITH_TAC);;

let REAL_LE_ANTISYM = prove
 (`!x y. x <= y /\ y <= x = (x = y)`,
  REAL_ARITH_TAC);;

let REAL_LET_ANTISYM = prove
 (`!x y. ~(x < y /\ y <= x)`,
  REAL_ARITH_TAC);;

let REAL_LTE_ANTSYM = prove
 (`!x y. ~(x <= y /\ y < x)`,
  REAL_ARITH_TAC);;

let REAL_NEG_LT0 = prove
 (`!x. (--x) < &0 = &0 < x`,
  REAL_ARITH_TAC);;

let REAL_NEG_GT0 = prove
 (`!x. &0 < (--x) = x < &0`,
  REAL_ARITH_TAC);;

let REAL_NEG_LE0 = prove
 (`!x. (--x) <= &0 = &0 <= x`,
  REAL_ARITH_TAC);;

let REAL_NEG_GE0 = prove
 (`!x. &0 <= (--x) = x <= &0`,
  REAL_ARITH_TAC);;

let REAL_LT_TOTAL = prove
 (`!x y. (x = y) \/ x < y \/ y < x`,
  REAL_ARITH_TAC);;

let REAL_LT_NEGTOTAL = prove
 (`!x. (x = &0) \/ (&0 < x) \/ (&0 < --x)`,
  REAL_ARITH_TAC);;

let REAL_LE_NEGTOTAL = prove
 (`!x. &0 <= x \/ &0 <= --x`,
  REAL_ARITH_TAC);;

let REAL_LE_01 = prove
 (`&0 <= &1`,
  REAL_ARITH_TAC);;

let REAL_LT_01 = prove
 (`&0 < &1`,
  REAL_ARITH_TAC);;

let REAL_LE_LADD = prove
 (`!x y z. (x + y) <= (x + z) = y <= z`,
  REAL_ARITH_TAC);;

let REAL_LE_RADD = prove
 (`!x y z. (x + z) <= (y + z) = x <= y`,
  REAL_ARITH_TAC);;

let REAL_LT_ADD2 = prove
 (`!w x y z. w < x /\ y < z ==> (w + y) < (x + z)`,
  REAL_ARITH_TAC);;

let REAL_LE_ADD2 = prove
 (`!w x y z. w <= x /\ y <= z ==> (w + y) <= (x + z)`,
  REAL_ARITH_TAC);;

let REAL_LE_ADD = prove
 (`!x y. &0 <= x /\ &0 <= y ==> &0 <= (x + y)`,
  REAL_ARITH_TAC);;

let REAL_LT_ADD = prove
 (`!x y. &0 < x /\ &0 < y ==> &0 < (x + y)`,
  REAL_ARITH_TAC);;

let REAL_LT_LNEG = prove
 (`!x y. --x < y = &0 < x + y`,
  REAL_ARITH_TAC);;

let REAL_LT_RNEG = prove
 (`!x y. x < --y = x + y < &0`,
  REAL_ARITH_TAC);;

let REAL_LT_ADDNEG = prove
 (`!x y z. y < (x + (--z)) = (y + z) < x`,
  REAL_ARITH_TAC);;

let REAL_LT_ADDNEG2 = prove
 (`!x y z. (x + (--y)) < z = x < (z + y)`,
  REAL_ARITH_TAC);;

let REAL_LT_ADD1 = prove
 (`!x y. x <= y ==> x < (y + &1)`,
  REAL_ARITH_TAC);;

let REAL_SUB_ADD = prove
 (`!x y. (x - y) + y = x`,
  REAL_ARITH_TAC);;

let REAL_SUB_ADD2 = prove
 (`!x y. y + (x - y) = x`,
  REAL_ARITH_TAC);;

let REAL_SUB_REFL = prove
 (`!x. x - x = &0`,
  REAL_ARITH_TAC);;

let REAL_SUB_0 = prove
 (`!x y. (x - y = &0) = (x = y)`,
  REAL_ARITH_TAC);;

let REAL_LE_DOUBLE = prove
 (`!x. &0 <= x + x = &0 <= x`,
  REAL_ARITH_TAC);;

let REAL_LE_NEGL = prove
 (`!x. (--x <= x) = (&0 <= x)`,
  REAL_ARITH_TAC);;

let REAL_LE_NEGR = prove
 (`!x. (x <= --x) = (x <= &0)`,
  REAL_ARITH_TAC);;

let REAL_NEG_EQ0 = prove
 (`!x. (--x = &0) = (x = &0)`,
  REAL_ARITH_TAC);;

let REAL_NEG_0 = prove
 (`--(&0) = &0`,
  REAL_ARITH_TAC);;

let REAL_NEG_SUB = prove
 (`!x y. --(x - y) = y - x`,
  REAL_ARITH_TAC);;

let REAL_SUB_LT = prove
 (`!x y. &0 < x - y = y < x`,
  REAL_ARITH_TAC);;

let REAL_SUB_LE = prove
 (`!x y. &0 <= (x - y) = y <= x`,
  REAL_ARITH_TAC);;

let REAL_ADD_SUB = prove
 (`!x y. (x + y) - x = y`,
  REAL_ARITH_TAC);;

let REAL_NEG_EQ = prove
 (`!x y. (--x = y) = (x = --y)`,
  REAL_ARITH_TAC);;

let REAL_NEG_MINUS1 = prove
 (`!x. --x = (--(&1)) * x`,
  REAL_ARITH_TAC);;

let REAL_LT_IMP_NE = prove
 (`!x y. x < y ==> ~(x = y)`,
  REAL_ARITH_TAC);;

let REAL_LE_ADDR = prove
 (`!x y. x <= x + y = &0 <= y`,
  REAL_ARITH_TAC);;

let REAL_LE_ADDL = prove
 (`!x y. y <= x + y = &0 <= x`,
  REAL_ARITH_TAC);;

let REAL_LT_ADDR = prove
 (`!x y. x < x + y = &0 < y`,
  REAL_ARITH_TAC);;

let REAL_LT_ADDL = prove
 (`!x y. y < x + y = &0 < x`,
  REAL_ARITH_TAC);;

let REAL_SUB_SUB = prove
 (`!x y. (x - y) - x = --y`,
  REAL_ARITH_TAC);;

let REAL_LT_ADD_SUB = prove
 (`!x y z. (x + y) < z = x < (z - y)`,
  REAL_ARITH_TAC);;

let REAL_LT_SUB_RADD = prove
 (`!x y z. (x - y) < z = x < z + y`,
  REAL_ARITH_TAC);;

let REAL_LT_SUB_LADD = prove
 (`!x y z. x < (y - z) = (x + z) < y`,
  REAL_ARITH_TAC);;

let REAL_LE_SUB_LADD = prove
 (`!x y z. x <= (y - z) = (x + z) <= y`,
  REAL_ARITH_TAC);;

let REAL_LE_SUB_RADD = prove
 (`!x y z. (x - y) <= z = x <= z + y`,
  REAL_ARITH_TAC);;

let REAL_LT_NEG = prove
 (`!x y. --x < --y = y < x`,
  REAL_ARITH_TAC);;

let REAL_LE_NEG = prove
 (`!x y. --x <= --y = y <= x`,
  REAL_ARITH_TAC);;

let REAL_ADD2_SUB2 = prove
 (`!a b c d. (a + b) - (c + d) = (a - c) + (b - d)`,
  REAL_ARITH_TAC);;

let REAL_SUB_LZERO = prove
 (`!x. &0 - x = --x`,
  REAL_ARITH_TAC);;

let REAL_SUB_RZERO = prove
 (`!x. x - &0 = x`,
  REAL_ARITH_TAC);;

let REAL_LET_ADD2 = prove
 (`!w x y z. w <= x /\ y < z ==> (w + y) < (x + z)`,
  REAL_ARITH_TAC);;

let REAL_LTE_ADD2 = prove
 (`!w x y z. w < x /\ y <= z ==> (w + y) < (x + z)`,
  REAL_ARITH_TAC);;

let REAL_LET_ADD = prove
 (`!x y. &0 <= x /\ &0 < y ==> &0 < (x + y)`,
  REAL_ARITH_TAC);;

let REAL_LTE_ADD = prove
 (`!x y. &0 < x /\ &0 <= y ==> &0 < (x + y)`,
  REAL_ARITH_TAC);;

let REAL_SUB_LNEG = prove
 (`!x y. (--x) - y = --(x + y)`,
  REAL_ARITH_TAC);;

let REAL_SUB_RNEG = prove
 (`!x y. x - (--y) = x + y`,
  REAL_ARITH_TAC);;

let REAL_SUB_NEG2 = prove
 (`!x y. (--x) - (--y) = y - x`,
  REAL_ARITH_TAC);;

let REAL_SUB_TRIANGLE = prove
 (`!a b c. (a - b) + (b - c) = a - c`,
  REAL_ARITH_TAC);;

let REAL_EQ_SUB_LADD = prove
 (`!x y z. (x = y - z) = (x + z = y)`,
  REAL_ARITH_TAC);;

let REAL_EQ_SUB_RADD = prove
 (`!x y z. (x - y = z) = (x = z + y)`,
  REAL_ARITH_TAC);;

let REAL_SUB_SUB2 = prove
 (`!x y. x - (x - y) = y`,
  REAL_ARITH_TAC);;

let REAL_ADD_SUB2 = prove
 (`!x y. x - (x + y) = --y`,
  REAL_ARITH_TAC);;

let REAL_EQ_IMP_LE = prove
 (`!x y. (x = y) ==> x <= y`,
  REAL_ARITH_TAC);;

let REAL_POS_NZ = prove
 (`!x. &0 < x ==> ~(x = &0)`,
  REAL_ARITH_TAC);;

let REAL_DIFFSQ = prove
 (`!x y. (x + y) * (x - y) = (x * x) - (y * y)`,
  REAL_ARITH_TAC);;

let REAL_EQ_NEG2 = prove
 (`!x y. (--x = --y) = (x = y)`,
  REAL_ARITH_TAC);;

let REAL_LT_NEG2 = prove
 (`!x y. --x < --y = y < x`,
  REAL_ARITH_TAC);;

let REAL_SUB_LDISTRIB = prove
 (`!x y z. x * (y - z) = x * y - x * z`,
  REAL_ARITH_TAC);;

let REAL_SUB_RDISTRIB = prove
 (`!x y z. (x - y) * z = x * z - y * z`,
  REAL_ARITH_TAC);;

(* ------------------------------------------------------------------------- *)
(* Theorems about "abs".                                                     *)
(* ------------------------------------------------------------------------- *)

let REAL_ABS_ZERO = prove
 (`!x. (abs(x) = &0) = (x = &0)`,
  REAL_ARITH_TAC);;

let REAL_ABS_NUM = prove
 (`abs(&n) = &n`,
   REWRITE_TAC[REAL_POS; real_abs]);;

let REAL_ABS_0 = prove
 (`abs(&0) = &0`,
  REAL_ARITH_TAC);;

let REAL_ABS_1 = prove
 (`abs(&1) = &1`,
  REAL_ARITH_TAC);;

let REAL_ABS_NEG = prove
 (`!x. abs(--x) = abs(x)`,
  REAL_ARITH_TAC);;

let REAL_ABS_TRIANGLE = prove
 (`!x y. abs(x + y) <= abs(x) + abs(y)`,
  REAL_ARITH_TAC);;

let REAL_ABS_TRIANGLE_LE = prove
 (`!x y z.abs(x) + abs(y - x) <= z ==> abs(y) <= z`,
  REAL_ARITH_TAC);;

let REAL_ABS_TRIANGLE_LT = prove
 (`!x y z.abs(x) + abs(y - x) < z ==> abs(y) < z`,
  REAL_ARITH_TAC);;

let REAL_ABS_POS = prove
 (`!x. &0 <= abs(x)`,
  REAL_ARITH_TAC);;

let REAL_ABS_SUB = prove
 (`!x y. abs(x - y) = abs(y - x)`,
  REAL_ARITH_TAC);;

let REAL_ABS_NZ = prove
 (`!x. ~(x = &0) = &0 < abs(x)`,
  REAL_ARITH_TAC);;

let REAL_ABS_ABS = prove
 (`!x. abs(abs(x)) = abs(x)`,
  REAL_ARITH_TAC);;

let REAL_ABS_LE = prove
 (`!x. x <= abs(x)`,
  REAL_ARITH_TAC);;

let REAL_ABS_REFL = prove
 (`!x. (abs(x) = x) = &0 <= x`,
  REAL_ARITH_TAC);;

let REAL_ABS_BETWEEN = prove
 (`!x y d. &0 < d /\ ((x - d) < y) /\ (y < (x + d)) = abs(y - x) < d`,
  REAL_ARITH_TAC);;

let REAL_ABS_BOUND = prove
 (`!x y d. abs(x - y) < d ==> y < (x + d)`,
  REAL_ARITH_TAC);;

let REAL_ABS_STILLNZ = prove
 (`!x y. abs(x - y) < abs(y) ==> ~(x = &0)`,
  REAL_ARITH_TAC);;

let REAL_ABS_CASES = prove
 (`!x. (x = &0) \/ &0 < abs(x)`,
  REAL_ARITH_TAC);;

let REAL_ABS_BETWEEN1 = prove
 (`!x y z. x < z /\ (abs(y - x)) < (z - x) ==> y < z`,
  REAL_ARITH_TAC);;

let REAL_ABS_SIGN = prove
 (`!x y. abs(x - y) < y ==> &0 < x`,
  REAL_ARITH_TAC);;

let REAL_ABS_SIGN2 = prove
 (`!x y. abs(x - y) < --y ==> x < &0`,
  REAL_ARITH_TAC);;

let REAL_ABS_CIRCLE = prove
 (`!x y h. abs(h) < (abs(y) - abs(x)) ==> abs(x + h) < abs(y)`,
  REAL_ARITH_TAC);;

let REAL_SUB_ABS = prove
 (`!x y. (abs(x) - abs(y)) <= abs(x - y)`,
  REAL_ARITH_TAC);;

let REAL_ABS_SUB_ABS = prove
 (`!x y. abs(abs(x) - abs(y)) <= abs(x - y)`,
  REAL_ARITH_TAC);;

let REAL_ABS_BETWEEN2 = prove
 (`!x0 x y0 y. x0 < y0 /\ &2 * abs(x - x0) < (y0 - x0) /\
                          &2 * abs(y - y0) < (y0 - x0)
        ==> x < y`,
  REAL_ARITH_TAC);;

let REAL_ABS_BOUNDS = prove
 (`!x k. abs(x) <= k = --k <= x /\ x <= k`,
  REAL_ARITH_TAC);;

(* ------------------------------------------------------------------------- *)
(* To simplify backchaining, just as in the natural number case.             *)
(* ------------------------------------------------------------------------- *)

let REAL_LE_IMP =
  let pth = PURE_ONCE_REWRITE_RULE
              [TAUT `(a /\ b ==> c) = (a ==> b ==> c)`] REAL_LE_TRANS in
  fun th -> GEN_ALL(MATCH_MP pth (SPEC_ALL th));;

let REAL_LET_IMP =
  let pth = PURE_ONCE_REWRITE_RULE
              [TAUT `(a /\ b ==> c) = (a ==> b ==> c)`] REAL_LET_TRANS in
  fun th -> GEN_ALL(MATCH_MP pth (SPEC_ALL th));;

(* ------------------------------------------------------------------------- *)
(* Now a bit of nonlinear stuff.                                             *)
(* ------------------------------------------------------------------------- *)

let REAL_OF_NUM_POW = prove
 (`!x n. (&x) pow n = &(x EXP n)`,
  GEN_TAC THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[real_pow; EXP; REAL_OF_NUM_MUL]);;

let REAL_ABS_MUL = prove
 (`!x y. abs(x * y) = abs(x) * abs(y)`,
  REPEAT GEN_TAC THEN
  DISJ_CASES_TAC (SPEC `x:real` REAL_LE_NEGTOTAL) THENL
   [ALL_TAC;
    GEN_REWRITE_TAC (RAND_CONV o LAND_CONV) [GSYM REAL_ABS_NEG]] THEN
  (DISJ_CASES_TAC (SPEC `y:real` REAL_LE_NEGTOTAL) THENL
    [ALL_TAC;
     GEN_REWRITE_TAC (RAND_CONV o RAND_CONV) [GSYM REAL_ABS_NEG]]) THEN
  ASSUM_LIST(MP_TAC o MATCH_MP REAL_LE_MUL o end_itlist CONJ o rev) THEN
  REWRITE_TAC[REAL_MUL_LNEG; REAL_MUL_RNEG; REAL_NEG_NEG] THEN DISCH_TAC THENL
   [ALL_TAC;
    GEN_REWRITE_TAC LAND_CONV [GSYM REAL_ABS_NEG];
    GEN_REWRITE_TAC LAND_CONV [GSYM REAL_ABS_NEG];
    ALL_TAC] THEN
  ASM_REWRITE_TAC[real_abs; REAL_MUL_LNEG; REAL_MUL_RNEG; REAL_NEG_NEG]);;

let REAL_POW_LE = prove
 (`!x n. &0 <= x ==> &0 <= (x pow n)`,
  REPEAT STRIP_TAC THEN SPEC_TAC(`n:num`,`n:num`) THEN
  INDUCT_TAC THEN REWRITE_TAC[real_pow; REAL_POS] THEN
  MATCH_MP_TAC REAL_LE_MUL THEN ASM_REWRITE_TAC[]);;

let REAL_POW_LT = prove
 (`!x n. &0 < x ==> &0 < (x pow n)`,
  REPEAT STRIP_TAC THEN SPEC_TAC(`n:num`,`n:num`) THEN
  INDUCT_TAC THEN REWRITE_TAC[real_pow; REAL_LT_01] THEN
  MATCH_MP_TAC REAL_LT_MUL THEN ASM_REWRITE_TAC[]);;

let REAL_ABS_POW = prove
 (`!x n. abs(x pow n) = abs(x) pow n`,
  GEN_TAC THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[real_pow; REAL_ABS_NUM; REAL_ABS_MUL]);;

let REAL_POW_NEG = prove
 (`!x n. (--x) pow n = if EVEN n then x pow n else --(x pow n)`,
  GEN_TAC THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[real_pow; EVEN] THEN
  ASM_CASES_TAC `EVEN n` THEN
  ASM_REWRITE_TAC[REAL_MUL_RNEG; REAL_MUL_LNEG; REAL_NEG_NEG]);;

let REAL_LE_LMUL = prove
 (`!x y z. &0 <= x /\ y <= z ==> x * y <= x * z`,
  ONCE_REWRITE_TAC[REAL_ARITH `x <= y = &0 <= y - x`] THEN
  REWRITE_TAC[GSYM REAL_SUB_LDISTRIB; REAL_SUB_RZERO; REAL_LE_MUL]);;

let REAL_LE_RMUL = prove
 (`!x y z. x <= y /\ &0 <= z ==> x * z <= y * z`,
  MESON_TAC[REAL_MUL_SYM; REAL_LE_LMUL]);;

let REAL_LT_LMUL = prove
 (`!x y z. &0 < x /\ y < z ==> x * y < x * z`,
  ONCE_REWRITE_TAC[REAL_ARITH `x < y = &0 < y - x`] THEN
  REWRITE_TAC[GSYM REAL_SUB_LDISTRIB; REAL_SUB_RZERO; REAL_LT_MUL]);;

let REAL_LT_RMUL = prove
 (`!x y z. x < y /\ &0 < z ==> x * z < y * z`,
  MESON_TAC[REAL_MUL_SYM; REAL_LT_LMUL]);;

let REAL_EQ_MUL_LCANCEL = prove
 (`!x y z. (x * y = x * z) = (x = &0) \/ (y = z)`,
  REPEAT GEN_TAC THEN
  ONCE_REWRITE_TAC[REAL_ARITH `(x = y) = (x - y = &0)`] THEN
  REWRITE_TAC[GSYM REAL_SUB_LDISTRIB; REAL_ENTIRE; REAL_SUB_RZERO]);;

let REAL_EQ_MUL_RCANCEL = prove
 (`!x y z. (x * z = y * z) = (x = y) \/ (z = &0)`,
  ONCE_REWRITE_TAC[REAL_MUL_SYM] THEN
  REWRITE_TAC[REAL_EQ_MUL_LCANCEL] THEN
  MESON_TAC[]);;

let REAL_MUL_LINV_UNIQ = prove
 (`!x y. (x * y = &1) ==> (inv(y) = x)`,
  REPEAT GEN_TAC THEN
  ASM_CASES_TAC `y = &0` THEN
  ASM_REWRITE_TAC[REAL_MUL_RZERO; REAL_OF_NUM_EQ; ARITH_EQ] THEN
  FIRST_ASSUM(SUBST1_TAC o SYM o MATCH_MP REAL_MUL_LINV) THEN
  ASM_REWRITE_TAC[REAL_EQ_MUL_RCANCEL] THEN
  DISCH_THEN(ACCEPT_TAC o SYM));;

let REAL_MUL_RINV_UNIQ = prove
 (`!x y. (x * y = &1) ==> (inv(x) = y)`,
  ONCE_REWRITE_TAC[REAL_MUL_SYM] THEN
  MATCH_ACCEPT_TAC REAL_MUL_LINV_UNIQ);;

let REAL_INV_INV = prove
 (`!x. inv(inv x) = x`,
  GEN_TAC THEN ASM_CASES_TAC `x = &0` THEN
  ASM_REWRITE_TAC[REAL_INV_0] THEN
  MATCH_MP_TAC REAL_MUL_RINV_UNIQ THEN
  MATCH_MP_TAC REAL_MUL_LINV THEN
  ASM_REWRITE_TAC[]);;

let REAL_INV_EQ_0 = prove
 (`!x. (inv(x) = &0) = (x = &0)`,
  GEN_TAC THEN EQ_TAC THEN DISCH_TAC THEN ASM_REWRITE_TAC[REAL_INV_0] THEN
  ONCE_REWRITE_TAC[GSYM REAL_INV_INV] THEN ASM_REWRITE_TAC[REAL_INV_0]);;

let REAL_LT_INV = prove
 (`!x. &0 < x ==> &0 < inv(x)`,
  GEN_TAC THEN
  REPEAT_TCL DISJ_CASES_THEN ASSUME_TAC (SPEC `inv(x)` REAL_LT_NEGTOTAL) THEN
  ASM_REWRITE_TAC[] THENL
   [RULE_ASSUM_TAC(REWRITE_RULE[REAL_INV_EQ_0]) THEN ASM_REWRITE_TAC[];
    DISCH_TAC THEN SUBGOAL_THEN `&0 < --(inv x) * x` MP_TAC THENL
     [MATCH_MP_TAC REAL_LT_MUL THEN ASM_REWRITE_TAC[];
      REWRITE_TAC[REAL_MUL_LNEG]]] THEN
  SUBGOAL_THEN `inv(x) * x = &1` SUBST1_TAC THENL
   [MATCH_MP_TAC REAL_MUL_LINV THEN
    UNDISCH_TAC `&0 < x` THEN REAL_ARITH_TAC;
    REWRITE_TAC[REAL_LT_RNEG; REAL_ADD_LID; REAL_OF_NUM_LT; ARITH]]);;

let REAL_LT_INV_EQ = prove
 (`!x. &0 < inv x = &0 < x`,
  GEN_TAC THEN EQ_TAC THEN REWRITE_TAC[REAL_LT_INV] THEN
  GEN_REWRITE_TAC (funpow 2 RAND_CONV) [GSYM REAL_INV_INV] THEN
  REWRITE_TAC[REAL_LT_INV]);;

let REAL_INV_NEG = prove
 (`!x. inv(--x) = --(inv x)`,
  GEN_TAC THEN ASM_CASES_TAC `x = &0` THEN
  ASM_REWRITE_TAC[REAL_NEG_0; REAL_INV_0] THEN
  MATCH_MP_TAC REAL_MUL_LINV_UNIQ THEN
  REWRITE_TAC[REAL_MUL_LNEG; REAL_MUL_RNEG; REAL_NEG_NEG] THEN
  MATCH_MP_TAC REAL_MUL_LINV THEN ASM_REWRITE_TAC[]);;

let REAL_LE_INV_EQ = prove
 (`!x. &0 <= inv x = &0 <= x`,
  REWRITE_TAC[REAL_LE_LT; REAL_LT_INV_EQ; REAL_INV_EQ_0] THEN
  MESON_TAC[REAL_INV_EQ_0]);;

let REAL_LE_INV = prove
 (`!x. &0 <= x ==> &0 <= inv(x)`,
  REWRITE_TAC[REAL_LE_INV_EQ]);;

let REAL_MUL_RINV = prove
 (`!x. ~(x = &0) ==> (x * inv(x) = &1)`,
  ONCE_REWRITE_TAC[REAL_MUL_SYM] THEN
  REWRITE_TAC[REAL_MUL_LINV]);;

let REAL_INV_1 = prove
 (`inv(&1) = &1`,
  MATCH_MP_TAC REAL_MUL_RINV_UNIQ THEN
  REWRITE_TAC[REAL_MUL_LID]);;

let REAL_DIV_1 = prove
 (`!x. x / &1 = x`,
  REWRITE_TAC[real_div; REAL_INV_1; REAL_MUL_RID]);;

let REAL_ABS_INV = prove
 (`!x. abs(inv x) = inv(abs x)`,
  GEN_TAC THEN CONV_TAC SYM_CONV THEN
  ASM_CASES_TAC `x = &0` THEN ASM_REWRITE_TAC[REAL_INV_0; REAL_ABS_0] THEN
  MATCH_MP_TAC REAL_MUL_RINV_UNIQ THEN
  REWRITE_TAC[GSYM REAL_ABS_MUL] THEN
  POP_ASSUM(SUBST1_TAC o MATCH_MP REAL_MUL_RINV) THEN
  REWRITE_TAC[REAL_ABS_1]);;

let REAL_ABS_DIV = prove
 (`!x y. abs(x / y) = abs(x) / abs(y)`,
  REWRITE_TAC[real_div; REAL_ABS_INV; REAL_ABS_MUL]);;

let REAL_INV_MUL = prove
 (`!x y. inv(x * y) = inv(x) * inv(y)`,
  REPEAT GEN_TAC THEN
  MAP_EVERY ASM_CASES_TAC [`x = &0`; `y = &0`] THEN
  ASM_REWRITE_TAC[REAL_INV_0; REAL_MUL_LZERO; REAL_MUL_RZERO] THEN
  MATCH_MP_TAC REAL_MUL_LINV_UNIQ THEN
  ONCE_REWRITE_TAC[AC REAL_MUL_AC `(a * b) * (c * d) = (a * c) * (b * d)`] THEN
  EVERY_ASSUM(SUBST1_TAC o MATCH_MP REAL_MUL_LINV) THEN
  REWRITE_TAC[REAL_MUL_LID]);;

let REAL_INV_DIV = prove
 (`!x y. inv(x / y) = y / x`,
  REWRITE_TAC[real_div; REAL_INV_INV; REAL_INV_MUL] THEN
  MATCH_ACCEPT_TAC REAL_MUL_SYM);;

let REAL_POW_MUL = prove
 (`!x y n. (x * y) pow n = (x pow n) * (y pow n)`,
  GEN_TAC THEN GEN_TAC THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[real_pow; REAL_MUL_LID; REAL_MUL_AC]);;

let REAL_POW_INV = prove
 (`!x n. (inv x) pow n = inv(x pow n)`,
  GEN_TAC THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[real_pow; REAL_INV_1; REAL_INV_MUL]);;

let REAL_POW_DIV = prove
 (`!x y n. (x / y) pow n = (x pow n) / (y pow n)`,
  REWRITE_TAC[real_div; REAL_POW_MUL; REAL_POW_INV]);;

let REAL_POW_ADD = prove
 (`!x m n. x pow (m + n) = x pow m * x pow n`,
  GEN_TAC THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[ADD_CLAUSES; real_pow; REAL_MUL_LID; REAL_MUL_ASSOC]);;

let REAL_LT_IMP_NZ = prove
 (`!x. &0 < x ==> ~(x = &0)`,
  REAL_ARITH_TAC);;

let REAL_LT_REFL = prove
 (`!x. ~(x < x)`,
  REAL_ARITH_TAC);;

let REAL_LT_LCANCEL_IMP = prove
 (`!x y z. &0 < x /\ x * y < x * z ==> y < z`,
  REPEAT GEN_TAC THEN
  DISCH_THEN(fun th -> ASSUME_TAC(CONJUNCT1 th) THEN MP_TAC th) THEN DISCH_THEN
   (MP_TAC o uncurry CONJ o (MATCH_MP REAL_LT_INV F_F I) o CONJ_PAIR) THEN
  DISCH_THEN(MP_TAC o MATCH_MP REAL_LT_LMUL) THEN
  POP_ASSUM(ASSUME_TAC o MATCH_MP REAL_MUL_LINV o MATCH_MP REAL_LT_IMP_NZ) THEN
  ASM_REWRITE_TAC[REAL_MUL_ASSOC; REAL_MUL_LID]);;

let REAL_LT_RCANCEL_IMP = prove
 (`!x y z. &0 < z /\ x * z < y * z ==> x < y`,
  ONCE_REWRITE_TAC[REAL_MUL_SYM] THEN REWRITE_TAC[REAL_LT_LCANCEL_IMP]);;

let REAL_LE_LCANCEL_IMP = prove
 (`!x y z. &0 < x /\ x * y <= x * z ==> y <= z`,
  REPEAT GEN_TAC THEN REWRITE_TAC[REAL_LE_LT; REAL_EQ_MUL_LCANCEL] THEN
  ASM_CASES_TAC `x = &0` THEN ASM_REWRITE_TAC[REAL_LT_REFL] THEN
  STRIP_TAC THEN ASM_REWRITE_TAC[] THEN DISJ1_TAC THEN
  MATCH_MP_TAC REAL_LT_LCANCEL_IMP THEN
  EXISTS_TAC `x:real` THEN ASM_REWRITE_TAC[]);;

let REAL_LE_RCANCEL_IMP = prove
 (`!x y z. &0 < z /\ x * z <= y * z ==> x <= y`,
  ONCE_REWRITE_TAC[REAL_MUL_SYM] THEN REWRITE_TAC[REAL_LE_LCANCEL_IMP]);;

let REAL_MUL_2 = prove
 (`!x. &2 * x = x + x`,
  REAL_ARITH_TAC);;

let REAL_POW_EQ_0 = prove
 (`!x n. (x pow n = &0) = (x = &0) /\ ~(n = 0)`,
  GEN_TAC THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[NOT_SUC; real_pow; REAL_ENTIRE] THENL
   [REAL_ARITH_TAC;
    CONV_TAC TAUT]);;

let REAL_LE_MUL2 = prove
 (`!w x y z. &0 <= w /\ w <= x /\ &0 <= y /\ y <= z
             ==> w * y <= x * z`,
  REPEAT STRIP_TAC THEN MATCH_MP_TAC REAL_LE_TRANS THEN
  EXISTS_TAC `w * z` THEN CONJ_TAC THENL
   [MATCH_MP_TAC REAL_LE_LMUL; MATCH_MP_TAC REAL_LE_RMUL] THEN
  ASM_REWRITE_TAC[] THEN
  MATCH_MP_TAC REAL_LE_TRANS THEN EXISTS_TAC `y:real` THEN
  ASM_REWRITE_TAC[]);;

let REAL_LT_MUL2 = prove
 (`!w x y z. &0 <= w /\ w < x /\ &0 <= y /\ y < z
             ==> w * y < x * z`,
  REPEAT STRIP_TAC THEN MATCH_MP_TAC REAL_LET_TRANS THEN
  EXISTS_TAC `w * z` THEN CONJ_TAC THENL
   [MATCH_MP_TAC REAL_LE_LMUL; MATCH_MP_TAC REAL_LT_RMUL] THEN
  ASM_REWRITE_TAC[] THENL
   [MATCH_MP_TAC REAL_LT_IMP_LE THEN ASM_REWRITE_TAC[];
    MATCH_MP_TAC REAL_LET_TRANS THEN EXISTS_TAC `y:real` THEN
    ASM_REWRITE_TAC[]]);;

let REAL_LE_SQUARE = prove
 (`!x. &0 <= x * x`,
  GEN_TAC THEN DISJ_CASES_TAC(SPEC `x:real` REAL_LE_NEGTOTAL) THEN
  POP_ASSUM(fun th -> MP_TAC(MATCH_MP REAL_LE_MUL (CONJ th th))) THEN
  REWRITE_TAC[REAL_MUL_LNEG; REAL_MUL_RNEG; REAL_NEG_NEG]);;

let REAL_LT_SQUARE = prove
 (`!x. (&0 < x * x) = ~(x = &0)`,
  GEN_TAC THEN REWRITE_TAC[REAL_LT_LE; REAL_LE_SQUARE] THEN
  GEN_REWRITE_TAC (LAND_CONV o RAND_CONV) [EQ_SYM_EQ] THEN
  REWRITE_TAC[REAL_ENTIRE]);;

let REAL_INV_LE_1 = prove
 (`!x. &1 <= x ==> inv(x) <= &1`,
  GEN_TAC THEN DISCH_TAC THEN
  MATCH_MP_TAC REAL_LE_RCANCEL_IMP THEN
  EXISTS_TAC `x:real` THEN
  SUBGOAL_THEN `&0 < x` ASSUME_TAC THENL
   [POP_ASSUM MP_TAC THEN REAL_ARITH_TAC;
    SUBGOAL_THEN `inv(x) * x = &1` ASSUME_TAC THENL
     [MATCH_MP_TAC REAL_MUL_LINV THEN
      UNDISCH_TAC `&0 < x` THEN REAL_ARITH_TAC;
      ASM_REWRITE_TAC[REAL_MUL_LID]]]);;

let REAL_POW_LE_1 = prove
 (`!n x. &1 <= x ==> &1 <= x pow n`,
  INDUCT_TAC THEN REPEAT STRIP_TAC THEN
  ASM_REWRITE_TAC[real_pow; REAL_LE_REFL] THEN
  GEN_REWRITE_TAC LAND_CONV [GSYM REAL_MUL_LID] THEN
  MATCH_MP_TAC REAL_LE_MUL2 THEN
  ASM_REWRITE_TAC[REAL_LE_01] THEN
  FIRST_ASSUM MATCH_MP_TAC THEN ASM_REWRITE_TAC[]);;

let REAL_POW_1_LE = prove
 (`!n x. &0 <= x /\ x <= &1 ==> x pow n <= &1`,
  INDUCT_TAC THEN REPEAT STRIP_TAC THEN
  ASM_REWRITE_TAC[real_pow; REAL_LE_REFL] THEN
  GEN_REWRITE_TAC RAND_CONV [GSYM REAL_MUL_LID] THEN
  MATCH_MP_TAC REAL_LE_MUL2 THEN
  ASM_REWRITE_TAC[] THEN CONJ_TAC THENL
   [MATCH_MP_TAC REAL_POW_LE THEN ASM_REWRITE_TAC[];
    FIRST_ASSUM MATCH_MP_TAC THEN ASM_REWRITE_TAC[]]);;

let REAL_POW_1 = prove
 (`!x. x pow 1 = x`,
  REWRITE_TAC[num_CONV `1`] THEN
  REWRITE_TAC[real_pow; REAL_MUL_RID]);;

let REAL_POW_2 = prove
 (`!x. x pow 2 = x * x`,
  REWRITE_TAC[num_CONV `2`] THEN
  REWRITE_TAC[real_pow; REAL_POW_1]);;

let REAL_POW_ONE = prove
 (`!n. &1 pow n = &1`,
  INDUCT_TAC THEN ASM_REWRITE_TAC[real_pow; REAL_MUL_LID]);;

let REAL_LT_INV2 = prove
 (`!x y. &0 < x /\ x < y ==> inv(y) < inv(x)`,
  REPEAT STRIP_TAC THEN MATCH_MP_TAC REAL_LT_RCANCEL_IMP THEN
  EXISTS_TAC `x * y` THEN CONJ_TAC THENL
   [MATCH_MP_TAC REAL_LT_MUL THEN
    POP_ASSUM_LIST(MP_TAC o end_itlist CONJ) THEN REAL_ARITH_TAC;
    SUBGOAL_THEN `(inv x * x = &1) /\ (inv y * y = &1)` ASSUME_TAC THENL
     [CONJ_TAC THEN MATCH_MP_TAC REAL_MUL_LINV THEN
      POP_ASSUM_LIST(MP_TAC o end_itlist CONJ) THEN REAL_ARITH_TAC;
      ASM_REWRITE_TAC[REAL_MUL_ASSOC; REAL_MUL_LID] THEN
      GEN_REWRITE_TAC (LAND_CONV o LAND_CONV) [REAL_MUL_SYM] THEN
      ASM_REWRITE_TAC[GSYM REAL_MUL_ASSOC; REAL_MUL_RID]]]);;

let REAL_LE_INV2 = prove
 (`!x y. &0 < x /\ x <= y ==> inv(y) <= inv(x)`,
  REPEAT GEN_TAC THEN REWRITE_TAC[REAL_LE_LT] THEN
  ASM_CASES_TAC `x:real = y` THEN ASM_REWRITE_TAC[] THEN
  STRIP_TAC THEN DISJ1_TAC THEN MATCH_MP_TAC REAL_LT_INV2 THEN
  ASM_REWRITE_TAC[]);;

let REAL_DOWN = prove
 (`!d. &0 < d ==> ?e. &0 < e /\ e < d`,
  GEN_TAC THEN DISCH_TAC THEN EXISTS_TAC `d / &2` THEN
  ASSUME_TAC(REAL_ARITH `&0 < &2`) THEN
  ASSUME_TAC(MATCH_MP REAL_MUL_LINV (REAL_ARITH `~(&2 = &0)`)) THEN
  CONJ_TAC THEN MATCH_MP_TAC REAL_LT_RCANCEL_IMP THEN EXISTS_TAC `&2` THEN
  ASM_REWRITE_TAC[real_div; GSYM REAL_MUL_ASSOC; REAL_MUL_RID] THEN
  UNDISCH_TAC `&0 < d` THEN REAL_ARITH_TAC);;

let REAL_DOWN2 = prove
 (`!d1 d2. &0 < d1 /\ &0 < d2 ==> ?e. &0 < e /\ e < d1 /\ e < d2`,
  REPEAT GEN_TAC THEN STRIP_TAC THEN
  DISJ_CASES_TAC(SPECL [`d1:real`; `d2:real`] REAL_LE_TOTAL) THENL
   [MP_TAC(SPEC `d1:real` REAL_DOWN);
    MP_TAC(SPEC `d2:real` REAL_DOWN)] THEN
  ASM_REWRITE_TAC[] THEN
  DISCH_THEN(X_CHOOSE_THEN `e:real` STRIP_ASSUME_TAC) THEN
  EXISTS_TAC `e:real` THEN
  POP_ASSUM_LIST(MP_TAC o end_itlist CONJ) THEN
  REAL_ARITH_TAC);;

let REAL_POW_LE2 = prove
 (`!n x y. &0 <= x /\ x <= y ==> x pow n <= y pow n`,
  INDUCT_TAC THEN REWRITE_TAC[real_pow; REAL_LE_REFL] THEN
  REPEAT STRIP_TAC THEN MATCH_MP_TAC REAL_LE_MUL2 THEN
  ASM_REWRITE_TAC[] THEN CONJ_TAC THENL
   [MATCH_MP_TAC REAL_POW_LE THEN ASM_REWRITE_TAC[];
    FIRST_ASSUM MATCH_MP_TAC THEN ASM_REWRITE_TAC[]]);;

let REAL_POW_MONO = prove
 (`!m n x. &1 <= x /\ m <= n ==> x pow m <= x pow n`,
  REPEAT GEN_TAC THEN REWRITE_TAC[LE_EXISTS] THEN
  DISCH_THEN(CONJUNCTS_THEN2 ASSUME_TAC MP_TAC) THEN
  DISCH_THEN(X_CHOOSE_THEN `d:num` SUBST1_TAC) THEN
  REWRITE_TAC[REAL_POW_ADD] THEN
  GEN_REWRITE_TAC LAND_CONV [GSYM REAL_MUL_RID] THEN
  MATCH_MP_TAC REAL_LE_LMUL THEN CONJ_TAC THENL
   [MATCH_MP_TAC REAL_LE_TRANS THEN EXISTS_TAC `&1` THEN
    REWRITE_TAC[REAL_OF_NUM_LE; ARITH] THEN
    MATCH_MP_TAC REAL_POW_LE_1 THEN ASM_REWRITE_TAC[];
    MATCH_MP_TAC REAL_POW_LE_1 THEN ASM_REWRITE_TAC[]]);;

(* ------------------------------------------------------------------------- *)
(* Remainder of the integer arithmetic operations.                           *)
(* ------------------------------------------------------------------------- *)

let REAL_INT_ABS_CONV =
  let pth = prove
   (`(abs(--(&x)) = &x) /\
     (abs(&x) = &x)`,
    REWRITE_TAC[REAL_ABS_NEG; REAL_ABS_NUM]) in
  GEN_REWRITE_CONV I [pth];;

let REAL_INT_POW_CONV =
  let n = `n:num` and x = `x:num` in
  let pth1,pth2 = (CONJ_PAIR o prove)
   (`(&x pow n = &(x EXP n)) /\
     ((--(&x)) pow n = if EVEN n then &(x EXP n) else --(&(x EXP n)))`,
    REWRITE_TAC[REAL_OF_NUM_POW; REAL_POW_NEG]) in
  let tth = prove
   (`((if T then x:real else y) = x) /\ ((if F then x:real else y) = y)`,
    REWRITE_TAC[]) in
  let neg_tm = `--` in
  (GEN_REWRITE_CONV I [pth1] THENC RAND_CONV NUM_EXP_CONV) ORELSEC
  (GEN_REWRITE_CONV I [pth2] THENC
   RATOR_CONV(RATOR_CONV(RAND_CONV NUM_EVEN_CONV)) THENC
   GEN_REWRITE_CONV I [tth] THENC
   (fun tm -> if rator tm = neg_tm then RAND_CONV(RAND_CONV NUM_EXP_CONV) tm
              else RAND_CONV NUM_EXP_CONV  tm));;

let REAL_INT_RED_CONV = FIRST_CONV
  [REAL_INT_LE_CONV; REAL_INT_LT_CONV; REAL_INT_GE_CONV;
   REAL_INT_GT_CONV; REAL_INT_EQ_CONV;
   REAL_INT_NEG_CONV; REAL_INT_ABS_CONV;
   REAL_INT_ADD_CONV; REAL_INT_SUB_CONV;
   REAL_INT_MUL_CONV; REAL_INT_POW_CONV];;

let REAL_INT_REDUCE_CONV =
  let conv = GEN_REWRITE_CONV I (basic_rewrites()) in
  DEPTH_CONV (conv ORELSEC REAL_INT_RED_CONV);;

(* ------------------------------------------------------------------------- *)
(* Stuff about sums.                                                         *)
(* ------------------------------------------------------------------------- *)

let SUM_TWO = prove
 (`!f n p. Sum(0,n) f + Sum(n,p) f = Sum(0,n + p) f`,
  GEN_TAC THEN GEN_TAC THEN INDUCT_TAC THEN
  REWRITE_TAC[Sum; REAL_ADD_RID; ADD_CLAUSES] THEN
  ASM_REWRITE_TAC[REAL_ADD_ASSOC]);;

let SUM_DIFF = prove
 (`!f m n. Sum(m,n) f = Sum(0,m + n) f - Sum(0,m) f`,
  REPEAT GEN_TAC THEN REWRITE_TAC[REAL_EQ_SUB_LADD] THEN
  ONCE_REWRITE_TAC[REAL_ADD_SYM] THEN MATCH_ACCEPT_TAC SUM_TWO);;

let ABS_SUM = prove
 (`!f m n. abs(Sum(m,n) f) <= Sum(m,n) (\n. abs(f n))`,
  GEN_TAC THEN GEN_TAC THEN INDUCT_TAC THEN
  REWRITE_TAC[Sum; REAL_ABS_0; REAL_LE_REFL] THEN BETA_TAC THEN
  MATCH_MP_TAC REAL_LE_TRANS THEN
  EXISTS_TAC `abs(Sum(m,n) f) + abs(f(m + n))` THEN
  ASM_REWRITE_TAC[REAL_ABS_TRIANGLE; REAL_LE_RADD]);;

let SUM_LE = prove
 (`!f g m n. (!r. m <= r /\ r < n + m ==> f(r) <= g(r))
        ==> (Sum(m,n) f <= Sum(m,n) g)`,
  EVERY(replicate GEN_TAC 3) THEN
  INDUCT_TAC THEN REWRITE_TAC[Sum; REAL_LE_REFL] THEN
  DISCH_TAC THEN MATCH_MP_TAC REAL_LE_ADD2 THEN CONJ_TAC THEN
  FIRST_ASSUM MATCH_MP_TAC THENL
   [GEN_TAC THEN DISCH_TAC THEN FIRST_ASSUM MATCH_MP_TAC THEN
    ASM_REWRITE_TAC[ADD_CLAUSES] THEN
    MATCH_MP_TAC LT_TRANS THEN EXISTS_TAC `(n:num) + m`;
    GEN_REWRITE_TAC (RAND_CONV o RAND_CONV) [ADD_SYM]] THEN
  ASM_REWRITE_TAC[ADD_CLAUSES; LE_ADD; LT]);;

let SUM_EQ = prove
 (`!f g m n. (!r. m <= r /\ r < (n + m) ==> (f(r) = g(r)))
        ==> (Sum(m,n) f = Sum(m,n) g)`,
  REPEAT GEN_TAC THEN DISCH_TAC THEN REWRITE_TAC[GSYM REAL_LE_ANTISYM] THEN
  CONJ_TAC THEN MATCH_MP_TAC SUM_LE THEN GEN_TAC THEN
  DISCH_THEN(fun th -> MATCH_MP_TAC REAL_EQ_IMP_LE THEN
    FIRST_ASSUM(SUBST1_TAC o C MATCH_MP th)) THEN REFL_TAC);;

let SUM_POS = prove
 (`!f. (!n. &0 <= f(n)) ==> !m n. &0 <= Sum(m,n) f`,
  GEN_TAC THEN DISCH_TAC THEN GEN_TAC THEN INDUCT_TAC THEN
  REWRITE_TAC[Sum; REAL_LE_REFL] THEN
  MATCH_MP_TAC REAL_LE_ADD THEN ASM_REWRITE_TAC[]);;

let SUM_POS_GEN = prove
 (`!f m. (!n. m <= n ==> &0 <= f(n)) ==>
        !n. &0 <= Sum(m,n) f`,
  REPEAT GEN_TAC THEN DISCH_TAC THEN INDUCT_TAC THEN
  REWRITE_TAC[Sum; REAL_LE_REFL] THEN
  MATCH_MP_TAC REAL_LE_ADD THEN
  ASM_REWRITE_TAC[] THEN FIRST_ASSUM MATCH_MP_TAC THEN
  MATCH_ACCEPT_TAC LE_ADD);;

let SUM_ABS = prove
 (`!f m n. abs(Sum(m,n) (\m. abs(f m))) = Sum(m,n) (\m. abs(f m))`,
  GEN_TAC THEN GEN_TAC THEN REWRITE_TAC[REAL_ABS_REFL] THEN
  SPEC_TAC(`m:num`,`m:num`) THEN MATCH_MP_TAC SUM_POS THEN BETA_TAC THEN
  REWRITE_TAC[REAL_ABS_POS]);;

let SUM_ABS_LE = prove
 (`!f m n. abs(Sum(m,n) f) <= Sum(m,n)(\n. abs(f n))`,
  GEN_TAC THEN GEN_TAC THEN INDUCT_TAC THEN
  REWRITE_TAC[Sum; REAL_ABS_0; REAL_LE_REFL] THEN
  MATCH_MP_TAC REAL_LE_TRANS THEN
  EXISTS_TAC `abs(Sum(m,n) f) + abs(f(m + n))` THEN
  REWRITE_TAC[REAL_ABS_TRIANGLE] THEN BETA_TAC THEN
  ASM_REWRITE_TAC[REAL_LE_RADD]);;

let SUM_ZERO = prove
 (`!f N. (!n. n >= N ==> (f(n) = &0)) ==>
         (!m n. m >= N ==> (Sum(m,n) f = &0))`,
  REPEAT GEN_TAC THEN DISCH_TAC THEN
  MAP_EVERY X_GEN_TAC [`m:num`; `n:num`] THEN REWRITE_TAC[GE] THEN
  DISCH_THEN(X_CHOOSE_THEN `d:num` SUBST1_TAC o REWRITE_RULE[LE_EXISTS]) THEN
  SPEC_TAC(`n:num`,`n:num`) THEN INDUCT_TAC THEN REWRITE_TAC[Sum] THEN
  ASM_REWRITE_TAC[REAL_ADD_LID] THEN FIRST_ASSUM MATCH_MP_TAC THEN
  REWRITE_TAC[GE; GSYM ADD_ASSOC; LE_ADD]);;

let SUM_ADD = prove
 (`!f g m n. Sum(m,n) (\n. f(n) + g(n)) = Sum(m,n) f + Sum(m,n) g`,
  EVERY(replicate GEN_TAC 3) THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[Sum; REAL_ADD_LID; REAL_ADD_AC]);;

let SUM_CMUL = prove
 (`!f c m n. Sum(m,n) (\n. c * f(n)) = c * Sum(m,n) f`,
  EVERY(replicate GEN_TAC 3) THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[Sum; REAL_MUL_RZERO] THEN BETA_TAC THEN
  REWRITE_TAC[REAL_ADD_LDISTRIB]);;

let SUM_NEG = prove
 (`!f n d. Sum(n,d) (\n. --(f n)) = --(Sum(n,d) f)`,
  GEN_TAC THEN GEN_TAC THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[Sum; REAL_NEG_0] THEN
  BETA_TAC THEN REWRITE_TAC[REAL_NEG_ADD]);;

let SUM_SUB = prove
 (`!f g m n. Sum(m,n)(\n. (f n) - (g n)) = Sum(m,n) f - Sum(m,n) g`,
  REPEAT GEN_TAC THEN REWRITE_TAC[real_sub; GSYM SUM_NEG; GSYM SUM_ADD]);;

let SUM_SUBST = prove
 (`!f g m n. (!p. m <= p /\ p < (m + n) ==> (f p = g p))
        ==> (Sum(m,n) f = Sum(m,n) g)`,
  EVERY (replicate GEN_TAC 3) THEN INDUCT_TAC THEN REWRITE_TAC[Sum] THEN
  ASM_REWRITE_TAC[] THEN DISCH_TAC THEN BINOP_TAC THEN
  FIRST_ASSUM MATCH_MP_TAC THENL
   [GEN_TAC THEN DISCH_TAC THEN FIRST_ASSUM MATCH_MP_TAC THEN
    ASM_REWRITE_TAC[ADD_CLAUSES; LT_SUC_LE] THEN
    MATCH_MP_TAC LT_IMP_LE THEN ASM_REWRITE_TAC[];
    REWRITE_TAC[LE_ADD] THEN ONCE_REWRITE_TAC[ADD_SYM] THEN
    REWRITE_TAC[LT_SUC_LE; LE_REFL; ADD_CLAUSES]]);;

let SUM_NSUB = prove
 (`!n f c. Sum(0,n) f - (&n * c) = Sum(0,n)(\p. f(p) - c)`,
  INDUCT_TAC THEN REWRITE_TAC[Sum; REAL_MUL_LZERO; REAL_SUB_REFL] THEN
  REWRITE_TAC[ADD_CLAUSES; GSYM REAL_OF_NUM_SUC; REAL_ADD_RDISTRIB] THEN
  REPEAT GEN_TAC THEN POP_ASSUM(fun th -> REWRITE_TAC[GSYM th]) THEN
  REWRITE_TAC[real_sub; REAL_NEG_ADD; REAL_MUL_LID; REAL_ADD_AC]);;

let SUM_BOUND = prove
 (`!f K m n. (!p. m <= p /\ p < (m + n) ==> (f(p) <= K))
        ==> (Sum(m,n) f <= (&n * K))`,
  EVERY (replicate GEN_TAC 3) THEN INDUCT_TAC THEN
  REWRITE_TAC[Sum; REAL_MUL_LZERO; REAL_LE_REFL] THEN
  DISCH_TAC THEN REWRITE_TAC[GSYM REAL_OF_NUM_SUC; REAL_ADD_RDISTRIB] THEN
  MATCH_MP_TAC REAL_LE_ADD2 THEN CONJ_TAC THENL
   [FIRST_ASSUM MATCH_MP_TAC THEN GEN_TAC THEN DISCH_TAC THEN
    FIRST_ASSUM MATCH_MP_TAC THEN ASM_REWRITE_TAC[] THEN
    REWRITE_TAC[ADD_CLAUSES; LT_SUC_LE; LE_REFL] THEN
    MATCH_MP_TAC LT_IMP_LE THEN ASM_REWRITE_TAC[];
    REWRITE_TAC[REAL_MUL_LID] THEN FIRST_ASSUM MATCH_MP_TAC THEN
    REWRITE_TAC[ADD_CLAUSES; LE_ADD; LT_SUC_LE; LE_REFL]]);;

let SUM_GROUP = prove
 (`!n k f. Sum(0,n)(\m. Sum(m * k,k) f) = Sum(0,n * k) f`,
  INDUCT_TAC THEN REWRITE_TAC[Sum; MULT_CLAUSES] THEN
  REPEAT GEN_TAC THEN BETA_TAC THEN ASM_REWRITE_TAC[] THEN
  REWRITE_TAC[ADD_CLAUSES; SUM_TWO]);;

let SUM_1 = prove
 (`!f n. Sum(n,1) f = f(n)`,
  REPEAT GEN_TAC THEN
  REWRITE_TAC[num_CONV `1`; Sum; ADD_CLAUSES; REAL_ADD_LID]);;

let SUM_2 = prove
 (`!f n. Sum(n,2) f = f(n) + f(n + 1)`,
  REPEAT GEN_TAC THEN CONV_TAC(REDEPTH_CONV num_CONV) THEN
  REWRITE_TAC[Sum; ADD_CLAUSES; REAL_ADD_LID]);;

let SUM_OFFSET = prove
 (`!f n k. Sum(0,n)(\m. f(m + k)) = Sum(0,n + k) f - Sum(0,k) f`,
  REPEAT GEN_TAC THEN
  GEN_REWRITE_TAC (RAND_CONV o ONCE_DEPTH_CONV) [ADD_SYM] THEN
  REWRITE_TAC[GSYM SUM_TWO; REAL_ADD_SUB] THEN
  SPEC_TAC(`n:num`,`n:num`) THEN INDUCT_TAC THEN REWRITE_TAC[Sum] THEN
  BETA_TAC THEN ASM_REWRITE_TAC[ADD_CLAUSES] THEN AP_TERM_TAC THEN
  AP_TERM_TAC THEN MATCH_ACCEPT_TAC ADD_SYM);;

let SUM_REINDEX = prove
 (`!f m k n. Sum(m + k,n) f = Sum(m,n)(\r. f(r + k))`,
  EVERY(replicate GEN_TAC 3) THEN INDUCT_TAC THEN REWRITE_TAC[Sum] THEN
  ASM_REWRITE_TAC[REAL_EQ_ADD_LCANCEL] THEN REWRITE_TAC[ADD_AC]);;

let SUM_0 = prove
 (`!m n. Sum(m,n)(\r. &0) = &0`,
  GEN_TAC THEN INDUCT_TAC THEN REWRITE_TAC[Sum] THEN
  BETA_TAC THEN ASM_REWRITE_TAC[REAL_ADD_LID]);;

let SUM_CANCEL = prove
 (`!f n d. Sum(n,d) (\n. f(SUC n) - f(n)) = f(n + d) - f(n)`,
  GEN_TAC THEN GEN_TAC THEN INDUCT_TAC THEN
  ASM_REWRITE_TAC[Sum; ADD_CLAUSES; REAL_SUB_REFL] THEN
  BETA_TAC THEN ONCE_REWRITE_TAC[REAL_ADD_SYM] THEN
  REWRITE_TAC[real_sub; REAL_NEG_ADD; REAL_ADD_ASSOC] THEN
  AP_THM_TAC THEN AP_TERM_TAC THEN
  REWRITE_TAC[GSYM REAL_ADD_ASSOC; REAL_ADD_LINV; REAL_ADD_RID]);;
