(* ========================================================================= *)
(* Syntax sugaring; theory of pairing, with a bit of support.                *)
(* ========================================================================= *)

let LET_DEF = new_definition
 `LET (f:A->B) x = f x`;;

let LET_END_DEF = new_definition
 `LET_END (t:A) = t`;;

let GABS_DEF = new_definition
 `GABS (P:A->bool) = $@ P`;;

let GEQ_DEF = new_definition
 `GEQ a b = (a:A = b)`;;

(* ------------------------------------------------------------------------- *)
(* Pair type.                                                                *)
(* ------------------------------------------------------------------------- *)

let mk_pair_def = new_definition
  `mk_pair (x:A) (y:B) = \a b. (a = x) /\ (b = y)`;;

let PAIR_EXISTS_THM = prove
 (`?x. ?(a:A) (b:B). x = mk_pair a b`,
  MESON_TAC[]);;

let prod_tybij = new_type_definition
  "prod" ("ABS_prod","REP_prod") PAIR_EXISTS_THM;;

let REP_ABS_PAIR = prove
 (`!(x:A) (y:B). REP_prod (ABS_prod (mk_pair x y)) = mk_pair x y`,
  MESON_TAC[prod_tybij]);;

parse_as_infix (",",(14,"right"));;

let COMMA_DEF = new_definition
 `(x:A),(y:B) = ABS_prod(mk_pair x y)`;;

let FST_DEF = new_definition
 `FST (p:A#B) = @x. ?y. p = x,y`;;

let SND_DEF = new_definition
 `SND (p:A#B) = @y. ?x. p = x,y`;;

let PAIR_EQ = prove
 (`!(x:A) (y:B) a b. (x,y = a,b) = (x = a) /\ (y = b)`,
  REPEAT GEN_TAC THEN EQ_TAC THENL
   [REWRITE_TAC[COMMA_DEF] THEN
    DISCH_THEN(MP_TAC o AP_TERM `REP_prod:A#B->A->B->bool`) THEN
    REWRITE_TAC[REP_ABS_PAIR] THEN REWRITE_TAC[mk_pair_def; FUN_EQ_THM];
    ALL_TAC] THEN
  MESON_TAC[]);;

let PAIR_SURJECTIVE = prove
 (`!p:A#B. ?x y. p = x,y`,
  GEN_TAC THEN REWRITE_TAC[COMMA_DEF] THEN
  MP_TAC(SPEC `REP_prod p :A->B->bool` (CONJUNCT2 prod_tybij)) THEN
  REWRITE_TAC[CONJUNCT1 prod_tybij] THEN
  DISCH_THEN(X_CHOOSE_THEN `a:A` (X_CHOOSE_THEN `b:B` MP_TAC)) THEN
  DISCH_THEN(MP_TAC o AP_TERM `ABS_prod:(A->B->bool)->A#B`) THEN
  REWRITE_TAC[CONJUNCT1 prod_tybij] THEN DISCH_THEN SUBST1_TAC THEN
  MAP_EVERY EXISTS_TAC [`a:A`; `b:B`] THEN REFL_TAC);;

let FST = prove
 (`!(x:A) (y:B). FST(x,y) = x`,
  REPEAT GEN_TAC THEN REWRITE_TAC[FST_DEF] THEN
  MATCH_MP_TAC SELECT_UNIQUE THEN GEN_TAC THEN BETA_TAC THEN
  REWRITE_TAC[PAIR_EQ] THEN EQ_TAC THEN
  STRIP_TAC THEN ASM_REWRITE_TAC[] THEN
  EXISTS_TAC `y:B` THEN ASM_REWRITE_TAC[]);;

let SND = prove
 (`!(x:A) (y:B). SND(x,y) = y`,
  REPEAT GEN_TAC THEN REWRITE_TAC[SND_DEF] THEN
  MATCH_MP_TAC SELECT_UNIQUE THEN GEN_TAC THEN BETA_TAC THEN
  REWRITE_TAC[PAIR_EQ] THEN EQ_TAC THEN
  STRIP_TAC THEN ASM_REWRITE_TAC[] THEN
  EXISTS_TAC `x:A` THEN ASM_REWRITE_TAC[]);;

let PAIR = prove
 (`!x:A#B. FST x,SND x = x`,
  GEN_TAC THEN
  (X_CHOOSE_THEN `a:A` (X_CHOOSE_THEN `b:B` SUBST1_TAC)
     (SPEC `x:A#B` PAIR_SURJECTIVE)) THEN
  REWRITE_TAC[FST; SND]);;

(* ------------------------------------------------------------------------- *)
(* Syntax operations.                                                        *)
(* ------------------------------------------------------------------------- *)

let is_pair = is_binary ",";;

let dest_pair = dest_binary ",";;

let mk_pair(l,r) =
  let ptm =
    mk_const(",",[type_of l,aty; type_of r,bty]) in
  mk_comb(mk_comb(ptm,l),r);;

(* ------------------------------------------------------------------------- *)
(* Extend basic rewrites; extend new_definition to allow paired varstructs.  *)
(* ------------------------------------------------------------------------- *)

extend_basic_rewrites [FST; SND; PAIR];;

let new_definition =
  let depair =
    let rec depair gv arg =
      try let l,r = dest_pair arg in
          (depair (list_mk_icomb "FST" [gv]) l) @
          (depair (list_mk_icomb "SND" [gv]) r)
      with Failure _ -> [gv,arg] in
    fun arg -> let gv = genvar(type_of arg) in
               gv,depair gv arg in
  fun tm ->
    let avs,def = strip_forall tm in
    let l,r = dest_eq(snd(strip_forall def)) in
    let fn,args = strip_comb l in
    let gargs,reps = (I F_F Union) (unzip(map depair args)) in
    let l' = list_mk_comb(fn,gargs) and r' = subst reps r in
    let th1 = new_definition (mk_eq(l',r')) in
    let slist = zip args gargs in
    let th2 = INST slist (SPEC_ALL th1) in
    let xreps = map (subst slist o fst) reps in
    let threps = map (SYM o PURE_REWRITE_CONV[FST; SND]) xreps in
    let th3 = TRANS th2 (SYM(SUBS_CONV threps r)) in
    GEN_ALL (GENL avs th3);;

(* ------------------------------------------------------------------------- *)
(* A few more useful definitions.                                            *)
(* ------------------------------------------------------------------------- *)

let CURRY_DEF = new_definition
 `CURRY(f:A#B->C) x y = f(x,y)`;;

let UNCURRY_DEF = new_definition
 `!f x y. UNCURRY(f:A->B->C)(x,y) = f x y`;;

let PASSOC_DEF = new_definition
 `!f x y z. PASSOC (f:(A#B)#C->D) (x,y,z) = f ((x,y),z)`;;

(* ------------------------------------------------------------------------- *)
(* Treatment of paired abstractions.                                         *)
(* ------------------------------------------------------------------------- *)

let PAIRED_BETA_CONV =
  let pth = (EQT_ELIM o REWRITE_CONV [EXISTS_THM; GABS_DEF])
   `!P:A->bool. $? P ==> P($GABS P)`
  and pth1 = GSYM PASSOC_DEF and pth2 = GSYM UNCURRY_DEF in
  let dest_geq = dest_binary "GEQ" in
  let GEQ_RULE = CONV_RULE(REWR_CONV(GSYM GEQ_DEF))
  and UNGEQ_RULE = CONV_RULE(REWR_CONV GEQ_DEF) in
  let rec UNCURRY_CONV fn vs =
    try let l,r = dest_pair vs in
        try let r1,r2 = dest_pair r in
            let lr = mk_pair(l,r1) in
            let th0 = UNCURRY_CONV fn (mk_pair(lr,r2)) in
            let th1 = ISPECL [rator(rand(concl th0));l;r1;r2] pth1 in
            TRANS th0 th1
        with Failure _ ->
            let th0 = UNCURRY_CONV fn l in
            let fn' = rand(concl th0) in
            let th1 = UNCURRY_CONV fn' r in
            let th2 = ISPECL [rator fn';l;r] pth2 in
            TRANS (TRANS (AP_THM th0 r) th1) th2
    with Failure _ -> REFL(mk_comb(fn,vs)) in
  fun tm ->
    try BETA_CONV tm with Failure _ ->
    let gabs,args = dest_comb tm in
    let fn,bod = dest_binder "GABS" gabs in
    let avs,eqv = strip_forall bod in
    let l,r = dest_geq eqv in
    let pred = list_mk_abs(avs,r) in
    let th0 = rev_itlist
      (fun v th -> CONV_RULE(RAND_CONV BETA_CONV) (AP_THM th v))
      avs (REFL pred) in
    let th1 = TRANS (SYM(UNCURRY_CONV pred (rand l))) th0 in
    let th1a = GEQ_RULE th1 in 
    let etm = list_mk_icomb "?" [rand gabs] in
    let th2 = EXISTS(etm,rator (lhand(concl th1a))) (GENL avs th1a) in
    let th3 = SPECL (striplist dest_pair args) (BETA_RULE(MATCH_MP pth th2)) in
    UNGEQ_RULE th3;;

(* ------------------------------------------------------------------------- *)
(* A tactic that is often useful.                                            *)
(* ------------------------------------------------------------------------- *)

let GEN_PAIR_TAC =
  let th = GSYM PAIR in
  W(prefix THEN GEN_TAC o SUBST1_TAC o C ISPEC th o fst o dest_forall o snd);;
