#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------





 # Default option settings:

option add *Fileselect.action OK widgetDefault
option add *Fileselect.command {} widgetDefault
option add *Fileselect.directory . widgetDefault
option add *Fileselect.dirVariable {} widgetDefault
option add *Fileselect.dirOnly 0 widgetDefault
option add *Fileselect.filter fileselect:filter widgetDefault
option add *Fileselect.message "Select a file:" widgetDefault
option add *Fileselect.new 0 widgetDefault
option add *Fileselect.pattern "*" widgetDefault
option add *Fileselect.textVariable {} widgetDefault
        # Place button text in resources so that the user can customize it.
option add *Fileselect.b.c.text "Cancel" widgetDefault
option add *Fileselect.d.head.text "Directory:" widgetDefault
option add *Fileselect.patt.head.text "Filter:" widgetDefault
option add *Fileselect.patt.head.width 10 widgetDefault
option add *Fileselect.patt.head.anchor w widgetDefault
option add *Fileselect.patt.entry.relief sunken widgetDefault
option add *Fileselect.f.head.text "File name:" widgetDefault
option add *Fileselect.f.head.width 10 widgetDefault
option add *Fileselect.f.head.anchor w widgetDefault
option add *Fileselect.f.entry.Font -*-courier-medium-r-*-120-* widgetDefault
option add *Fileselect.f.entry.relief sunken widgetDefault
option add *Fileselect.l.list.relief sunken widgetDefault
        # The indented geometry options are in the following list because the
        # packer appears to override the non-indented ones.
option add *Fileselect.e.height 20 widgetDefault
        option add *Fileselect.e.geometry 1x20 widgetDefault
option add *Fileselect.f.r.width 40 widgetDefault
        option add *Fileselect.f.r.geometry 40x1 widgetDefault
option add *Fileselect.patt.r.width 40 widgetDefault
        option add *Fileselect.patt.r.geometry 40x1 widgetDefault
option add *Fileselect.l.l.width 40 widgetDefault
        option add *Fileselect.l.l.geometry 40x1 widgetDefault
option add *Fileselect.l.r.width 40 widgetDefault
        option add *Fileselect.l.r.geometry 40x1 widgetDefault
option add *Fileselect.x.l.width 40 widgetDefault
        option add *Fileselect.x.l.geometry 40x1 widgetDefault
option add *Fileselect.x.r.width 60 widgetDefault
        option add *Fileselect.x.r.geometry 40x1 widgetDefault
        # Following option will be replaced someday with `width' and `height'
option add *Fileselect.l.list.geometry 32x10 widgetDefault

proc fileselect {w args} {
        upvar #0 $w data

        # Make the window

        frame $w -class Fileselect
        bind $w <Destroy> "+catch \{fileselect:destroy $w\}"

        # Defaults

        set action [option get $w action Action]
        set command [option get $w command Command]
        set dirname [option get $w directory Directory]
        set dirvar [option get $w dirVariable DirVariable]
        set dironly [option get $w dirOnly DirOnly]
        set filter  [option get $w filter Filter]
        set message [option get $w message Message]
        set newfile [option get $w new New]
        set pattern [option get $w pattern Pattern]
        set textvar [option get $w textVariable TextVariable]

        # Parse command line

        while {[llength $args] > 0} {
                set string [lindex $args 0]
                set args [lrange $args 1 end]
                case $string in {
                        {-a -action} {
                                set action [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-c -command} {
                                set command [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-d -directory} {
                                set dirname [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-dironly} {
                                set dironly [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-dv -dirvariable} {
                                set dirvar [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-f -filter} {
                                set filter [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-m -message} {
                                set message [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-n -new} {
                                set newfile [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-p -pattern} {
                                set pattern [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-textv -textvariable} {
                                set textvar [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        default {
                                destroy $w
                                error "$string: unknown option"
                        }
                }
        }

        # Store options in globals

        set data(command) $command
        set data(dirname) $dirname
        set data(dirvariable) $dirvar
        set data(dironly) $dironly
        set data(filter) $filter
        set data(new) $newfile
        set data(pattern) $pattern
        set data(textvariable) $textvar

        # Create the subwindows

        # Prompt message

        global gui_flags
        pack [fontcheck label $w.m -text $message] -side top -expand yes -anchor w

        # Directory name

        pack [frame $w.d]
        pack [fontcheck label $w.d.head] -side left
        pack [fontcheck label $w.d.name -width 40 -anchor w] -side left

        # Entry for file name

        pack [frame $w.f] -side top -expand yes -fill x
        pack [fontcheck label $w.f.head] -side left
        pack [fontcheck entry $w.f.entry  -font $gui_flags(font,textentry)] -side left -expand yes -fill x
        pack [frame $w.f.r] -side left
        
        pack [frame $w.patt] -side top -expand yes -fill x
        pack [fontcheck label $w.patt.head] -side left
        pack [fontcheck entry $w.patt.entry  -font $gui_flags(font,textentry)] -side left -expand yes -fill x
        pack [frame $w.patt.r] -side left
        
        bind $w.patt.entry <Return> "fileselect:select $w ."
        # Pad


        # Scrollbar for the file list

        pack [frame $w.x] -side top -expand yes -fill x
        pack [frame $w.x.l] -side left
        pack [scrollbar $w.x.s -command "$w.l.list xview" -orient horizontal] -side left -expand yes -fill x
        pack [frame $w.x.r] -side left

        # Listbox to show files

        pack [frame $w.l] -side top -expand yes -fill both
        pack [frame $w.l.l] -side left
        pack [scrollbar $w.l.scroll -command "$w.l.list yview"] -side left -fill y
        pack [fontcheck listbox $w.l.list -yscroll "$w.l.scroll set" -xscroll "$w.x.s set"] -side left -expand yes -fill both
        pack [frame $w.l.r] -side left

        bind $w.l.list <ButtonRelease-1> {
                fileselect:copyEntry [winfo parent [winfo parent %W]] \
                                [fileselect_slash [%W get [%W nearest %y]]]
        }
        bind $w.l.list <Double-Button-1> {
                fileselect:select [winfo parent [winfo parent %W]] \
                                [fileselect_slash [%W get [%W nearest %y]]]
        }

        # Buttons

        pack [frame $w.b] -side bottom -pady 10
        pack [fontcheck button $w.b.o -command "fileselect:ok $w" \
                -text $action] \
            -side left -padx 10
        if $data(dironly) {
            pack [fontcheck button $w.b.open_dir \
                        -command "fileselect:open_dir $w" \
                        -text "Open Dir"] \
                -side left -padx 10
        }
        pack [fontcheck button $w.b.filt -command "fileselect:select $w ." \
                -text "Filter"  \
                -font $gui_flags(font,buttons)] \
            -side left -padx 10
        pack [fontcheck button $w.b.c \
                -command "fileselect:cancel $w" \
                -font $gui_flags(font,buttons)] \
            -side left -padx 10

        # Fill in the default file name

        $w.patt.entry insert 0 $pattern
        if {$textvar != ""} {
                upvar #0 $textvar tv
                if {[info exists tv]} {
                        $w.f.entry delete 0 end
                        $w.f.entry insert 0 [file tail $tv]
                }
        }
        if {$dirvar != ""} {
                upvar #0 $dirvar dv
                if {[info exists dv] && $dv != "" 
                    && [file isdirectory $dv] && [file readable $dv]} {
                        set dirname $dv
                }
        }

        # Move into the specified directory
        wm protocol [winfo toplevel $w] WM_DELETE_WINDOW "set data(textvariable) {}"

        after 1 fileselect:dir $w [list $dirname/]

        focus $w.f.entry
}

# Procedure:    fileselect:cancel
#
# Synopsis:
#       Cancel a file selection request.
#
# Usage:
#c      fileselect:cancel pathName
#
# Parameters:
#c      pathName
#               Path name of a file selecction dialog
#
# Return value:
#       None.
#
# Description:
#       `fileselect:cancel' is invoked when the user presses the `Cancel'
#       button in a file selection dialog.  It sets the text variable to
#       the null string, executes the selection command giving the null string
#       as an argument, and returns.

proc fileselect:cancel {w} {
        upvar #0 $w data
        if {$data(textvariable) != ""} {
                upvar #0 $data(textvariable) tv
                set tv {}
        }
        if {$data(command) != ""} {
                uplevel #0 $data(command) {}
        }
}

# Procedure:    fileselect:copyEntry
#
# Synopsis:
#       Select an entry from the listbox in a file selection dialog.
#
# Usage:
#c      fileselect:copyEntry pathName text
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box
#c      text
#               File name.
#
# Return value:
#       None.
#
# Description:
#       `fileselect:copyEntry' is invoked in response to a change of the
#       selection within a file selection dialog box.  It copies
#       the current selection (given as the second parameter) to the
#       entry box so that the user can edit it if desired.

proc fileselect:copyEntry {w text} {
        catch {$w.f.entry delete 0 end}
        catch {$w.f.entry insert 0 $text}
}

# Procedure:    fileselect:destroy
#
# Synopsis:
#       Clean up when a file selection dialog is destroyed.
#
# Usage:
#c      fileselect:destroy pathName
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box.
#
# Return value:
#       None.
#
# Description:
#       `fileselect:destroy' is invoked when a file selection dialog box is
#       destroyed.  It removes all the private variables associated with the
#       dialog.

proc fileselect:destroy w {
        upvar #0 $w data
        unset data
}

# Procedure:    fileselect:dir
#
# Synopsis:
#       Traverse to a new directory in a file selection dialog.
#
# Usage:
#c      fileselect:dir pathName directory
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box.
#c      directory
#               Directory name to read
#
# Return value:
#       None.
#
# Description:
#       `fileselect:dir' is invoked when the directory on display in a file
#       selection dialog box changes.  It scans the files in the new directory,
#       matches them against the pattern and filter, and rebuilds the list
#       box.  It locks the application in a modal dialog while scanning,
#       in order to avoid spurious selections.

proc fileselect:dir {w dirname} {
        upvar #0 $w data

        set pattern [$w.patt.entry get]
        set filter $data(filter)
        set status [catch {exec /bin/sh -cef "cd $dirname. && pwd"} dirname]
        set status [catch {glob $dirname} dirname]
        if {$status} {
                errormessage $w.error $dirname
                return
        }
        global busy
        incr busy
        global feedback
        set feedback [list $w "Scanning $dirname"]
#       modalDialog transient label $w.info -text "Scanning $dirname"
        set data(dirname) $dirname/
        if {$data(dirvariable) != ""} {
                upvar #0 $data(dirvariable) dv
                set dv $dirname
        }

        if [string length $dirname]>40 {
           $w.d.name config -text "...[string range $dirname [expr [string length $dirname]-37] end]"
        } else {
           $w.d.name config -text $dirname
        }
        catch {$w.f.entry delete 0 end}
        catch {$w.l.list delete 0 end}
#       set status [catch {split [exec ls -a $dirname] \n} fileList]
        set status [catch {glob -nocomplain $dirname/*} fileList]
        set fileList [lsort $fileList]
        if {$status == 0} {
                set fileList [linsert $fileList 0 . ..]
                set toinsertA ""
                set toinsertB ""
                foreach filename $fileList {
                        set filename [file tail $filename]
                        if [file isdirectory $dirname/$filename] {
                                lappend toinsertA $filename/
                        } else {
                            if {!$data(dironly) && [$filter $dirname/$filename]} {
                                set matches 0
                                foreach p $pattern {
                                    if [string match $p $filename] {
                                        set matches 1
                                        break
                                    }
                                }
                                if {$matches} {
                                        lappend toinsertB $filename
                                }
                            }
                        }
                }
                if {$toinsertA!=""} {
                        eval [list $w.l.list insert end] $toinsertA
                }
                if {$toinsertB!=""} {
                        eval [list $w.l.list insert end] $toinsertB
                }
        }
        incr busy -1
#       modalDialog.end $w.info
}               

# Procedure:    fileselect:filter
#
# Synopsis:
#       Default filter for a file selection dialog.
#
# Usage:
#c      fileselect:filter fileName
#
# Parameters:
#c      fileName
#               Path name of a file
#
# Return value:
#       0 if the file should be suppressed, 1 if it should be displayed.
#
# Description:
#       fileselect:filter is the default filter function in a file selection
#       dialog box if the user doesn't supply one.  It returns 1 if the
#       supplied file exists, and 0 otherwise.

proc fileselect:filter {filename} {
        return [file exists $filename]
}

# Procedure:    fileselect:ok
#
# Synopsis:
#       Handle the `OK' button in a file selection dialog.
#
# Usage:
#c      fileselect:ok pathName
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box.
#
# Return value:
#       None
#
# Description:
#       `fileselect:ok' is invoked when the user presses `OK' or double
#       clicks in a file selection dialog box.  It retrieves the file name
#       from the listbox or entry as appropriate, makes sure that a file
#       has been specified, and calls `fileselect:select' to select it.

proc fileselect:ok w {
        upvar #0 $w data
        if $data(dironly) {
            set f [$w.f.entry get]
            if {$f == ""} {
                set s [$w.l.list curselection]
                if {[llength $s] == 0} {
                        set f ""
                } elseif {[llength $s] > 1} {
                        errormessage $w.error "Please select only one file."
                        return
                } else {
                    set f [$w.l.list get $s]
                }
            }
            set f $data(dirname)$f/
            set command $data(command)
            if {$data(textvariable) != ""} {
                upvar #0 $data(textvariable) tv
                set tv $f
            }
            if {$command != ""} {
                uplevel #0 $command $f
            }
        } else {
            set f [$w.f.entry get]
            if {$f == ""} {
                set s [$w.l.list curselection]
                if {[llength $s] == 0} {
                        errormessage $w.error "Please select a file name"
                        return
                } elseif {[llength $s] > 1} {
                        errormessage $w.error "Please select only one file."
                        return
                } else {
                    set f [$w.l.list get $s]
                }
            }
            fileselect:select $w $f
        }
}

#used only with -dironly on. DRS.

proc fileselect:open_dir { w } {
        upvar #0 $w data
            set f [$w.f.entry get]
            if {$f == ""} {
                set s [$w.l.list curselection]
                if {[llength $s] == 0} {
                        errormessage $w.error "Please select a file name"
                        return
                } elseif {[llength $s] > 1} {
                        errormessage $w.error "Please select only one file."
                        return
                } else {
                    set f [$w.l.list get $s]
                }
            }
            fileselect:select $w $f
}


# Procedure:    fileselect:select
#
# Synopsis:
#       Select a file or directory in a file selection dialog.
#
# Usage:
#c      fileselect:select pathName fileName
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box
#c      fileName
#               Name of a selected file or directory.
#
# Description:
#       `fileselect:select' is the general function that responds to the
#       `OK' button or to a double click in a file selection dialog box.
#       It is passed the file name being selected.  If the file is a
#       directory, it is scanned and opened.  If it is a plain file,
#       it is selected, the text variable is set, and the selection command
#       is executed.

proc fileselect:select {w filename} {
        upvar #0 $w data
        set command $data(command)
        set filter $data(filter)
        set dir $data(dirname)
        if {![regexp {^[/~]} $filename]} {
                set filename ${dir}$filename
        }
        if {!$data(new) && ![file exists $filename]} {
                errormessage $w.error "Can't open ``$filename''."
        } else {
                catch {$w.f.entry delete 0 end}
                if [file isdirectory $filename] {
                        fileselect:dir $w $filename/
                } else {
                        if {$data(textvariable) != ""} {
                                upvar #0 $data(textvariable) tv
                                set tv $filename
                        }
                        if {$command != ""} {
                                uplevel #0 $command $filename
                        }
                }
        }
}

# Procedure:    fileselect_slash
#
# Synopsis:
#       Strip a trailing slash from a directory name
#
# Usage:
#c      fileselect_slash fileName
#
# Parameters:
#c      fileName
#               Name of a file or directory, optionally with a trailing slash.
#
# Return value:
#       File name, with the trailing slash removed.
#
# Description:
#       fileselect_slash strips trailing slashes from file names.

proc fileselect_slash f {
        if [regexp {^(.*)/$} $f rubbish leader] {
                set f $leader
        }
        return $f
}

# Procedure: selectfile
#
# Synopsis:
#       Modal file selection dialog.
#
# Usage:
#c      selectFile ?argument...?
#
# Parameters:
#       Parameters are the same as for fileselect, except that the
#       widget path name should NOT be supplied, and the `-command' and
#       `-textvariable' options should not be used.
#
# Return value:
#       Name of the selected file.
#
# Errors:
#c      operation cancelled
#               Self-explanatory.
#
# Description:
#       selectfile provides a simple interface to fileselect, performing the
#       selection in a transient modal dialog.
#
# See also:
#c      fileselect

proc selectfile args {
        global selectfile_priv
        set w [eval modalDialog transient fileselect .fileselect $args \
                        -textvariable selectfile_priv]
        catch {unset selectfile_priv}
        tkwait variable selectfile_priv
        set selection $selectfile_priv
        unset selectfile_priv
        modalDialog.end $w
        return $selection
}



