
(*====================================================================
 *
 * HolTermPathReduction
 *
 * An accessor represents a composition of several RAND,RATOR,BVAR,BODY
 * actions.  It is an object with several fields:
 *	matcher: 
 *		A function to reduce a path through a given term
 *		down to the remaining path and the term accessed by
 *		the elements of the path used up.
 *		The function should fail for accesses not of
 * 		the given access kind.
 * 		
 *		A SIMPLE matcher matches only the exact sequence of movements
 * 		given.  The access_data returned is NONE.  
 *
 *		A STRIP matcher matches two or more repetitions
 *		of the given set of movements.  The access_data returned
 *		is again NONE.
 *
 *		An ASSOC matcher matches an access down into a
 *		associative-term such as
 *			"P /\ Q /\ R"
 *		The access_data returned is SOME n where n is
 *		the index into the list of
 *		the term accessed, starting with 0.
 *
 *		SIMPLE and STRIP matches against a term tm are dependent on 
 *		(tester tm) returning true, where tester is the first argument
 *		to SIMPLE and STRIP respectively.
 *
 *	accessor: 
 *		The text of a function which accesses the term
 *		along the given path.
 *                                 
 *	unique_name: 
 *              An upper case string uniquely representing the 
 * 		access action.  For instance RAND, RATOR,
 *		BODY_FORALL, STRIP_FORALL and so on.  These
 *		may be used to access a lookup table to convert
 * 		the access operation into other types of objects,
 *		e.g. window-library access operations.
 *
 * An updateable list of accessors is maintained.  This represents the
 * different types of access operations known to the HOL system.
 *
 *====================================================================*)
 

signature HolTermPathReduction_sig =
sig

    type access_data (* = int option *)
    type matcher (* = 
		(HolTermPaths.path * term) -> 
		(access_data * (HolTermPaths.path * term)) *)
    exception MATCH_FAIL;
    
    val SIMPLE : (term -> bool) -> HolTermPaths.branch list -> matcher;
    val STRIP : (term -> bool) -> HolTermPaths.branch list -> matcher;
    val ASSOC : (term -> bool) -> matcher;
    val repeat_discard : matcher -> matcher;
    val or : (matcher * matcher) -> matcher;
    
    
    type accessor; (* = {
		matcher: matcher,
		accessor: string,
		unique_name: string
	} *)
    type access; (* = {
     		accessor: accessor,
		data: int option
    *)

    type path; (* = access list *)
    exception REDUCE_PATH
    val reduce_term_path : 
		(HolTermPaths.path * term) -> 
		path * (HolTermPaths.path * term)
    
    val add_accessor : accessor -> unit;
    val accessors : unit -> accessor list;
    
end;

