(*--------------------------------------------------------------------------*)
(*                  Copyright (c) Donald Syme 1992                          *)
(*                  All rights reserved                                     *)
(*                                                                          *)
(* Donald Syme, hereafter referred to as `the Author', retains the copyright*)
(* and all other legal rights to the Software contained in this file,       *)
(* hereafter referred to as `the Software'.                                 *)
(*                                                                          *)
(* The Software is made available free of charge on an `as is' basis. No    *)
(* guarantee, either express or implied, of maintenance, reliability,       *)
(* merchantability or suitability for any purpose is made by the Author.    *)
(*                                                                          *)
(* The user is granted the right to make personal or internal use of the    *)
(* Software provided that both:                                             *)
(* 1. The Software is not used for commercial gain.                         *)
(* 2. The user shall not hold the Author liable for any consequences        *)
(*    arising from use of the Software.                                     *)
(*                                                                          *)
(* The user is granted the right to further distribute the Software         *)
(* provided that both:                                                      *)
(* 1. The Software and this statement of rights are not modified.           *)
(* 2. The Software does not form part or the whole of a system distributed  *)
(*    for commercial gain.                                                  *)
(*                                                                          *)
(* The user is granted the right to modify the Software for personal or     *)
(* internal use provided that all of the following conditions are observed: *)
(* 1. The user does not distribute the modified software.                   *)
(* 2. The modified software is not used for commercial gain.                *)
(* 3. The Author retains all rights to the modified software.               *)
(*                                                                          *)
(* Anyone seeking a licence to use this software for commercial purposes is *)
(* invited to contact the Author.                                           *)
(*--------------------------------------------------------------------------*)




(*-----------------------------------------------------------------------
 * use "hol90_richtext/src/interface_maps.sig"; 
 * use "hol90_richtext/src/interface_maps.sml"; 
 *
 *-----------------------------------------------------------------------*)
structure InterfaceMaps : InterfaceMaps_sig = struct

type look = string 
type kind = string
exception INTERFACE_MAP;
type interface_map = {
   external_to_internal : (look * string) list -> string,
   internal_to_external : (kind * string) -> (look * string) list
};

structure Regular = struct
   type interface_map = interface_map
   type mapdata = {
      mappings : ((kind * string) * (look * string) list) list,
      default_looks : (kind * look) list,
      default_look : look option
   }
   type map = {
      add_mappings : ((kind * string) * (look * string) list) list -> unit,
      add_default_looks : (kind * look) list -> unit,
      set_default_look : look -> unit,
      remove : ((kind * string) * (look * string)) -> unit,
      remove_default : (kind * look) -> unit,
      external_to_internal : (look * string) list -> string,
      internal_to_external : (kind * string) -> (look * string) list,
      clear : unit -> unit,
      data : unit -> mapdata
   };
   fun add_mappings ((r:mapdata ref) as ref {default_looks,mappings,default_look}) l = 
        r := {default_looks=default_looks,mappings=mappings@l,default_look=default_look}
   fun add_default_looks ((r:mapdata ref)  as ref {default_looks,mappings,default_look}) l = 
        r := {default_looks=default_looks@l,mappings=mappings,default_look=default_look}
   fun set_default_look ((r:mapdata ref)  as ref {default_looks,mappings,default_look}) l =
        r := {default_looks=default_looks,mappings=mappings,default_look=SOME l}
   fun remove r l = raise INTERFACE_MAP
   fun remove_default r l = raise INTERFACE_MAP
   fun external_to_internal ((r:mapdata ref) as ref {default_looks,mappings,default_look}) (externals:(look * string) list) =
      case (assoc2 externals mappings) of
         SOME ((_,string),_) => string
       | NONE => itlist (concat o #2) externals ""
   fun internal_to_external ((r:mapdata ref)  as ref {default_looks,mappings,default_look}) (kind,string) =
      case (assoc1 (kind,string) mappings) of
         SOME (_,res) => res
       | NONE => 
      case (assoc1 kind default_looks) of
         SOME (_,res) => [(res,string)]
       | NONE => 
      case default_look of
         SOME default => [(default,string)]
       | NONE => raise INTERFACE_MAP
   fun clear ((r:mapdata ref)  as ref {default_looks,mappings,default_look}) () =
        r := {default_looks=[],mappings=[],default_look=NONE}
   fun new () = 
      let val data = ref {default_looks=[],mappings=[],default_look=NONE}
      in {
       add_mappings=add_mappings data,
       add_default_looks=add_default_looks data,
       set_default_look=set_default_look data,
       remove=remove data,
       remove_default=remove_default data,
       external_to_internal=external_to_internal data,
       internal_to_external=internal_to_external data,
       clear=clear data,
       data=(fn () => !data)
      }
      end
   fun restrict (map:map as {external_to_internal,internal_to_external, ...}) =
      {external_to_internal=external_to_internal,internal_to_external=internal_to_external}
end;

structure Sum = struct
   type interface_map = interface_map
   type mapdata = {
      maps : (string * interface_map) list
   }
   type map = {
      add : (string * interface_map) -> unit,
      remove : string -> unit,
      external_to_internal : (look * string) list -> string,
      internal_to_external : (kind * string) -> (look * string) list,
      data : unit -> mapdata
   };
   fun add (r as ref {maps}) map = r := {maps=map::maps}
   fun remove (r as ref {maps}) s = r := {maps=filter (fn (s',_) => s <> s') maps}
   fun external_to_internal (r as ref ({maps}:mapdata)) s = tryfind (fn (_,{external_to_internal,...}) => external_to_internal s) maps
   fun internal_to_external (r as ref ({maps}:mapdata)) s = tryfind (fn (_,{internal_to_external,...}) => internal_to_external s) maps
   fun new () = 
      let val data = ref {maps = []}
      in {
         add=add data,
         remove=remove data,
         external_to_internal=external_to_internal data,
         internal_to_external=internal_to_external data,
         data=(fn () => !data)
      }
      end
   fun restrict (map:map as {external_to_internal,internal_to_external, ...}) =
      {external_to_internal=external_to_internal,internal_to_external=internal_to_external}
end;

end;

