(* matching.ml                                           (c) R.J.Boulton 1990 *)
(*----------------------------------------------------------------------------*)


structure TRS_matching: TRS_matching_sig =
struct

datatype thmkind = Axiom | Definition | Theorem;
(* Datatype for a full representation of a theorem *)



(* Datatype for representing theorem patterns *)

(* The first seven constructors generate representations for theorem patterns => *)
(* The rest combine or modify such representations =>                            *)

datatype thmpattern_rep = Kind' of thmkind
                       | Any'
                       | None'
                       | Thryname' of TRS_name.namepattern
                       | Thmname' of TRS_name.namepattern
                       | Conc' of TRS_struct.termpattern
                       | HypP' of TRS_struct.termpattern list
                       | HypF' of TRS_struct.termpattern list
                       | Side' of side_condition
                       | Andalso' of thmpattern_rep * thmpattern_rep
                       | Orelse' of thmpattern_rep * thmpattern_rep
                       | Not' of thmpattern_rep
                       | Where' of thmpattern_rep * thmpattern_rep;


(* Abstract datatype for theorem patterns *)

(* There are two types of theorem pattern clause =>                             *)

(* There are main clauses, in which tests are performed on a foundthm => All of *)
(* the constructors are allowed in this type of clause, though in principle,  *)
(* side-condition tests should not be => Side-condition tests within main       *)
(* clauses are re-interpreted as follows:                                     *)
(*                                                                            *)
(*    Side <side-condition>                                                   *)
(*                                                                            *)
(* is interpreted as                                                          *)
(*                                                                            *)
(*    (Conc (autotermpattern `conc:bool`)) Where (Side <side-condition>)      *)
(*                                                                            *)
(* This only makes sense if <side-condition> tests `conc` =>                    *)

(* Only "Side', "Andalso', "Orelse', and "Not' constructors are permitted     *)
(* within a side-condition clause =>                                            *)

(* "Where' is used to link these two types of clause => Its first argument is a *)
(* main clause => Its second argument is a side-condition clause => Note that     *)
(* "Where' cannot occur within a side-condition clause =>                       *)

(* All of these constraints are imposed by the abstract datatype, which uses  *)
(* the type "thmpattern_rep' as its representing type =>                        *)

datatype thmpattern = THMPATTERN of thmpattern_rep;
    
exception sidematch_FAIL;
exception BAD_SIDE_CONDITION;
infix Where;
infix Orelse;
infix Andalso;


local

(* The following auxiliary matching functions are local to the abstract type *)
(* definition => Hence, they are hidden from the user =>                         *)

(* "mainmatch' is used for processing main clauses of theorem patterns, given *)
(* a foundthm to match against => For the first six cases of clause type,       *)
(* auxiliary functions are called => Note that these and "mainmatch' itself are *)
(* lazy => That is they require a null argument before they actually perform    *)
(* any computation =>                                                           *)

(* Side-condition clauses are re-interpreted when they occur within a main    *)
(* clause, as described at the beginning of this abstract type definition =>    *)

(* "Andalso' and "Orelse' call "mainmatch' recursively on their two arguments *)
(* and use subsidiary functions to combine the results => "Not' calls           *)
(* "mainmatch' on its argument and then calls a subsidiary function to        *)
(* process the result => "Where' calls "mainmatch' on its first argument, and   *)
(* then passes the result along with its second argument to a function which  *)
(* deals with the side-condition clause =>                                      *)

fun mainmatch thmp_rep fthm () =

   (* : (thmpattern_rep -> foundthm -> unit -> result_of_match) *)

   case thmp_rep
   of (Kind' x)  => (kindfn x fthm ())
    | Any'  => TRS_struct.Match_null
    | None'  => TRS_struct.Nomatch
    | (Thryname' x)  => (thrynamefn x fthm ())
    | (Thmname' x)  => (thmnamefn x fthm ())
    | (Conc' x)  => (concfn x fthm ())
    | (HypP' x)  => (hypPfn x fthm ())
    | (HypF' x)  => (hypFfn x fthm ())
    | (Side' _)  => (mainmatch
                     (Where' ((Conc' o TRS_struct.autotermpattern) (Term.mk_var {Name="conc",Ty=Type.mk_type {Args=[],Tyop="bool"}}),thmp_rep))
                     fthm
                     ()
                  )
    | (Andalso' (x,y))  => (andalsofn
                            (mainmatch x fthm)
                            (mainmatch y fthm)
                            ()
                         )
    | (Orelse' (x,y))  => (TRS_struct.approms
                           (mainmatch x fthm)
                           (mainmatch y fthm)
                           ()
                        )
    | (Not' x)  => (notfn (mainmatch x fthm) ())
    | (Where' (x,y))  => (wherefn y (mainmatch x fthm) ())

(* "sidematch' is used for processing side-condition clauses, given an       *)
(* environment which consists of a single matching => All side-condition tests *)
(* within the clause are applied to this matching =>                           *)

(* Tests on the foundthm itself are prohibited (there is no foundthm         *)
(* available to test) => This means that the first six cases for theorem       *)
(* patterns all cause failures =>                                              *)

(* If the side-condition clause is simply a side-condition, the side-        *)
(* condition is applied to the environment => If the test succeeds, the        *)
(* result is passed back up => If not, "TRS_struct.Nomatch' is passed back up =>            *)

(* "Andalso', "Orelse' and "Not' cause "sidematch' to be called recursively, *)
(* and the results of these calls are processed further by subsidiary        *)
(* functions => "Where' is prohibited within side-condition clauses =>           *)

(* The failures due to illegal constructor use should never occur because    *)
(* the abstract datatype will prevent such constructions =>                    *)

and sidematch thmp_rep env () =

   (* : (thmpattern_rep -> matching -> unit -> result_of_match) *)

   case thmp_rep
   of (Kind' _)  => (raise sidematch_FAIL)
    | (Thryname' _)  => raise sidematch_FAIL
    | (Thmname' _)  => raise sidematch_FAIL
    | (Conc' _)  => raise sidematch_FAIL
    | (HypP' _)  => raise sidematch_FAIL
    | (HypF' _)  => raise sidematch_FAIL
    | Any'  => TRS_struct.Match_null
    | None'  => TRS_struct.Nomatch
    | (Side' x)  => ((x env) handle TRS_extents.NO_MATCH => (TRS_struct.Nomatch))
    | (Andalso' (x,y))  => (andalsofn
                            (sidematch x env)
                            (sidematch y env)
                            ()
                         )
    | (Orelse' (x,y))  => (TRS_struct.approms
                           (sidematch x env)
                           (sidematch y env)
                           ()
                        )
    | (Not' x)  => (notfn (sidematch x env) ())
    | (Where' _)  => raise sidematch_FAIL

(* "andalsofn' is used for ANDing two "result_of_matches' together =>          *)

(* The first argument is applied to () => If the result is "TRS_struct.Nomatch', then the *)
(* result of the whole evaluation is "TRS_struct.Nomatch' => If not, the second argument  *)
(* is treated similarly => If both the arguments contain matchings, the        *)
(* function attempts to join the two "heads' => If this succeeds, the result   *)
(* becomes the "head' of the combined "result_of_match' => If not, the result  *)
(* is discarded =>                                                             *)

(* The rest of the "result_of_match' is (when required) obtained by calling  *)
(* "andalsofn' recursively, firstly on the original first argument and the   *)
(* "tail' of the second, and then on the tail of the first and the original  *)
(* second argument => The two resulting "result_of_matches' are appended using *)
(* "TRS_struct.approms' =>                                                                *)

(* The overall effect of this is to combine a "list' of n matchings with a   *)
(* "list' of m matchings to form a "list' of all the possible combinations   *)
(* of matchings which can be joined successfully (maximum length n * m) =>     *)

and andalsofn rom1fn rom2fn () =

   (* : ((unit -> result_of_match) -> (unit -> result_of_match) ->           *)
   (*                                             (unit -> result_of_match)) *)

   case (rom1fn ())
   of (TRS_struct.Nomatch)  => (TRS_struct.Nomatch)
    | (TRS_struct.Match (m1,romfn1)) =>
         (case (rom2fn ())
          of (TRS_struct.Nomatch)  => (TRS_struct.Nomatch)
           | (TRS_struct.Match (m2,romfn2)) =>
                ( let val rest = (TRS_struct.approms
                               (andalsofn rom1fn romfn2)
                               (andalsofn romfn1 rom2fn)
                            )
                 in ( (TRS_struct.Match (TRS_struct.join_matchings m1 m2,rest))
                    handle TRS_extents.NO_MATCH => (rest ())
                    )
		 end
                )
         )

(* "notfn' simply negates a "result_of_match', discarding any matchings, *)
(* since they make no sense when negated => "Not' can therefore be very    *)
(* destructive =>                                                          *)

and notfn rom1fn () =

   (* : ((unit -> result_of_match) -> (unit -> result_of_match)) *)

   case (rom1fn ())
   of (TRS_struct.Nomatch)  => (TRS_struct.Match_null)
    | (TRS_struct.Match _)  => (TRS_struct.Nomatch)

(* "wherefn' is used for handling side-condition clauses =>               *)

(* It passes each matching in the "result_of_match' it is given to the  *)
(* theorem pattern => The matchings are passed in turn as environments =>   *)
(* The evaluation proceeds only as far as is necessary, but the         *)
(* potential to continue it is retained =>                                *)

(* "sidematch' is used to test the theorem pattern under each of the    *)
(* environments => It returns a "result_of_match' => Only those matchings   *)
(* consistent with the environment should be retained => That is, any     *)
(* wildcard which appears in the environment as well as in the matching *)
(* should match to the same object in both cases => "andalsofn' is used   *)
(* to perform this checking =>                                            *)

(* The "result_of_matches' generated for each environment are appended  *)
(* using "TRS_struct.approms' =>                                                     *)

and wherefn thmp_rep rom1fn () =

   (* : (thmpattern_rep -> (unit -> result_of_match) ->                 *)
   (*                                        (unit -> result_of_match)) *)

   case (rom1fn ())
   of (TRS_struct.Nomatch)  => (TRS_struct.Nomatch)
    | (TRS_struct.Match (m,romfn))  => (TRS_struct.approms
                             (andalsofn
                                ( fn () => TRS_struct.Match (m,( fn () => TRS_struct.Nomatch)))
                                (sidematch thmp_rep m))
                             (wherefn thmp_rep romfn)
                             ()
                          )


(* "kindfn' tests the kind of a found theorem => *)

and kindfn knd fthm () =

   (* : (thmkind -> foundthm -> (unit -> result_of_match)) *)

   TRS_struct.bool_to_rom (knd = (fst fthm))


(* "thrynamefn' uses a "TRS_name.namepattern' to test the name of the theory to which *)
(* a found theorem belongs =>                                                  *)

and thrynamefn nmp fthm () =

   (* : (TRS_name.namepattern -> foundthm -> (unit -> result_of_match)) *)

   TRS_struct.bool_to_rom (TRS_name.namematch nmp ((fst o snd) fthm))


(* "thmnamefn' uses a "TRS_name.namepattern' to test the name of a found theorem => *)

and thmnamefn nmp fthm () =

   (* : (TRS_name.namepattern -> foundthm -> (unit -> result_of_match)) *)

   TRS_struct.bool_to_rom (TRS_name.namematch nmp ((fst o snd o snd) fthm))


(* "concfn' tests the conclusion of a found theorem against a TRS_struct.termpattern => *)

(* The conclusion is extracted and then matched against the TRS_struct.termpattern => *)
(* If the match succeeds, the matching is made into a "result_of_match' => *)
(* Otherwise, "TRS_struct.Nomatch' is returned as the "result_of_match' =>            *)

and concfn patt fthm () =

   (* : (TRS_struct.termpattern -> foundthm -> (unit -> result_of_match)) *)

   (TRS_struct.Match (TRS_struct.make_matching patt ((Thm.concl o snd o snd o snd) fthm),( fn () => TRS_struct.Nomatch)))
   handle TRS_extents.NO_MATCH => TRS_struct.Nomatch


(* "hypPfn' tests the hypotheses of a found theorem against a list of          *)
(* TRS_struct.termpatterns => Not all of the hypotheses need to be matched for the match to *)
(* succeed =>                                                                    *)

(* The list of hypotheses is extracted from the found theorem => If there are *)
(* more TRS_struct.termpatterns than hypotheses, "TRS_struct.Nomatch' is returned => Otherwise,     *)
(* "hypfn' is used to test the TRS_struct.termpatterns against the hypotheses =>         *)

and hypPfn pattl fthm () =

   (* : (TRS_struct.termpattern list -> foundthm -> (unit -> result_of_match)) *)

    let val hypl = (Thm.hyp o snd o snd o snd) fthm
   in if ((length pattl) > (length hypl))
      then TRS_struct.Nomatch
      else hypfn pattl hypl ()
   end

(* "hypFfn' tests the hypotheses of a found theorem against a list of      *)
(* TRS_struct.termpatterns => All of the hypotheses need to be matched for the match to *)
(* succeed =>                                                                *)

(* The list of hypotheses is extracted from the found theorem => If there are    *)
(* the same number of TRS_struct.termpatterns as there are hypotheses, "hypfn' is used to *)
(* test the TRS_struct.termpatterns against the hypotheses => Otherwise, "TRS_struct.Nomatch' is       *)
(* returned =>                                                                   *)

and hypFfn pattl fthm () =

   (* : (TRS_struct.termpattern list -> foundthm -> (unit -> result_of_match)) *)

    let val hypl = (Thm.hyp o snd o snd o snd) fthm
   in if ((length pattl) = (length hypl))
      then hypfn pattl hypl ()
      else TRS_struct.Nomatch
   end
(* "hypfn' tests a list of TRS_struct.termpatterns against a list of hypotheses *)

(* The result is a "result_of_match' => A subsidiary function is used to allow *)
(* backtracking =>                                                             *)

(* "hypmatch' takes four arguments plus a null argument to provide "lazy'    *)
(* evaluation => The first argument is an accumulated matching for the         *)
(* wildcards bound so far => The second argument is a list of hypotheses left  *)
(* unmatched => This has to be remembered while the various ways of matching   *)
(* them are attempted => The third argument is the list of patterns not yet    *)
(* matched => The fourth argument is the list of hypotheses which have not yet *)
(* been tried against the head of the pattern list =>                          *)

(* When the pattern list is empty, the accumulated matching is made into a   *)
(* "result_of_match', and returned as result => If the list of hypotheses runs *)
(* out before the patterns, "TRS_struct.Nomatch' is returned =>                           *)

(* If the head of the pattern list matches the head of the hypothesis list,  *)
(* and the resulting matching is consistent with the accumulated matching,   *)
(* the head of the hypothesis list is removed from the previous level's list *)
(* and "hypmatch' is called recursively to attempt a new level of match => Any *)
(* other ways of matching are found as described below, and are appended to  *)
(* the result of this call =>                                                  *)

(* Any other ways of matching are found by a recursive call to "hypmatch'    *)
(* with all of the original arguments except that the fourth argument is the *)
(* tail of the original list => The result of this call becomes the result of  *)
(* the original call if the head of the pattern list did not match the head  *)
(* of the hypothesis list =>                                                   *)

and hypfn pattl hypl () =

   (* : (TRS_struct.termpattern list -> term list -> (unit -> result_of_match)) *)

    let fun hypmatch m prevtl pl terml () =

      (* : (matching -> term list -> TRS_struct.termpattern list -> term list ->        *)
      (*                                          (unit -> result_of_match)) *)

      if (null pl)
      then TRS_struct.Match(m,( fn () => TRS_struct.Nomatch))
      else if (null terml)
           then TRS_struct.Nomatch
           else ( let val rest = hypmatch m prevtl pl (tl terml)
                 in (( let val newm = TRS_struct.join_matchings m
                                    (TRS_struct.make_matching (hd pl) (hd terml))
                      in ( let val newtl = filter ( fn x => not (x = (hd terml))) prevtl
                          in TRS_struct.approms
                                (hypmatch newm newtl (tl pl) newtl)
                                rest
                                ()
			  end
                         )   
		      end
                     )	 
		    
                    handle TRS_extents.NO_MATCH => rest ()
                    )
		 end
                )   
   in hypmatch TRS_struct.null_matching hypl pattl hypl ()
   end

in

   fun show_thmpattern (THMPATTERN thmp) = thmp;

       (* : (thmpattern -> thmpattern_rep) *)

   fun Kind knd = THMPATTERN (Kind' knd);

       (* : thmpattern *)

   val None  = THMPATTERN None';
   val Any  = THMPATTERN Any';

       (* : (thmkind -> thmpattern) *)


   fun  Thryname nmp = THMPATTERN (Thryname' nmp);

       (* : (TRS_name.namepattern -> thmpattern) *)

   fun  Thmname nmp = THMPATTERN (Thmname' nmp);

       (* : (TRS_name.namepattern -> thmpattern) *)

   fun  Conc patt = THMPATTERN (Conc' patt);

       (* : (TRS_struct.termpattern -> thmpattern) *)

   fun  HypP pattl = THMPATTERN (HypP' pattl);

       (* : (TRS_struct.termpattern list -> thmpattern) *)

   fun  HypF pattl = THMPATTERN (HypF' pattl);

       (* : (TRS_struct.termpattern list -> thmpattern) *)

   fun  Side x = THMPATTERN (Side' x);

       (* : (TRS_struct.side_condition -> thmpattern) *)

   fun  (THMPATTERN thmp1) Andalso (THMPATTERN thmp2) =

       (* : ((thmpattern * thmpattern) -> thmpattern) *)

       THMPATTERN (Andalso' (thmp1,thmp2));

   fun  (THMPATTERN thmp1) Orelse (THMPATTERN thmp2) =

       (* : ((thmpattern * thmpattern) -> thmpattern) *)

       THMPATTERN (Orelse' (thmp1,thmp2));

   fun  Not (THMPATTERN thmp) = THMPATTERN (Not' thmp);

       (* : (thmpattern -> thmpattern) *)

   fun  (THMPATTERN thmp1) Where (THMPATTERN thmp2) =

       (* : ((thmpattern * thmpattern) -> thmpattern) *)

       (* Function to check that a side-condition clause is legal *)

       (* The function either returns "true' or fails => The failure which     *)
       (* occurs in the body of "Where' if "is_legal_sidecond' returns false *)
       (* is therefore unnecessary =>                                          *)

        let fun is_legal_sidecond thmp_rep =

          (* : (thmpattern_rep -> bool) *)

          case thmp_rep
          of (Kind' _)  => raise BAD_SIDE_CONDITION
           | (Thryname' _) => raise BAD_SIDE_CONDITION
           | (Thmname' _) => raise BAD_SIDE_CONDITION
           | (Conc' _)  => raise BAD_SIDE_CONDITION
           | (HypP' _)  => raise BAD_SIDE_CONDITION
           | (HypF' _)  => raise BAD_SIDE_CONDITION
           | Any'  => raise BAD_SIDE_CONDITION
           | None'  => raise BAD_SIDE_CONDITION
           | (Side' _)  => true
           | (Andalso' (thmp_rep1,thmp_rep2)) =>
                ((is_legal_sidecond thmp_rep1) andalso (is_legal_sidecond thmp_rep2))
           | (Orelse' (thmp_rep1,thmp_rep2)) =>
                ((is_legal_sidecond thmp_rep1) andalso (is_legal_sidecond thmp_rep2))
           | (Not' thmp_rep1)  => (is_legal_sidecond thmp_rep1)
           | (Where' _)  => raise BAD_SIDE_CONDITION

       in if (is_legal_sidecond thmp2)
          then THMPATTERN
                  (Where' (thmp1,thmp2))
          else raise BAD_SIDE_CONDITION
       end;
       
       
   (* Function to test a theorem pattern against a foundthm *)

   (* It calls "mainmatch' to attempt the matching => "mainmatch' returns a *)
   (* "result_of_match', which "thmmatch' converts to a Boolean value =>    *)

   fun  thmmatch (THMPATTERN thmp) fthm =

       (* : (thmpattern -> foundthm -> bool) *)

       TRS_struct.rom_to_bool (mainmatch thmp fthm ());

end;
(* Infix declarations to make construction of theorem patterns nicer *)


(* Function to filter a list of theorems using a theorem pattern *)

 fun thmfilter thmp fthml = filter (thmmatch thmp) fthml;

   (* : (thmpattern -> foundthm list -> foundthm list) *)

end;

(*----------------------------------------------------------------------------*)

type foundthm = TRS_matching.thmkind * (string * (string * Thm.thm));

