(****************************************************************************)
(*                                                                          *)
(*               Copyright 1994, 1995 University of Cambridge               *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : poly-ml-2.06m.sml                                        *)
(* DESCRIPTION   : General purpose functions (for Poly ML version 2.06m).   *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 31st January 1994                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 28th February 1995                                       *)
(****************************************************************************)

structure Portable : PORTABLE =
struct

val chr = chr;
val ord = ord;
val explode = explode;
val implode = implode;


datatype 'a option = NONE | SOME of 'a;

exception Fail of string;


fun can f x = (f x; true) handle _ => false;

fun cannot f x = not (can f x);

fun funpow f n x = if (n <= 0) then x else funpow f (n - 1) (f x);


exception Hd;
fun hd [] = raise Hd
  | hd (x::_) = x;

exception Tl;
fun tl [] = raise Tl
  | tl (_::xs) = xs;

fun null [] = true
  | null (_::_) = false;

fun length [] = 0
  | length (x::xs) = 1 + length xs;

fun flat [] = []
  | flat (x::xs) = x @ (flat xs);


fun fold f [] z = z
  | fold f (x::xs) z = f (x,fold f xs z);

fun revfold f [] z = z
  | revfold f (x::xs) z = revfold f xs (f (x,z));


exception Nth;
fun nth ([],_) = raise Nth
  | nth (x::_,0) = x
  | nth (_::xs,n) = nth (xs,n-1);

exception NthHead;
fun nthhead (_,0) = []
  | nthhead ([],_) = raise NthHead
  | nthhead (x::xs,n) = x::(nthhead (xs,n - 1));

exception NthTail;
fun nthtail (xs,0) = xs
  | nthtail ([],_) = raise NthTail
  | nthtail (x::xs,n) = nthtail (xs,n - 1);

fun last l = nth (l,length l - 1);

fun rotate n [] = []
  | rotate n l =
   let val pos = n mod (length l)
   in  nthtail (l,pos) @ nthhead (l,pos)
   end;


fun exists p [] = false
  | exists p (x::xs) = (p x) orelse (exists p xs);

fun forall p [] = true
  | forall p (x::xs) = (p x) andalso (forall p xs);

fun filter p [] = []
  | filter p (x::xs) = if p x then x::(filter p xs) else filter p xs;


exception Find;
fun find p [] = raise Find
  | find p (x::xs) = if (p x) then x else find p xs;

exception NOT_FOUND;

fun assoc x [] = raise NOT_FOUND
  | assoc x ((x',y)::rest) = if (x = x') then y else assoc x rest;

fun rev_assoc y [] = raise NOT_FOUND
  | rev_assoc y ((x,y')::rest) = if (y = y') then x else rev_assoc y rest;


fun zip ([],[]) = []
  | zip (x::xs,y::ys) = (x,y)::(zip (xs,ys))
  | zip _ = raise Fail "zip";

fun unzip [] = ([],[])
  | unzip ((x,y)::xys) =
   let val (xs,ys) = unzip xys
   in  (x::xs,y::ys)
   end;


fun member_component f x l = exists (fn y => f y = x) l;

val member = member_component (fn x => x);

fun intersect l1 l2 = filter (fn x => member x l2) l1;

fun subtract l1 l2 = filter (fn x => not (member x l2)) l1;

fun union l1 l2 = l1 @ (subtract l2 l1);

fun set_equal l1 l2 =
   (forall (fn x => member x l2) l1) andalso
   (forall (fn x => member x l1) l2);

fun remove_duplicates p l =
   let fun remove [] keep = keep
         | remove (x::xs) keep =
          if (exists (fn x' => p (x,x')) keep)
          then remove xs keep
          else remove xs (x::keep)
   in  rev (remove l [])
   end;

fun setify l = remove_duplicates (op =) l;


fun mapcan f [] = []
  | mapcan f (x::xs) =
   let val y = f x in y::(mapcan f xs) end handle _ => mapcan f xs;

fun duplicate_components f l =
   let fun dupl ds [] = rev ds
         | dupl ds (x::xs) =
          if (member_component f (f x) ds) orelse (member_component f (f x) xs)
          then dupl (x::ds) xs
          else dupl ds xs
   in  dupl [] l
   end;

fun duplicates l =
   let fun dupl [] = []
         | dupl (x::xs) = if (member x xs) then (x :: dupl xs) else dupl xs
   in  setify (dupl l)
   end;

fun split_at x [] = ([],[])
  | split_at x (l as y::ys) =
   if (y = x)
   then ([],l)
   else let val (l1,l2) = split_at x ys
        in  (y::l1,l2)
        end;

fun upto from to =
   if (from > to)
   then []
   else from::(upto (from + 1) to);

fun map_thread f ([],env) = ([],env)
  | map_thread f (x::xs,env) =
   let val (y,env') = f (x,env)
       val (ys,env'') = map_thread f (xs,env')
   in  (y::ys,env'')
   end;

fun flat_map_thread f (xs,env) =
   let val (ys,env') = map_thread f (xs,env)
   in  (flat ys,env')
   end;

fun initial_length f [] = 0
  | initial_length f (x::xs) = if (f x) then 1 + initial_length f xs else 0;

fun common_head (lists as (_,[])) = ([],lists)
  | common_head (lists as ([],_)) = ([],lists)
  | common_head (lists as (x::xs,y::ys)) =
   if (x = y)
   then let val (commons,remainders) = common_head (xs,ys)
        in (x::commons,remainders)
        end
   else ([],lists);

fun is_subsequence [] _ = true
  | is_subsequence (_::_) [] = false
  | is_subsequence (sub as x::xs) (y::ys) =
   if (x = y)
   then is_subsequence xs ys
   else is_subsequence sub ys;

fun pairings f (xs,ys) =
   flat (map (fn x => map (fn y => f (x,y)) ys) xs);

fun replace_or_add combine id_of l1 l2 =
   let fun replace [] xs' = xs'
         | replace (x::xs) xs' =
          let val id = id_of x
              val x' = find (fn x' => id_of x' = id) l2
          in  replace xs ((combine x x')::xs')
          end
          handle Find => replace xs (x::xs')
       val ids = map id_of l1
   in  (rev (replace l1 [])) @ (filter (fn x => not (member (id_of x) ids)) l2)
   end;


fun string_of_int i =
   let fun digits 0 = ""
         | digits i = digits (i div 10) ^ chr (ord("0") + (i mod 10))
   in  if (i = 0)
       then "0"
       else if (i < 0)
            then "~" ^ digits (~i)
            else digits i
   end;

exception IntOfString;
fun int_of_string s =
   let val zero = ord("0")
       fun digit (c,result) =
          if ("0" <= c) andalso (c <= "9")
          then ord(c) - zero + 10 * result
          else raise IntOfString
   in  case (explode s)
       of ["~"] => raise IntOfString
        | [] => raise IntOfString
        | "~"::cs => ~(revfold digit cs 0)
        | cs => revfold digit cs 0
   end;

exception Substring;
fun substring (s,i,n) =
   PolyML.StringBuiltIns.substring (s,i+1,n)
   handle PolyML.StringBuiltIns.Substring => raise Substring;

fun ordof (s,i) = ord (substring (s,i,1)) handle Substring => raise Ord;


fun min (i,j) : int = if (i < j) then i else j;

fun max (i,j) : int = if (i > j) then i else j;


fun inc r = r := !r + 1;

fun dec r = r := !r - 1;

end;
