(*--------------------------------------------------------------------------*)
(*                  Copyright (c) Donald Syme 1992                          *)
(*                  All rights reserved                                     *)
(*                                                                          *)
(* Donald Syme, hereafter referred to as `the Author', retains the copyright*)
(* and all other legal rights to the Software contained in this file,       *)
(* hereafter referred to as `the Software'.                                 *)
(*                                                                          *)
(* The Software is made available free of charge on an `as is' basis. No    *)
(* guarantee, either express or implied, of maintenance, reliability,       *)
(* merchantability or suitability for any purpose is made by the Author.    *)
(*                                                                          *)
(* The user is granted the right to make personal or internal use of the    *)
(* Software provided that both:                                             *)
(* 1. The Software is not used for commercial gain.                         *)
(* 2. The user shall not hold the Author liable for any consequences        *)
(*    arising from use of the Software.                                     *)
(*                                                                          *)
(* The user is granted the right to further distribute the Software         *)
(* provided that both:                                                      *)
(* 1. The Software and this statement of rights are not modified.           *)
(* 2. The Software does not form part or the whole of a system distributed  *)
(*    for commercial gain.                                                  *)
(*                                                                          *)
(* The user is granted the right to modify the Software for personal or     *)
(* internal use provided that all of the following conditions are observed: *)
(* 1. The user does not distribute the modified software.                   *)
(* 2. The modified software is not used for commercial gain.                *)
(* 3. The Author retains all rights to the modified software.               *)
(*                                                                          *)
(* Anyone seeking a licence to use this software for commercial purposes is *)
(* invited to contact the Author.                                           *)
(*--------------------------------------------------------------------------*)




structure PretermOps: PretermOps_sig =
struct

type preterm = Parse_support.Preterm.preterm
type preterm_in_env = Parse_support.preterm_in_env


open Parse_support;
open Parse_support.Preterm;

fun forall f (h::t) = f h andalso forall f t
  | forall f [] = true;
fun is_sys_type ty = 
   forall is_sys_type (#Args (dest_type ty)) 
   handle _ => false;

fun preterm_to_preterm_in_env (Var {Name,Ty}) =
   if (is_sys_type Ty)
   then make_atom Name
   else make_constrained (make_atom Name) Ty
 | preterm_to_preterm_in_env (Const {Name,Ty}) =
   if (is_sys_type Ty)
   then make_atom Name
   else make_constrained (make_atom Name) Ty
 | preterm_to_preterm_in_env (Comb {Rator,Rand}) =
   list_make_comb [preterm_to_preterm_in_env Rator, preterm_to_preterm_in_env Rand]
 | preterm_to_preterm_in_env (Abs {Body=Body,Bvar=Var {Name,Ty}}) =
   bind_term "\\" [make_binding_occ Name] (preterm_to_preterm_in_env Body)
 | preterm_to_preterm_in_env (Constrained (tm,ty)) =
   make_constrained (preterm_to_preterm_in_env tm) ty
 | preterm_to_preterm_in_env _ = raise (Fail "preterm_to_preterm_in_env");


fun term_to_preterm tm =
 case (dest_term tm) of
     VAR v => 
        Var v
   | CONST c => 
        Const c
   | COMB {Rator,Rand} =>
        Comb{Rator=term_to_preterm Rator, Rand=term_to_preterm Rand}
   | LAMB {Bvar,Body} =>
        Abs{Bvar=term_to_preterm Bvar,Body=term_to_preterm Body}
 handle _ => raise (Fail "term_to_preterm");


fun dest_Var (Var {Name,Ty}) = Name
  | dest_Var _       = raise (Fail "dest_Var");

fun dest_Const (Const {Name,Ty}) = Name
  | dest_Const _         = raise (Fail "dest_Const");

fun dest_Comb (Comb {Rator,Rand}) = (Rator,Rand)
  | dest_Comb _        = raise (Fail "dest_Comb");

fun dest_Abs (Abs {Bvar=Var {Name,Ty},Body}) = (Name,Body)
  | dest_Abs _       = raise (Fail "dest_Abs");

fun dest_Constrained (Constrained x) = x
  | dest_Constrained _               = raise (Fail "dest_Constrained");

fun dest_Antiq (Antiq p) = p
  | dest_Antiq _         = raise (Fail "dest_Antiq");

val is_Var      = can dest_Var
and is_Const    = can dest_Const
and is_Comb     = can dest_Comb
and is_Abs      = can dest_Abs
and is_Constrained    = can dest_Constrained
and is_Antiq = can dest_Antiq;

val preterm_rator = #1 o dest_Comb
and preterm_rand  = #2 o dest_Comb;

val preterm_bvar  = #1 o dest_Abs
and preterm_body  = #2 o dest_Abs;

fun strip_Comb p =
 let fun f (Comb {Rator,Rand}) =
            let val (oper,args) = f Rator in (oper,Rand::args) end
       | f p = (p,[])
 in
 (I ## rev)(f p) 
 end;

fun dest_2App s p =
   case p of 
     Comb {Rator=Comb{Rator=Const{Name=oper,Ty=_},Rand=p1},Rand=p2} => 
       (if s = oper then (p1,p2) else raise Match)
   | _ => raise Match
   handle _ => raise (Fail ("dest_2App -- " ^ s))

fun is_2App s = can (dest_2App s);

val dest_Pair = dest_2App ","
val is_Pair = is_2App ","

fun dest_List (Const {Name="NIL",Ty=_}) = []
  | dest_List (Comb {Rator=Comb{Rator=Const{Name="CONS",Ty=_},Rand=h},Rand=rest}) = 
      (h::dest_List rest)
  | dest_List _ = raise (Fail "dest_List")

val is_List = can dest_List

fun dest_Tuple (Comb {Rator=Comb{Rator=Const{Name=",",Ty=_},Rand=x},Rand=rest}) = 
      (x::dest_Tuple rest)
  | dest_Tuple x = [x]
fun dest_Varstruct p = map dest_Var (dest_Tuple p)
fun is_Varstruct p = forall is_Var (dest_Tuple p)

val dest_Conj = dest_2App "/\\"
val is_Conj = is_2App "/\\"
val dest_Disj = dest_2App "\\/"
val is_Disj = is_2App "\\/"
val dest_Imp = dest_2App "==>"
val is_Imp = is_2App "==>"
val dest_Eq = dest_2App "="
val is_Eq = is_2App "="

fun dest_Forall (Comb {Rator=Const {Name="!",Ty=_},Rand=Abs{Bvar=Var {Name=v,Ty=_},Body}}) = (v,Body)
  | dest_Forall _ = raise (Fail "dest_Forall")
val is_Forall = can dest_Forall
fun dest_Exists (Comb {Rator=Const {Name="?",Ty=_},Rand=Abs{Bvar=Var {Name=v,Ty=_},Body}}) = (v,Body)
  | dest_Exists _ = raise (Fail "dest_Exists")
val is_Exists = can dest_Exists
fun Conjuncts p = 
    let val (p1,p2) = dest_Conj p in 
       (Conjuncts p1@Conjuncts p2)
    end
    handle _ => [p]
fun Disjuncts p = 
    let val (p1,p2) = dest_Disj p in 
       (Disjuncts p1@Disjuncts p2)
    end
    handle _ => [p]

val mk_Var = make_atom
val mk_Const = make_atom
fun mk_Constrained (p,t) = make_constrained p t
fun list_mk_Comb (p,l) = list_make_comb (p::l)
fun mk_Comb(Rator,Rand) = list_mk_Comb (Rator,[Rand])
fun mk_Abs (Bvar,Body) =
   bind_term "\\" [make_binding_occ Bvar] Body
fun mk_Pabs ([Bvar],Body) =
       mk_Abs(Bvar,Body)
  | mk_Pabs ((Bvar::Varstruct),Body) =
       mk_Comb(mk_Const "UNCURRY",
               mk_Abs(Bvar,mk_Pabs(Varstruct,Body)))
  | mk_Pabs _ = raise (Fail "mk_Pabs")
fun mk_Pair(p1,p2) = list_mk_Comb (mk_Const ",",[p1,p2])
fun mk_Tuple pl = end_itlist (curry mk_Pair) pl
fun mk_Varstruct sl = mk_Tuple (map mk_Var sl)
fun mk_Conj(p1,p2) = list_mk_Comb (mk_Const "/\\",[p1,p2])
val list_mk_Conj = end_itlist (curry mk_Conj)
fun mk_Disj(p1,p2) = list_mk_Comb (mk_Const "\\/",[p1,p2])
val list_mk_Disj = end_itlist (curry mk_Disj)
fun mk_Imp(p1,p2) = list_mk_Comb (mk_Const "==>",[p1,p2])
fun mk_Eq(p1,p2) = list_mk_Comb (mk_Const "=",[p1,p2])
fun mk_Forall(v,p) = mk_Comb(mk_Const "!",mk_Abs(v,p))
fun mk_Exists(v,p) = mk_Comb(mk_Const "?",mk_Abs(v,p))

fun list_mk_Forall(vl,p) = itlist (curry mk_Forall) vl p
fun list_mk_Exists(vl,p) = itlist (curry mk_Exists) vl p

fun mk_Pforall (vs,body) = mk_Comb(mk_Const "!",mk_Pabs(vs,body))
fun mk_Pexists (vs,body) = mk_Comb(mk_Const "?",mk_Pabs(vs,body))

(*----------------------------------------------------------------------------*)
(* ([(`v1`,`t1`);...;(`vn`,`tn`)], `p`)                                       *)
(* --->                                                                       *)
(* `!v1...vn. v1::t1 /\ ... /\ vn::tn ==> p`                                  *)
(*----------------------------------------------------------------------------*)

fun list_mk_ResForall(resvl,p) = 
    list_mk_Forall (map #1 resvl,
                    mk_Imp(list_mk_Conj (map (fn (v,res) => list_mk_Comb (mk_Const "IN",[mk_Var v,res])) resvl),
                           p))

(*----------------------------------------------------------------------------*)
(* ([(`v1`,`t1`);...;(`vn`,`tn`)], `p`)                                       *)
(* --->                                                                       *)
(* `?v1...vn. v1::t1 /\ ... /\ vn::tn /\ p`                                   *)
(*----------------------------------------------------------------------------*)
fun list_mk_ResExists(resvl,p) = 
    list_mk_Exists (map #1 resvl,
                    mk_Imp(list_mk_Conj (map (fn (v,res) => list_mk_Comb (mk_Const "IN",[mk_Var v,res])) resvl),
                           p))

fun mk_List [] = mk_Const "NIL"
  | mk_List (h::t) = list_mk_Comb (mk_Const "CONS",[h,mk_List t])


(*----------------------------------------------------------------------------*)
(* preterm_frees computes the free variables in a preterm.                    *)
(*----------------------------------------------------------------------------*)

fun Frees p =
 case p
  of Var {Name,Ty} => [Name]
  |  Const c => []
  |  Comb{Rator,Rand} => union (Frees Rator) (Frees Rand)
  |  Abs{Bvar=Var {Name,Ty},Body} => subtract (Frees Body) [Name]
  |  Constrained(p1,ty) => Frees p1
  |  Antiq t => map (#Name o dest_var) (free_vars t)
  |  _ => raise (Fail "Frees");

(*----------------------------------------------------------------------------*)
(* Preterm substitution. No renaming; variable capture causes failure.        *)
(*----------------------------------------------------------------------------*)

(*----------------------------------------------------------------------------*)
(* var_capture l v1 v2 is true if substituting l in p2 would result in a      *)
(* variable capture by p1.                                                    *)
(*----------------------------------------------------------------------------*)

(*

fun var_capture l p1 p2 =
 (map 
   (fn v => case (assoc2 v l) of
               SOME (x,_) = if (mem p1 (Frees x)) then raise Fail
               NONE => false)
   (Frees p2);
  false
 ) ? true;

fun Subst l p =
 case p
  of (Var v)       . fst(rev_assoc p l) ? p
  |  (Const c)     . p
  |  (Comb(p1,p2)) . Comb
                              (Subst l p1, Subst l p2)
  |  (Abs(p1,p2))  . if var_capture l p1 p2
                             then raise (Fail "Subst: variable capture"
                             else Abs(p1, Subst l p2)
  |  (Constrained(p1,ty)). Constrained(Subst l p1,ty)
  |  (Antiq t)  . p;
*)


(*----------------------------------------------------------------------------*)
(* Prime (dash) a preterm variable.                                           *)
(*----------------------------------------------------------------------------*)

   fun replicate x 0 = []
     | replicate x n = (x::replicate x (n-1))
   fun repeat_count f x = 
         let val a = f x val (r,n) = repeat_count f a in (r,n+1) end
         handle _ => (x,0)
   exception TAKEPRIME
   fun takeprime_Var s = 
      let val (f,b) = front_n_back (explode s) in 
      if (b="'") then (implode f) else raise TAKEPRIME end;
   fun takeprime_Varstruct vs = map takeprime_Var vs

   fun primes n = implode (replicate "'" n)
   fun prime_Var (s,n) = s ^ (primes n)
   fun prime_Varstruct (vs,n) = let val s = primes n in map (fn v => v ^ s) vs end

   val unprime_Var = repeat_count takeprime_Var
   val unprime_Varstruct = repeat_count takeprime_Varstruct

   val isprime_Var = can takeprime_Var

end;


