(****************************************************************************)
(*                                                                          *)
(*               Copyright 1994, 1995 University of Cambridge               *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(* Z.pp.support.sml,v 1.2 1995/09/07 01:57:12 drs1004 Exp                                                                     *)
(* DESCRIPTION   : Support code for HOL pretty-printer.                     *)
(* AUTHOR        : R.J.Boulton/Donald Syme                                  *)
(*                                                                          *)
(****************************************************************************)

(*====================================================================*)
(* ZExternals                                                       *)
(*====================================================================*)

structure ZExternals =
struct
   val abbreviate_schemas_ref = ref true;
   fun abbreviate_schemas () = !abbreviate_schemas_ref;
   fun set_abbreviate_schemas b = abbreviate_schemas_ref := b;
   
   val extract_schemas_ref = ref true;
   fun extract_schemas () = !extract_schemas_ref;
   fun set_extract_schemas b = extract_schemas_ref := b;
   
   val subscriptZ_ref = ref true;
   fun subscriptZ () = !subscriptZ_ref;
   fun set_subscriptZ b = subscriptZ_ref := b;
   
end;


(*----------------------------------------------------------------------------
 - use "hol90_Z/pp/Z.pp.support.sml";  
 - val lab = {Branches=[],Kind=Portable.NONE,Look=Portable.NONE};
 - val s1 = rhs (concl (definition "BirthdayBook" "AddBirthday"));;
 - ZDestructors.SCHEMA (s1,lab);
 - val s2 = rhs (concl (definition "BirthdayBook" "BirthdayBook"));;
 - val delta_s2 = ZDestructors.mk_delta {schema=s2,varstruct= #varstruct (dest_pabs s2)};;
 - ZDestructors.dest_delta delta_s2;
 - ZDestructors.DELTA (delta_s2,lab);
 - ZDestructors.dest_delta (mk_delta {schema=(--`BirthdayBook`--),varstruct= #varstruct (dest_pabs s2)});

 - --`\(known,birthday). SCHEMA [BirthdayBook (known,birthday)] [] []`--;;
 - Z.dest_SCHEMA it;;
 - ZDestructors.SCHEMA (it,lab);;
 - ZDestructors.SCHEMA (--`SCHEMA [x IN NN] [EVEN x]`--,1);;
 - ZDestructors.SCHEMA (--`SCHEMA [x IN NN; y IN NN] [EVEN x; ODD y]`--,lab);;
 - ZDestructors.SCHEMA (--`SCHEMA [x IN NN; x' IN NN; y IN NN] [EVEN x; EVEN x'; ODD y]`--,1);;
 -
 - ZDestructors.SCHEMA_APPLICATION t;
 -----------------------------------------------------------------------------*)

signature ZDestructors_sig = sig
    structure PPLabels : PPLabels_sig
    structure HolBranches : HolBranches_sig
    structure HolTermKinds : HolTermKinds_sig

    val SCHEMA : term * HolBranches.branch PPLabels.label -> (term * HolBranches.branch PPLabels.label) list * (term * HolBranches.branch PPLabels.label) list * (term * HolBranches.branch PPLabels.label) list
    val SCHEMA_BINDER : term * HolBranches.branch PPLabels.label -> string * (term * HolBranches.branch PPLabels.label) * (term * HolBranches.branch PPLabels.label)
    val SCHEMA_BINOP : term * HolBranches.branch PPLabels.label -> string * (term * HolBranches.branch PPLabels.label) * (term * HolBranches.branch PPLabels.label)
(*    val SCHEMA_NOT : term * HolBranches.branch PPLabels.label -> (term * HolBranches.branch PPLabels.label) * (term * HolBranches.branch PPLabels.label) *)

    val SCHEMA_USAGE : term * HolBranches.branch PPLabels.label -> term * HolBranches.branch PPLabels.label
    val SCHEMA_DEFINITION : thm * HolBranches.branch PPLabels.label -> (term * HolBranches.branch PPLabels.label) * (term * HolBranches.branch PPLabels.label) 
    val DELTA : term * HolBranches.branch PPLabels.label -> term * HolBranches.branch PPLabels.label
    val XI : term * HolBranches.branch PPLabels.label -> term * HolBranches.branch PPLabels.label
end;

   
structure ZDestructors: ZDestructors_sig =
struct
   structure PPLabels = PPLabels;
   structure HolBranches = HolBranches;
   structure HolTermKinds = HolTermKinds;
   open PPLabels;
   open HolTermKinds;
   open HolBranches.HolTermPaths;
   open HolBranches;
   open Z;
   open Zxi_delta;

   fun replicate x 0 = []
     | replicate x n = (x::replicate x (n-1))


   fun label_list (h::t) label = 
          ((h,label_aBranches [TERM_BRANCH RAND,TERM_BRANCH RATOR] 
              (label_uKind (kind_of_term h) label))::label_list t (label_aBranch (TERM_BRANCH RAND) label))
     | label_list [] label = []

   fun path_through_gabs varstruct = 
       (TERM_BRANCH BODY)::(flatten (replicate [TERM_BRANCH BODY,TERM_BRANCH RAND] (length (strip_pair varstruct) - 1)))

   (* nb. path not correct *)
   fun DELTA (tm,label) =
      let val schema = #const (const_decl (dest_delta tm))
          val path = map TERM_BRANCH [RATOR,RAND,RATOR]@(path_through_gabs (varstruct_for_schema tm)) in
         (schema,label_uKind (kind_of_term schema) (label_aBranches path label))
      end;

   fun XI (tm,label) =
      let val schema =  #const (const_decl (dest_xi tm))
          val path = map TERM_BRANCH [RATOR,RAND,RATOR,RAND,RATOR,RATOR]@(path_through_gabs (varstruct_for_schema tm)) in
         (schema,label_uKind (kind_of_term schema) (label_aBranches path label))
      end;

   fun SCHEMA_BINDER (tm,label) =
      let val ({body,schema},binder) =  
                  (dest_pred_schema_forall tm, "!")
                  handle _ => (dest_pred_schema_exists tm, "?")
          val bodypath = map TERM_BRANCH [RAND]@path_through_gabs (varstruct_for_schema (rand tm))@map TERM_BRANCH [RAND]
          val schemapath = map TERM_BRANCH [RATOR,RAND,RAND,RATOR]@path_through_gabs (varstruct_for_schema (rand tm))@map TERM_BRANCH [RAND] in
         (binder,
          (schema,label_aBranches schemapath label),
          (body,label_aBranches bodypath label))
      end;

   fun SCHEMA_BINOP (tm,label) =
      let val ((l,r),binop) =  
                  (dest_schema_or tm, "\\/")
                  handle _ => (dest_schema_and tm, "/\\")
                  handle _ => (dest_schema_imp tm, "==>")
          val lpath = map TERM_BRANCH [RAND,RATOR]@path_through_gabs (varstruct_for_schema (rand tm))
          val rpath = map TERM_BRANCH [RAND]@path_through_gabs (varstruct_for_schema (rand tm)) in
         (binop,
          (l,label_aBranches lpath label),
	  (r,label_aBranches rpath label))
      end;

   fun SCHEMA_USAGE (tm,label) = 
      let val schema = dest_schema_usage tm in
         (schema,label_uKind (kind_of_term schema) (label_aBranches [TERM_BRANCH RATOR] label))
      end;

   fun SCHEMA (tm,label) = 
      let val {schemas,body,decs}=dest_SCHEMA tm 
          val ptvs = path_through_gabs (varstruct_for_schema tm) in
         (label_list schemas (label_aBranches (map TERM_BRANCH [RAND,RATOR,RATOR]@ptvs) label),
          label_list decs (label_aBranches (map TERM_BRANCH [RAND,RATOR]@ptvs) label),
          label_list body (label_aBranches (map TERM_BRANCH [RAND]@ptvs) label))
      end;
   exception SCHEMA_DEFINITION_EXN;

   fun SCHEMA_DEFINITION (thm,label) = 
      let val {lhs=schema_con,rhs=schema_def} = dest_eq (concl thm) in
         if (null (hyp thm) andalso is_defined_schema (#Name (dest_const schema_con))) 
         then 
            ((schema_con,label_uKind (kind_of_term schema_con) (label_aBranches [TERM_BRANCH RAND,TERM_BRANCH RATOR, CONCL] label)),
             (schema_def,label_uKind (kind_of_term schema_def) (label_aBranches [TERM_BRANCH RAND, CONCL] label)))
         else (raise SCHEMA_DEFINITION_EXN)
      end;
       
end;
   
(*====================================================================*)
(*                                                                    *)
(*====================================================================*)

structure HolExternals = struct
	open HolExternals;
	open ZExternals;
end;
   
structure HolDestructors = struct
	open HolDestructors;
	open ZDestructors;
end;
   

