#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




#----------------------------------------------------------------------------
#
# WIDGET CLASS TkTheoryViewer
#
# OPTIONS
#       -theory
#               The initial theory to display
#
# COMMANDS
#
#----------------------------------------------------------------------------

proc TkTheoryViewer { w args } {
        global feedback
        global TkTheoryViewer_flags
        global gui_flags
        global vals

        set vals($w,main) 1
        for {set i 0} {$i<[llength $args]} {incr i} {
            case [lindex $args $i] -theory {
                incr i
                set theory [lindex $args $i]
            } -main {
                incr i
                set vals($w,main) [lindex $args $i]
            } default {
                error "unrecognized arg [lindex $args $i]"
            }
        }

        global busy
        incr busy 1

        toplevel $w -class TkTheoryViewer
        wm minsize $w 1 1
        catch {wm withdraw $w} 

        if ![info exists theory] { set theory [ML current_theory()] }
        
        set vals($w,theory) $theory
                                                              
        set feedback [list {} "Displaying theory $theory"]

        # 3. Make the internals. First make a whole lot of frame
        # to put the different subcomponents in.


        # 3a.  Make the Parents frame.

        if $vals($w,main) {
            pack [collapsible TheoryTree $w.theorytree \
                -title "Theory Tree" \
                -visible 1 \
                -multiselect 1 \
                -height 150 \
                -width 650 \
                -relief sunken \
                -command "TkTheoryViewer::change_theory $w"] \
            -side top -fill both
        }

        # 3b. Make the Theorems, Axioms and Definitions Frames.
        #
        # When the frames are expanded/collapsed we must alter the
        # packing on the frame so they don't take up extra space.  Hence
        # the -collapseCommand argument.

            pack [collapsible frame $w.axiom \
                        -title "Axioms:" \
                        -collapseCommand "pack config $w.axiom -expand " \
                        -visible 0] \
                -fill both -expand no
            pack [theoremsframe $w.axiom.b.tf] \
                -fill both -expand yes -side bottom
            $w.axiom.b.tf richtextobjects configure \
                -selectionvar vals($w,axiom,selection)
            $w.axiom.b.tf richtextobjects richtext text configure \
                -height 6 \
                -width 65

            set vals($w,axiom,selection) ""


            pack [collapsible frame $w.theorem \
                        -title "Theorems:" \
                        -collapseCommand "pack config $w.theorem -expand " \
                        -visible 0] \
                -fill both -expand no
            pack [theoremsframe $w.theorem.b.tf] \
                -fill both -expand yes -side bottom
            $w.theorem.b.tf richtextobjects configure \
                -selectionvar vals($w,theorem,selection)
            set vals($w,theorem,selection) ""
            $w.theorem.b.tf richtextobjects richtext text configure \
                -height 6 \
                -width 65
            pack [buttonbar $w.theorem.b.bb] -fill x
            $w.theorem.b.bb addbutton sensitive button \
                        -sensitivevar vals($w,theorem,selection) \
                        -sensitiveexpr "\[llength \$vals($w,theorem,selection)\]!=0" \
                        -text "Replay Proof" \
                        -command "TkTheoryViewer::replay_object $w $w.theorem.b.tf"
            foreach package $TkTheoryViewer_flags(proof_packages) {
                $w.theorem.b.bb addbutton button \
                    -text [lindex $package 1] \
                    -command "newwin [lindex $package 0]"
            }


            pack [collapsible frame $w.definition \
                        -title "Definitions:" \
                        -collapseCommand "pack config $w.definition -expand " \
                        -visible 0] \
                -fill both -expand no
            pack [theoremsframe $w.definition.b.tf] \
                -fill both -expand yes -side bottom
            $w.definition.b.tf richtextobjects configure \
                -selectionvar vals($w,definition,selection)
            set vals($w,definition,selection) ""
            $w.definition.b.tf richtextobjects richtext text configure \
                -height 6 \
                -width 65
            pack [buttonbar $w.definition.b.bb] -fill x
            $w.definition.b.bb addbutton sensitive button \
                        -sensitivevar vals($w,definition,selection) \
                        -sensitiveexpr "\[llength \$vals($w,definition,selection)\]!=0" \
                        -text "Replay Definition" \
                        -command "TkTheoryViewer::replay_object $w $w.definition.b.tf"
            foreach package $TkTheoryViewer_flags(definition_packages) {
                $w.definition.b.bb addbutton button \
                    -text [lindex $package 1] \
                    -command "newwin [lindex $package 0]"
            }
        
        # 4. Display the theory into these internals.

        TkTheoryViewer::display_theory $w

        TkTheoryViewer::remake_menus $w

        # 5. Destruction stuff.

        bind $w <Destroy> "+TkTheoryViewer::upon_destroy $w %W"
        wm protocol $w WM_DELETE_WINDOW "TkTheoryViewer::verify_destroy $w"


        # 6. 

        update
        wm deiconify $w
        tkwait visibility $w
        update
        if $vals($w,main) {
            focus_goToFirst $w.theorytree
        } else {
            focus_goToFirst $w.axiom
        }
        

        ML -type void "TclCurrentTheoryNotification.add_client \n(TclCurrentTheoryNotification.mk_client([ml_string TkTheoryViewer],\n[ml_string $w]))"
        set feedback [list {} "Please wait..."]

        incr busy -1
        return .
}


#----------------------------------------------------------------------------
# TkTheoryViewer::change_theory
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::change_theory { w theories } {
        global vals
        global busy
        global feedback
        incr busy 1
        if {[llength $theories]==1} {
            set theory [lindex $theories 0]
            if {$vals($w,theory)!=$theory} {
                set feedback [list $w "Opening theory $theory..."]
                TkTheoryViewer::undisplay_theory $w
                set vals($w,theory) $theory
                TkTheoryViewer::display_theory $w
            }
        }
        incr busy -1
}



#----------------------------------------------------------------------------
# TkTheoryViewer::verify_destroy 
#
# Called when the user tries to destroy the TkTheoryViewer via
# WM_DELETE_WINDOW window manager protocol or by the "Close" menu option.
#----------------------------------------------------------------------------

proc TkTheoryViewer::verify_destroy { w } {
        global gui_flags
        if $gui_flags(debug) { puts "TkTheoryViewer::verify_destroy (a)"}
        global busy
        incr busy
        global feedback
        set feedback [list $w "Please wait..."]
        wm withdraw $w
        after 1 "destroy $w"
        incr busy -1
        if $gui_flags(debug) { puts "TkTheoryViewer::verify_destroy (b)"}
}


#----------------------------------------------------------------------------
# TkTheoryViewer::upon_destroy - called just as the TkTheoryViewer is destroyed.
#----------------------------------------------------------------------------

proc TkTheoryViewer::upon_destroy { w realw } {
    if {$w==$realw} {
        global vals
        global gui_flags
        if $gui_flags(debug) { puts "TkTheoryViewer::upon_destroy (a)"}
        ML -type void "TclCurrentTheoryNotification.remove_client \n(TclCurrentTheoryNotification.mk_client([ml_string TkTheoryViewer],\n[ml_string $w]))"
        if $gui_flags(debug) { puts "TkTheoryViewer::upon_destroy (b)" }
        unset_vals_for_widget $w
        if $gui_flags(debug) { puts "TkTheoryViewer::upon_destroy (c)" }
    }
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::menu_entries_for_packages { w packages } {
        set entries ""
        foreach package $packages {
            set p [lindex $package 0]
            set text [lindex $package 1]
            lappend entries [list command "$text..." "newwin [set p]" {}]
        }
        return $entries

}


#----------------------------------------------------------------------------
# remake_menus
#
#
# 1. Determine the mode and whether we need to do anything or not.
# Also find the list of theories to show within
# Open Other =>.
#
# Nb. do not call "ancestors".  It takes ages.
#
# 
# 2. Set up the menus using MakeMenus
#
# 3. Create the Open Other cascade menu.  There is a bug here, as
# when new parents and libraries are added they are not reflected
# in this menu.  They will, however, always be available from the
# menu for the current theory.
# 
# 4. Adjust the disabled/enabled states of the menu entries under the View
# menu according to the values of vals($w,exist_<objtype>)
#
# Note these menu options are in existence in all modes, hence we do not
# have to check if the menu options exist.
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::remake_title { w } {
        global gui_flags
        global vals
        if [ML -type bool "draft_mode()"] {
            set vals($w,mode) draft
        } else {
            set vals($w,mode) proof
        }
        if $vals($w,main) {
            set title "TkHol: \[$vals($w,theory)\]"
        } else {
            set title "\[$vals($w,theory)\]"
        }

        wm title $w $title
        wm iconname $w $title
}

proc TkTheoryViewer::remake_menus { w } {
        global vals 
        global TkTheoryViewer_flags     
        global gui_flags     

        foreach win [info commands $w.menu.*] {
            catch {destroy $win}
        }
        set viewmenu_entries ""
        lappend viewmenu_entries [list command "Expand All" "TkTheoryViewer::expandall $w" 0]
        if $vals($w,main) {
            lappend viewmenu_entries [list sep]
            lappend viewmenu_entries [list command "Rescan Theory Tree" "TheoryTree::remake_hierarchy $w.theorytree.b" 0]
        }
        set viewmenu [list view "View" 0 left $viewmenu_entries]

        set helpmenu [list help "Help" 0 left [Help::menu_entries TkTheoryViewer]]

        set theoremsmenu [list theorems "Proofs" 0 left [TkTheoryViewer::menu_entries_for_packages $w $TkTheoryViewer_flags(proof_packages)]]
        set definitionsmenu [list definitions "Definitions" 0 left [TkTheoryViewer::menu_entries_for_packages $w $TkTheoryViewer_flags(definition_packages)]]
        set axiomsmenu [list axioms "Axioms" 0 left [TkTheoryViewer::menu_entries_for_packages $w $TkTheoryViewer_flags(axiom_packages)]]

        set searchmenu                                                  \
        [list search "Search" 0 left                                    \
            [list                                                       \
                [list command "Selection..." "newwin TkTRS -sources \[hierarchy::selection $w.theorytree.b.hier\]" 0]                   \
                [list command "Ancestors of selection..." "newwin TkTRS -sources \[hierarchy::selection $w.theorytree.b.hier\] -ancestors 1" 0] \
                [list command "Descendants of selection..." "newwin TkTRS -sources \[hierarchy::selection $w.theorytree.b.hier\] -descendants 1" 0] \
                [list command "All Theories..." "incr busy; newwin TkTRS -sources \[ML -type stringlist \[hol_ancestry\]\]; incr busy -1" 0]    \
            ]                                                           \
        ]

        set entries ""
        foreach preference_group [preferences::groups] {
            lappend entries [list command "$preference_group..." "preferences -group [list $preference_group]" {}]
        }
        lappend entries [list sep]
        lappend entries [list command "Save All Options" "PreferencesSave" 0]
        set preferencesmenu [list preferences "Options" 0 left $entries ]

        set entries [TkTheoryViewer::menu_entries_for_packages $w $TkTheoryViewer_flags(other_packages)]
        set entries [linsert $entries 0 [list command "Load Package..." "TkTheoryViewer::load_package $w" 0]]
        set packagesmenu [list other "Packages" 0 left $entries ]

        set filemenu_entries ""
        if $vals($w,main) {
            lappend filemenu_entries [list command "New Theory..." "TkTheoryViewer::new_theory $w" 0]
#           lappend filemenu_entries [list command "Load Theory..." "TkTheoryViewer::load_theory $w" 5]
#           lappend filemenu_entries [list command "Add Parent..." "TkTheoryViewer::new_parent $w" 0]
            lappend filemenu_entries [list command "Save Theory" "TkTheoryViewer::export_theory $w" 0]
            lappend filemenu_entries [list sep]
            lappend filemenu_entries [list command "Load ML File..." "TkTheoryViewer::load_mlfile $w" 5]
            lappend filemenu_entries [list command "Load Library..." "TkTheoryViewer::load_library $w" 0]
            if {$vals($w,mode)=="draft"} {
                lappend filemenu_entries [list command "Proof Mode" "TkTheoryViewer::enter_mode $w proof" 0]
            } else {
                lappend filemenu_entries [list command "Draft Mode" "TkTheoryViewer::enter_mode $w draft" 0]
            }
            lappend filemenu_entries [list sep]
            lappend filemenu_entries [list command "Save HOL Image..." "TkTheoryViewer::save_hol $w" 0]
            lappend filemenu_entries [list sep]
            lappend filemenu_entries [list command "Browse/Replay Source For Theory" "TkTheoryViewer::analyse_files_for_theory $w" 0]
            lappend filemenu_entries [list command "Browse/Replay External ML File..." "TkTheoryViewer::analyse_file $w" 0]
            lappend filemenu_entries [list sep]
        }
        lappend filemenu_entries [list cascade "View Other" $w.menu.file.m.ancestors 0]
        lappend filemenu_entries [list cascade "View Other (new window)" $w.menu.file.m.ancestors2 0]
        lappend filemenu_entries [list sep]
        lappend filemenu_entries [list command "Close Window" "TkTheoryViewer::verify_destroy $w" 0]
        if $vals($w,main) {
            lappend filemenu_entries [list command "Exit" "after 1 exit" 1]
        }

        set filemenu [list file "File" 0 left $filemenu_entries]

        lappend menus $filemenu
        lappend menus $viewmenu
        if $vals($w,main) {
            if {[llength [lindex $searchmenu 4]]>0} { lappend menus $searchmenu }
            if {[llength [lindex $axiomsmenu 4]]>0} { lappend menus $axiomsmenu }
            if {[llength [lindex $definitionsmenu 4]]>0} { lappend menus $definitionsmenu }
            if {[llength [lindex $theoremsmenu 4]]>0} { lappend menus $theoremsmenu }
            lappend menus $preferencesmenu
            lappend menus $packagesmenu $helpmenu
        }
        MakeMenus $w $menus


        set full_ancestors [lsort [ML -type stringlist [hol_ancestry]]]
        foreach menu [list $w.menu.file.m.ancestors $w.menu.file.m.ancestors2] {
            menu $menu
            set ancestors $full_ancestors
            while {[llength $ancestors]!=0} {
                set head [lrange $ancestors 0 29]
                set ancestors [lrange $ancestors 30 end]
                foreach ancestor $head {
                    if {$menu=="$w.menu.file.m.ancestors"} {
                        set command "TkTheoryViewer::change_theory $w $ancestor"
                    } else {
                        set command "newwin TkTheoryViewer -theory $ancestor -main 0"
                    }
                    $menu add command -label $ancestor  \
                    -font $gui_flags(font,menus) \
                    -command $command   \
                    -underline 0
                }
                if {[llength $ancestors]!=0} {
                    $menu add cascade -label "More"     \
                    -menu $menu.menu \
                    -underline 0
                    set menu $menu.menu
                   menu $menu
                }
            }
        }
        

#       if [hol90] { 
#           $w.menu.view.m entryconfigure "*Rich*" -state disabled
#       }

}


#----------------------------------------------------------------------------
# Handle different menu actions...
#----------------------------------------------------------------------------


proc TkTheoryViewer::expandall { w } {
        global busy
        incr busy
        foreach kind [list theorem definition axiom types constants] { 
            TkTheoryViewer::expandkind $w $kind
        }
        incr busy -1
}



proc TkTheoryViewer::expandkind { w kind } {
        if {[winfo exists $w.$kind.b]} { 
            global busy
            incr busy
            global feedback
            set feedback [list $w "Expanding [set kind]s"]
            $w.$kind.b.tf richtextobjects expandall
            incr busy -1
        }
}

proc TkTheoryViewer::analyse_files_for_theory { w } {
       global vals
       set status [catch {set files [ML -type stringlist "FileAnalysis.files_for_theory [ml_string $vals($w,theory)]"]} err]
       if $status {
           errormessage .error "The script(s) for the theory \"$vals($w,theory)\" could not be found. (error: $err"
           return
       }
       foreach file $files {
           newwin SourceBrowser -file $file
       }
}

proc TkTheoryViewer::analyse_file { w } {
        set file [selectfile -new 1 -dirvariable pwd -pattern *.sml -message "Select a theory construction file to analyse and browse:"]
        if {$file != ""} {
            newwin SourceBrowser -file $file
        }
}


proc TkTheoryViewer::replay_object { w theoremsframe } {
        global vals
        global busy
        incr busy
        global feedback
        global gui_flags
        set selection [$theoremsframe richtextobjects selection]
        # puts "selection = $selection"
        set selectedthm [lindex $selection 0]
        set objspec [$theoremsframe richtextobjects objspec_for_object $selectedthm]
#       puts "objspec = $objspec"
        set thmspec [lindex $objspec 1]
        set thmname [lindex $thmspec 0]
        set theory [lindex $thmspec 2]
        set feedback [list $w "Parsing script for $theory"]
        
        set status [catch {set files [ML -type stringlist "FileAnalysis.files_for_theory [ml_string $vals($w,theory)]"]} err]
        if $status {
           errormessage .error "The script(s) for the theory \"$vals($w,theory)\" could not be found. (error: $err"
           incr busy -1
           return
        }
        foreach file $files {
           if [catch {set fragindex [ML -type int "
               let fun index_of_first_success f \[\] n = raise Fail \"index_of_first_success\"
                     | index_of_first_success f (h::t) n = (f h; n) handle _ => index_of_first_success f t (n+1)
               in
                   index_of_first_success (fn frag => assert (mem [ml_string $thmname]) (#pseudonyms frag))
                        (FileAnalysis.fragments_for_file [ml_string $file]) 1
               end"]} err] {
                puts stderr "err: $err"
               continue
           }
           set tool [ML -type string "#edit_tool (el $fragindex (FileAnalysis.fragments_for_file [ml_string $file]))"]
           set argscode "#data (el $fragindex (FileAnalysis.fragments_for_file [ml_string $file]))"
           # puts "tool = $tool"
           # puts "argscode = $argscode"
           if [catch {newwin $tool -initial $argscode -title $thmname} err] {
               errormessage $w.error "Sorry, $gui_flags(title) cannot replay the operatoin by which this object was created.  Replaying forward proof and some kinds of definitions is not yet supported. (error: $err)"
           }
           incr busy -1
           return
        }
        incr busy -1
        errormessage $w.error "The proof/definition for $selectedthm could not be found in any of the following files:\n[join $files \n]"
        return
}

proc TkTheoryViewer::undisplay_theory { w } {
        global vals 
        $w.theorem.b.tf richtextobjects setobjects {}
        $w.axiom.b.tf richtextobjects setobjects {}
        $w.definition.b.tf richtextobjects setobjects {}
}


proc TkTheoryViewer::display_theory { w } {
        global vals 
        global feedback
        global TkTheoryViewer_flags

        TkTheoryViewer::remake_title $w
        #
        # 1. Lookup what to display about the theory. Always display
        # definitions by default.
        #
        set theory $vals($w,theory)

        set vals($w,show_constants) 0
        set vals($w,show_types) 0

        set vals($w,show_parents) 1
        
        # Configure each of the sub-frames with information about what
        # to display.  Pack each of them into the display.

        if $vals($w,main) {
            TheoryTree::setselection $w.theorytree.b $vals($w,theory)
        }
        set theory $vals($w,theory)
        set mltheory [ml_string $theory]

        # set the defaults... I used to have theorems and 
        # definitions shown always by default.  I have now disabled
        # this.
        set vals($w,show_axioms) 0
        set vals($w,show_theorems) 0
        set vals($w,show_definitions) 0
        foreach pr [list [list axiom Axioms] [list theorem Theorems] [list definition Definitions]] {
            set objtype [lindex $pr 0]
            set label [lindex $pr 1]
            set feedback [list $w "Finding [set label]..."]
            set names [lsort [ML -type stringlist "map fst ([set objtype]s $mltheory)"]]
            set specs ""
            foreach name $names {
                lappend specs [list $name [list THM [list $name $objtype $theory]]]
            }
            set vals($w,show_[set objtype]s) [expr {$vals($w,show_[set objtype]s) || [llength $names]!=0}]
            eval [list $w.$objtype.b.tf richtextobjects setobjects] $specs
            $w.$objtype.b.tf configure \
                -trace [list $theory $objtype [list collapsible_show $w.$objtype]]
        }

        foreach objtype [list axiom theorem definition] {
            if {[set vals($w,show_[set objtype]s)]} {
                collapsible_show $w.$objtype
            } else {
                collapsible_hide $w.$objtype
            }
        }
        set vals($w,theory_is_current) [expr {[ML current_theory()]==$vals($w,theory)}]
}



#----------------------------------------------------------------------------
# TkTheoryViewer::enter_mode
#
# Called when the user selects a new mode from the menu.
#----------------------------------------------------------------------------

proc TkTheoryViewer::enter_mode { w mode } {
        global vals
        global busy
        incr busy 1
        global feedback
        set feedback [list $w "Entering $mode mode..."]
        if {$vals($w,mode)==$mode} {
            return
        } else {
            case $mode draft {
                ML -type void -log 1 "extend_theory (current_theory())"
            } proof {
               ML -type void -log 1 "close_theory()"
            }
        }       
        incr busy -1
}


#----------------------------------------------------------------------------
# TkTheoryViewer::current_theory_mode_change_Notify
#
# Gets called when the mode of the current theory is changed in
# the HOL session, either via the interface or some other means.  
# See the HolTheoryNotification package for
# how.
#
# TkTheoryViewer::theory_graph_change_Notify
#
# Gets called when the theory graph changes.  We context-switch
# to the current theory.
#
#----------------------------------------------------------------------------


proc TkTheoryViewer::current_theory_mode_change_Notify { w isdraft } {
        global busy
        incr busy 1
        TkTheoryViewer::remake_title $w
        TkTheoryViewer::remake_menus $w
        incr busy -1
}

proc TkTheoryViewer::theory_graph_change_Notify { w new_current } {
        global vals
        if $vals($w,main) {
            TheoryTree::remake_hierarchy $w.theorytree.b
            if {$vals($w,theory) != $new_current} {
                TkTheoryViewer::change_theory $w $new_current
            }
        }

}
                                                                         
#----------------------------------------------------------------------------
# TkTheoryViewer::load_theory
#
# Load a new theory as the current theory.  
# This will normally mean the current theory will have
# to enter "view" mode, hence recompute_mode is called for the
# current theory if everything else succeeds.
#----------------------------------------------------------------------------

                                                     
proc TkTheoryViewer::load_theory { w } {
        global vals
        if [ML -type bool "draft_mode()"] {
            errormessage $w.error "You cannot load a theory while in draft mode.\nFirst change to proof mode using the option under the File menu."
            return
        }
        set theory_pathname [selectfile -dirvariable pwd -pattern *.[hol_thryext] -message "Select a descendant theory:"]
        if {$theory_pathname!=""} {
            set theory_segment [file rootname [file tail $theory_pathname]]
            set theory_dir [file dirname $theory_pathname]/
            if {![ML -type bool "mem [ml_string $theory_dir] (!Globals.theory_path)"]} {
                    ML -log 1 -type any -check 1 "Globals.theory_path := ([ml_string $theory_dir])::(!Globals.theory_path)"
            }
            if [catch {ML -type void -log 1 -check 1 "load_theory [ml_string $theory_segment]"} err] {
                errormessage $w.err $err
            }
        }
}

#----------------------------------------------------------------------------
# TkTheoryViewer::load_mlfile
#
#----------------------------------------------------------------------------

                                                     
proc TkTheoryViewer::load_mlfile { w } {
        global feedback
        set file [selectfile -dirvariable pwd -pattern *.[ml_file_ext] -message "Select a ML file to load:"]
        if {$file!=""} {
            set feedback [list $w "Loading $file..."]
            if [hol90] { ML -toplevel 1 -log 1 -trace 1 "use \"$file\"" }
        }
}

#----------------------------------------------------------------------------
# TkTheoryViewer::new_parent
#
# Adds a parent to the existing current theory.
#
# Problem: Should add the new parent theory(ies) to the Open Other => 
# cascade menu of every theory.  This would be easier if the cascade
# menu was shared amongst all TkTheoryViewers.  This should happen
# via notifications on any new_parent calls.
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::new_parent { w } {
        global vals                             
        set theory_pathname [selectfile -dirvariable pwd -pattern *.[hol_thryext] -message "Select the new parent to add:"]
        if {$theory_pathname!=""} {
            set theory_segment [file rootname [file tail $theory_pathname]]
            ML -type void -log 1 -check 1 "new_parent [ml_string $theory_segment]"
        }
}


#----------------------------------------------------------------------------
# TkTheoryViewer::load_library
#
# Problem: Should add the new parent theory(ies) to the Open Other => 
# cascade menu of every theory.  This would be easier if the cascade
# menu was shared amongst all TkTheoryViewers.  This should happen
# via notifications on any new_parent calls.
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::load_library { w } {
        global vals                             
        global feedback
        set feedback [list $w "Looking for libraries..."]
        catch {unset vals($w,waiter)}
        modalDialog transient LoadLibraryBox $w.loadlib \
                -header "Select a library to load:" \
                -resultsvariable vals($w,waiter)
        tkwait variable vals($w,waiter)
        modalDialog.end $w.loadlib
        update
        global busy
        incr busy
        if {[lindex $vals($w,waiter) 0] == "Ok"} {
            set library [lindex $vals($w,waiter) 1]
            set feedback [list $w "Loading library $library..."]
            if [catch {hol_load_library $library} err] {
                errormessage $w.error $err; incr busy -1; return
            }
        }
        incr busy -1
}


#----------------------------------------------------------------------------
# TkTheoryViewer::load_package
#
# Adds a parent to the existing current theory.
#
# Problem: Should add the new parent theory(ies) to the Open Other => 
# cascade menu of every theory.  This would be easier if the cascade
# menu was shared amongst all TkTheoryViewers.  This should happen
# via notifications on any load_package calls.
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::load_package { w } {
        global vals                             
        set packagedir [selectfile \
                -dirvariable pwd \
                -message "Select the directory containing the package:"\
                -dironly 1 \
                -action "Add Package"]
        if {$packagedir!=""} {
            load_package $packagedir
        }
}


#----------------------------------------------------------------------------
# TkTheoryViewer::new_theory
#
# Creates a new theory.
#
# This will normally mean the current theory will have
# to enter "view" mode, hence recompute_mode is called for the
# current theory if everything else succeeds.
#----------------------------------------------------------------------------

proc TkTheoryViewer::new_theory { w } {
        global TkHol_flags
        global TkTheoryViewer_flags
        global busy
        global gui_flags
        set res [GetString $w.get "NewTheory" "Theory Name: "]
        if {[lindex $res 0]=="Ok"} {
            incr busy
            set newtheory [lindex $res 1]
            set newscript [Scripts::script_for_theory $newtheory]
            if [file exists $newscript] {
                switch -- [choice $w.verify \
                        -text "$newscript already exists.  Should $gui_flags(title) delete it?" \
                        -buttons [list Yes No Cancel]] {
                   Yes { exec rm $newscript }
                   No { }
                   Cancel { return }
                }
            }
            set newtheoryfile $newtheory.[hol_thryext]
            if [file exists $newtheoryfile] {
                switch -- [choice $w.verify2 \
                        -text "Theory file $newtheoryfile already exists.  Should $gui_flags(title) delete it?" \
                        -buttons [list Yes Cancel]] {
                   Yes { exec rm $newtheoryfile }
                   Cancel { return }
                }
            }
            set script "new_theory [ml_string $newtheory]"
            if [catch {ML -type void -log 1 $script} err] {
                errormessage $w.err $err; incr busy -1; return
            }
            set curr [Scripts::current_script]
            if {[choice $w.verify3 \
                        -text "Should $gui_flags(title) create and manage $curr for you?" \
                        -buttons [list [list "Yes Thanks" "Yes"] [list "No thanks, I'll do it myself" "No"]]]=="Yes"} {
                set TkHol_flags(managingScripts) 1
                Scripts::add_to_script "$script[ml_end_dec]"
#               newwin SourceBrowser -title "Record Of Construction: \[$curr\]" -file $curr
            } else {
                set TkHol_flags(managingScripts) 0
            }
            incr busy -1
        }
}


#----------------------------------------------------------------------------
# TkTheoryViewer::save_hol
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::save_hol { w } {
        global vals
        global feedback
        set image_pathname [selectfile -dirvariable pwd -new 1 -pattern * -message "Select a filename for the new executable:"]
        global busy
        incr busy
        set feedback [list $w "Saving the HOL image to disk..."]
        if {$image_pathname!=""} {
            if [hol90] {
                set function save_hol
            }
            if [catch {ML -type void "$function [ml_string $image_pathname]"} err] {
                errormessage $w.err $err; incr busy -1; return
            }
        }
        incr busy -1
}


#----------------------------------------------------------------------------
# TkTheoryViewer::export_theory
#
#----------------------------------------------------------------------------

proc TkTheoryViewer::export_theory { w } {
        global vals
        global feedback
        global busy
        incr busy
        set feedback [list $w "Exporting the current theory to disk..."]
        if [catch {ML -type void -log 1 "export_theory()"} err] {
            errormessage $w.err $err; incr busy -1; return
        }
        incr busy -1
}


#----------------------------------------------------------------------------
# Test routine used from interactive tcl.
#----------------------------------------------------------------------------

proc te { } {
   after 1 "
        source \$TkTheoryViewer_library/src/TkTheoryViewer.tcl
        source \$TkTheoryViewer_library/src/TheoryTree.tcl
        source \$hierarchy_library/src/hierarchy.tcl
        source \$RichText_library/src/TheoremsFrame.tcl
        source \$RichText_library/src/RichText.tcl
        newwin TkTheoryViewer -withfeedback 1
    "
}  



