#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




option add *shell.inputcolor blue widgetDefault
option add *shell.outputcolor black widgetDefault
option add *shell.command "uplevel #0" widgetDefault
option add *shell.initialtext "" widgetDefault
option add *shell.prompt "%" widgetDefault

proc shell { w args} {
        global TkHolShell_flags
        global vals
        set passon_args ""
        for {set i 0} {$i<[llength $args]} {incr i} {
            case [lindex $args $i]  -prompt {
                incr i
                set prompt [lindex $args $i]
            } -inputcolor {
                incr i
                set inputcolor [lindex $args $i]
            } -outputcolor {
                incr i
                set outputcolor [lindex $args $i]
            } -command {
                incr i
                set command [lindex $args $i]
            } -initialtext {
                incr i
                set initialtext [lindex $args $i]
            } default {
                lappend passon_args [lindex $args $i]
                incr i
                lappend passon_args [lindex $args $i]
            }
        }
        frame $w -class Shell
        eval [list text $w.text -relief sunken -borderwidth 2] $passon_args
        pack $w.text -expand yes -fill both

        if ![info exists inputcolor] { set inputcolor [option get $w inputcolor InputColor] } 
        if ![info exists outputcolor] { set outputcolor [option get $w outputcolor OutputColor] }
        if ![info exists command] { set command [option get $w command Command] }
        if ![info exists initialtext] { set initialtext [option get $w initialtext InitialText] }
        if ![info exists prompt] { set prompt [option get $w prompt InitialText] }

        set vals($w,prompt) $prompt
        $w.text tag configure output -foreground $outputcolor
        $w.text tag configure input -foreground $inputcolor
        
        # remove any standard bindings
        # what a hack....
        foreach event [bind Text] {
            global tk_version
            if ![regexp {FocusIn|FocusOut|Destroy} $event] {
                if {$tk_version > 4.0} {
                    bind $w.text $event break
                } else {
                    bind $w.text $event "set x 1"
                }
            }
        }
        # reestablish some primitive bindings we just removed
        # what a hack....

        shell::shell_bind $w

        bind $w.text <Return> "shell::execute $w"

        shell::output $w $initialtext
        shell::output $w $prompt
        shell::prepare_for_input $w
        $w.text yview 0.0

        set vals($w,command) $command
        return $w
}


proc shell::execute { w } {
        global vals
        set code [shell::latest_command $w] 
        shell::fix_latest_command $w
        shell::output $w \n
        update idletasks
        catch {eval $vals($w,command) [list $code]} result
        shell::output $w $result
        if {$vals($w,prompt) != ""} { shell::output $w "\n$vals($w,prompt)" }
        shell::prepare_for_input $w
}


proc shell::output { w text } {
        $w.text insert end $text
        $w.text yview -pickplace insert
        $w.text tag add output "end - [string length $text] chars" end
        $w.text tag remove input "end - [string length $text] chars" end
}
        
proc shell::prepare_for_input { w } {
        $w.text insert end " "
        $w.text tag remove output "end - 1 c" end
        $w.text tag add input "end - 1 c" end
        $w.text mark set insert end
        $w.text yview -pickplace insert
}


proc shell::latest_command { w } {
        global vals
        return [$w.text get "output.last + 1 c" end]
}


proc shell::fix_latest_command { w } {
        global vals
#       puts "text = [$w.text get "output.last + 1 c" end]"
        if [info exists vals($w,history_index)] {
            set endindex [expr [llength $vals($w,history)]-1]
            set text [$w.text get "output.last + 1 c" end]
            set vals($w,history) [lreplace $vals($w,history) $endindex $endindex $text]
        } else {
            lappend vals($w,history) [$w.text get "output.last + 1 c" end]
        }
        if {[llength $vals($w,history)]>100} {
            set vals($w,history) [lrange $vals($w,history) 1 end]
        }
        catch {unset vals($w,history_index)}
}


######################################################################
###  TEXT MOVEMENT COMMANDS
######################################################################

proc shell::replace_command { w text } {
        $w.text delete "output.last + 1 c" end
        $w.text insert end $text
        $w.text mark set insert end
        $w.text yview -pickplace insert
}


#We use the existence of vals($w,history_index) to determine if we are on
# the "last" (current) history line.  Maybe there's a better wayu to
# do this??
proc shell::prev_command { w } {
    global vals
    if ![info exists vals($w,history_index)] {
        lappend vals($w,history) [$w.text get "output.last + 1 c" end]
        set vals($w,history_index) [expr [llength $vals($w,history)]-1]
    }
#       puts "history = $vals($w,history)"
#    puts "historyi_index = $vals($w,history_index)"
    if {$vals($w,history_index) > 0} {
        incr vals($w,history_index) -1
        shell::replace_command $w [lindex $vals($w,history) $vals($w,history_index)]
    }
    if {$vals($w,history_index) == [llength $vals($w,history)]-1} {
        unset vals($w,history_index)
        set endindex [expr [llength $vals($w,history)]-1]
        set vals($w,history) [lreplace $vals($w,history) $endindex $endindex]
    }
}


proc shell::next_command { w } {
    global vals
    if ![info exists vals($w,history_index)] {
        return
    }
#    puts "history = $vals($w,history)"
#    puts "history_index = $vals($w,history_index)"
    if {$vals($w,history_index) < [expr [llength $vals($w,history)]-1]} {
        incr vals($w,history_index) 1
        shell::replace_command $w [lindex $vals($w,history) $vals($w,history_index)]
    }
    if {$vals($w,history_index) == [llength $vals($w,history)]-1} {
        unset vals($w,history_index)
        set endindex [expr [llength $vals($w,history)]-1]
        set vals($w,history) [lreplace $vals($w,history) $endindex $endindex]
    }
}


#############################################################################
# 
# This file is a set of tk event bindings to make a window seem "Shell-like".
# That is, text may be typed in but existing text may not be edited.
# It is based originally on the jbindings file by Jay Sekora, and thus
# his copyright stll applies to those bits I guess.
#
######################################################################


######################################################################
# shell::selection_if_any - return selection if it exists, else {}
#   this is from kjx@comp.vuw.ac.nz (R. James Noble)
#   defined elsewhere, but copied here for portability
######################################################################

proc shell::selection_if_any {} {
  if {[catch {selection get} s]} {return ""} {return $s}
}


######################################################################
####
###  GENERAL BINDING ROUTINES
####
######################################################################

######################################################################
# shell::bindings:init - initialise info for bindings
######################################################################

proc shell::bindings:init {} {
  global vals
  set vals(repeat_count) 1
  set vals(cutbuffer) {}
  set vals(dragscroll,txnd) 0
  set vals(dragscroll,delay) 50
  set vals(prefix) 0
  set vals(scanpaste_time) 0
  set vals(scanpaste_paste) 1
}


######################################################################
# shell::bindings:repeatable tclcode - execute tclcode one or more times
######################################################################

proc shell::bindings:repeatable { tclcode } {
  global vals
  
  set vals(prefix) 0                            ;# no longer collectig digits
  if {$vals(repeat_count) > 1} {
    for {set jri 0} {$jri < $vals(repeat_count)} {incr jri} {
      uplevel 1 "eval [list $tclcode]"          ;# variables in caller
    }
    set vals(repeat_count) 1
    } else {
    uplevel 1 "eval [list $tclcode]"            ;# variables in caller
  }
}

######################################################################
# shell::bindings:clear_count - clear argument count
######################################################################

proc shell::bindings:clear_count {} {
  global vals
  set vals(repeat_count) 1
}

######################################################################
# shell::bindings:repeat4 - set repeat count to four, or multiply by four
######################################################################

proc shell::bindings:repeat4 {} {
  global vals
  
  if {$vals(repeat_count) < 1} {
    set vals(repeat_count) 1
  }
  set vals(repeat_count) [expr {$vals(repeat_count)*4}]
}

######################################################################
# shell::bindings:start_number digit - start a numeric argument
#   invalid if not bound to (a sequence ending in) a digit key
######################################################################

proc shell::bindings:start_number { digit } {
  global vals
  set vals(prefix) 1                    ;# collecting # prefix
  set vals(repeat_count) [expr "$digit"]
}

######################################################################
# shell::bindings:continue_number digit - continue a numeric argument
#   invalid if not bound to a digit key
######################################################################

proc shell::bindings:continue_number { digit } {
  global vals
  set vals(repeat_count) [expr {($vals(repeat_count)*10)+$digit}]
}
  

######################################################################
####
###  GENERAL TEXT BINDING ROUTINES
####
######################################################################





######################################################################
# shell::insert_nondigit A w - insert A into text widget w, clear arg flag
######################################################################

proc shell::insert_nondigit { A w } {
  global vals

  if {"$A" != ""} {
    shell::bindings:repeatable {
      $w.text insert insert $A
      $w.text yview -pickplace insert
    }
  }
}


######################################################################
# shell::insert_digit A w - insert digit A into w, unless collecting arg
######################################################################

proc shell::insert_digit { A w } {
  global vals
    
  if $vals(prefix) {
    shell::bindings:continue_number $A
    return 0
  } else {
    if {"$A" != ""} {
      shell::bindings:repeatable {
        $w.text insert insert $A
        $w.text yview -pickplace insert
      }
    }
  }
}

######################################################################
###  TEXT MOUSE COMMANDS
######################################################################

# shell::paste_selection w - insert selection into w
proc shell::paste_selection { w args } {
  set selection [shell::selection_if_any]
  
  if {[string length $selection] != 0} {
    global vals
  
    $w.text insert insert $selection
    $w.text yview -pickplace insert
  }
}

######################################################################
# routines to let scanning and pasting to double-up on the same button
# based on code by Tom Phelps <phelps@cs.berkeley.edu>
######################################################################

# shell::start_scan_or_paste w x y t - start a drag or paste, recording
#   current location and time so we can later decide whether to paste or drag
# BIND TO <ButtonPress-N>
proc shell::start_scan_or_paste { w x y t } {
  global vals
  $w.text scan mark $y
  set vals(scanpaste_placey) $y
  set vals(scanpaste_time) $t
  set vals(scanpaste_paste) 1
}

# shell::continue_scan w x y t - scan the entry, and mark the fact that
#   we're scanning, not pasting.
# BIND TO <BN-Motion>
proc shell::continue_scan { w x y t } {
  global vals
  $w.text scan dragto $y
  set vals(scanpaste_paste) 0
}

# shell::end_scan_or_paste w x y t - if it's
#   been less than 500ms since button-down, paste selection
# BIND TO <ButtonRelease-N>
proc shell::end_scan_or_paste { w x y t } {
  global vals

# $vals(scanpaste_paste) &&

  if {[expr {$t-$vals(scanpaste_time)}] < 500} {
    # shell::paste_selection $w
    catch {
      $w.text insert insert [selection get]
      $w.text yview -pickplace insert
    }
  }
}

######################################################################
# routines to allow scrolling during text selection
# from raines@cgibm1.slac.stanford.edu (Paul E. Raines)
######################################################################

proc shell::start_sel { w loc } {
    global tk_priv
    # used by Text routine tk_textSelectTo
    set tk_priv(selectMode) char
    
    if {[$w.text compare $loc > output.last]} { 
        $w.text mark set insert $loc
    }
    $w.text mark set anchor $loc
    if {[lindex [$w.text config -state] 4] == "normal"} {focus $w.text}
}

# shell::drag_sel w loc - begin dragging out selection
proc shell::drag_sel { w loc } {
  global vals

  set ypos [lindex [split $loc ","] 1]
  if {$ypos > [winfo height $w.text]} {
    if {!$vals(dragscroll,txnd)} {
      after $vals(dragscroll,delay) shell::extend_sel $w
    }
    set vals(dragscroll,txnd) 1
    set vals(dragscroll,direction) down
  } else {
    if {$ypos < 0} {
      if {!$vals(dragscroll,txnd)} {
        after $vals(dragscroll,delay) shell::extend_sel $w
      }
      set vals(dragscroll,txnd) 1
      set vals(dragscroll,direction) up
    } else {
      set vals(dragscroll,txnd) 0
      set vals(dragscroll,direction) 0
    }
  }

   if {!$vals(dragscroll,txnd)} {
        tk_textSelectTo $w.text $loc
  }
}

# shell::extend_sel w - drag out a selection, scrolling if necessary
proc shell::extend_sel { w } {
  global vals

  if {$vals(dragscroll,txnd)} {
    if {$vals(dragscroll,direction) == "down"} {
      tk_textSelectTo $w.text sel.last+1l
      $w.text yview -pickplace sel.last+1l
    } else {
      if {$vals(dragscroll,direction) == "up"} {
        tk_textSelectTo $w.text sel.first-1l
        $w.text yview -pickplace sel.first-1l
      } else { return }
    }
    after $vals(dragscroll,delay) shell::extend_sel $w
  }
}

# shell::end_sel w - finish a selection
proc shell::end_sel { w } {
  global vals
  set vals(dragscroll,txnd) 0
}

######################################################################
###  TEXT SCROLLING COMMANDS - fragile - assume widget has a scrollbar
######################################################################
# fragile---assumes first word of yscrollcommand is name of scrollbar!
# should catch case of no yscrollcommand!
# ALSO---should handle arguments (scroll by line rather than windowful)


proc shell::extend_sel_with_move { command w } {
    if [catch {$w.text index sel.first} selfirst_prior] {
        set selfirst_prior [$w.text index insert]
    }
    if [catch {$w.text index sel.last} sellast_prior] {
        set sellast_prior [$w.text index insert]
    }
    set insert_prior [$w.text index insert]
    shell::[set command] $w
    if {[$w.text index insert] == $insert_prior} { return }
    if {[$w.text index insert] < $selfirst_prior} {
        $w.text tag add sel insert $sellast_prior
        return
    }
    if {[$w.text index insert] > $sellast_prior} {
        $w.text tag add sel $selfirst_prior insert
        return
    }
    $w.text tag remove sel $selfirst_prior $sellast_prior
    if {[$w.text index insert] < $insert_prior} {
        $w.text tag add sel $selfirst_prior insert
        return
    }
    if {[$w.text index insert] > $insert_prior} {
        $w.text tag add sel insert $sellast_prior
        return
    }
}


proc shell::unselect_sel_then_move { command w } {
    catch {$w.text tag remove sel sel.first sel.last}
    shell::[set command] $w
}




proc shell::scroll_down { w } {
  global vals
  shell::bindings:clear_count
  
  set yscrollcommand [lindex [$w.text configure -yscrollcommand] 4]
  set scrollbar [lindex $yscrollcommand 0]      ;# cross fingers and hope!
  
  set currentstate [$scrollbar get]
  # following is buggy if lines wrap:
  set newlinepos [expr {[lindex $currentstate 2] + [lindex $currentstate 1]}]
  $w.text yview insert
}

proc shell::scroll_up { w } {
  global vals
  shell::bindings:clear_count
  
  set yscrollcommand [lindex [$w.text configure -yscrollcommand] 4]
  set scrollbar [lindex $yscrollcommand 0]      ;# cross fingers and hope!

  set currentstate [$scrollbar get]
  # following is buggy if lines wrap:
  set newlinepos [expr {[lindex $currentstate 2] - [lindex $currentstate 1]}]
  $w.text yview insert
}



# shell::home w - move to start of input
proc shell::home { w } {
  shell::bindings:repeatable {$w.text mark set insert "output.last + 1 c"}
  $w.text yview -pickplace insert
}

# shell:end w - move to end of input
proc shell:end { w } {
  shell::bindings:repeatable {$w.text mark set insert {output.last lineend}}
  $w.text yview -pickplace insert
}


# shell::left w - move left
proc shell::left { w } {
  shell::bindings:repeatable {$w.text mark set insert {insert - 1 char}}
  if {[$w.text compare insert <= output.last]} { $w.text mark set insert "output.last + 1 c "}
  $w.text yview -pickplace insert
}

# shell::right w - move right
proc shell::right { w } {
  shell::bindings:repeatable {$w.text mark set insert {insert + 1 char}}
  $w.text yview -pickplace insert
}


# shell::word_left w - move back one word
proc shell::word_left { w } {
  shell::bindings:repeatable {
    while {[$w.text compare insert != 1.0] &&
           [string match "\[ \t\n\]" [$w.text get {insert - 1 char}]]} {
      $w.text mark set insert {insert - 1 char}
    }
    while {[$w.text compare insert != 1.0] &&
           ![string match "\[ \t\n\]" [$w.text get {insert - 1 char}]]} {
      $w.text mark set insert {insert - 1 char}
    }
    if {[$w.text compare insert <= output.last]} { $w.text mark set insert "output.last + 1 c "}
    $w.text yview -pickplace insert
  }
}

# shell::word_right w - move forward one word
proc shell::word_right { w } {
  shell::bindings:repeatable {
    while {[$w.text compare insert != end] &&
           [string match "\[ \t\n\]" [$w.text get insert]]} {
      $w.text mark set insert {insert + 1 char}
    }
    while {[$w.text compare insert != end] &&
           ![string match "\[ \t\n\]" [$w.text get insert]]} {
      $w.text mark set insert {insert + 1 char}
    }
    if {[$w.text compare insert <= output.last]} { $w.text mark set insert "output.last + 1 c "}
    $w.text yview -pickplace insert
  }
}

######################################################################
###  TEXT DELETION COMMANDS
######################################################################

# shell::delete_right w - delete character at insert
proc shell::delete_right { w } {
    if [$w.text compare insert != end] {
      global vals
  
      shell::bindings:repeatable {
        $w.text delete insert
      }
    }
}

# j:tb:delete_selection_or_left w - delete character before insert
proc shell::delete_selection_or_left { w } {
  if [catch {              
      global j_teb
      set j_teb(modified,$w) 1
  
      set j_teb(cutbuffer) [$w.text get sel.first sel.last]
      if [$w.text compare sel.first <= output.last] {
          if [$w.text compare sel.last <= output.last] {
              error dummy
          } else {
              $w.text delete "output.last + 2 c" sel.last
          }
      } else {
          $w.text delete sel.first sel.last
      }
  }] {
    if {[$w.text compare insert != 1.0] && [lsearch [$w.text tag names {insert - 2 c}] output] == -1} {
      global j_teb
      set j_teb(modified,$w) 1
  
      shell::bindings:repeatable {
          $w.text delete {insert - 1 char}
          $w.text yview -pickplace insert
      }
    }
  }
}


# shell::delete_left_word w - move back one word
proc shell::delete_left_word { w } {
  shell::bindings:repeatable {
    $w.text mark set del_to insert
    while {[$w.text compare insert != 1.0] && [lsearch [$w.text tag names {insert - 1 c}] output] == -1 &&
           [string match "\[ \t\n\]" [$w.text get {insert - 1 char}]]} {
      $w.text mark set insert {insert - 1 char}
    }
    while {[$w.text compare insert != 1.0] && [lsearch [$w.text tag names {insert - 1 c}] output] == -1 &&
           ![string match "\[ \t\n\]" [$w.text get {insert - 1 char}]]} {
      $w.text mark set insert {insert - 1 char}
    }
    $w.text delete insert del_to
    $w.text yview -pickplace insert
  }
}

# shell::delete_right_word w - move forward one word
proc shell::delete_right_word { w } {
  shell::bindings:repeatable {
    $w.text mark set del_from insert
    while {[$w.text compare insert != end] &&
           [string match "\[ \t\n\]" [$w.text get insert]]} {
      $w.text mark set insert {insert + 1 char}
    }
    while {[$w.text compare insert != end] &&
           ![string match "\[ \t\n\]" [$w.text get insert]]} {
      $w.text mark set insert {insert + 1 char}
    }
    $w.text delete del_from insert
    $w.text yview -pickplace insert
  }
}

######################################################################
###  TEXT EMACS DELETION COMMANDS
######################################################################

# shell::kill_line_again w - delete insert to end-of-line, appending cutbuffer
proc shell::kill_line_again { w } {
  global vals
  
  # if no argument, DON'T kill "\n" unless it's only thing at insert
  if {$vals(repeat_count) < 2} {
    shell::bindings:clear_count                         ;# in case it's eg -1
    if {[$w.text index insert] == [$w.text index {insert lineend}]} then {
      append vals(cutbuffer) [$w.text get insert]
      $w.text delete insert
    } else {
      append vals(cutbuffer) [$w.text get insert {insert lineend}]
      $w.text delete insert {insert lineend}
    }
  } else {
    # with argument, kill that many lines (including "\n")
    shell::bindings:repeatable {
      append vals(cutbuffer) [$w.text get insert {insert lineend + 1 char}]
      $w.text delete insert {insert lineend + 1 char}
    }
  }
  
  set vals(repeat_count) 1
}

# shell::kill_line w - delete insert to end-of-line, setting cutbuffer
#   (arg handled by called procedure)
proc shell::kill_line { w } {
  global vals
  set vals(cutbuffer) {}
  shell::kill_line_again $w
}

# shell::kill_selection w - delete selected region, setting cutbuffer
proc shell::kill_selection { w } {
  global vals

  shell::bindings:clear_count

  set vals(cutbuffer) {}
  catch {
    set vals(cutbuffer) [$w.text get sel.first sel.last]
    if [$w.text compare sel.first <= output.last] {
        $w.text delete "output.last + 2 c" sel.last
    } else {
        $w.text delete sel.first sel.last
    }
  }
}

# shell::yank w - insert contents of cutbuffer
###   handling of argument needs changed---not count, but not ignored
proc shell::yank { w } {
  global vals

  shell::bindings:clear_count
  
  $w.text insert insert $vals(cutbuffer)
  $w.text yview -pickplace insert
}


######################################################################
###  TEXT EMACS MARK COMMANDS
######################################################################

######################################################################
######################################################################
# ROUTINES TO SET UP TEXT BINDINGS
######################################################################
######################################################################

proc shell::shell_bind { w } {
        global gui_flags
        global shell_flags
        global vals
        bind $w.text <Control-Key>              "shell::bindings:clear_count"
        bind $w.text <Escape>           { }
  
        # mouse bindings (for scrolling selections)
        bind $w.text <ButtonPress-1>            "shell::start_sel $w @%x,%y"
        bind $w.text <B1-Motion>                "shell::drag_sel $w @%x,%y"
        bind $w.text <ButtonRelease-1>  "shell::end_sel $w"
  
        # mouse bindings (for scanning and pasting)
        bind $w.text <2>                        "shell::start_scan_or_paste $w %x %y %t"
        bind $w.text <B2-Motion>                "shell::continue_scan $w %x %y %t"
        bind $w.text <ButtonRelease-2>  "shell::end_scan_or_paste $w %x %y %t"

        # Microsft style movement keys
        bind $w.text <Left>             "shell::unselect_sel_then_move left $w"
        bind $w.text <Right>            "shell::unselect_sel_then_move right $w"
        bind $w.text <Up>               "shell::unselect_sel_then_move prev_command $w"
        bind $w.text <Down>             "shell::unselect_sel_then_move next_command $w"
        bind $w.text <Home>             "shell::unselect_sel_then_move home $w"
        bind $w.text <End>              "shell::unselect_sel_then_move end $w"
        bind $w.text <Next>             "shell::scroll_down $w"
        bind $w.text <Prior>            "shell::scroll_up $w"
        bind $w.text <Control-Left>     "shell::unselect_sel_then_move word_left $w"
        bind $w.text <Control-Right>    "shell::unselect_sel_then_move word_right $w"
        bind $w.text <Shift-Left>       "shell::extend_sel_with_move left $w"
        bind $w.text <Shift-Right>      "shell::extend_sel_with_move right $w"

        bind $w.text <Shift-Home>       "shell::extend_sel_with_move home $w"
        bind $w.text <Shift-End>        "shell::extend_sel_with_move end $w"
        bind $w.text <Shift-Control-Left>       "shell::extend_sel_with_move word_left $w"
        bind $w.text <Shift-Control-Right>      "shell::extend_sel_with_move word_right $w"


  # this is for stupid sun4 keyboard...
    if {[info exists gui_flags(arch)] 
        && $gui_flags(arch)=="sun4" 
        && [info exists shell_flags(modify_sun4)] 
        && $shell_flags(modify_sun4)} {
        bind $w.text <KP_8>             "shell::scroll_down $w"
        bind $w.text <KP_9>             "shell::scroll_up $w"
        bind $w.text <KP_7>             "shell::unselect_sel_then_move home $w"
        bind $w.text <KP_0>             "shell::unselect_sel_then_move end $w"
        bind $w.text <F30>              "shell::extend_sel_with_move left $w"
        bind $w.text <F32>              "shell::extend_sel_with_move right $w"
        bind $w.text <Shift-KP_7>               "shell::extend_sel_with_move home $w"
        bind $w.text <Shift-KP_0>               "shell::extend_sel_with_move end $w"

        # Microsoft style cut & paste
        bind $w.text <Shift-Insert>     "shell::paste_selection $w"
    }

        # Emacs control and function keys
        bind $w.text <Control-Key-i>    "shell::insert_nondigit \"\t\" $w"
        bind $w.text <Control-Key-j>    "shell::insert_nondigit \"\n\" $w"
  
        # Regular deletion keys
        bind $w.text <Control-Key-h>    "shell::delete_selection_or_left $w"
        bind $w.text <Delete>           "shell::delete_selection_or_left $w"
        bind $w.text <BackSpace>                "shell::delete_selection_or_left $w"
  
        # Emacs deletion keys
        bind $w.text <Escape><Delete>   "shell::delete_left_word $w"
        bind $w.text <Escape><Key-d>    "shell::delete_right_word $w"
        bind $w.text <Control-d>                "shell::delete_right $w"

#       bind $w.text <Return>           "shell::insert_nondigit \"\n\" $w"
        bind $w.text <Any-Key>          "shell::insert_nondigit %A $w"
  
        # Emacs argument count
        bind $w.text <Control-Key-u>            "shell::bindings:repeat4"
        foreach d {0 1 2 3 4 5 6 7 8 9} {
            bind $w.text <Control-Key-u><Key-$d>        "shell::bindings:start_number %A"
            bind $w.text <Escape><Key-$d>               "shell::bindings:start_number %A"
            bind $w.text <Key-$d>                       "shell::insert_digit %A $w"
        }
  
        # Emacs movement keys
        bind $w.text <Control-b>                        "shell::left $w"
        bind $w.text <Control-f>                        "shell::right $w"
        bind $w.text <Control-a>                        "shell::home $w"
        bind $w.text <Control-e>                        "shell:end $w"
        bind $w.text <Escape><b>                        "shell::word_left $w"
        bind $w.text <Escape><f>                        "shell::word_right $w"
  
        # Emacs scrolling keys
        bind $w.text <Control-v>                        "shell::scroll_down $w"
        bind $w.text <Escape><v>                        "shell::scroll_up $w"

        # Emacs deletion keys
        bind $w.text <Control-y>                        "shell::paste_selection $w"
  
        # Emacs keys
        bind $w.text <Control-x>                        {catch {unset vals(C-X-odd)}}
        bind $w.text <Control-x><Control-x>     "
            if {\[lsearch -exact \[array names vals\] C-X-odd\] != -1} {
                unset vals(C-X-odd)
            } else {
                shell::exchange-point-and-mark $w
                set vals(C-X-odd) 1
            }
        "
  
}

shell::bindings:init                            ;# set up global variables in shell bindings

