#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# Procedure:    scale:gainFocus
#
# Synopsis:
#       Internal procedure to handle keyboard focus directed to a scale widget.
#
# Usage:
#c      scale:gainFocus pathName
#
# Parameters:
#c      pathName
#               Path name of a scale widget.
#
# Return value:
#       None.
#
# Description:
#       scale:gainFocus handles directing keyboard focus to a scale widget.
#       It changes the `foreground' color of the scale to its `active
#       foreground'

proc scale:gainFocus w {
        global scale_priv
        set scale_priv(fg,$w) [lindex [$w config -foreground] 4]
        $w config -foreground [lindex [$w config -activeforeground] 4]
}

# Procedure:    scale:loseFocus
#
# Synopsis:
#       Internal procedure to handle keyboard focus directed away from a
#       scale widget.
#
# Usage:
#c      scale:loseFocus pathName
#
# Parameters:
#c      pathName
#               Path name of a scale widget.
#
# Return value:
#       None.
#
# Description:
#       scale:loseFocus handles directing keyboard focus away from a scale
#       widget. It changes the `foreground' color of the scale back to its
#       `normal foreground'

proc scale:loseFocus w {
        global scale_priv
        $w config -fg $scale_priv(fg,$w)
        unset scale_priv(fg,$w)
}

# Procedure:    scale:incr
#
# Synopsis:
#       Internal procedure to adjust the value in a scale widget.
#
# Usage:
#c      scale:incr pathName amount key
#
# Parameters:
#c      pathName
#               Path name of a scale widget.
#c      amount
#               Decimal integer giving the amount by which the scale is to
#               be adjusted.
#c      key
#               ASCII character corresponding to the key that was pressed.
#
# Return value:
#       None.
#
# Description:
#       scale:incr handles increasing and decreasing the value of a scale
#       in response to keyboard events.  The second argument is the amount
#       by which to adjust the scale.  The third is the key that requested
#       the adjustment; it is given because the arrow keys adjust in the
#       direction of the arrow (and the scale may be reversed!), while the
#       "+" and "-" keys adjust numerically.

proc scale:incr {w {amount 1} {key ""}} {
        set from [lindex [$w config -from] 4]
        set to [lindex [$w config -to] 4]
        if {$to < $from} {
                set min $to
                set max $from
        } else {
                set min $from
                set max $to
        }
        if {$to < $from && $key != "+" && $key != "-"} {
                set amount -amount
        } 
        set v [expr [$w get]+$amount]
        if {$v < $min} {
                set v $min
        }
        if {$v > $max} {
                set v $max
        }
        $w set $v
}

# Procedure:    scale_bindForTraversal
#
# Synopsis:
#       Establish keyboard bindings for scale widgets.
#
# Usage:
#c      scale_bindForTraversal pathOrClass...
#
# Parameters:
#c      pathOrClass
#               Either the path name of a widget, or the name of the Scale
#               class
#
# Return value:
#       None.
#
# Description:
#       scale_bindForTraversal establishes the default keyboard bindings for
#       either the specified set of widgets or for the Scale class.  It
#       includes the following:
#       - Color change to reflect the keyboard focus.
#       - <Left>, <Up>, <Right>, and <Down> arrows,
#         and the <plus> and <minus> keys, increment and decrement
#         the value of the scale as appropriate.
#       - Repeated presses of the keys above increment and decrement
#         by successively larger amounts.
#       - <Tab> and <Shift-Tab> are bound to keyboard traversal
#         of widgets.
#       - tk_bindForTraversal bindings are established.
#
# Notes:
#       This procedure is normally invoked from `init.tcl'; the user doesn't
#       generally have to call it unless building a customized set of
#       bindings.

proc scale_bindForTraversal args {
        foreach item $args {
                widget_bind $item GainFocus "scale:gainFocus"
                widget_bind $item LoseFocus "scale:loseFocus"
                bind $item <Any-Key-Left> "scale:incr %W -1 %A"
 #              bind $item <Any-Double-Key-Left> "scale:incr %W -5 %A"
 #              bind $item <Any-Triple-Key-Left> "scale:incr %W -25 %A"
                bind $item <Any-Key-Up> "scale:incr %W 1 %A"
 #              bind $item <Any-Double-Key-Up> "scale:incr %W 5 %A"
 #              bind $item <Any-Triple-Key-Up> "scale:incr %W 25 %A"
                bind $item <Any-Key-Right> "scale:incr %W 1 %A"
 #              bind $item <Any-Double-Key-Right> "scale:incr %W 5 %A"
 #              bind $item <Any-Triple-Key-Right> "scale:incr %W 25 %A"
                bind $item <Any-Key-Down> "scale:incr %W -1 %A"
 #              bind $item <Any-Double-Key-Down> "scale:incr %W -5 %A"
 #              bind $item <Any-Triple-Key-Down> "scale:incr %W -25 %A"
                bind $item <Any-Key-plus> "scale:incr %W 1 %A"
 #              bind $item <Any-Double-Key-plus> "scale:incr %W 5 %A"
 #              bind $item <Any-Triple-Key-plus> "scale:incr %W 25 %A"
                bind $item <Any-Key-minus> "scale:incr %W -1 %A"
 #              bind $item <Any-Double-Key-minus> "scale:incr %W -5 %A"
 #              bind $item <Any-Triple-Key-minus> "scale:incr %W -25 %A"
                focus_bindForTraversal $item
                tk_bindForTraversal $item
        }
}

