#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File: canvasAux.tcl

# Description:
#       The canvasAux.tcl file contains auxiliary procedures that handle the
#       bindings for canvas widgets.
#
#       At present, the only procedures that are supplied are the ones that
#       transfer the keyboard focus among text items in response to <Tab>
#       and <Shift-Tab> events.


# Procedure:    canvas_bindForTraversal
#
# Synopsis:
#       Keyboard traversal for canvas widgets.
#
# Usage:
#c      canvas_bindForTraversal pathName tagOrId ?-controlonly?
#
# Parameters:
#c      pathName
#               Name of a canvas widget.
#c      tagOrId
#               Item tag or identifier that identifies a set of items in
#               the canvas.
#c      -controlonly
#               Flag indicating that only Control Tab and Control Shift Tab
#               are to be bound.
#
# Return value:
#       None.
#
# Description:
#       The canvas_bindForTraversal function is used to establish traversal
#       key bindings for canvas items.  It should be called whenever keyboard
#       traversal is desired on a canvas item that accepts the focus.
#
#       The bindings that are established are-
#
#       - <Control-Tab> (and <Tab> if -controlonly is not specified) 
#       transfers keyboard focus to the next item in the application.
#
#       - <Control-Shift-Tab> (and <Shift-Tab> if -controlonly is not
#       specified) transfers keyboard focus to the previous item in
#       the application.
#
#       - <F10> and the <Alt-> keys are bound in the same way as
#       tk_bindForTraversal.
#
#       - The <Control-Return>, <Control-Enter>, and <Control-LineFeed> keys
#       (and their non-<Control-> counterparts if -controlonly is
#       not specified) invoke the current default action.

proc canvas_bindForTraversal {w tagOrId {controlonly ""}} {
        $w bind $tagOrId <Control-Tab> "canvas_goToNextItem %W"
        $w bind $tagOrId <Control-Shift-Tab> "canvas.goToPrevItem %W"
        catch {$w bind $tagOrId <Control-KP_Tab> "canvas_goToNextItem %W"}
        catch {
                $w bind $tagOrId \
                        <Control-Shift-KP_Tab> "canvas.goToPrevItem %W"
        }
        $w bind $tagOrId <Control-Key-j> {entry_invokeDefaultButton %W}
        $w bind $tagOrId <Control-Key-m> {entry_invokeDefaultButton %W}
        catch {
                $w bind $tagOrId <Control-Key-Linefeed> {
                        button_invokeDefault %W
                }
        }
        catch {
                $w bind $tagOrId <Control-Key-Return> {
                        button_invokeDefault %W
                }
        }
        catch {
                $w bind $tagOrId <Control-Key-KP_Enter> {
                        button_invokeDefault %W
                }
        }
        catch {
                $w bind $tagOrId <Control-Key-Enter> {
                        button_invokeDefault %W
                }
        }
        if {$controlonly != "-controlonly"} {
                $w bind $tagOrId <Tab> "canvas_goToNextItem %W"
                $w bind $tagOrId <Shift-Tab> "canvas_goToPrevItem %W"
                catch {
                        $w bind $tagOrId <Key-Linefeed> {
                                button_invokeDefault %W
                        }
                }
                catch {
                        $w bind $tagOrId <Key-Return> {
                                button_invokeDefault %W
                        }
                }
                catch {
                        $w bind $tagOrId <Key-KP_Enter> {
                                button_invokeDefault %W
                        }
                }
                catch {
                        $w bind $tagOrId <Key-Enter> {
                                button_invokeDefault %W
                        }
                }
        }
        $w bind $tagOrId <Alt-Key> {tk_traverseToMenu %W %A}
        $w bind $tagOrId <F10> {tk_firstMenu %W}
}

# Procedure: canvas_goToNextItem
#
# Synopsis:
#       Tab to next item in a canvas
#
# Usage:
#c      canvas_goToNextItem pathName
#
# Parameters:
#c      pathName
#               Path name of a canvas widget.
#
# Return value:
#       None.
#
# Description:
#       The canvas_goToNextItem procedure is invoked when the user
#       presses <Tab> or <Control-Tab> when keyboard focus is directed
#       to a canvas item.  It directs keyboard focus to the logical
#       `next item' within the canvas, or to the next item in the
#       application if the current focus is at the last item in the
#       canvas.

proc canvas_goToNextItem w {
        set id [$w focus]
        if {$id != ""} {
                focus_goToNext [list $w $id]
        } else {
                focus_goToNext $w
        }
}

# Procedure: canvas_goToPrevItem
#
# Synopsis:
#       Tab to previous item in a canvas
#
# Usage:
#c      canvas_goToPrevItem pathName
#
# Parameters:
#c      pathName
#               Path name of a canvas widget.
#
# Return value:
#       None.
#
# Description:
#       The canvas_goToPrevItem procedure is invoked when the user presses
#       Shift Tab or Control Shift Tab when keyboard focus is directed to
#       a canvas item.  It directs keyboard focus to the logical `next item'
#       within the canvas, or to the next item in the application if the
#       current focus is at the last item in the canvas.

proc canvas_goToPrevItem w {
        set id [$w focus]
        if {$id != ""} {
                focus_goToPrev [list $w $id]
        } else {
                focus_goToPrev $w
        }
}

