#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




proc SourceBrowser::delayedLoad { args } {
        global SourceBrowser_flags
        if ![info exists SourceBrowser_flags(loaded)] { return }
        if [hol90] {
            hol_load_library replay
        } 
        set SourceBrowser_flags(loaded) 1
}


#----------------------------------------------------------------------------
#
# WIDGET CLASS SourceBrowser
#
# OPTIONS
#       -file
#               The file to analyse and display.
#
# COMMANDS
#
#----------------------------------------------------------------------------

proc SourceBrowser { w args } {
        global feedback
        global SourceBrowser_flags
        global gui_flags
        global busy
        incr busy 1
        set feedback [list {} "Creating Source Browser..."]
        global vals

        SourceBrowser::delayedLoad

        for {set i 0} {$i<[llength $args]} {incr i} {
            case [lindex $args $i] -file {
                incr i
                set vals($w,file) [lindex $args $i]
            } default {
                error "unrecognized arg [lindex $args $i]"
            }
        }


        # 1. Create the window, withdraw it while we're constructing its internals

        set feedback [list {} "Displaying $vals($w,file)..."]
        toplevel $w -class SourceBrowser
        wm withdraw $w
        wm minsize $w 1 1
        
                                                              

        # 2. Make the project frame


        # 3. Make the fragments frame

        pack [collapsible frame $w.fragments \
                -title "Construction Sequence" \
                -collapseCommand "pack config $w.fragments -expand " \
                -visible 1] -expand yes -fill both -pady 5
        pack [buttonbar $w.fragments.b.bbar] -fill x
        pack [scrollable selectlistbox $w.fragments.b.list \
                -font $gui_flags(font,listboxes) \
                -geometry 36x7 \
                -multiselect 1 \
                -relief sunken \
                -export 0 \
                -borderwidth 2 \
                -selectionvar vals($w,selection)] -fill both -side left -expand yes
#       pack [frame $w.fragments.b.info] -expand yes -fill both -side right

        $w.fragments.b.bbar addbutton button \
                -text "Rescan File" \
                -command "SourceBrowser::rescan $w"

        $w.fragments.b.bbar addbutton sensitive button \
                -text "Execute" \
                -sensitivevar vals($w,selection) \
                -sensitiveexpr "\[llength \$vals($w,selection)\]!=0" \
                -command "SourceBrowser::execute_operation $w"
        $w.fragments.b.bbar addbutton sensitive button \
                -text "Fast Forward" \
                -sensitivevar vals($w,selection) \
                -sensitiveexpr "\[llength \$vals($w,selection)\]==1" \
                -command "SourceBrowser::fast_forward $w"
        $w.fragments.b.bbar addbutton sensitive button \
                -text "Skip" \
                -width 7 \
                -sensitivevar vals($w,selection) \
                -sensitiveexpr "\[llength \$vals($w,selection)\]==1" \
                -command "SourceBrowser::skip $w"
        $w.fragments.b.bbar addbutton sensitive button \
                -text "Replay Proof/Definition" \
                -sensitivevar vals($w,selection) \
                -sensitiveexpr "\[llength \$vals($w,selection)\]==1" \
                -command "SourceBrowser::replay_operation $w"

        trace variable vals($w,selection) w "SourceBrowser::display_code $w"

        # 4. Make the file frame

        pack [collapsible editor $w.editor \
                -title "Script File" \
                -collapseCommand "pack config $w.editor -expand " \
                -visible 1 \
                -onfileselectcommand "SourceBrowser::change_file $w %f" \
                -onsavecommand "SourceBrowser::change_file $w %f"] -expand yes -fill both -pady 5

        $w.editor.b text config -font $gui_flags(font,codeentry) \
                -width 80 \
                -height 10

        # 5. now set up the menus
        #

        set menus ""
        set entries [Editor::menu_entries $w.editor.b]
        lappend entries sep
        lappend entries [list command "Close Window" "destroy $w" 0]
        lappend menus [list file "File" 0 left $entries]

        set entries ""
        foreach preference_group [preferences::groups] {
            lappend entries [list command "$preference_group..." "preferences -group [list $preference_group]" {}]
        }
        lappend entries [list sep]
        lappend entries [list command "Save All Options" "PreferencesSave" 0]
        lappend menus [list preferences "Options" 0 left $entries ]

        trace variable SourceBrowser_flags(collapseLocals) w "SourceBrowser::rescan $w"
                      
        MakeMenus $w $menus

        # 6. Display the window.

        update
        wm deiconify $w
        tkwait visibility $w
        update

        # 7. Display the file into these internals.

        SourceBrowser::display_file $w
        widget_addBinding $w Destroy "SourceBrowser::upon_destroy $w"

        # 8. Tidy up...

        focus_goToFirst $w

        set feedback [list {} "Please wait..."]

        incr busy -1
        return $w
}


proc SourceBrowser::upon_destroy { w } {
        global vals
        global SourceBrowser_flags
        Scripts::deregister_trace_for_script $vals($w,file) SourceBrowser $w
        trace vdelete SourceBrowser_flags(collapseLocals) w "SourceBrowser::rescan $w"
        unset_vals_for_widget $w
}

#----------------------------------------------------------------------------
# SourceBrowser::change_file
#
#----------------------------------------------------------------------------

proc SourceBrowser::selectfile { w } {
        set file [selectfile -new 1 -dirvariable pwd -pattern *.sml -message "Select an ML file to analyse and browse:"]
        if {$file != ""} {
            SourceBrowser::change_file $w $file
        }
}

# nb. args get ignored.  This procedure is used as an argument to "trace".
proc SourceBrowser::rescan { w args } {
        global vals
        set lastsel $vals($w,selection)
        SourceBrowser::change_file $w $vals($w,file)
        if [catch {$w.fragments.b.list selectlistbox select from [lindex $lastsel 0];
            set vals($w,selection) [lindex $lastsel 0]
            $w.fragments.b.list selectlistbox yview [lindex $lastsel 0]
        } err] { puts stderr $err }
        
}


proc SourceBrowser::change_file { w file } {
        global vals
        global busy
        global feedback
        incr busy 1
        set file [lindex $file 0]
        set feedback [list $w "Opening file $vals($w,file)..."]
        SourceBrowser::undisplay_file $w
        set vals($w,file) $file
        SourceBrowser::display_file $w
        incr busy -1
}


proc SourceBrowser::script_change_notify { w file } {
        global vals
        global busy
        global feedback
        incr busy 1
        set file [lindex $file 0]
        set feedback [list $w "Opening file $vals($w,file)..."]
        SourceBrowser::undisplay_file $w
        set vals($w,file) $file
        SourceBrowser::display_file $w
        $w.fragments.b.list selectlistbox yview [expr [$w.fragments.b.list selectlistbox size]-6]
        incr busy -1
}

proc SourceBrowser::replay_operation { w } {
        global vals
        global busy
        incr busy
        global feedback
        set fragindex $vals($w,selection)
        set tool [ML -type string "#edit_tool (el ($fragindex+1) (FileAnalysis.fragments_for_file [ml_string $vals($w,file)]))"]
        set argscode "#data (el ($fragindex+1) (FileAnalysis.fragments_for_file [ml_string $vals($w,file)]))"
        newwin $tool -initial $argscode -title [$w.fragments.b.list selectlistbox get $fragindex]
        incr busy -1
}

proc SourceBrowser::execute_operation { w } {
        global vals
        global feedback
        global busy
        incr busy
        foreach fragindex $vals($w,selection) {
            set title [$w.fragments.b.list selectlistbox get $fragindex]
            set feedback [list $w $title]
            set range [ML -type intpair "(#2 o (fn SOME x => x | NONE => raise Match) o #origin) (el ($fragindex+1) (FileAnalysis.fragments_for_file [ml_string $vals($w,file)]))"]
            set f [open $vals($w,file)]
            seek $f [expr [lindex $range 0]-2]
            set code [read $f [expr [lindex $range 1]-[lindex $range 0]]]
            close $f
            set status [catch {ML -trace 1 -toplevel 1 $code} err]
            if $status {
                incr busy -1
                errormessage $w.error "$title - error:\n$err"
                return 0
            }
            if {[string index $title 0]!="+"} {
                $w.fragments.b.list selectlistbox delete $fragindex
                $w.fragments.b.list selectlistbox insert $fragindex "+$title"
            }
            if {$vals($w,selection) == [$w.fragments.b.list selectlistbox size]-1} {
                incr busy -1
                return 0
            }
        }
        SourceBrowser::skip $w
        incr busy -1
        return 1
}

proc SourceBrowser::fast_forward { w } {
        global vals 
        global busy
        incr busy
#       trace vdelete vals($w,selection) w "SourceBrowser::display_code $w"
        while {$vals($w,selection) <= [$w.fragments.b.list selectlistbox size]} {
            if ![SourceBrowser::execute_operation $w] { break }
        }
        incr busy -1
#       trace variable vals($w,selection) w "SourceBrowser::display_code $w"
#       set vals($w,selection) $vals($w,selection)
}

proc SourceBrowser::skip { w } {
        global vals 
        set fragindex [lindex $vals($w,selection) [expr [llength $vals($w,selection)]-1]]
        $w.fragments.b.list selectlistbox select from [expr $fragindex+1]
        $w.fragments.b.list selectlistbox yview [expr $fragindex-2]
        set vals($w,selection) [expr $fragindex+1]
}

proc SourceBrowser::undisplay_file { w } {
        global vals 
        $w.fragments.b.list selectlistbox delete 0 end
        Scripts::deregister_trace_for_script $vals($w,file) SourceBrowser $w
}


proc SourceBrowser::display_code { w args } {
        global vals 
        global feedback
        global SourceBrowser_flags
        global SourceBrowser_library

        if $vals($w,analyses_ok) {
            set fragindex [lindex $vals($w,selection) 0]
            set range [ML -type intpair "(#2 o (fn SOME x => x | NONE => raise Match) o #origin) (el ($fragindex+1) (FileAnalysis.fragments_for_file [ml_string $vals($w,file)]))"]

            if $SourceBrowser_flags(useLinefinder) {
                set realsource [exec perl $SourceBrowser_library/linefinder/linefinder.pl $vals($w,file) [expr [lindex $range 0]-1]]
                # puts "realsource = $realsource"
                set file [lindex $realsource 0]
                $w.editor.b loadfile $file
                wm title [winfo toplevel $w] $file
                $w.editor.b text yview [lindex $realsource 1].0
            } else {
                $w.editor.b text yview "1.0 + [lindex $range 0] chars"
            }
        } else {
            set errindex [lindex $vals($w,selection) 0]
            set error_range [lindex $vals($w,errors) $errindex]
            set start_of_error [lindex $error_range 0]
            set end_of_error [lindex $error_range 1]
            $w.editor.b text yview "$start_of_error - 3 lines"
            $w.editor.b text tag remove sel 1.0 end
            $w.editor.b text tag add sel $start_of_error $end_of_error
            $w.editor.b text mark set insert $start_of_error
        }
}

proc SourceBrowser::display_file { w } {
        global vals 
        global feedback
        global SourceBrowser_flags
        SourceBrowser::delayedLoad
        wm title $w [file tail $vals($w,file)]

        Scripts::register_trace_for_script $vals($w,file) SourceBrowser $w
        ML -type any "FileAnalysis.delete_cache_for_file [ml_string $vals($w,file)]"
        ML -type any "ScriptFragments.collapseLocals := [ml_bool $SourceBrowser_flags(collapseLocals)]"
        if $SourceBrowser_flags(makeSource) {
            if [catch {exec sh << "(cd [file dir $vals($w,file)] ; make [file tail $vals($w,file)])"} err] {
                information $w $err
            }
        }
        if ![file readable $vals($w,file)] {
            errormessage $w.error "$vals($w,file) does not exist or is not readable."
            return
        }
        $w.editor.b loadfile $vals($w,file)
        set vals($w,analyses_ok) [ML -type bool "FileAnalysis.file_analyses_ok [ml_string $vals($w,file)]"]

        if $vals($w,analyses_ok) {
            set descriptions [ML -type stringlist "map #describe (FileAnalysis.fragments_for_file [ml_string $vals($w,file)])"]
            eval $w.fragments.b.list selectlistbox insert 0 $descriptions
        } else {
            set errors [ML -type string "FileAnalysis.errors_for_file [ml_string $vals($w,file)]"]
            eval $w.fragments.b.list selectlistbox insert 0 [SourceBrowser::parse_errors $w $errors]
        }
        $w.fragments.b.list selectlistbox select from 0
        set vals($w,selection) 0
}

proc SourceBrowser::parse_errors { w errors } {
        global vals
        # puts "errors = $errors"
        
        set vals($w,errors) ""
        foreach error_line [split $errors \n] {
            if [regexp {([0-9]+\.[0-9]+).*([0-9]+\.[0-9]+)} $error_line dummy start_of_error end_of_error] {
                if [info exists prev_error] {
                    lappend vals($w,errors) $prev_errorpos
                    lappend descriptions [join $prev_error]
                    unset prev_error
                }
                set prev_errorpos [list "$start_of_error - 2 c" "$end_of_error - 2 c"]
            } 
            if [regexp {([0-9]+\.[0-9]+)} $error_line dummy start_of_error] {
                if [info exists prev_error] {
                    lappend vals($w,errors) $prev_errorpos
                    lappend descriptions [join $prev_error]
                    unset prev_error
                }
                set prev_errorpos [list "$start_of_error - 2 c" "$start_of_error - 1 c"]
            } 
            lappend prev_error $error_line
        }
        if [info exists prev_error] {
            lappend vals($w,errors) $prev_errorpos
            lappend descriptions [join $prev_error]
        }
        
}


#----------------------------------------------------------------------------
# Test routine used from interactive tcl.
#----------------------------------------------------------------------------

proc te { } {
   after 1 "
        source \$SourceBrowser_library/src/SourceBrowser.tcl
        source \$SourceBrowser_library/src/TheoryTree.tcl
        source \$hierarchy_library/src/hierarchy.tcl
        source \$RichText_library/src/TheoremsFrame.tcl
        source \$RichText_library/src/RichText.tcl
        newwin SourceBrowser -withfeedback 1
    "
}  




