#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




proc richtext {} {}

composite_define RichText {
        {-structuredtextvar structuredtextVar StructuredtextVar structuredtext composite_configPrivateNop}
        {-richtextvar richtextVar RichtextVar richtext composite_configPrivateNop}
        {-showtypesvar showtypesVar ShowtypesVar showtypes composite_configPrivateNop}
        {-selectiontypesvar selectiontypesVar SelectiontypesVar selectiontypes composite_configPrivateNop}
        {-editingvar editingVar EditingVar {} composite_configPrivateNop}
        {-focusvar focusVar FocusVar "" composite_configPrivateNop}
        {-cursorvar cursorVar CursorVar "" composite_configPrivateNop}
} RichText::initProc


proc RichText::initProc { w } {
        global RichText_flags
        global TheoremWidgets_flags
        global gui_flags
        upvar #0 config$w config
        upvar #0 $w data

        set data(numobjects) 0
        upvar #0 $config(structuredtextVar) structuredtext
        upvar #0 $config(showtypesVar) showtypes
        upvar #0 $config(selectiontypesVar) selectiontypes
        upvar #0 $config(richtextVar) richtext
        set richtext $TheoremWidgets_flags(mapsOnByDefault)
        set selectiontypes ""
        set structuredtext $RichText_flags(structuredtextOnByDefault)
        set showtypes 0
        
        pack [scrollable text $w.text \
                -font $gui_flags(font,codeentry) \
                -relief sunken \
                -borderwidth 2 \
                -state disabled] \
            -side left -expand yes -fill both
        if {$config(editingVar)==""} { set config(editingVar) [set w](editing) }
        RichText::adjust_editing $w 0
        set data(looks) [list $RichText_flags(default_look)]

        # command not available in constructor...
        RichText::reconfiguretags $w

        trace variable RichText_flags w "RichText::tag_flag_change_trigger $w"

        widget_addBinding $w Destroy "RichText::upon_destroy $w"

        return $w
}

composite_configFlag RichText selectiontypesVar { w arg value } {
        upvar #0 config$w config
        upvar #0 $config(selectiontypesVar) old_selectiontypes
        upvar #0 $value new_selectiontypes
        set new_selectiontypes $old_selectiontypes
        return private
}

composite_configFlag RichText editingVar { w arg value } {
        upvar #0 config$w config
        upvar #0 $config(editingVar) old_editing
        upvar #0 $value new_editingvar
        set new_editingvar $old_editing
        return private
}
composite_configFlag RichText richtextVar { w arg value } {
        upvar #0 config$w config
        upvar #0 $config(richtextVar) old_richtext
        upvar #0 $value new_richtext
        set new_richtext $old_richtext
        return private
}

composite_configFlag RichText showtypesVar { w arg value } {
        upvar #0 config$w config
        upvar #0 $config(showtypesVar) old_showtypes
        upvar #0 $value new_showtypes
        set new_showtypes $old_showtypes
        return private
}

composite_configFlag RichText structuredtextVar { w arg value } {
        upvar #0 config$w config
        upvar #0 $config(structuredtextVar) old_structuredtext
        upvar #0 $value new_structuredtext
        set new_structuredtext $old_structuredtext
        return private
}


proc RichText::upon_destroy { w } {
        global RichText_flags
        trace vdelete RichText_flags w "RichText::tag_flag_change_trigger $w"
        catch {trace vdelete focus w "RichText::focus_change $w"}
        catch {trace vdelete cursor w "RichText::cursor_change $w"}
}


proc RichText::tag_flag_change_trigger { w args } {
        RichText::reconfiguretags $w
}

proc RichText::reconfiguretags { w } {
        global RichText_flags
        global TheoremWidgets_flags
        upvar #0 $w data
        set l $data(looks)
        set data(looks) ""
        foreach look $l {
            RichText::configure_look_tag $w $look
        }
        set textw [$w.text text]

        # a hack - RichText widgets should be completely independent
        # of references to theorems etc.  Decoration tags should
        # be added / manipulated via some other interface.
        $textw tag configure DECORATE:thmname -foreground $TheoremWidgets_flags(color,thmname) -font $TheoremWidgets_flags(font,thmname) -underline 1
        $textw tag configure DECORATE:thmorigin -foreground $TheoremWidgets_flags(color,thmkind) -font $TheoremWidgets_flags(font,thmkind) 
        $textw tag configure DECORATE:sel \
                -background #b2dfee
        $textw tag configure DECORATE:focus \
                -background #b2dfee \
                -relief ridge \
                -borderwidth 2
        $textw tag configure DECORATE:cursor \
                -background #ffe4c4 \
                -relief ridge \
                -borderwidth 3
        
        
}


proc RichText::configure_look_tag { w look } {
        global RichText_flags
        # puts "w = $w, look = \"$look\""
        if ![regexp {(.*)\|(.*)} $look dummy parta partb] { error "Implementation error: Badly constructed look $look (didn't match ..|..)" }
        # puts "parta = $parta, partb = $partb"
        if ![regexp (.*)-(.*)-(.*)-(.*)-(.*) $parta dummy fontgroup wght slant sWdth sizegroup] { error "Implementation error: Badly constructed look $look (part A didn't match)" }
        if ![regexp (.*)-(.*)-(.*)-(.*)-(.*)-(.*)-(.*)-(.*)-(.*) $partb \
                dummy background bgstipple \
                borderwidth fgstipple foreground relief \
                underline subscript superscript] { error "Implementation error: Badly constructed look $look (part B didn't match)" }
        lappend data(looks) $look
        set family $RichText_flags(families,$fontgroup)
        set ptSz $RichText_flags(pointsizes,$sizegroup)
        set font "$family-$wght-$slant-$sWdth-*-*-$ptSz-*-*-*-*-*-*"
        set args ""
        lappend args -font $font
        if {$background!="*"} { lappend args -background $background }
        if {$bgstipple!="*"} { lappend args -bgstipple $bgstipple }
        if {$borderwidth!="*"} { lappend args -borderwidth $borderwidth }
        if {$fgstipple!="*"} { lappend args -fgstipple $fgstipple }
        if {$foreground!="*"} { lappend args -foreground $foreground }
        if {$relief!="*"} { lappend args -relief $relief }
        if {$underline!="*"} { lappend args -underline $underline }
#        if {$subscript!="*"} { lappend args -subscript $subscript }
#        if {$superscript!="*"} { lappend args -superscript $superscript }
        eval [list $w.text.b tag configure TMLOOK:$look] $args
}


proc RichText::adjust_editing { w new_editing } {
        upvar #0 $w data
        upvar #0 config$w config
        upvar #0 $config(editingVar) editing
        catch {set editing} val
        #puts "RichText::adjust_editing, w = $w, new_editing = $new_editing, config(editingVar) = $config(editingVar), editing = $val"
        if {(![info exists data(adjust_first)]) || (![info exists editing]) || ($editing != $new_editing)} {
            if $new_editing {
                $w.text.b config -state normal
                RichText::bind_for_normal_selection $w
            } else {
                $w.text.b config -state disabled
                if {$config(focusVar)==""} {
                    RichText::bind_for_subobject_selection $w
                } else {
                    upvar #0 $config(focusVar) focus
                    set focus ""
                    RichText::bind_for_subobject_focus $w
                }
            }
            set editing $new_editing
        }
        set data(adjust_first) 0
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------


proc RichText::new_objectid { w } {
        upvar #0 data$w data
        if ![info exists data(objectid_count)] {
            set data(objectid_count) 0
        }
        incr data(objectid_count)
        return OBJECT:$data(objectid_count)
}             


#----------------------------------------------------------------------------
# $w insertobject widget [-header text] objspec
#           
# When calculating the margins to use for HOL, we allow
# more leeway for richtext as it tends to take up more
# room.
#
# Hmmm.. as it turns out it doesn't seem we need to.  This
# will depend alot on font selection vis a vis the default
# font for the text window.  Even more so on the interface
# mapping that occurs!
#
# This is irritating.  It doesn't seem
# possible to find out the current width of the window
# after resizing except via winfo.  This means we have
# to recalculate the number of pixels/character by
# looking at the config settings and the requested width
# via winfo.
# 
#----------------------------------------------------------------------------

proc RichText::calculate_width { w } {
        set pixwidth [winfo width [$w.text text]]
        set reqwidth [winfo reqwidth [$w.text text]]
        set reqchars [lindex [$w.text.b config -width] 4]
        set width [expr {$pixwidth*$reqchars/$reqwidth - 5}]
        if {$width < 15} { set width 15 }
        return $width
}

composite_subcommand RichText insertobject { w args } {
        upvar #0 $w data
        upvar #0 config$w config
        global RichText_flags
        global TheoremWidgets_flags
        set textw [$w.text text]
        set oldstate [lindex [$textw config -state] 4]
        $textw config -state normal
        for {set i 0} {$i<[llength $args]} {incr i} {
            switch -- [lindex $args $i] -header {
                incr i
                set header [lindex $args $i]
            } -after {
                incr i
                set after [lindex $args $i]
            } default {
                if [info exists objspec] { error "too many arguments to RichText::insertthm" }
                set objspec [lindex $args $i]
            }
        }
        if ![info exists objspec] {
            error "too few arguments to insertobject - objspec must be given"
        }
        set objectid [RichText::new_objectid $w]
        # puts "new objectid = $objectid"
        if [info exists after] {
            # puts "after = $after"
            set firstindex [$textw index "$after.last + 2 c"]
        } else {
            set firstindex 1.0
        }


        # insert a newline and get rid of any spurious tags which will 
        # hang aroound when we insert new text.  Then position insert
        # cursor just before the newline character we've just added.

        #puts "firstindex = $firstindex"
        $textw mark set insert $firstindex
        #puts "tags at insert = [$textw tag names insert]"
        $textw insert insert \n
# RichText::InsertPlainText $w insert \n [list $objectid]
        set afternewline [$textw index insert]
        $textw mark set insert $firstindex
        #puts "tags at insert (before cleaning) = [$textw tag names insert]"
        #puts "tags at insert + 1 c (before cleaning) = [$textw tag names "insert + 1 c"]"


        set excesstags [$textw tag names insert]
        foreach excesstag $excesstags {
            $textw tag remove $excesstag insert
        }
        set excesstags [$textw tag names "insert - 1 c"]
        foreach excesstag $excesstags {
            $textw tag remove $excesstag "insert - 1 c"
        }
        #puts "tags at insert (after cleaning of newline) = [$textw tag names insert]"
        #puts "tags at insert + 1 c(after cleaning of newline) = [$textw tag names "insert + 1 c"]"
        #puts "tags at insert + 2 c(after cleaning of newline) = [$textw tag names "insert + 2 c"]"

        if ![info exists header] {
            set header [[lindex $objspec 0]::header [lindex $objspec 1]]
        }
        foreach tag_text_pair $header {
            set text [lindex $tag_text_pair 0]
            set tag [lindex $tag_text_pair 1]
            switch -- $tag bitmap {
                # -- not yet implemented -- should it ever be?
            } default {
                RichText::InsertPlainText $w insert $text [list $objectid DECORATE:$tag]
            }
        } 
        foreach tag_text_pair $header {
            set tag [lindex $tag_text_pair 1]
            switch -- $tag bitmap {
                # -- not yet implemented -- should it ever be?
            } default {
                $textw tag remove DECORATE:$tag "insert - 1 c"
            }
        } 
        #puts "tags at insert -1 c(after header) = [$textw tag names "insert - 1 c"]"
        #puts "tags at insert (after header) = [$textw tag names insert]"
        #puts "tags at insert + 1 c(after header) = [$textw tag names "insert + 1 c"]"
        #puts "tags at insert + 2 c(after header) = [$textw tag names "insert + 2 c"]"
        

        upvar #0 $config(richtextVar) richtext
        upvar #0 $config(structuredtextVar) structuredtext
        upvar #0 $config(showtypesVar) showtypes
        set initial_pos [split [$w.text.b index insert] .]

        set flags(width) [RichText::calculate_width $w]
        set flags(initial_line) [lindex $initial_pos 0]
        set flags(initial_char) [lindex $initial_pos 1]
        set flags(show_types) $showtypes

        if $structuredtext {
            if !$richtext {
                set flags(interface_maps) ""
            }
            set objcode [[lindex $objspec 0]::code [lindex $objspec 1]]
            set printer [[lindex $objspec 0]::structuredtext_printer flags]
            set text [ML -type stringlistlist "($printer)\n($objcode)"]
            RichText::InsertHol90StructuredText $w insert text $objectid
        } else {
            set flags(interface_maps) ""
            set objcode [[lindex $objspec 0]::code [lindex $objspec 1]]
            set printer [[lindex $objspec 0]::plaintext_printer flags]
            set text [ML -type string "($printer)\n($objcode)"]
            RichText::InsertPlainText $w insert "$text\n" $objectid
        }       
        #puts "tag ranges for new object (after text) = [$textw tag ranges $objectid]"
        $w yviewobject $objectid
        set data($objectid,spec) $objspec
        incr data(numobjects)
        $textw config -state $oldstate
        RichText::adjust_editing $w 0
        return $objectid
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------


composite_subcommand RichText editpreterm { w args } {
        set text [$w.text text]
        upvar #0 $w data
        global RichText_flags
        for {set i 0} {$i<[llength $args]} {incr i} {
            switch -- [lindex $args $i] default {
                if [info exists pretermspec] { error "too many arguments to RichText::editpreterm" }
                set pretermspec [lindex $args $i]
            }
        }
        set objectid "OBJECT:preterm"
        RichText::adjust_editing $w 1
        $text delete 1.0 end
        $text insert end [lindex $pretermspec 1]
        
        set data(numobjects) 1
        return $objectid
}

composite_subcommand RichText getpreterm { w args } {
        set text [$w.text text]
        $text get 1.0 end
}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

composite_subcommand RichText yviewobject { w tag } {
        set text [$w.text text]
        # objects need not have *any* characters
        catch {$text yview "$tag.first -1 lines"}
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

composite_subcommand RichText deleteobject { w tag } {
        set oldstate [lindex [$text config -state] 4]
        set text [$w.text text]
        $text config -state normal
        $text delete $tag.first $tag.last
        $text config -state $oldstate
        selection clear $w
}

composite_subcommand RichText deleteall { w  } {
        upvar #0 $w data
        set text [$w.text text]
        set oldstate [lindex [$text config -state] 4]
        set data(numobjects) 0
        $text config -state normal
        $text delete 0.0 end
        $text config -state $oldstate
        selection clear $w
}

composite_subcommand RichText text { w  args } {
        if {[llength $args] > 0} {
            eval $w.text text $args
        } else {
            return [$w.text text]
        }
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc RichText::InsertHol90StructuredText { w atindex textitemsvar extra_tags } {
        global RichText_flags
        upvar 1 $textitemsvar textitems

        set textw [$w.text text]
        if {[hol88] || !$RichText_flags(richtext)} { 
            error "RichText::InsertHol90StructuredText should not have been called"
        }
        set oldstate [lindex [$textw config -state] 4]
        $textw config -state normal

        $textw mark set insert $atindex
        $textw yview -pickplace "insert - 3 lines"
        set start [$textw index insert]
        foreach textitem $textitems {
            set text [lindex $textitem 4]
            $textw insert insert $text
        }
        foreach textitem $textitems {
            set from [lindex $textitem 0]
            set to [lindex $textitem 1]
            set where [lindex $textitem 2]
            set look [lindex $textitem 3]
            set text [lindex $textitem 4]
            if {$text==""} continue
            #puts "\nw = $w, text = $text, look = $look, from = $from, to = $to, pretags(from) = [$textw tag names $from], pretext = [$textw get $from "$to + 1 c"], where = $where, look = $look"
            if {[lindex [$textw tag config TMLOOK:$look -font] 4]==""} {
                RichText::configure_look_tag $w $look
            }
            #puts "tags at from (before cleaning) = [$textw tag names $from]"
            # puts "tags at from - 1 c(after inserting \{$text\}) = [$textw tag names "$from - 1 c"]"
            # puts "tags at from + 1 c(after inserting \{$text\}) = [$textw tag names "$from + 1 c"]"

            # the set of tags we added at the last chunk 
            # annoyingly hangs around on
            # the first character of the next chunk.  So
            # we have to explicitly remove them from the first
            # character. Bummer.
            if [info exists prev_tags] {
                foreach prev_tag $prev_tags {
                    $textw tag remove $prev_tag $from
                }
            }

            regexp {([^\.]*)\.(.*)} $to dummy tol toc
            if {$toc=="~1"} { incr tol -1; set toc 1000 }
            if {[string compare $from $to]==0} {
                $textw tag add TMWHERE:$where $from
                $textw tag add TMLOOK:$look $from
            } else {
                # puts "adding tag TMLOOK:$look, from = $from, tol = $tol, toc = $toc, to = $to, text = $text"
                $textw tag add TMWHERE:$where $from "$tol.$toc + 1 c"
                $textw tag add TMLOOK:$look $from "$tol.$toc + 1 c"
            }
            # puts "tags at from (after tagging \{$text\}) = [$textw tag names $from]"
            # puts "tags at from - 1 c(after tagging \{$text\}) = [$textw tag names "$from - 1 c"]"
            # puts "tags at from + 1 c(after tagging \{$text\}) = [$textw tag names "$from + 1 c"]"
            $textw tag lower TMWHERE:$where
            set prev_tags [list TMWHERE:$where TMLOOK:$look]
        }
        $textw insert insert "\n"
        if [info exists prev_tags] {
            foreach prev_tag $prev_tags {
                $textw tag remove $prev_tag "insert -1 c"
            }
        }
        foreach extra_tag $extra_tags {
            $textw tag add $extra_tag $start "insert -1 c"
        }
        $textw config -state $oldstate
}
        
#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc RichText::InsertPlainText { w atindex text extra_tags } {
        global RichText_flags
        set textw [$w.text text]
        set oldstate [lindex [$textw config -state] 4]
        $textw mark set insert $atindex
        $textw yview -pickplace "insert - 3 lines"
        $textw config -state normal
        lappend extra_tags $RichText_flags(default_look)
        $textw insert insert $text
        set len [string length $text]
#       foreach tag [$textw tag names "insert - $len c"] {
#           $textw tag remove $tag "insert -$len c - 1 c" "insert - 1 c"
#       } 
        if {$len==1} {
            foreach extra_tag $extra_tags {
                $textw tag add $extra_tag "insert - 1 c"
            }
        } else {
            foreach extra_tag $extra_tags {
                $textw tag add $extra_tag "insert - $len c - 1 c" "insert - 1 c"
            }
        }
        
        $textw config -state $oldstate
}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc RichText::tag_range { w tag first last } {
        # puts "first = $first, last = $last"
        set textw [$w.text text]

        $textw tag remove $tag 1.0 end
        regexp {([^\.]*)\.(.*)} $first dummy froml fromc
        regexp {([^\.]*)\.(.*)} $last dummy tol toc
        #puts "tol = $tol, toc = $toc"
        for {set line $froml} {$line <= $tol} {incr line; } {
            if {($line == $froml)} {
                set from $line.$fromc
            } else {
                set linetext [$textw get $line.0 "$line.0 lineend"]
                regexp "^(\[ \]*).*" $linetext dummy whitespace
                set from "$line.[string length $whitespace]"
            }
            if {$line==$tol} {
                set to "$line.$toc + 1 c"
            } else {
                set to "$line.0 lineend"
            }
        #puts "from = $from, to = $to"
            $textw tag add $tag $from $to
        }
}

composite_subcommand RichText subobject_at { w from to  } {

        set textw [$w.text text]
        global vals
        upvar #0 $w data
        upvar #0 config$w config
        if [$textw compare $from > $to] {
            set temp $from
            set from $to
            set to $temp
        } 
        regexp {([^\.]*)\.(.*)} $from dummy froml fromc
        regexp {([^\.]*)\.(.*)} $to dummy tol toc
        set paths ""
        # puts "froml = $froml, fromc = $fromc, tol = $tol, toc = $toc"
        for {set line $froml} {$line <= $tol} {incr line; } {
           set numchars [lindex [split [$textw index "$line.0 lineend"] .] 1]
           for {set char [expr "($line == $froml)?$fromc:0"] } \
               {($line == $tol)?($char <= $toc):($char < $numchars)} \
               {incr char} {
              set tags [$textw tag names $line.$char]
              # puts "tags at $line.$char = $tags"
              foreach tag $tags {
                 if [info exists tagdone($tag)] continue
                 set tagdone($tag) 1
                 if [regexp ^OBJECT:.*\$ $tag] {
                          if {[info exists object] && $object!=$tag} {
                              # puts "crosses object boundaries"
                              return ""
                          }
                          set object $tag
                 }
                 if [regexp ^TMWHERE:(.*)\$ $tag blah path] {
                     lappend paths $path
                          if {![info exists shortest_path] || [string length $shortest_path]>[string length $path]} {
                             set shortest_path $path
                          }
                 }
              }
           }
        }
        
        #------------------------------ 
        # We now have all the relevant paths, and the shortest of
        # all these as well.  shortest_path is a good upper-bound
        # on the longest common path.  
        # Starting with the shortest_path we keep making the path
        # one shorter until we find one that matches with all
        # the paths, which will be the longest common path.
        #------------------------------ 
        
        if ![info exists object] {
            return ""
        }
        # puts "paths = $paths, shortest_path = $shortest_path"
        if {$paths == ""} { return "" }
        set shortest_path_list [split $shortest_path -]
        for {set i [llength $shortest_path_list]} {$i > 0} {incr i} {
           set longest_common_path [join [lrange $shortest_path_list 0 [expr $i-1]] -]
           set ok 1
           foreach path $paths {
               # these string matches check to see if the current longest_common_path candidate 
               # matches the head of "path"
               if {$longest_common_path!=$path && ![string match [set longest_common_path]-* $path]} {
                  set ok 0
                  break
               }
           }
           if $ok break
        }
        if {$i==0} {
           set longest_common_path ""
        }
        return [list $object [split $longest_common_path -]]
}

#------------------------------ 
# compute the smallest range of text over the object
# which is covered by longest_common_path.
#------------------------------ 
composite_subcommand RichText range_of_subobject { w object branchlist } {
        upvar #0 $w data
        upvar #0 config$w config
        set branchtag [join $branchlist -]
        set textw [$w.text text]

        # optimization when 1 object is shown, and using focus interaction
        # model.  This is a hack.  It doesn't work for non-focus
        # interaction as sub-objects are not leaves in other cases,
        # whereas in focus interaction sub-objects are always
        # leaves.  This stuff needs to be reorganised anyway, but now I
        # understand what's going on at least.

        if {$data(numobjects)==1 && $config(focusVar)!=""} {
            #puts "branchtag = $branchtag"
            set ranges [$textw tag ranges TMWHERE:$branchtag] 
            if {[llength $ranges]==0} { error "sub-object $branchtag of $object does not exist (1)" }
            set first [lindex $ranges 0] 
            #puts "ranges = $ranges"
            set last [$textw index "[lindex $ranges [expr [llength $ranges ]-1]] - 1 c"]
            return [list $first $last]
        }

        set object_ranges [$textw tag ranges $object]
        # puts "object_ranges = $object_ranges"
        set on_first 1
        set first_set 0
        set last_set 0
        foreach object_range $object_ranges {
            if $on_first {
               set object_from $object_range
               set on_first 0
               continue
            } else {
               set object_to $object_range
               set on_first 1
            } 
            
            regexp {([^\.]*)\.(.*)} $object_from dummy froml fromc
            regexp {([^\.]*)\.(.*)} $object_to dummy tol toc
            
            # puts "object_from = $object_from, object_to = $object_to"
            for {set line $froml} {$line <= $tol} {incr line; } {
               set numchars [lindex [split [$textw index "$line.0 lineend"] .] 1]
               # puts "line = $line, numchars = $numchars"
               for {set char [expr {($line == $froml)?$fromc:0}] } \
                   {($line == $tol)?($char < $toc):($char < $numchars)} \
                   {incr char} {
                  set tags [$textw tag names $line.$char]
                  # puts "line = $line, char = $char, tags = $tags, branchtag = $branchtag"
                  if {[lsearch -exact $tags TMWHERE:[set branchtag]]!=-1 ||
                      ($branchtag!="" && [lsearch -glob $tags TMWHERE:[set branchtag]-*]!=-1) ||
                      ($branchtag=="" && [lsearch -glob $tags TMWHERE:*]!=-1)} {
                      if {!$first_set || [$textw compare $line.$char < $first]} {
                          set first $line.$char
                          set first_set 1
                      }
                      if {!$last_set || [$textw compare $line.$char > $last]} {
                          set last $line.$char
                          set last_set 1
                      }
                  }
               }
            }
        }    
        if ![info exists first] { error "sub-object $branchtag does not exist" }
        # puts "first = $first, last = $last"
        
        set res [list $first $last]
        return $res

}

composite_subcommand RichText tag_subobject { w subobject tag } {
        $w text tag raise $tag
        set object [lindex $subobject 0]
        set path [lindex $subobject 1]
        set firstlast [$w range_of_subobject $object $path]
        RichText::tag_range $w $tag [lindex $firstlast 0] [lindex $firstlast 1]
}



#--------------------------------------------------------------------------
# normal-selection interaction model
#--------------------------------------------------------------------------
proc RichText::bind_for_normal_selection { w } {
        set textw [$w.text text]
        tk_bindForTraversal $textw
        bind $textw <ButtonPress-1> [bind Text <ButtonPress-1>]
        bind $textw <ButtonRelease-1> [bind Text <ButtonRelease-1>]
        bind $textw <B1-Motion> [bind Text <B1-Motion>]
        bind $textw <Double-1> [bind Text <Double-1>]
        bind $textw <FocusIn> [bind Text <FocusIn>]
        bind $textw <FocusOut> [bind Text <FocusOut>]
        $textw tag remove DECORATE:cursor 1.0 end
        $textw tag remove DECORATE:focus 1.0 end
        $textw tag configure DECORATE:sel -underline 0 -relief raised -borderwidth 1
}


#--------------------------------------------------------------------------
# subobject-selection interaction model
#
# double clicking selects the smallest subobject.
#
#--------------------------------------------------------------------------

proc RichText::select_smallest_subobject { w from to } {
        set textw [$w.text text]
        global vals
        upvar #0 $w data
        upvar #0 config$w config

        #------------------------------ 
        # find the smallest subobject covering the selection
        #------------------------------ 
        set subobject [$w subobject_at $from $to]
        if {$subobject==""} { return "" }
        set object [lindex $subobject 0]
        set path [lindex $subobject 1]
        
        #------------------------------ 
        # we allow arbitrary selection types to be created by
        # object-class functions "select_subobject".
        #------------------------------ 
        
        selection own $w "catch \{$textw tag remove DECORATE:sel 1.0 end\}; set $config(selectiontypesVar) {}"
        $w tag_subobject $subobject DECORATE:sel
        upvar #0 $config(selectiontypesVar) selectiontypes
        set selectiontypes [[lindex $data($object,spec) 0]::select_subobject $w [lindex $data($object,spec) 1] $path]
}



proc RichText::bind_for_subobject_selection { w } {
        upvar #0 $w data
        set textw [$w.text text]
        tk_bindForTraversal $textw
        bind $textw <Double-1> "
            RichText::select_smallest_subobject $w \[%W index @%x,%y\] \[%W index @%x,%y\]
        "
        bind $textw <ButtonPress-1> {
            %W mark set insert @%x,%y
            %W mark set anchor insert
            if {[lindex [%W config -state] 4] == "normal"} {
                focus_goTo %W
            }
        }
        bind $textw <B1-Motion> "
            RichText::select_smallest_subobject $w \[%W index anchor\] \[%W index @%x,%y\]
        "
        bind $textw <ButtonRelease-1> { nop }
        $textw tag configure DECORATE:sel -underline 1 \
                -relief flat
        $textw tag remove DECORATE:cursor 1.0 end
        $textw tag remove DECORATE:focus 1.0 end
        $textw tag remove sel 1.0 end
}




#--------------------------------------------------------------------------
# subobject-focus interaction model
# 
# Only leaf nodes may be highlighted by the focus.  We find the
# next node in a particular direction by looking through the text in that
# direction for the first different TMWHERE:* tag.
#
# This kind of assumes only one object is being displayed... It could
# be written for more.
#
# Assumes each node in the text is only tagged with one TMWHERE:* tag,
# which should *always* be the case!
#--------------------------------------------------------------------------

proc tag_is_leaf { tag } {
        set path [split $tag -]
        return [expr {[lindex $path [expr [llength $path]-1]]=="L"}]
}

proc RichText::search { w from origpath include_this_line forward_or_back } {
        # puts "from = $from, origpath = $origpath, include_this_line = $include_this_line, forward_or_back = $forward_or_back"
        set textw [$w text]
        regexp {([^\.]*)\.(.*)} $from dummy froml fromc
        if {$forward_or_back=="back"} {
            set increment -1
            set line_comparison {$line > 0}
            set char_initialisation {set char $numchars}
            set char_comparison {$char > 0}
        } else {
            set increment 1
            set numlines [lindex [split [$textw index end] .] 0]
            set line_comparison {$line <= $numlines}
            set char_initialisation {set char 0}
            set char_comparison {$char <= $numchars}
        }
        # puts "numlines = $numlines"
        set origtag [join $origpath -]
        set textw [$w.text text]
        set line $froml
        set char $fromc
        if {!$include_this_line} { incr line $increment }
        for {} $line_comparison {incr line $increment} {
           set numchars [lindex [split [$textw index "$line.0 lineend"] .] 1]
           if {$line!=$froml} $char_initialisation
           for {} $char_comparison {incr char $increment} {
              set tags [$textw tag names $line.$char]
              #puts "line = $line, char = $char, tags = $tags, origtag = $origtag"
              foreach tag $tags {
                 if [info exists tagdone($tag)] continue
                 set tagdone($tag) 1
                 if [regexp ^OBJECT:.*\$ $tag] {
                     set object $tag
                 }
                 if {[info exists object] && [regexp ^TMWHERE:(.*)\$ $tag blah pathtag]} {
                        if {[tag_is_leaf $pathtag] && $pathtag!=$origtag} {
              # puts "found different tag: $pathtag"
                           return [list $object [split $pathtag -]]
                        }
                 }
              }
           }
        }
        error "search: no object found"
}

proc RichText::search_from_object { w subobject include_this_line forward_or_back } {
        set origobject [lindex $subobject 0]
        set origpath [lindex $subobject 1]
        set origrange [$w range_of_subobject $origobject $origpath]
        if {$forward_or_back=="back"} {
            set from [lindex $origrange 0]
        } else {
            set from [lindex $origrange 1]
        }
        RichText::search $w $from $origpath $include_this_line $forward_or_back
}



proc RichText::cursor_at { w where } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set newcursor [$w subobject_at $where $where]
        # puts "where = $where, newcursor = $newcursor"
        if {$newcursor=="" || $cursor==$newcursor} { return }
        set newpath [lindex $newcursor 1]
        # check the selected subobject ends in L, i.e. that it is cursorable.
        if {[lindex $newpath [expr [llength $newpath]-1]]!="L"} { return }

        set? cursor $newcursor
}

proc RichText::focus_at { w where } {
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        set oldfocus $focus
        set newfocus [$w subobject_at $where $where]
        # puts "where = $where, newfocus = $newfocus"
        if {$newfocus=="" || $focus==$newfocus} { return }
        set newpath [lindex $newfocus 1]
        # check the selected subobject ends in L, i.e. that it is focusable.
        if {[lindex $newpath [expr [llength $newpath]-1]]!="L"} { return }

        if [catch {set? focus $newfocus}] {
            set focus $oldfocus
        }
}

proc RichText::focus_at_cursor { w } {
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        upvar #0 $config(cursorVar) cursor
        set oldfocus $focus
        if [catch {set? focus $cursor}] {
            set focus $oldfocus
        }
}


#proc RichText::cursor_up1 { w old_cursor } {
#       upvar #0 config$w config
#       upvar #0 $config(cursorVar) cursor
#       set object [lindex $old_cursor 0]
#       set old_path [lindex $old_cursor 1]
#       set old_path_length [llength $old_path]
#               # take off trailing 0 and prev. index
#       set new_path [lrange $old_path 0 [expr $old_path_length-3]]
#       lappend new_path 0
#       if ![catch {set cursor [list $object $new_path]}] { return }
#       error "can't go up"
#}
#
#proc RichText::cursor_down1 { w old_cursor } {
#       upvar #0 config$w config
#       upvar #0 $config(cursorVar) cursor
#       set old_path [lindex $old_cursor 1]
#       set object [lindex $old_cursor 0]
#
#               # take off trailing 0
#       set old_path_length [llength $old_path]
#       set new_path [lrange $old_path 0 [expr $old_path_length-2]]
#       lappend new_path 1 0
#       # puts "new_path = $new_path"
#       if ![catch {set? cursor [list $object $new_path]}] { return }
#       # puts "can't go down"
#        error "can't go down"
#}
#proc RichText::cursor_leftright { w old_cursor howfar } {
#       upvar #0 config$w config
#       upvar #0 $config(cursorVar) cursor
#       set old_path [lindex $old_cursor 1]
#       set object [lindex $old_cursor 0]
#
#       set old_path_length [llength $old_path]
#       if {$old_path_length==0} { return }
#       set back_of_path [lindex $old_path [expr $old_path_length-2]]
#               # take off trailing 0 and prev. index
#       set new_path [lrange $old_path 0 [expr $old_path_length-3]]
#       incr back_of_path $howfar
#       lappend new_path $back_of_path 0
#       if ![catch {set? cursor [list $object $new_path]}] { return }
#       error "can't go left/right $howfar"
#          
#}


proc RichText::cursor_up { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set old_cursor $cursor
        # puts "cursor = $cursor"
        catch {set? cursor [RichText::search_from_object $w $old_cursor 0 back]}
        $w text yview "DECORATE:cursor.first - 2 lines"
}
proc RichText::cursor_down { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set old_cursor $cursor
        # puts "cursor = $cursor"
        catch {set? cursor [RichText::search_from_object $w $old_cursor 0 forward]}
        $w text yview "DECORATE:cursor.first - 2 lines"
}
proc RichText::cursor_left { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set old_cursor $cursor
        catch {set? cursor [RichText::search_from_object $w $old_cursor 1 back]}
        $w text yview "DECORATE:cursor.first - 2 lines"
}
proc RichText::cursor_right { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        # puts "cursor = $cursor"
        set old_cursor $cursor
        catch {set? cursor [RichText::search_from_object $w $old_cursor 1 forward]}
        $w text yview "DECORATE:cursor.first - 2 lines"
}

proc RichText::cursor_home { w } {
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set? cursor [RichText::search $w 1.0 dummy 1 forward]
        $w text yview "DECORATE:cursor.first"
}


proc RichText::focus_change { w args } {
        global errorInfo
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        upvar #0 $config(cursorVar) cursor
        set textw [$w text]
        catch {$textw tag remove DECORATE:focus 1.0 end}
        if [catch {$w tag_subobject $focus DECORATE:focus} err] {
            puts "error in tag_subobject $errorInfo"; error $err $errorInfo
        }
        update
        set? cursor $focus
        $textw tag raise DECORATE:focus DECORATE:cursor
        $textw yview "DECORATE:focus.first - 2 lines"
}

proc RichText::cursor_change { w args } {
        global errorInfo
        upvar #0 config$w config
        upvar #0 $config(cursorVar) cursor
        set textw [$w text]
        catch {$textw tag remove DECORATE:cursor 1.0 end}
        if [catch {$w tag_subobject $cursor DECORATE:cursor} err] {
            # puts "error in tag_subobject $errorInfo"; error $err $errorInfo
        }
        $textw tag raise DECORATE:focus DECORATE:cursor
#       $textw yview "DECORATE:cursor.first - 2 lines"
}

proc RichText::bind_for_subobject_focus { w } {
        upvar #0 $w data
        set textw [$w.text text]
        tk_bindForTraversal $textw
        upvar #0 config$w config
        upvar #0 $config(focusVar) focus
        upvar #0 $config(cursorVar) cursor
        trace variable focus w "RichText::focus_change $w"
        trace variable cursor w "RichText::cursor_change $w"
        bind $textw <Double-1> {}
        bind $textw <ButtonPress-1> "focus_goTo %W; RichText::cursor_at $w \[%W index @%x,%y\]"
        bind $textw <B1-Motion> "RichText::cursor_at $w \[%W index @%x,%y\]"
        bind $textw <ButtonRelease-1> "grab release %W; RichText::focus_at $w \[%W index @%x,%y\]"
        bind $textw <Up> "RichText::cursor_up $w"
        bind $textw <Left> "RichText::cursor_left $w"
        bind $textw <Down> "RichText::cursor_down $w"
        bind $textw <Right> "RichText::cursor_right $w"
        bind $textw <Home> "RichText::cursor_home $w"
        bind $textw <Return> "RichText::focus_at_cursor $w"
        bind $textw <FocusIn> { %W tag configure DECORATE:cursor -borderwidth 4; %W tag configure DECORATE:focus -borderwidth 4 -relief ridge }
        bind $textw <FocusOut> { %W tag configure DECORATE:cursor -borderwidth 2; %W tag configure DECORATE:focus -borderwidth 2 -relief ridge }
        $textw tag remove DECORATE:sel 1.0 end
        $textw tag remove sel 1.0 end
}

