%============================================================================%
%                    New inductive definitions package                       %
%                                                                            %
%                     For HOL88, version 2.02 or 2.03                        %
%                                                                            %
%                     (c) John Harrison, 8th May 1995                        %
%============================================================================%

timer true;;

top_print print_all_thm;;

%============================================================================%
% Part 0: Convenient utilities.                                              %
%============================================================================%

%----------------------------------------------------------------------------%
% Special case of "union", intended to be reasonably efficient.              %
%----------------------------------------------------------------------------%

let insert x l =
  if mem x l then l else x.l;;

%----------------------------------------------------------------------------%
% Repetition.                                                                %
%----------------------------------------------------------------------------%

letrec repeat f x =
  (let y = f x in repeat f y) ? x;;

%----------------------------------------------------------------------------%
% Apply a destructor as many times as elements in list.                      %
%----------------------------------------------------------------------------%

letrec nsplit dest clist x =
  if clist = [] then [],x else
  let l,r = dest x in
  let ll,y = nsplit dest (tl clist) r in
  l.ll,y;;

%----------------------------------------------------------------------------%
% Strip off exactly n arguments from combination.                            %
%----------------------------------------------------------------------------%

let strip_ncomb =
  letrec strip(n,tm,acc) =
    if n < 1 then tm,acc else
    let l,r = dest_comb tm in
    strip(n - 1,l,r.acc) in
  \n tm. strip(n,tm,[]);;

%----------------------------------------------------------------------------%
% Share out list according to pattern in list-of-lists.                      %
%----------------------------------------------------------------------------%

letrec shareout pat all =
  if pat = [] then [] else
  let l,r = chop_list (length (hd pat)) all in
  l.(shareout (tl pat) r);;

%----------------------------------------------------------------------------%
% Cons-free version of assoc.                                                %
%----------------------------------------------------------------------------%

letrec assoc2 x (l,r) =
  if x = hd l then hd r else assoc2 x (tl l,tl r);;

%----------------------------------------------------------------------------%
% Grabbing left operand of a binary operator (or something coextensive!)     %
%----------------------------------------------------------------------------%

let land = rand o rator;;

%----------------------------------------------------------------------------%
% Gets all variables (free and/or bound) in a term.                          %
%----------------------------------------------------------------------------%

let variables =
  letrec vars(acc,tm) =
    if is_var tm then insert tm acc
    else if is_const tm then acc
    else if is_abs tm then
      let v,bod = dest_abs tm in
      vars(insert v acc,bod)
    else
      let l,r = dest_comb tm in
      vars(vars(acc,l),r) in
  \tm. vars([],tm);;

%----------------------------------------------------------------------------%
% Gentler version of "variant" which allows same name but different types.   %
%----------------------------------------------------------------------------%

letrec variant av v =
  if mem v av then
    let n,t = dest_var v in
    variant av (mk_var(n^`'`,t))
  else v;;

%----------------------------------------------------------------------------%
% Produces a sequence of variants, considering previous inventions.          %
%----------------------------------------------------------------------------%

letrec variants av vs =
  if vs = [] then [] else
  let vh = variant av (hd vs) in vh.(variants (vh.av) (tl vs));;

%----------------------------------------------------------------------------%
% Produce a set of reasonably readable arguments, using variants if needed.  %
%----------------------------------------------------------------------------%

let make_args =
  letrec margs n avoid tys =
    if tys = [] then [] else
    let v = variant avoid (mk_var(`a`^(string_of_int n),hd tys)) in
    v.(margs (n + 1) (v.avoid) (tl tys)) in
  margs 0;;

%----------------------------------------------------------------------------%
% Grabs conclusion of rule, whether or not it has an antecedant.             %
%----------------------------------------------------------------------------%

let getconcl tm =
  let bod = repeat (snd o dest_forall) tm in
  snd(dest_imp bod) ? bod;;

%----------------------------------------------------------------------------%
% Crude but adequate tautology prover.                                       %
%----------------------------------------------------------------------------%

let TAUT =
  let val w t = type_of t = ":bool" & can (find_term is_var) t & free_in t w in
  C (curry prove)
  (REPEAT GEN_TAC THEN (REPEAT o CHANGED_TAC o W)
   (C $THEN (REWRITE_TAC[]) o BOOL_CASES_TAC o hd o sort (uncurry free_in) o
    W(find_terms o val) o snd));;

%----------------------------------------------------------------------------%
% Expand lambda-term function definition with its arguments.                 %
%----------------------------------------------------------------------------%

let RIGHT_BETAS =
  rev_itlist (\a. CONV_RULE (RAND_CONV BETA_CONV) o C AP_THM a);;

%----------------------------------------------------------------------------%
% Likewise, but quantify afterwards.                                         %
%----------------------------------------------------------------------------%

let HALF_BETA_EXPAND args th =
  GENL args (RIGHT_BETAS args th);;

%----------------------------------------------------------------------------%
% Universally quantify both sides of equation.                               %
%----------------------------------------------------------------------------%

let MK_FORALL =
  let bty = ":bool" in
  \v th.
    let atm = mk_const(`!`,
      mk_type(`fun`,[mk_type(`fun`,[type_of v;bty]); bty])) in
  AP_TERM atm (ABS v th);;

%----------------------------------------------------------------------------%
% MK_CONJ - Conjoin equations                                                %
%----------------------------------------------------------------------------%

let MK_CONJ =
  let andtm = "$/\" in
  \eq1 eq2. MK_COMB(AP_TERM andtm eq1,eq2);;

%----------------------------------------------------------------------------%
% Simple case of EXISTS: A |- P[x]  ->  A |- ?x. P[x]                        %
%----------------------------------------------------------------------------%

let SIMPLE_EXISTS v th =
  EXISTS (mk_exists(v,concl th),v) th;;

%----------------------------------------------------------------------------%
% Simple case of CHOOSE: P[x] |- Q  ->  ?x. P[x] |- Q                        %
%----------------------------------------------------------------------------%

let SIMPLE_CHOOSE v th =
  CHOOSE(v,ASSUME (mk_exists(v,hd(hyp th)))) th;;

%----------------------------------------------------------------------------%
% Simple case of DISJ_CASES: P |- R, Q |- R  ->  P \/ Q |- R                 %
%----------------------------------------------------------------------------%

let SIMPLE_DISJ_CASES th1 th2 =
  DISJ_CASES (ASSUME(mk_disj(hd(hyp th1),hd(hyp th2)))) th1 th2;;

%----------------------------------------------------------------------------%
% Converse, i.e. P \/ Q |- R  ->  P |- R, Q |- R                             %
%----------------------------------------------------------------------------%

let SIMPLE_DISJ_PAIR th =
  let l,r = dest_disj(hd(hyp th)) in
  PROVE_HYP (DISJ1 (ASSUME l) r) th,PROVE_HYP (DISJ2 l (ASSUME r)) th;;

%----------------------------------------------------------------------------%
% Iterated FORALL_IMP_CONV: (!x1..xn. P[xs] ==> Q) -> (?x1..xn. P[xs]) ==> Q %
%----------------------------------------------------------------------------%

let FORALL_IMPS_CONV tm =
  let avs,bod = strip_forall tm in
  let th1 = DISCH tm (UNDISCH(SPEC_ALL(ASSUME tm))) in
  let th2 = itlist SIMPLE_CHOOSE avs th1 in
  let tm2 = hd(hyp th2) in
  let th3 = DISCH tm2 (UNDISCH th2) in
  let th4 = ASSUME (concl th3) in
  let ant = land bod in
  let th5 = itlist SIMPLE_EXISTS avs (ASSUME ant) in
  let th6 = GENL avs (DISCH ant (MP th4 th5)) in
  IMP_ANTISYM_RULE (DISCH_ALL th3) (DISCH_ALL th6);;

%----------------------------------------------------------------------------%
%    (!x1..xn. P1[xs] ==> Q[xs]) /\ ... /\ (!x1..xn. Pm[xs] ==> Q[xs])       %
% -> (!x1..xn. P1[xs] \/ ... \/ Pm[xs] ==> Q[xs])                            %
%----------------------------------------------------------------------------%

let AND_IMPS_CONV tm =
  let ths = CONJUNCTS(ASSUME tm) in
  let avs = fst(strip_forall(concl(hd ths))) in
  let thl = map (DISCH tm o UNDISCH o SPEC_ALL) ths in
  let th1 = end_itlist SIMPLE_DISJ_CASES thl in
  let tm1 = hd(hyp th1) in
  let th2 = GENL avs (DISCH tm1 (UNDISCH th1)) in
  let tm2 = concl th2 in
  let th3 = DISCH tm2 (UNDISCH (SPEC_ALL (ASSUME tm2))) in
  let thts,tht =  nsplit SIMPLE_DISJ_PAIR (tl ths) th3 in
  let proc_fn th =
    let t = hd(hyp th) in GENL avs (DISCH t (UNDISCH th)) in
  let th4 = itlist (CONJ o proc_fn) thts (proc_fn tht) in
  IMP_ANTISYM_RULE (DISCH_ALL th2) (DISCH_ALL th4);;

%----------------------------------------------------------------------------%
% AC equivalence for conjunctions.                                           %
%----------------------------------------------------------------------------%

let CONJ_AC =
  letrec CONJ_BUILD thl tm =
   (let l,r = dest_conj tm in
    CONJ (CONJ_BUILD thl l) (CONJ_BUILD thl r))
   ? find (aconv tm o concl) thl in
  \(l,r).
    let lthms = CONJUNCTS (ASSUME l)
    and rthms = CONJUNCTS (ASSUME r) in
    let lth = CONJ_BUILD lthms r
    and rth = CONJ_BUILD rthms l in
    IMP_ANTISYM_RULE (DISCH_ALL lth) (DISCH_ALL rth);;

%----------------------------------------------------------------------------%
%      A, x = t |- P[x]                                                      %
%     ------------------ EXISTS_EQUATION                                     %
%        A |- ?x. P[x]                                                       %
%----------------------------------------------------------------------------%

let EXISTS_EQUATION =
  let pth =
    let th1 = MP (SPEC "t:*" (ASSUME "!x:*. (x = t) ==> P x")) (REFL "t:*") in
    let th2 = MP (ISPECL ["P:*->bool"; "t:*"] SELECT_AX) th1 in
    let th3 = AP_THM EXISTS_DEF "P:*->bool" in
    let th4 = TRANS th3 (BETA_CONV (rand(concl th3))) in
    GENL ["P:*->bool"; "t:*"] (DISCH_ALL (EQ_MP (SYM th4) th2)) in
  \tm th.
    let l,r = dest_eq tm in
    let P = mk_abs(l,concl th) in
    let th1 = BETA_CONV(mk_comb(P,l)) in
    let th2 = ISPECL [P; r] pth in
    let th3 = EQ_MP (SYM th1) th in
    let th4 = GEN l (DISCH tm th3) in
    MP th2 th4;;

%----------------------------------------------------------------------------%
% GEN_PAIR_TAC - Like GEN_TAC but "pairs" the relevant variable              %
%----------------------------------------------------------------------------%

let GEN_PAIR_TAC =
  W($THEN GEN_TAC o SUBST1_TAC o SYM o
    C ISPEC PAIR o fst o dest_forall o snd);;

%============================================================================%
% Part 1: The main part of the inductive definitions package.                %
%============================================================================%

%----------------------------------------------------------------------------%
% Translates a single clause to have variable arguments, simplifying.        %
%----------------------------------------------------------------------------%

let canonicalize_clause =
  let calculate_simp_sequence =
    letrec getequs(avs,plis) =
      if plis = [] then [] else
      let h.t = plis in
      let r = snd h in
      if mem r avs then
        h.(getequs(avs,filter ($not o curry$=r o snd) t))
      else
        getequs(avs,t) in
    \avs plis.
      let oks = getequs(avs,plis) in
      oks,subtract plis oks in
  \(cls,args).
    let avs,bimp = strip_forall cls in
    let ant,con = dest_imp bimp ? "T",bimp in
    let rel,xargs = strip_comb con in
    let plis = com(args,xargs) in
    let yes,no = calculate_simp_sequence avs plis in
    let nvs = filter ($not o C mem (map snd yes)) avs in
    let eth =
      if is_imp bimp then
        let atm = itlist (curry mk_conj o mk_eq) (yes@no) ant in
        let ths,tth = nsplit CONJ_PAIR plis (ASSUME atm) in
        let thl = map (\t. find (\th. lhs(concl th) = t) ths) args in
        let th0 = MP (SPECL avs (ASSUME cls)) tth in
        let th1 = rev_itlist (C (curry MK_COMB)) thl (REFL rel) in
        let th2 = EQ_MP (SYM th1) th0 in
        let th3 = INST yes (DISCH atm th2) in
        let tm4 = funpow (length yes) rand (land(concl th3)) in
        let th4 = itlist (CONJ o REFL o fst) yes (ASSUME tm4) in
        let th5 = GENL args (GENL nvs (DISCH tm4 (MP th3 th4))) in
        let th6 = SPECL nvs (SPECL (map snd plis) (ASSUME (concl th5))) in
        let th7 = itlist (CONJ o REFL o snd) no (ASSUME ant) in
        let th8 = GENL avs (DISCH ant (MP th6 th7)) in
        IMP_ANTISYM_RULE (DISCH_ALL th5) (DISCH_ALL th8)
      else
        let atm = list_mk_conj(map mk_eq (yes@no)) in
        let ths = CONJUNCTS (ASSUME atm) in
        let thl = map (\t. find (\th. lhs(concl th) = t) ths) args in
        let th0 = SPECL avs (ASSUME cls) in
        let th1 = rev_itlist (C (curry MK_COMB)) thl (REFL rel) in
        let th2 = EQ_MP (SYM th1) th0 in
        let th3 = INST yes (DISCH atm th2) in
        let tm4 = funpow (length yes) rand (land(concl th3)) in
        let th4 = itlist (CONJ o REFL o fst) yes (ASSUME tm4) in
        let th5 = GENL args (GENL nvs (DISCH tm4 (MP th3 th4))) in
        let th6 = SPECL nvs (SPECL (map snd plis) (ASSUME (concl th5))) in
        let th7 = LIST_CONJ (map (REFL o snd) no) in
        let th8 = GENL avs (MP th6 th7) in
        IMP_ANTISYM_RULE (DISCH_ALL th5) (DISCH_ALL th8) in
   let ftm = funpow (length args) (body o rand) (rand(concl eth)) in
   TRANS eth (itlist MK_FORALL args (FORALL_IMPS_CONV ftm));;

%----------------------------------------------------------------------------%
% Canonicalizes the set of clauses, disjoining compatible antecedants.       %
%----------------------------------------------------------------------------%

let canonicalize_clauses clauses =
  let concls = map getconcl clauses in
  let uncs = map strip_comb concls in
  let rels = itlist (insert o fst) uncs [] in
  let xargs = map (snd o C assoc uncs) rels in
  let closed = list_mk_conj clauses in
  let avoids = variables closed in
  let flargs = make_args avoids (map type_of (end_itlist append xargs)) in
  let vargs = shareout xargs flargs in
  let cargs = map (C assoc2 (rels,vargs) o fst) uncs in
  let cthms = map2 canonicalize_clause (clauses,cargs) in
  let pclauses = map (rand o concl) cthms in
  let collectclauses tm =
    mapfilter (\t. if fst t = tm then snd t else fail)
              (com(map fst uncs,pclauses)) in
  let clausell = map collectclauses rels in
  let cclausel = map list_mk_conj clausell in
  let cclauses = list_mk_conj cclausel
  and oclauses = list_mk_conj pclauses in
  let pth = TRANS (end_itlist MK_CONJ cthms) (CONJ_AC (oclauses,cclauses)) in
  TRANS pth (end_itlist MK_CONJ (map AND_IMPS_CONV cclausel));;

%----------------------------------------------------------------------------%
% Proves non-schematic relations defined by canonical rules exist.           %
%----------------------------------------------------------------------------%

let prove_canon_relations_exists clauses =
  let closed = list_mk_conj clauses in
  let clauses = conjuncts closed in
  let vargs,bodies = split(map strip_forall clauses) in
  let ants,concs = split(map dest_imp bodies) in
  let rels = map (repeat rator) concs in
  let avoids = variables closed in
  let rels' = variants avoids rels in
  let crels = com(rels',rels) in
  let prime_fn = subst crels in
  let closed' = prime_fn closed in
  let mk_def (arg,con) =
    mk_eq(repeat rator con,
      list_mk_abs(arg,list_mk_forall(rels',mk_imp(closed',prime_fn con)))) in
  let deftms = map2 mk_def (vargs,concs) in
  let defthms = map2 (uncurry HALF_BETA_EXPAND) (vargs,map ASSUME deftms) in
  let mk_ind (args,th) =
    let th1 = fst(EQ_IMP_RULE(SPEC_ALL th)) in
    let ant = land(concl th1) in
    let th2 = SPECL rels' (UNDISCH th1) in
    GENL args (DISCH ant (UNDISCH th2)) in
  let indthms = map2 mk_ind (vargs,defthms) in
  let indthmr = LIST_CONJ indthms in
  let indthm = GENL rels' (DISCH closed' indthmr) in
  let mconcs = map2 (\(a,t). list_mk_forall(a,mk_imp(t,prime_fn t)))
    (vargs,ants) in
  let monotm = mk_imp(concl indthmr,list_mk_conj mconcs) in
  let monothm = ASSUME(list_mk_forall(rels,list_mk_forall(rels',monotm))) in
  let closthm = ASSUME closed' in
  let monothms = CONJUNCTS
    (MP (SPEC_ALL monothm) (MP (SPECL rels' indthm) closthm)) in
  let closthms = CONJUNCTS closthm in
  let prove_rule (mth,cth,dth) =
    let avs,bod = strip_forall(concl mth) in
    let th1 = IMP_TRANS (SPECL avs mth) (SPECL avs cth) in
    let th2 = GENL rels' (DISCH closed' (UNDISCH th1)) in
    let th3 = EQ_MP (SYM (SPECL avs dth)) th2 in
    GENL avs (DISCH (land bod) th3) in
  let rulethms = map2 prove_rule (monothms,com(closthms,defthms)) in
  let rulethm = LIST_CONJ rulethms in
  let dtms = map2 list_mk_abs (vargs,ants) in
  let double_fn = subst (com(dtms,rels)) in
  let mk_unbetas (tm,dtm) =
    let avs,bod = strip_forall tm in
    let il,r = dest_comb bod in
    let i,l = dest_comb il in
    let bth = RIGHT_BETAS avs (REFL dtm) in
    let munb = AP_THM (AP_TERM i bth) r in
    let iunb = AP_TERM (mk_comb(i,double_fn l)) bth in
    let junb = AP_TERM (mk_comb(i,r)) bth in
    let quantify = itlist MK_FORALL avs in
    (quantify munb,quantify iunb,quantify junb) in
  let unths = map2 mk_unbetas (clauses,dtms) in
  let irthm = EQ_MP (SYM(end_itlist MK_CONJ (map fst unths))) rulethm in
  let mrthm = MP (SPECL rels (SPECL dtms monothm)) irthm in
  let imrth = EQ_MP (SYM(end_itlist MK_CONJ (map (fst o snd) unths))) mrthm in
  let ifthm = MP (SPECL dtms indthm) imrth in
  let fthm = EQ_MP (end_itlist MK_CONJ (map (snd o snd) unths)) ifthm in
  let mk_case (th1,th2) =
    let avs = fst(strip_forall(concl th1)) in
    GENL avs (IMP_ANTISYM_RULE (SPEC_ALL th1) (SPEC_ALL th2)) in
  let casethm = LIST_CONJ (map2 mk_case (CONJUNCTS fthm,CONJUNCTS rulethm)) in
  let xthm = CONJ rulethm (CONJ indthm casethm) in
  itlist EXISTS_EQUATION deftms xthm;;

%----------------------------------------------------------------------------%
% General case for nonschematic relations, leaving monotonicity hyp.         %
%----------------------------------------------------------------------------%

let manual_prove_nonschematic_inductive_relations_exist clauses =
  let canonthm = canonicalize_clauses clauses in
  let canonthm' = SYM canonthm in
  let pclosed = rand(concl canonthm) in
  let pclauses = conjuncts pclosed in
  let rawthm = prove_canon_relations_exists pclauses in
  let rels,thmbody = strip_exists(concl rawthm) in
  let allthms = ASSUME thmbody in
  let rulethm,otherthms = CONJ_PAIR allthms in
  let indthm,casethm = CONJ_PAIR otherthms in
  let rulethm' = EQ_MP canonthm' rulethm in
  let rels',indbody = strip_forall(concl indthm) in
  let il,r = dest_comb indbody in
  let i,l = dest_comb il in
  let th = AP_THM (AP_TERM i (INST (com(rels',rels)) canonthm)) r in
  let indthm' = EQ_MP
    (itlist MK_FORALL rels'
      (AP_THM (AP_TERM i (INST (com(rels',rels)) canonthm')) r)) indthm in
  let tempthm = CONJ rulethm' (CONJ indthm' casethm) in
  PROVE_HYP rawthm
   (itlist SIMPLE_CHOOSE rels (itlist SIMPLE_EXISTS rels tempthm));;

%============================================================================%
% Part 2: Tactic-integrated tools for proving monotonicity automatically.    %
%============================================================================%

let MONO_AND = TAUT "(A ==> B) /\ (C ==> D) ==> (A /\ C ==> B /\ D)";;

let MONO_OR = TAUT "(A ==> B) /\ (C ==> D) ==> (A \/ C ==> B \/ D)";;

let MONO_IMP = TAUT "(B ==> A) /\ (C ==> D) ==> ((A ==> C) ==> (B ==> D))";;

let MONO_NOT = TAUT "(B ==> A) ==> (~A ==> ~B)";;

let MONO_ALL = PROVE
 ("(!x:*. P x ==> Q x) ==> ($! P ==> $! Q)",
  REPEAT DISCH_TAC THEN
  GEN_REWRITE_TAC RAND_CONV [] [GSYM ETA_AX] THEN
  GEN_TAC THEN FIRST_ASSUM MATCH_MP_TAC THEN
  SPEC_TAC("x:*","x:*") THEN ASM_REWRITE_TAC[ETA_AX]);;

let MONO_EXISTS = PROVE
 ("(!x:*. P x ==> Q x) ==> ($? P ==> $? Q)",
  REPEAT DISCH_TAC THEN GEN_REWRITE_TAC RAND_CONV [] [GSYM ETA_AX] THEN
  EXISTS_TAC "$@(P:*->bool)" THEN FIRST_ASSUM MATCH_MP_TAC THEN
  RULE_ASSUM_TAC(BETA_RULE o REWRITE_RULE[EXISTS_DEF]) THEN
  ASM_REWRITE_TAC[]);;

let MONO_ALL_EL = PROVE
 ("(!x:*. P x ==> Q x) ==> (ALL_EL P l ==> ALL_EL Q l)",
  DISCH_TAC THEN SPEC_TAC("l:* list","l:* list") THEN
  LIST_INDUCT_TAC THEN REWRITE_TAC[ALL_EL] THEN
  REPEAT STRIP_TAC THEN FIRST_ASSUM MATCH_MP_TAC THEN
  ASM_REWRITE_TAC[]);;

%----------------------------------------------------------------------------%
% Simplified version of MATCH_MP_TAC to avoid quantifier troubles.           %
%----------------------------------------------------------------------------%

let (BACKCHAIN_TAC th :tactic) =
  let match_fn = PART_MATCH (snd o dest_imp) th in
  \(asl,w).
    let th1 = match_fn w in
    let ant,con = dest_imp(concl th1) in
    ([asl,ant],\[t]. MATCH_MP th1 t);;

%----------------------------------------------------------------------------%
%   ?- (\x. P[x]) x1 .. xn ==> (\y. Q[y]) x1 .. xn                           %
% ==================================================                         %
%     ?- !x1. P[x1] x2 .. xn ==> Q[x1] x2 .. xn                              %
%----------------------------------------------------------------------------%

let MONO_ABS_TAC =
  let imp = "$==>" in
  \(asl,w).
    let ant,con = dest_imp w in
    let vars = snd(strip_comb con) in
    let rnum = length vars - 1 in
    let hd1,args1 = strip_ncomb rnum ant
    and hd2,args2 = strip_ncomb rnum con in
    let th1 = rev_itlist (C AP_THM) args1 (BETA_CONV hd1)
    and th2 = rev_itlist (C AP_THM) args1 (BETA_CONV hd2) in
    let th3 = MK_COMB(AP_TERM imp th1,th2) in
    CONV_TAC(REWR_CONV th3) (asl,w);;

%----------------------------------------------------------------------------%
%   ?- UNCURRY P x1 .. xn ==> UNCURRY Q x1 .. xn                             %
% ================================================                           %
%    ?- !x y. P x y x2 .. xn ==> Q x y x2 .. xn                              %
%----------------------------------------------------------------------------%

let MONO_UNCURRY_TAC =
  let UNCURRY_LEMMA = PROVE
   ("!(R:*->**->***) x. UNCURRY R x = R (FST x) (SND x)",
    GEN_TAC THEN GEN_PAIR_TAC THEN
    PURE_REWRITE_TAC[UNCURRY_DEF] THEN
    REWRITE_TAC[])
  and imp = "$==>" in
  \(asl,w).
    let ant,con = dest_imp w in
    let vars = snd(strip_comb con) in
    let rnum = length vars - 2 in
    let hd1,args1 = strip_ncomb rnum ant
    and hd2,args2 = strip_ncomb rnum con in
    let x0 = rand hd1 in
    let x0n,x0ty = dest_var x0 in
    let [xty;yty] = snd(dest_type x0ty) in
    let avs = itlist union (map variables asl) (variables w) in
    let x = variant avs (mk_var(x0n,xty))
    and y = variant avs (mk_var(x0n,yty)) in
    let core1 = hd(snd(strip_comb ant))
    and core2 = hd vars in
    let ant' = list_mk_comb(core1,x.y.args1)
    and con' = list_mk_comb(core2,x.y.args1) in
    let asm = list_mk_forall([x;y],mk_imp(ant',con')) in
    let fst_tm = mk_const(`FST`,mk_type(`fun`,[x0ty; xty]))
    and snd_tm = mk_const(`SND`,mk_type(`fun`,[x0ty; yty])) in
    let th1 = SPECL [mk_comb(fst_tm,x0);mk_comb(snd_tm,x0)] (ASSUME asm) in
    let th2a = rev_itlist (C AP_THM) args1 (ISPECL[core1; x0] UNCURRY_LEMMA)
    and th2b = rev_itlist (C AP_THM) args1 (ISPECL[core2; x0] UNCURRY_LEMMA) in
    let th2 = SYM(MK_COMB(AP_TERM imp th2a,th2b)) in
    let th3 = DISCH_ALL (EQ_MP th2 th1) in
    BACKCHAIN_TAC th3 (asl,w);;

%----------------------------------------------------------------------------%
% Collection, so users can add their own rules.                              %
%                                                                            %
% As a simple speedup, the tactics are indexed by head operator in the       %
% relevant expression. If there isn't a head constant, use the empty string. %
%----------------------------------------------------------------------------%

letref mono_tactics =
 [`/\\`,BACKCHAIN_TAC MONO_AND THEN CONJ_TAC;
  `\\/`,BACKCHAIN_TAC MONO_OR THEN CONJ_TAC;
  `!`,BACKCHAIN_TAC MONO_ALL;
  `?`,BACKCHAIN_TAC MONO_EXISTS;
  `ALL_EL`,BACKCHAIN_TAC MONO_ALL_EL;
  `UNCURRY`,MONO_UNCURRY_TAC;
  ``,MONO_ABS_TAC;
  `==>`,BACKCHAIN_TAC MONO_IMP THEN CONJ_TAC;
  `~`,BACKCHAIN_TAC MONO_NOT];;

let APPLY_MONOTAC =
  let IMP_REFL = TAUT "!p. p ==> p" in
  \(asl,w).
    let a,c = dest_imp w in
    if aconv a c then ACCEPT_TAC (SPEC a IMP_REFL) (asl,w) else
    let cn = fst(dest_const(repeat rator c)) ? `` in
    tryfind (\(k,t). if k = cn then t (asl,w) else fail) mono_tactics;;

%----------------------------------------------------------------------------%
% Tactics to prove monotonicity automatically.                               %
%----------------------------------------------------------------------------%

let MONO_STEP_TAC =
  REPEAT GEN_TAC THEN APPLY_MONOTAC;;

let MONO_TAC =
  REPEAT MONO_STEP_TAC THEN ASM_REWRITE_TAC[];;

%============================================================================%
% Part 4: Final user wrapper round the package, including schemas.           %
%============================================================================%

%----------------------------------------------------------------------------%
% Case of nonschematic relations.                                            %
%----------------------------------------------------------------------------%

let prove_nonschematic_inductive_relations_exist clauses =
  let th0 = manual_prove_nonschematic_inductive_relations_exist clauses
            ? failwith `Badly formed or trivial induction rule(s)` in
  let th1 = prove(hd(hyp th0),
                  REPEAT GEN_TAC THEN
                  DISCH_THEN(REPEAT_TCL CONJUNCTS_THEN ASSUME_TAC) THEN
                  REPEAT CONJ_TAC THEN
                  MONO_TAC)
            ? failwith `Can't prove monotonicity property` in
  PROVE_HYP th1 th0;;

%----------------------------------------------------------------------------%
% Schematic relations.                                                       %
%                                                                            %
% All relations in a given call must have the same schematic args (if they   %
% aren't mutually inductive, use separate definitions), which must occur as  %
% the first arguments to each relation, in the same order(!)                 %
%----------------------------------------------------------------------------%

let prove_inductive_relations_exist =
  letrec pare_comb qvs tm =
    if intersect (frees tm) qvs = [] & forall is_var (snd(strip_comb tm))
    then tm
    else pare_comb qvs (rator tm) in
  \clauses.
    let schem = map (\cls. let avs,bod = strip_forall cls in
                           pare_comb avs (snd(dest_imp bod) ? bod)) clauses in
    let schems = setify schem in
    if is_var(hd schem) then
      prove_nonschematic_inductive_relations_exist clauses else
    if not length(setify (map (snd o strip_comb) schems)) = 1
    then failwith `Schematic variables not used consistently` else
    let avoids = variables (list_mk_conj clauses) in
    let hack_fn tm = mk_var(fst(dest_var(repeat rator tm)),type_of tm) in
    let grels = variants avoids (map hack_fn schems) in
    let crels = com(grels,schems) in
    let clauses' = map (subst crels) clauses in
    let th0 = prove_nonschematic_inductive_relations_exist clauses' in
    let th1 = GENL (snd(strip_comb(hd schems))) th0 in
    let evs = fst(strip_exists(concl th0)) in
    let nms = map (repeat rator o snd o C assoc crels) evs in
    rev_itlist (CONV_RULE o ONCE_DEPTH_CONV o X_SKOLEM_CONV) nms th1;;

%----------------------------------------------------------------------------%
% Deals with stupid theory/parsing status implementation in HOL88.           %
%----------------------------------------------------------------------------%

let make_guided_specification name flags th =
  let evs = fst(strip_exists(concl th)) in
  let sfs = map (\v. rev_assoc (fst(dest_var v)) flags) evs in
  new_specification name sfs th;;

%----------------------------------------------------------------------------%
% Definition functions, nonschematic and schematic.                          %
%----------------------------------------------------------------------------%

let new_nonschematic_inductive_definition name flags clauses =
  let th0 = prove_nonschematic_inductive_relations_exist clauses in
  let th1 = make_guided_specification name flags th0 in
  (I # CONJ_PAIR) (CONJ_PAIR th1);;

let new_inductive_definition name flags clauses =
  let th0 = prove_inductive_relations_exist clauses in
  let th1 = make_guided_specification name flags th0 in
  let avs = fst(strip_forall(concl th1)) in
  (I # CONJ_PAIR) (CONJ_PAIR (SPEC_ALL th1));;
