(* ========================================================================= *)
(* Abstract type of HOL types and functions for manipulating them.           *)
(* ========================================================================= *)

type hol_type = Tyvar of string
              | Tyapp of string *  hol_type list;;

(* ------------------------------------------------------------------------- *)
(* List of current type constants with their arities.                        *)
(*                                                                           *)
(* Initially we just have the boolean type and the function space            *)
(* constructor. Later on we add as primitive the type of individuals.        *)
(* All other new types result from definitional extension.                   *)
(* ------------------------------------------------------------------------- *)

let the_type_constants = ref ["bool",0; "fun",2];;

(* ------------------------------------------------------------------------- *)
(* Lookup function for type constants. Returns arity if it succeeds.         *)
(* ------------------------------------------------------------------------- *)

let get_type_arity s = assoc s (!the_type_constants);;

(* ------------------------------------------------------------------------- *)
(* Declare a new type.                                                       *)
(* ------------------------------------------------------------------------- *)

let new_type(name,arity) =
  if can get_type_arity name then
    failwith ("new_type: type "^name^" has already been declared")
  else the_type_constants := (name,arity)::(!the_type_constants);;

(* ------------------------------------------------------------------------- *)
(* Basic type constructors.                                                  *)
(* ------------------------------------------------------------------------- *)

let mk_type(tyop,args) =
  let arity = try get_type_arity tyop with Failure _ ->
    failwith ("mk_type: type "^tyop^" has not been defined") in
  if arity = length args then
    Tyapp(tyop,args)
  else failwith ("mk_type: incorrect number of arguments to "^tyop);;

let mk_vartype v = Tyvar(v);;

(* ------------------------------------------------------------------------- *)
(* The following are common enough to deserve their own bindings.            *)
(* ------------------------------------------------------------------------- *)

let bool_ty = mk_type("bool",[]);;

let mk_fun_ty ty1 ty2 = mk_type("fun",[ty1; ty2]);;

let aty = mk_vartype "A";;

let bty = mk_vartype "B";;

(* ------------------------------------------------------------------------- *)
(* Basic type destructors.                                                   *)
(* ------------------------------------------------------------------------- *)

let dest_type =
  fun (Tyapp p) -> p
    | (Tyvar _) -> failwith "dest_type: type variable not constructor";;

let dest_vartype =
  fun (Tyapp _) -> failwith "dest_vartype: type constructor not variable"
    | (Tyvar s) -> s;;

(* ------------------------------------------------------------------------- *)
(* Basic type discriminators.                                                *)
(* ------------------------------------------------------------------------- *)

let is_type = can dest_type;;

let is_vartype = can dest_vartype;;

(* ------------------------------------------------------------------------- *)
(* Return the type variables in a type and in a list of types.               *)
(* ------------------------------------------------------------------------- *)

let rec tyvars =
    fun (Tyapp(_,args)) -> tyvarsl args
      | (Tyvar v as tv) -> [tv]

and tyvarsl l = itlist (union o tyvars) l [];;

(* ------------------------------------------------------------------------- *)
(* Substitute types for type variables.                                      *)
(*                                                                           *)
(* NB: (1) Non-variables in subst list are just ignored (a check would be    *)
(*         repeated many times), as are repetitions.                         *)
(*     (2) It raises "Unchanged" where applicable.                           *)
(* ------------------------------------------------------------------------- *)

let rec type_subst i =
  fun (Tyapp(tycon,args)) -> Tyapp(tycon,qmap (type_subst i) args)
    | (Tyvar v as tv) -> try rev_assoc tv i with Failure _ -> raise Unchanged;;
