(* ========================================================================= *)
(* Abstract type of theorems and primitive inference rules.                  *)
(* ========================================================================= *)

type thm = Sequent of (term list * term);;

(* ------------------------------------------------------------------------- *)
(* Useful to have term union modulo alpha-conversion for assumption lists.   *)
(* ------------------------------------------------------------------------- *)

let rec term_union l1 l2 =
  match l1 with
    [] -> l2
  | (h::t) -> let subun = term_union t l2 in
              if exists (aconv h) subun then subun else h::subun;;

(* ------------------------------------------------------------------------- *)
(* Basic theorem destructors.                                                *)
(* ------------------------------------------------------------------------- *)

let dest_thm (Sequent s) = s;;

let hyp (Sequent(asl,c)) = asl;;

let concl (Sequent(asl,c)) = c;;

(* ------------------------------------------------------------------------- *)
(* Equivalence properties of equality.                                       *)
(* ------------------------------------------------------------------------- *)

let REFL tm =
  Sequent([],mk_eq(tm,tm));;

let SYM (Sequent(asl,c)) =
  try let l,r = dest_eq c in
      Sequent(asl,mk_eq(r,l))
  with Failure _ -> failwith "SYM: Not an equation";;

let TRANS (Sequent(asl1,c1)) (Sequent(asl2,c2)) =
  try let l,m1 = dest_eq c1 in
      let m2,r = dest_eq c2 in
      if aconv m1 m2 then Sequent(term_union asl1 asl2,mk_eq(l,r)) else fail()
  with Failure _ -> failwith "TRANS";;

(* ------------------------------------------------------------------------- *)
(* Congruence properties of equality.                                        *)
(* ------------------------------------------------------------------------- *)

let MK_COMB(Sequent(asl1,c1),Sequent(asl2,c2)) =
  try let l1,r1 = dest_eq c1
      and l2,r2 = dest_eq c2 in
      Sequent(term_union asl1 asl2,mk_eq(mk_comb(l1,l2),mk_comb(r1,r2)))
  with Failure _ -> failwith "MK_COMB";;

let ABS v (Sequent(asl,c)) =
  let l,r = try dest_eq c with Failure _ -> failwith "ABS: not an equation" in
  if mem v (freesl asl) then failwith "ABS: variable is free in assumptions"
  else Sequent(asl,mk_eq(mk_abs(v,l),mk_abs(v,r)));;

(* ------------------------------------------------------------------------- *)
(* Lambda-calculus (alpha-conversion is implicit and derivable).             *)
(* ------------------------------------------------------------------------- *)

let BETA_CONV tm =
  try let f,arg = dest_comb tm in
      let v,bod = dest_abs f in
      let tm' = if arg = v then bod else vsubst[arg,v] bod in
      Sequent([],mk_eq(tm,tm'))
  with Failure _ -> failwith "BETA_CONV: Not a beta-redex";;

(* ------------------------------------------------------------------------- *)
(* Primitive rules of inference: assumptions and implication.                *)
(* ------------------------------------------------------------------------- *)

let ASSUME tm =
  if fst(dest_type(type_of tm)) = "bool" then Sequent([tm],tm)
  else failwith "ASSUME: not a proposition";;

let DISCH a (Sequent(asl,c)) =
  Sequent(filter (prefix not o aconv a) asl,mk_imp(a,c));;

let MP (Sequent(asl1,c1)) (Sequent(asl2,c2)) =
  let ant,cons = try dest_imp c1
                 with Failure _ -> failwith "MP: not an implication" in
  if aconv ant c2 then Sequent(term_union asl1 asl2,cons)
  else failwith "MP: antecedent of first thm not aconv to concl of second";;

(* ------------------------------------------------------------------------- *)
(* Connections between boolean equality and implication.                     *)
(* ------------------------------------------------------------------------- *)

let EQ_MP (Sequent(asl1,eq)) (Sequent(asl2,c)) =
  try let l,r = dest_eq eq in
      if aconv l c then Sequent(term_union asl1 asl2,r) else fail()
  with Failure _ -> failwith "EQ_MP";;

let IMP_ANTISYM_RULE (Sequent(asl1,c1)) (Sequent(asl2,c2)) =
  let ant1,cons1 = dest_imp c1
  and ant2,cons2 = dest_imp c2 in
  if aconv ant1 cons2 & aconv ant2 cons1 then
    Sequent(term_union asl1 asl2,mk_eq(ant1,cons1))
  else failwith "IMP_ANTISYM_RULE: Implications don't correspond";;

(* ------------------------------------------------------------------------- *)
(* Type and term instantiation.                                              *)
(* ------------------------------------------------------------------------- *)

let INST_TYPE theta (Sequent(asl,c)) =
  if intersect (itlist (union o type_vars_in_term) asl [])
               (map snd theta) = []
  then Sequent(asl,inst theta c)
  else failwith "INST_TYPE: type variable(s) free in assumptions";;

let INST theta (Sequent(asl,c)) =
  if intersect (freesl asl) (map snd theta) = []
  then Sequent(asl,vsubst theta c)
  else failwith "INST: variable(s) free in assumptions";;

(* ------------------------------------------------------------------------- *)
(* Handling of axioms.                                                       *)
(* ------------------------------------------------------------------------- *)

let the_axioms = ref ([]:thm list);;

let axioms() = !the_axioms;;

let new_axiom tm =
  if fst(dest_type(type_of tm)) = "bool" then
    let th = Sequent([],tm) in
     (the_axioms := th::(!the_axioms); th)
  else failwith "new_axiom: Not a proposition";;

(* ------------------------------------------------------------------------- *)
(* Handling of (term) definitions.                                           *)
(*                                                                           *)
(* The function "new_basic_definition" implements the following rule:        *)
(*                                                                           *)
(*         |- P t                                                            *)
(*       ---------- [t is a closed term; c is a new constant]                *)
(*         |- P c                                                            *)
(*                                                                           *)
(* This is now the primitive form of (term) constant definition.             *)
(*                                                                           *)
(* It avoids the hacks and duplications caused by the non-primitive nature   *)
(* of the existential quantifier.                                            *)
(* ------------------------------------------------------------------------- *)

let the_definitions = ref ([]:thm list);;

let definitions() = !the_definitions;;

let new_basic_definition cname (Sequent(asl,c)) =
  if not asl = [] then
    failwith "new_definition: Assumptions not allowed in theorem" else
  let pred,arg = dest_comb c in
  if not frees arg = [] then
    failwith "new_definition: Argument to predicate must be a closed term" else
  if not subset (type_vars_in_term pred) (tyvars(type_of arg)) then
    failwith "new_definition: Type variables not reflected in constant" else
  try let th = find (fun th -> fst(dest_const(rand(concl th))) = cname)
                    (!the_definitions) in
      let oldpred = rator(concl th) in
      if aconv pred oldpred then (warn true "benign redefinition"; th)
      else failwith "new_definition: Constant already defined"
  with Failure "find" ->
       let ty = type_of arg in
       new_constant(cname,ty);
       let cn = mk_const(cname,[]) in
       let th = Sequent([],mk_comb(pred,cn)) in
       the_definitions := th::(!the_definitions); th;;

(* ------------------------------------------------------------------------- *)
(* Handling of type definitions.                                             *)
(*                                                                           *)
(* This function now involves no logical constants beyond equality.          *)
(*                                                                           *)
(*             |- P t                                                        *)
(*    ---------------------------                                            *)
(*        |- abs(rep a) = a                                                  *)
(*     |- P r = (rep(abs r) = r)                                             *)
(*                                                                           *)
(* Where "abs" and "rep" are new constants with the nominated names.         *)
(* ------------------------------------------------------------------------- *)

let new_basic_type_definition =
  let tydef_cache = ref [] in
  fun tyname ((absname,repname) as binames) (Sequent(asl,c) as th) ->
    try let retval = assoc (tyname,binames,th) (!tydef_cache) in
        warn true "benign type redefinition; returning old theorems"; retval
    with Failure _ ->
    if exists (can get_const_type) [absname; repname] then
      failwith "new_basic_type_definition: Constant(s) already in use" else
    if not asl = [] then
      failwith "new_basic_type_definition: Assumptions in theorem" else
    let P,x = try dest_comb c
              with Failure _ ->
                failwith "new_basic_type_definition: Not a combination" in
    if not frees P = [] then
      failwith "new_basic_type_definition: Predicate is not closed" else
    let tyvars = type_vars_in_term P in
    let _ = try new_type(tyname,length tyvars)
            with Failure _ ->
                failwith "new_basic_type_definition: Type already defined" in
    let aty = mk_type(tyname,tyvars)
    and rty = type_of x in
    let abs = new_constant(absname,mk_fun_ty rty aty); mk_const(absname,[])
    and rep = new_constant(repname,mk_fun_ty aty rty); mk_const(repname,[]) in
    let a = mk_var("a",aty) and r = mk_var("r",rty) in
    let retval =
      Sequent([],mk_eq(mk_comb(abs,mk_comb(rep,a)),a)),
      Sequent([],mk_eq(mk_comb(P,r),mk_eq(mk_comb(rep,mk_comb(abs,r)),r))) in
    tydef_cache := ((tyname,binames,th),retval)::(!tydef_cache);
    retval;;
