(* ========================================================================= *)
(* Abstract type of HOL name-carrying terms and manipulation functions.      *)
(* ========================================================================= *)

type term = Var of string * hol_type
          | Const of string * hol_type
          | Comb of term * term
          | Abs of term * term;;

(* ------------------------------------------------------------------------- *)
(* List of term constants and their types.                                   *)
(*                                                                           *)
(* We begin with the basic logical connectives of equality and implication.  *)
(* Later, the Hilbert choice operator is added. All other new constants      *)
(* arise by definitional extension.                                          *)
(* ------------------------------------------------------------------------- *)

let the_term_constants =
   ref ["=",  mk_fun_ty aty (mk_fun_ty aty bool_ty);
        "==>",mk_fun_ty bool_ty (mk_fun_ty bool_ty bool_ty)];;

(* ------------------------------------------------------------------------- *)
(* Gets type of constant if it succeeds.                                     *)
(* ------------------------------------------------------------------------- *)

let get_const_type s = assoc s (!the_term_constants);;

(* ------------------------------------------------------------------------- *)
(* Declare a new constant.                                                   *)
(* ------------------------------------------------------------------------- *)

let new_constant(name,ty) =
  if can get_const_type name then
    failwith ("new_constant: constant "^name^" has already been declared")
  else the_term_constants := (name,ty)::(!the_term_constants);;

(* ------------------------------------------------------------------------- *)
(* Finds the type of a term (assumes it is well-typed).                      *)
(* ------------------------------------------------------------------------- *)

let type_of =
  let chase n ty = funpow n (hd o tl o snd o dest_type) ty in
  let rec tyof n tm =
    match tm with
      Var(_,ty) -> chase n ty
    | Const(_,ty) -> chase n ty
    | Comb(s,_) -> tyof (n+1) s
    | Abs(x,t) -> if n = 0 then chase n (mk_type("fun",[tyof 0 x; tyof 0 t]))
                           else tyof (n-1) t in
  tyof 0;;

(* ------------------------------------------------------------------------- *)
(* Tests for alpha-convertibility (equality ignoring names in abstractions). *)
(* ------------------------------------------------------------------------- *)

let rec paconv env tm1 tm2 =
  match tm1 with
    Var(_) -> (try tm2 = assoc tm1 env
               with Failure _ -> tm1 = tm2 & not can (rev_assoc tm1) env)
  | Const(_) -> tm1 = tm2
  | Comb(s1,t1) ->
       (match tm2 with Comb(s2,t2) -> paconv env s1 s2 & paconv env t1 t2
                      | _ -> false)
  | Abs(x1,t1) ->
       (match tm2 with Abs(x2,t2) -> type_of x1 = type_of x2 &
                                     paconv ((x1,x2)::env) t1 t2
                     | _ -> false);;

let aconv tm1 tm2 =
  tm1 = tm2 or paconv [] tm1 tm2;;

(* ------------------------------------------------------------------------- *)
(* Primitive discriminators.                                                 *)
(* ------------------------------------------------------------------------- *)

let is_var = fun (Var _) -> true | _ -> false;;

let is_const = fun (Const _) -> true | _ -> false;;

let is_abs = fun (Abs _) -> true | _ -> false;;

let is_comb = fun (Comb _) -> true | _ -> false;;

(* ------------------------------------------------------------------------- *)
(* Primitive constructors.                                                   *)
(* ------------------------------------------------------------------------- *)

let mk_var(v,ty) = Var(v,ty);;

let mk_const(name,theta) =
  let uty = try get_const_type name with Failure _ ->
    failwith "mk_const: not a constant name" in
  Const(name,qtry (type_subst theta) uty);;

let mk_abs((bvar,_) as p) =
  match bvar with
    Var(_) -> Abs(p)
  | _ -> failwith "mk_abs: not a variable";;

let mk_comb((f,a) as p) =
  let fty = type_of f and aty = type_of a in
  match (dest_type fty) with
    "fun",[ty1;_] -> if ty1 = aty then Comb(p)
                     else failwith "mk_comb: types do not agree"
  | _ -> failwith "mk_comb: rator is not a function";;

(* ------------------------------------------------------------------------- *)
(* Primitive destructors.                                                    *)
(* ------------------------------------------------------------------------- *)

let dest_var =
  fun (Var p) -> p | _ -> failwith "dest_var: not a variable";;

let dest_const =
  fun (Const p) -> p | _ -> failwith "dest_const: not a constant";;

let dest_comb =
  fun (Comb p) -> p | _ -> failwith "dest_comb: not a combination";;

let dest_abs =
  fun (Abs p) -> p | _ -> failwith "dest_abs: not an abstraction";;

(* ------------------------------------------------------------------------- *)
(* Some useful derived syntax constructs for combinations and abstractions.  *)
(* ------------------------------------------------------------------------- *)

let rator tm =
  try fst(dest_comb tm)
  with Failure _ -> failwith "rator: Not a combination";;

let rand tm =
  try snd(dest_comb tm)
  with Failure _ -> failwith "rand: Not a combination";;

let bndvar tm =
  try fst(dest_abs tm)
  with Failure _ -> failwith "bndvar: Not an abstraction";;

let body tm =
  try snd(dest_abs tm)
  with Failure _ -> failwith "body: Not an abstraction";;

let list_mk_comb(h,t) = rev_itlist (C (curry mk_comb)) t h;;

let list_mk_abs(vs,bod) = itlist (curry mk_abs) vs bod;;

let strip_comb = rev_splitlist dest_comb;;

let strip_abs = splitlist dest_abs;;

let lhand = rand o rator;;

(* ------------------------------------------------------------------------- *)
(* Generic syntax to deal with some binary operators.                        *)
(*                                                                           *)
(* Note that "mk_binary" only works for monomorphic functions.               *)
(* ------------------------------------------------------------------------- *)

let is_binary s tm =
  try fst(dest_const(rator(rator tm))) = s
  with Failure _ -> false;;

let dest_binary s tm =
  try let il,r = dest_comb tm in
      let i,l = dest_comb il in
      if fst(dest_const i) = s then l,r else fail()
  with Failure _ -> failwith "dest_binary";;

let mk_binary s =
  let c = mk_const(s,[]) in
  fun (l,r) -> try mk_comb(mk_comb(c,l),r)
               with Failure _ -> failwith "mk_binary";;

(* ------------------------------------------------------------------------- *)
(* Derived syntax for implication and equality.                              *)
(* ------------------------------------------------------------------------- *)

let is_imp = is_binary "==>";;

let dest_imp tm =
  try dest_binary "==>" tm with Failure _ -> failwith "dest_imp";;

let mk_imp tm =
  try mk_binary "==>" tm with Failure _ -> failwith "mk_imp";;

let is_eq = is_binary "=";;

let dest_eq tm =
  try dest_binary "=" tm with Failure _ -> failwith "dest_eq";;

let mk_eq (l,r) =
  try let ty = type_of l in
      let eq_tm = mk_const("=",[ty,aty]) in
      mk_comb(mk_comb(eq_tm,l),r)
  with Failure _ -> failwith "mk_eq";;

let lhs = fst o dest_eq;;

let rhs = snd o dest_eq;;

(* ------------------------------------------------------------------------- *)
(* Finds the variables free in a term (list of terms).                       *)
(* ------------------------------------------------------------------------- *)

let rec frees tm =
  match tm with
    Var p -> [tm]
  | Const _ -> []
  | Abs(bv,bod) -> subtract (frees bod) [bv]
  | Comb(s,t) -> union (frees s) (frees t);;

let freesl tml = itlist (union o frees) tml [];;

(* ------------------------------------------------------------------------- *)
(* Finds the type variables (free) in a term.                                *)
(* ------------------------------------------------------------------------- *)

let rec type_vars_in_term tm =
  match tm with
    Var(_,ty)        -> tyvars ty
  | Const(_,ty)      -> tyvars ty
  | Comb(s,t)        -> union (type_vars_in_term s) (type_vars_in_term t)
  | Abs(Var(_,ty),t) -> union (tyvars ty) (type_vars_in_term t);;

(* ------------------------------------------------------------------------- *)
(* For name-carrying syntax, we need this early.                             *)
(* ------------------------------------------------------------------------- *)

let rec variant avoid v =
  let s,ty = dest_var v in
  if mem v avoid then variant avoid (mk_var(s^"'",ty))
  else v;;

(* ------------------------------------------------------------------------- *)
(* Substitution primitive (substitution for variables only!)                 *)
(* ------------------------------------------------------------------------- *)

let vsubst =
  let mk_qcomb = qcomb(fun (x,y) -> Comb(x,y)) in
  let rec vsubst theta tm =
    match tm with
      Var(v)  -> (try snd(rev_assoc tm theta) with Failure _ -> raise Unchanged)
    | Const(_) -> raise Unchanged
    | Comb(p) -> mk_qcomb (vsubst theta) (p)
    | Abs(_) -> fst(vasubst theta tm)
  and vasubst theta tm =
    match tm with
      Var(v)  -> (try snd(rev_assoc tm theta),[tm]
                  with Failure _ -> raise Unchanged)
    | Const(_) -> raise Unchanged
    | Comb(l,r) -> (try let l',vs = vasubst theta l in
                        try let r',vt = vasubst theta r in
                            Comb(l',r'),union vs vt
                        with Unchanged -> Comb(l',r),vs
                    with Unchanged ->
                        let r',vt = vasubst theta r in Comb(l,r'),vt)
    | Abs(v,bod) -> let theta' = filter (prefix not o prefix=v o snd) theta in
                    if theta' = [] then raise Unchanged else
                    let bod',vs = vasubst theta' bod in
                    let tms = map
                      (eval o fst o C rev_assoc theta') vs in
                    if exists (mem v) tms then
                      let fvs = itlist union tms (subtract (frees bod) vs) in
                      let v' = variant fvs v in
                      let bod',vars' = vasubst
                        (((eager [v'],v'),v)::theta') bod in
                      Abs(v',bod'),subtract vars' [v]
                    else
                      Abs(v,bod'),vs in
  fun theta ->
    if theta = [] then (fun tm -> tm) else
    let atheta = map
      (fun (t,x) -> if type_of t = snd(dest_var x)
                    then (lazy frees t,t),x
                    else failwith "vsubst: Bad substitution list") theta in
    qtry(vsubst atheta);;

(* ------------------------------------------------------------------------- *)
(* Type instantiation primitive.                                             *)
(* ------------------------------------------------------------------------- *)

exception Clash of term;;

let inst =
  let mk_qcomb = qcomb (fun (x,y) -> Comb(x,y)) in
  let rec inst env tyin tm =
    match tm with
      Var(n,ty)   -> let tm' = Var(n,type_subst tyin ty) in
                     if try rev_assoc tm' env = tm with Failure _ -> true
                     then tm' else raise (Clash tm')
    | Const(c,ty) -> Const(c,type_subst tyin ty)
    | Comb(p)     -> mk_qcomb (inst env tyin) p
    | Abs(y,t)    -> try let y' = inst [] tyin y in
                         let env' = (y,y')::env in
                         try Abs(y',qtry (inst env' tyin) t)
                         with (Clash(w') as ex) ->
                         if not w' = y' then raise ex else
                         let ifrees = map (inst [] tyin) (frees t) in
                         let y'' = variant ifrees y' in
                         let z = Var(fst(dest_var y''),snd(dest_var y)) in
                         inst env tyin (Abs(z,vsubst[z,y] t))
                     with Unchanged ->
                         Abs(y,inst ((y,y)::env) tyin t) in
  fun tyin -> if tyin = [] then fun tm -> tm
              else qtry (inst [] tyin);;
