(* ========================================================================= *)
(* Simplistic prettyprinter. Uses the CAML Light boxes.                      *)
(* ========================================================================= *)

#open "format";;

set_max_boxes 100;;

let backquote_char = "`";;

let IS_BINDER s = parses_as_binder s;;

let IS_INFIX s = can get_infix_status s;;

let FIXITY s = fst(get_infix_status s);;

let ARIGHT s =
  match snd(get_infix_status s) with
  "right" -> true | _ -> false;;

let NAME_OF tm =
  try fst(dest_const tm) with Failure _ ->
  try fst(dest_var tm) with Failure _ -> "";;

(* ------------------------------------------------------------------------- *)
(* This stuff is temporary (it's defined later in the build sequence, and    *)
(* duplicated here). Eventually the printer will print preterms              *)
(* ------------------------------------------------------------------------- *)

let dest_binder s tm =
  try if fst(dest_const(rator tm)) = s
      then dest_abs(rand tm) else fail()
  with Failure _ -> failwith "dest_binder";;

let dest_exists = dest_binder "?";;

let strip_exists = splitlist dest_exists;;

let dest_conj = dest_binary "/\\";;

let dest_binop op tm =
  try let f,r = dest_comb tm in
      let xop,l = dest_comb f in
      if xop = op then l,r else fail()
  with Failure _ -> failwith "dest_binop";;

let dest_cons tm = dest_binary "CONS" tm;;

let dest_list tm =
  try let tms,nil = splitlist dest_cons tm in
      if fst(dest_const nil) = "NIL" then tms else fail()
  with Failure _ -> failwith "dest_list";;

let dest_numeral =
  let rec dest_num tm =
    if try fst(dest_const tm) = "_0" with Failure _ -> false then Int 0 else
    let l,r = dest_comb tm in
    let n = Int 2 */ dest_num r in
    let cn = fst(dest_const l) in
    if cn = "BIT0" then n
    else if cn = "BIT1" then n +/ Int 1
    else fail() in
  fun tm -> try let l,r = dest_comb tm in
                if fst(dest_const l) = "NUMERAL" then dest_num r else fail()
            with Failure _ -> failwith "dest_numeral";;

(* ------------------------------------------------------------------------- *)
(* Printer for types.                                                        *)
(* ------------------------------------------------------------------------- *)

let print_type =
  let bracket flag s = if flag then "("^s^")" else s in
  let seplist sep = end_itlist (fun x y -> x^sep^y) in
  let rec sot bra ty =
    try dest_vartype ty with Failure _ ->
    match dest_type ty with
      con,[] -> con
    | "fun",args -> bracket bra (seplist "->" (map (sot true) args))
    | "sum",args -> bracket bra (seplist "+" (map (sot true) args))
    | "prod",args -> bracket bra (seplist "#" (map (sot true) args))
    | con,args -> (bracket true (seplist "," (map (sot false) args))^con) in
  fun ty -> print_string ("`:"^(sot false ty)^"`");;

(* ------------------------------------------------------------------------- *)
(* Printer for terms.                                                        *)
(* ------------------------------------------------------------------------- *)

let print_term =
  let rec print_term prec tm =
    try print_string(string_of_num(dest_numeral tm)) with Failure _ ->
    try let tms = dest_list tm in
        print_string "[";
        print_term_sequence "; " tms;
        print_string "]"
    with Failure _ ->
    let hop,args = strip_comb tm in
    if is_abs hop & args = [] then print_binder prec hop else
    let s0 = NAME_OF hop in
    let s = try assoc s0 (!the_overloaded) with Failure _ -> s0 in
    try if not (s = "ESPEC") then fail() else
        let mems = map rand (striplist (dest_binary "\/") (body(rand tm))) in
        print_string "{";
        print_term_sequence ", " mems;
        print_string "}"
    with Failure _ ->
    try if not (s = "GSPEC") then fail() else
        let bod = snd(strip_exists(body(rand tm))) in
        let babs,fabs' = dest_conj bod in
        print_string "{";
        print_term 0 (rand fabs');
        print_string " | ";
        print_term 0 babs;
        print_string "}"
    with Failure _ ->
    try if not (s = "GABS") then fail() else
        let vs,bod = dest_gabs tm in
         (if prec = 0 then open_hvbox 4
          else (open_hvbox 5; print_string "(");
          print_string "\\";
          print_term 0 vs;
          print_string ". ";
          print_term 0 bod;
          if prec = 0 then () else print_string ")";
          close_box())
    with Failure _ ->
    try let eqs,bod = dest_let tm in
        open_hvbox 0;
        print_string "let ";
        print_term 0 (mk_eq(hd eqs));
        do_list (fun (v,t) -> print_string " and ";
                              print_term 0 (mk_eq(v,t)))
                (tl eqs);
        print_string " in";
        print_break(1,0);
        print_term 0 bod;
        close_box()
    with Failure _ ->
    if s = "COND" & length args = 3 then
      (if prec = 0 then () else print_string "(";
       open_hvbox (-1);
       print_string "if ";
       print_term 0 (hd args);
       print_break(0,0);
       print_string " then ";
       print_term 0 (hd(tl args));
       print_break(0,0);
       print_string " else ";
       print_term 0 (hd(tl(tl args)));
       close_box();
       if prec = 0 then () else print_string ")")
    else if is_prefix s & length args = 1 then
      (print_string s; print_term 999 (hd args))
    else if IS_BINDER s & length args = 1 & is_abs (hd args) then
      print_binder prec tm
    else if IS_INFIX s & length args = 2 then
      let bargs =
        if ARIGHT s then
          let tms,tmt = splitlist (dest_binop hop) tm in tms@[tmt]
        else
          let tmt,tms = rev_splitlist (dest_binop hop) tm in tmt::tms in
      let newprec = FIXITY s in
      (if newprec <= prec then
        (open_hvbox 1; print_string "(")
       else open_hvbox 0;
       print_term newprec (hd bargs);
       do_list (fun x -> if s = "," then ()
                         else if s = "==>" then print_break(1,0)
                         else print_string " ";
                         let s' =
                           if s = "Seq" then ";"
                           else reverse_interface_map s in
                         print_string s';
                         if s = "," then print_break(0,0)
                         else if s = "==>" then print_string " "
                         else print_break(1,0);
                         print_term newprec x) (tl bargs);
       if newprec <= prec then print_string ")" else ();
       close_box())
    else if (is_const hop or is_var hop) & args = [] then
      let s' = if IS_BINDER s or IS_INFIX s
                or s = "COND" or s = "~" then "$"^s else s in
      print_string (reverse_interface_map s')
    else
      let l,r = dest_comb tm in
      (open_hvbox 0;
       if prec = 1000 then print_string "(" else ();
       print_term 999 l;
       (if try fst(dest_const l) = "real_of_num" with Failure _ -> false
        then () else print_space());
       print_term 1000 r;
       if prec = 1000 then print_string ")" else ();
       close_box())

  and print_term_sequence sep tms =
    if tms = [] then () else
    (print_term 14 (hd tms);
     let ttms = tl tms in
     if ttms = [] then ()
     else (print_string sep; print_term_sequence sep ttms))

  and print_binder prec tm =
    let s,(vs,bod) =
      if is_abs tm then "\\",strip_abs tm else
      let st,abs = dest_comb tm in
      let s = NAME_OF st in
      s,splitlist (dest_binder s) tm in
    (if prec = 0 then open_hvbox 4
     else (open_hvbox 5; print_string "(");
     print_string (reverse_interface_map s);
     do_list (fun x -> print_string (fst(dest_var x));
                       print_string " ") (butlast vs);
     print_string (fst(dest_var (last vs)));
     print_string ".";
     if length vs = 1 then print_string " " else print_space();
     print_term 0 bod;
     if prec = 0 then () else print_string ")";
     close_box())

  and print_program_expression prec tm =
     print_term prec (snd(dest_gabs tm))

  and print_program_assignment prec tm =
     let vt,et = dest_gabs tm in
     let vs = striplist (dest_binary ",") vt
     and es = striplist (dest_binary ",") et in
     let ps =
       itlist2 (fun v e acc -> if e = v then acc else (v,e)::acc) vs es [] in
     let ls,rs = unzip ps in
     (open_hvbox 0;
      print_term_sequence "," ls;
      print_string " := ";
      print_term_sequence "," rs;
      close_box()) in
  print_term 0;;

(* ------------------------------------------------------------------------- *)
(* Print term with quotes.                                                   *)
(* ------------------------------------------------------------------------- *)

let print_qterm tm =
  print_string "`";
  print_term tm;
  print_string "`";;

(* ------------------------------------------------------------------------- *)
(* Printer for theorems.                                                     *)
(* ------------------------------------------------------------------------- *)

let print_thm th =
  let asl,tm = dest_thm th in
  (if not asl = [] then
    (print_term(hd asl);
     do_list (fun x -> print_string ","; print_space(); print_term x) (tl asl);
     print_space())
   else ();
   open_hbox();
   print_string "|- ";
   print_term tm;
   close_box());;

(* ------------------------------------------------------------------------- *)
(* Now install all these printers in the toplevel loop.                      *)
(* ------------------------------------------------------------------------- *)

install_printer "print_type";;

install_printer "print_qterm";;

install_printer "print_thm";;
