(* ========================================================================= *)
(*                               HOL LIGHT                                   *)
(*                                                                           *)
(*              Reimplementation in CAML-light of the HOL core.              *)
(*                                                                           *)
(*                  (c) John Harrison, 1994, 1995, 1996.                     *)
(*                                                                           *)
(*                     Draws on work of, among others:                       *)
(*                                                                           *)
(*           Mike Gordon, Tom Melham, Konrad Slind and Donald Syme.          *)
(* ========================================================================= *)

(* ------------------------------------------------------------------------- *)
(* Basic environment of library functions; link with quotation filter.       *)
(* ------------------------------------------------------------------------- *)

include "lib.ml";;      (* Basic library functions a la HOL88                *)

#open "unix";;
#open "num";;

let loadt s =
  let tmpdir = try sys__getenv "TMPDIR" with Not_found -> "/tmp" in
  let uniqueid = string_of_int(unix__getpid()) in
  let suffix = filename__basename s in
  let tmpname = filename__concat tmpdir (uniqueid^"-"^suffix) in
  sys__system_command("cat "^s^" | filter.$ARCH >"^tmpname);
  (try include tmpname
   with ex -> (sys__remove tmpname; raise ex));
  sys__remove tmpname;;

(* ------------------------------------------------------------------------- *)
(* The logical core.                                                         *)
(* ------------------------------------------------------------------------- *)

include "type.ml";;     (* Abstract type of HOL types                        *)
include "term.ml";;     (* Abstract type of HOL terms                        *)
include "thm.ml";;      (* Abstract type of HOL theorems: deductive system!  *)

(* ------------------------------------------------------------------------- *)
(* Some extra support stuff needed outside the core.                         *)
(* ------------------------------------------------------------------------- *)

include "basics.ml";;   (* Basic "prelogical" stuff                          *)
include "nets.ml";;     (* Term nets                                         *)

(* ------------------------------------------------------------------------- *)
(* The interface.                                                            *)
(* ------------------------------------------------------------------------- *)

include "preterm.ml";;  (* Preterms and their interconversion with terms     *)
include "parser.ml";;   (* Lexer and parser                                  *)
include "printer.ml";;  (* Crude prettyprinter                               *)

(* ------------------------------------------------------------------------- *)
(* Higher level deductive system.                                            *)
(* ------------------------------------------------------------------------- *)

loadt "equal.ml";;      (* Basic equality reasoning and conversionals        *)
loadt "bool.ml";;       (* Boolean theory and basic derived rules            *)
loadt "drule.ml";;      (* Additional derived rules                          *)
loadt "tactics.ml";;    (* Tactics, tacticals and goal stack                 *)
loadt "itab.ml";;       (* Toy prover for intuitionistic logic               *)
loadt "simp.ml";;       (* Basic rewriting and simplification tools.         *)
loadt "theorems.ml";;   (* Additional theorem (mainly for quantifiers) etc.  *)
loadt "ind-defs.ml";;   (* Derived rules for inductive definitions           *)
loadt "class.ml";;      (* Classical reasoning: Choice and Extensionality    *)
loadt "trivia.ml";;     (* Some very basic theories, e.g. type ":one"        *)
loadt "canon.ml";;      (* Tools for putting terms in canonical forms        *)
loadt "meson.ml";;      (* First order automation: MESON (model elimination) *)
loadt "mizar.ml";;      (* Mizar-style proofs.                               *)
loadt "pair.ml";;       (* Theory of pairs                                   *)
loadt "num.ml";;        (* Axiom of Infinity, definition of natural numbers  *)
loadt "wf.ml";;         (* Theory of wellfounded relations                   *)
loadt "recursion.ml";;  (* Tools for primitive recursion on inductive types  *)
loadt "arith.ml";;      (* Natural number arithmetic                         *)
loadt "calc_num.ml";;   (* Calculation with natural numbers                  *)
loadt "ind-types.ml";;  (* Tools for defining inductive types                *)
loadt "list.ml";;       (* Theory of lists                                   *)
loadt "quot.ml";;       (* Derived rules for defining quotient types         *)
loadt "realax.ml";;     (* Definition of real numbers                        *)
loadt "real.ml";;       (* Derived properties of reals, linear decision proc *)
loadt "calc_rat.ml";;   (* Calculation with rational numbers                 *)
loadt "int.ml";;        (* Definition of integers                            *)
loadt "sets.ml";;       (* Basic set theory.                                 *)

(* ------------------------------------------------------------------------- *)
(* Real work.                                                                *)
(* ------------------------------------------------------------------------- *)

(***************
loadt "Work/analysis.ml";;  (* Real analysis                                 *)
loadt "Work/poly.ml";;      (* Theory of polynomials                         *)
loadt "Work/sturm.ml";;     (* Squarefree decomposition; Sturm's theorem     *)
loadt "Work/drang.ml";;     (* Accuracy of polynomial approximations         *)
 ***************)                                                            
                                                                             
(**********                                                                  
loadt "Work/prog.ml";;      (* Imperative programming language               *)
loadt "Work/ieee.ml";;      (* Formalization of IEEE-754                     *)
loadt "Work/tang.ml";;      (* Verification of Tang's exponential            *)
 **********)
