(* ========================================================================= *)
(* Basic equality reasoning including conversionals.                         *)
(* ========================================================================= *)

type conv == term->thm;;

(* ------------------------------------------------------------------------- *)
(* Rule allowing easy instantiation of polymorphic proformas.                *)
(* ------------------------------------------------------------------------- *)

let PINST tyin tmin =
  let iterm_fn = INST (map (I F_F inst tyin) tmin)
  and itype_fn = INST_TYPE tyin in
  fun th -> try iterm_fn (itype_fn th)
            with Failure _ -> failwith "PINST";;

(* ------------------------------------------------------------------------- *)
(* Really basic stuff for equality.                                          *)
(* ------------------------------------------------------------------------- *)

let ALPHA tm1 tm2 =
  try TRANS (REFL tm1) (REFL tm2)
  with Failure _ -> failwith "ALPHA";;

let AP_TERM tm th =
  try MK_COMB(REFL tm,th)
  with Failure _ -> failwith "AP_TERM";;

let AP_THM th tm =
  try MK_COMB(th,REFL tm)
  with Failure _ -> failwith "AP_THM";;

let ALPHA_CONV v tm =
  let res = alpha v tm in
  ALPHA tm res;;

let GEN_ALPHA_CONV v tm =
  if is_abs tm then ALPHA_CONV v tm else
  let b,abs = dest_comb tm in
  AP_TERM b (ALPHA_CONV v abs);;

let MK_BINOP op (lth,rth) =
  MK_COMB(AP_TERM op lth,rth);;

(* ------------------------------------------------------------------------- *)
(* Terminal conversion combinators.                                          *)
(* ------------------------------------------------------------------------- *)

let (NO_CONV:conv) = fun tm -> failwith "NO_CONV";;

let (ALL_CONV:conv) = REFL;;

(* ------------------------------------------------------------------------- *)
(* Combinators for sequencing, trying, repeating etc. conversions.           *)
(* ------------------------------------------------------------------------- *)

let (THENC:conv -> conv -> conv) =
  fun conv1 conv2 t ->
    let th1 = conv1 t in
    let th2 = conv2 (rand(concl th1)) in
    TRANS th1 th2;;

#infix "THENC";;

let (ORELSEC:conv -> conv -> conv) =
  fun conv1 conv2 t ->
    try conv1 t with Failure _ -> conv2 t;;

#infix "ORELSEC";;

let (FIRST_CONV:conv list -> conv) = end_itlist (prefix ORELSEC);;

let (EVERY_CONV:conv list -> conv) =
  fun l -> itlist (prefix THENC) l ALL_CONV;;

let REPEATC =
  let rec REPEATC conv t =
    ((conv THENC (REPEATC conv)) ORELSEC ALL_CONV) t in
  (REPEATC:conv->conv);;

let (CHANGED_CONV:conv->conv) =
  fun conv tm ->
    let th = conv tm in
    let l,r = dest_eq (concl th) in
    if aconv l r then failwith "CHANGED_CONV" else th;;

let TRY_CONV conv = conv ORELSEC ALL_CONV;;

(* ------------------------------------------------------------------------- *)
(* Subterm conversions.                                                      *)
(* ------------------------------------------------------------------------- *)

let (RATOR_CONV:conv->conv) =
  fun conv tm ->
    let l,r = dest_comb tm in AP_THM (conv l) r;;

let (RAND_CONV:conv->conv) =
  fun conv tm ->
    let l,r = dest_comb tm in AP_TERM l (conv r);;

let LAND_CONV = RATOR_CONV o RAND_CONV;;

let (COMB2_CONV: conv->conv->conv) =
  fun lconv rconv tm -> let l,r = dest_comb tm in MK_COMB(lconv l,rconv r);;

let COMB_CONV = W COMB2_CONV;;

let (ABS_CONV:conv->conv) =
  fun conv tm ->
    let v,bod = dest_abs tm in
    let th = conv bod in
    try ABS v th with Failure _ ->
    let gv = genvar(type_of v) in
    let gbod = vsubst[gv,v] bod in
    let gth = ABS gv (conv gbod) in
    let gtm = concl gth in
    let l,r = dest_eq gtm in
    let v' = variant (frees gtm) v in
    let l' = alpha v' l and r' = alpha v' r in
    EQ_MP (ALPHA gtm (mk_eq(l',r'))) gth;;

let BINDER_CONV conv tm =
  try ABS_CONV conv tm
  with Failure _ -> RAND_CONV(ABS_CONV conv) tm;;

let SUB_CONV =
  fun conv -> (COMB_CONV conv) ORELSEC (ABS_CONV conv) ORELSEC REFL;;

let BINOP_CONV conv tm =
  let lop,r = dest_comb tm in
  let op,l = dest_comb lop in
  MK_COMB(AP_TERM op (conv l),conv r);;

(* ------------------------------------------------------------------------- *)
(* Faster depth conversions using failure rather than returning a REFL.      *)
(* ------------------------------------------------------------------------- *)

let THENQC conv1 conv2 tm =
  try let th1 = conv1 tm in
      try let th2 = conv2(rand(concl th1)) in TRANS th1 th2
      with Failure _ -> th1
  with Failure _ -> conv2 tm;;

let THENCQC conv1 conv2 tm =
  let th1 = conv1 tm in
  try let th2 = conv2(rand(concl th1)) in TRANS th1 th2
  with Failure _ -> th1;;

let rec REPEATQC conv tm =
  THENCQC conv (REPEATQC conv) tm;;

let COMB2_QCONV conv1 conv2 tm =
  let l,r = dest_comb tm in
  try let th1 = conv1 l in
      try let th2 = conv2 r in MK_COMB(th1,th2)
      with Failure _ -> AP_THM th1 r
  with Failure _ -> AP_TERM l (conv2 r);;

let COMB_QCONV = W COMB2_QCONV;;

let SUB_QCONV conv tm =
  if is_abs tm then ABS_CONV conv tm
  else COMB_QCONV conv tm;;

let rec ONCE_DEPTH_QCONV conv tm =
  prefix ORELSEC conv (SUB_QCONV (ONCE_DEPTH_QCONV conv)) tm;;

let rec DEPTH_QCONV conv tm =
  THENQC (SUB_QCONV (DEPTH_QCONV conv))
         (REPEATQC conv) tm;;

let rec REDEPTH_QCONV conv tm =
  THENQC (SUB_QCONV (REDEPTH_QCONV conv))
         (THENCQC conv (REDEPTH_QCONV conv)) tm;;

let rec TOP_DEPTH_QCONV conv tm =
  THENQC (REPEATQC conv)
         (THENCQC (SUB_QCONV (TOP_DEPTH_QCONV conv))
                  (THENCQC conv (TOP_DEPTH_QCONV conv))) tm;;

let rec TOP_SWEEP_QCONV conv tm =
  THENQC (REPEATQC conv)
         (SUB_QCONV (TOP_SWEEP_QCONV conv)) tm;;

(* ------------------------------------------------------------------------- *)
(* Standard conversions.                                                     *)
(* ------------------------------------------------------------------------- *)

let (ONCE_DEPTH_CONV: conv->conv) = fun c -> TRY_CONV (ONCE_DEPTH_QCONV c)
and (DEPTH_CONV: conv->conv) = fun c -> TRY_CONV (DEPTH_QCONV c)
and (REDEPTH_CONV: conv->conv) = fun c -> TRY_CONV (REDEPTH_QCONV c)
and (TOP_DEPTH_CONV: conv->conv) = fun c -> TRY_CONV (TOP_DEPTH_QCONV c)
and (TOP_SWEEP_CONV: conv->conv) = fun c -> TRY_CONV (TOP_SWEEP_QCONV c);;

(* ------------------------------------------------------------------------- *)
(* Single bottom-up pass, starting at the topmost success!                   *)
(* ------------------------------------------------------------------------- *)

let (SINGLE_DEPTH_CONV:conv->conv) =
  let rec SINGLE_DEPTH_CONV conv tm =
    try conv tm
    with Failure _ -> (SUB_CONV (SINGLE_DEPTH_CONV conv) THENC (TRY_CONV conv)) tm in
  SINGLE_DEPTH_CONV;;

(* ------------------------------------------------------------------------- *)
(* Apply at leaves of op-tree; NB any failures at leaves cause failure.      *)
(* ------------------------------------------------------------------------- *)

let rec DEPTH_BINOP_CONV op conv tm =
  try let l,r = dest_binop op tm in
      let lth = DEPTH_BINOP_CONV op conv l
      and rth = DEPTH_BINOP_CONV op conv r in
      MK_COMB(AP_TERM op lth,rth)
  with Failure "dest_binop" -> conv tm;;

(* ------------------------------------------------------------------------- *)
(* Symmetry conversion.                                                      *)
(* ------------------------------------------------------------------------- *)

let SYM_CONV tm =
  try let th1 = SYM(ASSUME tm) in
      let tm' = concl th1 in
      let th2 = SYM(ASSUME tm') in
      IMP_ANTISYM_RULE (DISCH tm th1) (DISCH tm' th2)
  with Failure _ -> failwith "SYM_CONV";;

(* ------------------------------------------------------------------------- *)
(* Conversion to a rule.                                                     *)
(* ------------------------------------------------------------------------- *)

let CONV_RULE (conv:conv) th =
  EQ_MP (conv(concl th)) th;;

(* ------------------------------------------------------------------------- *)
(* Substitution conversion.                                                  *)
(* ------------------------------------------------------------------------- *)

let SUBS_CONV ths tm =
  try if ths = [] then REFL tm else
      let lefts = map (lhand o concl) ths in
      let gvs = map (genvar o type_of) lefts in
      let pat = subst (zip gvs lefts) tm in
      let abs = list_mk_abs(gvs,pat) in
      let th = rev_itlist
        (fun y x -> CONV_RULE (RAND_CONV BETA_CONV THENC LAND_CONV BETA_CONV)
                              (MK_COMB(x,y))) ths (REFL abs) in
      if rand(concl th) = tm then REFL tm else th
  with Failure _ -> failwith "SUBS_CONV";;

(* ------------------------------------------------------------------------- *)
(* Get a few rules.                                                          *)
(* ------------------------------------------------------------------------- *)

let BETA_RULE = CONV_RULE(REDEPTH_CONV BETA_CONV);;

let GSYM = CONV_RULE(ONCE_DEPTH_CONV SYM_CONV);;

let SUBS ths = CONV_RULE (SUBS_CONV ths);;
