(* ========================================================================= *)
(* Reasonably efficient conversions for various canonical forms.             *)
(* ========================================================================= *)

let PRESIMP_CONV =
  GEN_REWRITE_CONV DEPTH_CONV
   [NOT_CLAUSES; AND_CLAUSES; OR_CLAUSES; IMP_CLAUSES; EQ_CLAUSES;
    FORALL_SIMP; EXISTS_SIMP; EXISTS_OR_THM; FORALL_AND_THM;
    LEFT_EXISTS_AND_THM; RIGHT_EXISTS_AND_THM;
    LEFT_FORALL_OR_THM; RIGHT_FORALL_OR_THM];;

(* ------------------------------------------------------------------------- *)
(* Conversions for NNF, either angling for good DNF or CNF.                  *)
(* ------------------------------------------------------------------------- *)

let NNF_CONV,NNFC_CONV =
  let NOT_EXISTS_UNIQUE_THM = prove
   (`~(?!x. P x) = (!x. ~P x) \/ ?x x'. P x /\ P x' /\ ~(x = x')`,
    REWRITE_TAC[EXISTS_UNIQUE_THM; DE_MORGAN_THM; NOT_EXISTS_THM] THEN
    REWRITE_TAC[NOT_FORALL_THM; NOT_IMP; CONJ_ASSOC]) in
  let common_tauts =
    [TAUT `~(~p) = p`;
     TAUT `~(p /\ q) = ~p \/ ~q`;
     TAUT `~(p \/ q) = ~p /\ ~q`;
     TAUT `~(p ==> q) = p /\ ~q`;
     TAUT `p ==> q = ~p \/ q`;
     NOT_FORALL_THM;
     NOT_EXISTS_THM;
     EXISTS_UNIQUE_THM;
     NOT_EXISTS_UNIQUE_THM]
  and dnf_tauts =
    map TAUT [`~(p = q) = (p /\ ~q) \/ (~p /\ q)`;
              `(p = q) = (p /\ q) \/ (~p /\ ~q)`]
  and cnf_tauts =
    map TAUT [`~(p = q) = (p \/ q) /\ (~p \/ ~q)`;
              `(p = q) = (p \/ ~q) /\ (~p \/ q)`] in
  let NNF_CONV =
    GEN_REWRITE_CONV TOP_SWEEP_CONV (common_tauts @ dnf_tauts)
  and NNFC_CONV =
    GEN_REWRITE_CONV TOP_SWEEP_CONV (common_tauts @ cnf_tauts) in
  let rec SINGLE_SWEEP_CONV conv tm =
    try let th = conv tm in
        let tm' = rand(concl th) in
        let th' = if is_abs tm' then NNFC_CONV tm'
                  else SUB_CONV (SINGLE_SWEEP_CONV conv) tm' in
        TRANS th th'
    with Failure _ ->
        if is_abs tm then NNFC_CONV tm else
        SUB_CONV (SINGLE_SWEEP_CONV conv) tm in
  NNF_CONV,
  SINGLE_SWEEP_CONV (GEN_REWRITE_CONV I (common_tauts @ dnf_tauts));;

(* ------------------------------------------------------------------------- *)
(* Skolemize a term already in NNF (doesn't matter if it's not prenex).      *)
(* ------------------------------------------------------------------------- *)

let SKOLEM_CONV =
  GEN_REWRITE_CONV REDEPTH_CONV
   [RIGHT_AND_EXISTS_THM;
    LEFT_AND_EXISTS_THM;
    OR_EXISTS_THM;
    RIGHT_OR_EXISTS_THM;
    LEFT_OR_EXISTS_THM;
    SKOLEM_THM];;

(* ------------------------------------------------------------------------- *)
(* Put a term already in Skolem NF in prenex (Skolem) form.                  *)
(* ------------------------------------------------------------------------- *)

let PRENEX_CONV =
  GEN_REWRITE_CONV REDEPTH_CONV
   [AND_FORALL_THM;
    LEFT_AND_FORALL_THM;
    RIGHT_AND_FORALL_THM;
    LEFT_OR_FORALL_THM;
    RIGHT_OR_FORALL_THM];;

(* ------------------------------------------------------------------------- *)
(* Converse operation, i.e. try to miniscope a Skolem NF formula.            *)
(* ------------------------------------------------------------------------- *)

let MINISCOPE_CONV =
  let rewrites = itlist (mk_rewrites false) [FORALL_AND_THM; FORALL_SIMP] [] in
  let DISJ_ACI_CONV = AC DISJ_ACI
  and DISJ_LEFT_CONV = REWR_CONV LEFT_FORALL_OR_THM in
  let MINISCOPE_FORALL_CONV tm =
    let avs,bod = strip_forall tm in
    if not avs = [] & is_disj bod then
      let djs = disjuncts bod in
      let dtm = find (fun t -> exists (fun v -> not vfree_in v t) avs) djs in
      let yvs,nvs = partition (fun v -> vfree_in v dtm) avs in
      let avs' = yvs @ nvs in
      let tm' = list_mk_forall(avs',bod) in
      let th1 =
        if avs = avs' then REFL tm else
        let th1a = GENL avs' (SPECL avs (ASSUME tm))
        and th1b = GENL avs (SPECL avs' (ASSUME tm')) in
        IMP_ANTISYM_RULE (DISCH_ALL th1a) (DISCH_ALL th1b) in
      let ydjs,ndjs = partition (vfree_in (last avs')) djs in
      let bod' = mk_disj(list_mk_disj ydjs,list_mk_disj ndjs) in
      let th2 = DISJ_ACI_CONV (mk_eq(bod,bod')) in
      let th3 = itlist MK_FORALL avs' th2 in
      let th4 = funpow (length(tl avs)) BINDER_CONV DISJ_LEFT_CONV
          (rand(concl th3)) in
      TRANS th1 (TRANS th3 th4)
    else failwith "MINISCOPE_FORALL_CONV" in
  let net =
    net_of_conv `!x. P x` MINISCOPE_FORALL_CONV
      (itlist (net_of_thm false) rewrites empty_net) in
  GENERAL_REWRITE_CONV false REDEPTH_CONV net [];;

(* ------------------------------------------------------------------------- *)
(* Put a prenex NNF term into CNF.                                           *)
(* ------------------------------------------------------------------------- *)

let PROP_CNF_CONV =
  GEN_REWRITE_CONV REDEPTH_CONV
   [TAUT `a \/ (b /\ c) = (a \/ b) /\ (a \/ c)`;
    TAUT `(a /\ b) \/ c = (a \/ c) /\ (b \/ c)`;
    GSYM CONJ_ASSOC; GSYM DISJ_ASSOC];;

(* ------------------------------------------------------------------------- *)
(* Likewise DNF.                                                             *)
(* ------------------------------------------------------------------------- *)

let PROP_DNF_CONV =
  GEN_REWRITE_CONV REDEPTH_CONV
   [TAUT `a /\ (b \/ c) = (a /\ b) \/ (a /\ c)`;
    TAUT `(a \/ b) /\ c = (a /\ c) \/ (b /\ c)`;
    GSYM CONJ_ASSOC; GSYM DISJ_ASSOC];;

(* ------------------------------------------------------------------------- *)
(* Refuter tactic to prove goal by doing something to its assumed negation.  *)
(* ------------------------------------------------------------------------- *)

let REFUTE_THEN =
  let conv = REWR_CONV(TAUT `p = ~p ==> F`) in
  fun ttac -> CONV_TAC conv THEN DISCH_THEN ttac;;

(* ------------------------------------------------------------------------- *)
(* Splitter for a refutation procedure.                                      *)
(* ------------------------------------------------------------------------- *)

let rec SPLIT_TAC splitlevel gl =
  (REPEAT(FIRST_X_ASSUM(CONJUNCTS_THEN ASSUME_TAC)) THEN
   (fun gl -> if splitlevel < 1 then ALL_TAC gl else
              try (FIRST_X_ASSUM(DISJ_CASES_THEN ASSUME_TAC) THEN
                   SPLIT_TAC (splitlevel-1)) gl
              with Failure _ -> ALL_TAC gl)) gl;;

(* ------------------------------------------------------------------------- *)
(* Equate lambda-reduced and universally quantified applied definitions.     *)
(* ------------------------------------------------------------------------- *)

let EQ_ABS_CONV =
  let pth = prove
   (`(f:A->B = \x. t x) = (!x. f x = t x)`,
    REWRITE_TAC[FUN_EQ_THM]) in
  let cnv = REWR_CONV pth in
  let rec EQ_ABS_CONV tm =
    try (cnv THENC BINDER_CONV EQ_ABS_CONV) tm
    with Failure _ -> REFL tm in
  EQ_ABS_CONV;;

(* ------------------------------------------------------------------------- *)
(* Eliminate beta redexes and lambdas in equations.                          *)
(* ------------------------------------------------------------------------- *)

let DELAMB_CONV =
  let pth = prove
   (`(((\x. s x) = t) = (!x. s x = t x)) /\
     ((s = \x. t x) = (!x. s x = t x))`,
    REWRITE_TAC[FUN_EQ_THM]) in
  let qconv =
    THENQC (TOP_DEPTH_QCONV BETA_CONV)
           (REPEATQC (THENCQC (GEN_REWRITE_CONV ONCE_DEPTH_QCONV [pth])
                              (TRY_CONV(TOP_DEPTH_QCONV BETA_CONV)))) in
  TRY_CONV qconv;;

(* ------------------------------------------------------------------------- *)
(* Find all the head operator arities in a (NNF) formula.                    *)
(* ------------------------------------------------------------------------- *)

let rec get_heads lconsts tm (cheads,vheads as sofar) =
  try let v,bod = dest_forall tm in
      get_heads (subtract lconsts [v]) bod sofar
  with Failure _ -> try
      let l,r = try dest_conj tm with Failure _ -> dest_disj tm in
      get_heads lconsts l (get_heads lconsts r sofar)
  with Failure _ -> try
      let tm' = dest_neg tm in
      get_heads lconsts tm' sofar
  with Failure _ ->
      let hop,args = strip_comb tm in
      let len = length args in
      let newheads =
        if is_const hop or mem hop lconsts
        then (insert (hop,len) cheads,vheads)
        else if len > 0 then (cheads,insert (hop,len) vheads) else sofar in
      itlist (get_heads lconsts) args newheads;;

let get_thm_heads th sofar =
  get_heads (freesl(hyp th)) (concl th) sofar;;

(* ------------------------------------------------------------------------- *)
(* Fix up the arities of head operators to be consistent.                    *)
(* ------------------------------------------------------------------------- *)

let GEN_FOL_CONV,FOL_CONV =
  let APP_CONV =
    let th = prove
     (`!(f:A->B) x. f x = I f x`,
      REWRITE_TAC[I_THM]) in
    REWR_CONV th in
  let rec APP_N_CONV n tm =
    if n = 1 then APP_CONV tm
    else (RATOR_CONV (APP_N_CONV (n - 1)) THENC APP_CONV) tm in
  let rec FOL_CONV hddata tm =
    if is_forall tm then BINDER_CONV (FOL_CONV hddata) tm
    else if is_conj tm or is_disj tm then BINOP_CONV (FOL_CONV hddata) tm else
    let op,args = strip_comb tm in
    let th = rev_itlist (C (curry MK_COMB))
                        (map (FOL_CONV hddata) args) (REFL op) in
    let tm' = rand(concl th) in
    let n = try length args - assoc op hddata with Failure _ -> 0 in
    if n = 0 then th
    else TRANS th (APP_N_CONV n tm') in
  let GEN_FOL_CONV (cheads,vheads) =
    let hddata =
      if vheads = [] then
        let hops = setify (map fst cheads) in
        let getmin h =
          let ns = mapfilter
            (fun (k,n) -> if k = h then n else fail()) cheads in
          if length ns < 2 then fail() else h,end_itlist min ns in
        mapfilter getmin hops
      else
        map (fun t -> if is_const t & fst(dest_const t) = "="
                      then t,2 else t,0)
            (setify (map fst (vheads @ cheads))) in
    FOL_CONV hddata in
  let FOL_CONV tm =
    let headsp = get_heads [] tm ([],[]) in
    GEN_FOL_CONV headsp tm in
  GEN_FOL_CONV,FOL_CONV;;

(* ------------------------------------------------------------------------- *)
(* Wrap this up as a tactic, applied to all assumptions.                     *)
(* ------------------------------------------------------------------------- *)

let ASM_FOL_TAC (asl,w) =
  let headsp = itlist (get_thm_heads o snd) asl ([],[]) in
  RULE_ASSUM_TAC(CONV_RULE(GEN_FOL_CONV headsp)) (asl,w);;
