(* ========================================================================= *)
(* Calculation with rationals.                                               *)
(* ========================================================================= *)

let gcd_num n1 n2 =
  num_of_big_int
   (big_int__gcd_big_int (big_int_of_num n1)
                         (big_int_of_num n2));;

(* ------------------------------------------------------------------------- *)
(* Various handy lemmas.                                                     *)
(* ------------------------------------------------------------------------- *)

let RAT_LEMMA1 = prove
 (`~(y1 = &0) /\ ~(y2 = &0) ==>
      ((x1 / y1) + (x2 / y2) = (x1 * y2 + x2 * y1) * inv(y1) * inv(y2))`,
  STRIP_TAC THEN REWRITE_TAC[real_div; REAL_ADD_RDISTRIB] THEN BINOP_TAC THENL
   [REWRITE_TAC[GSYM REAL_MUL_ASSOC] THEN AP_TERM_TAC THEN ONCE_REWRITE_TAC
     [AC REAL_MUL_AC `a * b * c = (b * a) * c`];
    REWRITE_TAC[REAL_MUL_ASSOC] THEN AP_THM_TAC THEN AP_TERM_TAC] THEN
  GEN_REWRITE_TAC LAND_CONV [GSYM REAL_MUL_RID] THEN
  REWRITE_TAC[GSYM REAL_MUL_ASSOC] THEN REWRITE_TAC[REAL_EQ_MUL_LCANCEL] THEN
  DISJ2_TAC THEN CONV_TAC SYM_CONV THEN MATCH_MP_TAC REAL_MUL_RINV THEN
  ASM_REWRITE_TAC[]);;

let RAT_LEMMA2 = prove
 (`&0 < y1 /\ &0 < y2 ==>
      ((x1 / y1) + (x2 / y2) = (x1 * y2 + x2 * y1) * inv(y1) * inv(y2))`,
  DISCH_TAC THEN MATCH_MP_TAC RAT_LEMMA1 THEN POP_ASSUM MP_TAC THEN
  REAL_ARITH_TAC);;

let RAT_LEMMA3 = prove
 (`&0 < y1 /\ &0 < y2 ==>
      ((x1 / y1) - (x2 / y2) = (x1 * y2 - x2 * y1) * inv(y1) * inv(y2))`,
  DISCH_THEN(MP_TAC o GEN_ALL o MATCH_MP RAT_LEMMA2) THEN
  REWRITE_TAC[real_div] THEN DISCH_TAC THEN
  ASM_REWRITE_TAC[real_sub; GSYM REAL_MUL_LNEG]);;

let RAT_LEMMA4 = prove
 (`&0 < y1 /\ &0 < y2 ==> (x1 / y1 <= x2 / y2 = x1 * y2 <= x2 * y1)`,
  let lemma = prove
   (`&0 < y ==> (&0 <= x * y = &0 <= x)`,
    DISCH_TAC THEN EQ_TAC THEN DISCH_TAC THENL
     [SUBGOAL_THEN `&0 <= x * (y * inv y)` MP_TAC THENL
       [REWRITE_TAC[REAL_MUL_ASSOC] THEN MATCH_MP_TAC REAL_LE_MUL THEN
        ASM_REWRITE_TAC[] THEN MATCH_MP_TAC REAL_LE_INV THEN
        MATCH_MP_TAC REAL_LT_IMP_LE THEN ASM_REWRITE_TAC[];
        SUBGOAL_THEN `y * inv y = &1` (fun th ->
          REWRITE_TAC[th; REAL_MUL_RID]) THEN
        MATCH_MP_TAC REAL_MUL_RINV THEN
        UNDISCH_TAC `&0 < y` THEN REAL_ARITH_TAC];
      MATCH_MP_TAC REAL_LE_MUL THEN ASM_REWRITE_TAC[] THEN
      MATCH_MP_TAC REAL_LT_IMP_LE THEN ASM_REWRITE_TAC[]]) in
  ONCE_REWRITE_TAC[CONJ_SYM] THEN DISCH_TAC THEN
  ONCE_REWRITE_TAC[REAL_ARITH `a <= b = &0 <= b - a`] THEN
  FIRST_ASSUM(fun th -> REWRITE_TAC[MATCH_MP RAT_LEMMA3 th]) THEN
  MATCH_MP_TAC EQ_TRANS THEN
  EXISTS_TAC `&0 <= (x2 * y1 - x1 * y2) * inv y2` THEN
  REWRITE_TAC[REAL_MUL_ASSOC] THEN CONJ_TAC THEN
  MATCH_MP_TAC lemma THEN MATCH_MP_TAC REAL_LT_INV THEN
  ASM_REWRITE_TAC[]);;

let RAT_LEMMA5 = prove
 (`&0 < y1 /\ &0 < y2 ==> ((x1 / y1 = x2 / y2) = (x1 * y2 = x2 * y1))`,
  REPEAT DISCH_TAC THEN REWRITE_TAC[GSYM REAL_LE_ANTISYM] THEN
  MATCH_MP_TAC(TAUT `(a = a') /\ (b = b') ==> (a /\ b = a' /\ b')`) THEN
  CONJ_TAC THEN MATCH_MP_TAC RAT_LEMMA4 THEN ASM_REWRITE_TAC[]);;

(* ------------------------------------------------------------------------- *)
(* Conversions between rational and (raw) integer pairs.                     *)
(* ------------------------------------------------------------------------- *)

let mk_ratconst =
  let mk_divop = mk_binop `$/` in
  let mk_raw_ratconst (m,n) =
    mk_divop (mk_intconst m) (mk_intconst n) in
  fun (m,n) ->
    if n =/ Int 0 then failwith "mk_ratconst: Zero denominator"
    else if n </ Int 0 then mk_raw_ratconst(minus_num m,minus_num n)
    else mk_raw_ratconst(m,n);;

let dest_ratconst =
  let dest_divop = dest_binop `$/` in
  fun tm -> try dest_intconst tm,Int 1 with Failure _ ->
            let l,r = dest_divop tm in
            dest_intconst l,dest_intconst r;;

let is_ratconst =
  let dest_divop = dest_binop `$/` in
  fun tm -> try let l,r = dest_divop tm in
                is_intconst l & is_intconst r
            with Failure _ -> is_intconst tm;;

(* ------------------------------------------------------------------------- *)
(* Create trivial rational from integer.                                     *)
(* ------------------------------------------------------------------------- *)

let REAL_INT_RAT_CONV =
  let pth = prove
   (`(&x = &x / &1) /\
     (--(&x) = --(&x) / &1)`,
    REWRITE_TAC[REAL_DIV_1]) in
  TRY_CONV(GEN_REWRITE_CONV I [pth]);;

let REAL_INT_RAT_UNOP_CONV = RAND_CONV REAL_INT_RAT_CONV;;

let REAL_INT_RAT_BINOP_CONV = COMB2_CONV (RAND_CONV REAL_INT_RAT_CONV)
                                         REAL_INT_RAT_CONV;;

let REAL_RAT_INT_CONV =
  let pth = prove
   (`x / &1 = x`,
    REWRITE_TAC[REAL_DIV_1]) in
  TRY_CONV(GEN_REWRITE_CONV I [pth]);;

(* ------------------------------------------------------------------------- *)
(* Conversion to prove an integer constant is positive.                      *)
(* ------------------------------------------------------------------------- *)

let REAL_INT_POS_CONV = EQT_ELIM o REAL_INT_LT_CONV;;

let REAL_INT_POS_PROVE th =
  let l,r = dest_imp(concl th) in
  MP th (REAL_INT_POS_CONV l);;

(* ------------------------------------------------------------------------- *)
(* Relational operations.                                                    *)
(* ------------------------------------------------------------------------- *)

let REAL_RAT_LE_CONV =
  let pth = prove
   (`&0 < y1 ==> &0 < y2 ==> (x1 / y1 <= x2 / y2 = x1 * y2 <= x2 * y1)`,
    REWRITE_TAC[TAUT `(a ==> b ==> c) = (a /\ b ==> c)`; RAT_LEMMA4])
  and x1 = `x1:real` and x2 = `x2:real`
  and y1 = `y1:real` and y2 = `y2:real`
  and dest_le = dest_binop `$<=`
  and dest_div = dest_binop `$/` in
  let RAW_REAL_RAT_LE_CONV tm =
    let l,r = dest_le tm in
    let lx,ly = dest_div l
    and rx,ry = dest_div r in
    let th0 = INST [lx,x1; ly,y1; rx,x2; ry,y2] pth in
    let th1 = REAL_INT_POS_PROVE (REAL_INT_POS_PROVE th0) in
    let th2 = (COMB2_CONV (RAND_CONV REAL_INT_MUL_CONV) REAL_INT_MUL_CONV
               THENC REAL_INT_LE_CONV) (rand(concl th1)) in
    TRANS th1 th2 in
  REAL_INT_RAT_BINOP_CONV THENC RAW_REAL_RAT_LE_CONV;;

let REAL_RAT_LT_CONV =
  let pth = prove
   (`&0 < y1 ==> &0 < y2 ==> (x1 / y1 < x2 / y2 = x1 * y2 < x2 * y1)`,
    REWRITE_TAC[TAUT `(a ==> b ==> c) = (b /\ a ==> c)`] THEN
    GEN_REWRITE_TAC (RAND_CONV o ONCE_DEPTH_CONV) [GSYM REAL_NOT_LE] THEN
    REWRITE_TAC[TAUT `(~a = ~b) = (a = b)`; RAT_LEMMA4])
  and x1 = `x1:real` and x2 = `x2:real`
  and y1 = `y1:real` and y2 = `y2:real`
  and dest_lt = dest_binop `$<`
  and dest_div = dest_binop `$/` in
  let RAW_REAL_RAT_LT_CONV tm =
    let l,r = dest_lt tm in
    let lx,ly = dest_div l
    and rx,ry = dest_div r in
    let th0 = INST [lx,x1; ly,y1; rx,x2; ry,y2] pth in
    let th1 = REAL_INT_POS_PROVE (REAL_INT_POS_PROVE th0) in
    let th2 = (COMB2_CONV (RAND_CONV REAL_INT_MUL_CONV) REAL_INT_MUL_CONV
               THENC REAL_INT_LT_CONV) (rand(concl th1)) in
    TRANS th1 th2 in
  REAL_INT_RAT_BINOP_CONV THENC RAW_REAL_RAT_LT_CONV;;

let REAL_RAT_GE_CONV =
  GEN_REWRITE_CONV I [real_ge] THENC REAL_RAT_LE_CONV;;

let REAL_RAT_GT_CONV =
  GEN_REWRITE_CONV I [real_gt] THENC REAL_RAT_LT_CONV;;

let REAL_RAT_EQ_CONV =
  let pth = prove
   (`&0 < y1 ==> &0 < y2 ==> ((x1 / y1 = x2 / y2) = (x1 * y2 = x2 * y1))`,
    REWRITE_TAC[TAUT `(a ==> b ==> c) = (a /\ b ==> c)`; RAT_LEMMA5])
  and x1 = `x1:real` and x2 = `x2:real`
  and y1 = `y1:real` and y2 = `y2:real`
  and dest_eq = dest_binop `$= :real->real->bool`
  and dest_div = dest_binop `$/` in
  let RAW_REAL_RAT_EQ_CONV tm =
    let l,r = dest_eq tm in
    let lx,ly = dest_div l
    and rx,ry = dest_div r in
    let th0 = INST [lx,x1; ly,y1; rx,x2; ry,y2] pth in
    let th1 = REAL_INT_POS_PROVE (REAL_INT_POS_PROVE th0) in
    let th2 = (COMB2_CONV (RAND_CONV REAL_INT_MUL_CONV) REAL_INT_MUL_CONV
               THENC REAL_INT_EQ_CONV) (rand(concl th1)) in
    TRANS th1 th2 in
  REAL_INT_RAT_BINOP_CONV THENC RAW_REAL_RAT_EQ_CONV;;

(* ------------------------------------------------------------------------- *)
(* The unary operations; all easy.                                           *)
(* ------------------------------------------------------------------------- *)

let REAL_RAT_NEG_CONV =
  let pth = prove
   (`(--(&0) = &0) /\
     (--(--(&n)) = &n) /\
     (--(&m / &n) = --(&m) / &n) /\
     (--(--(&m) / &n) = &m / &n)`,
    REWRITE_TAC[real_div; REAL_INV_NEG; REAL_MUL_LNEG; REAL_NEG_NEG;
     REAL_NEG_0])
  and ptm = `--` in
  let conv1 = GEN_REWRITE_CONV I [pth] in
  fun tm -> try conv1 tm
            with Failure _ -> try
                let l,r = dest_comb tm in
                if l = ptm & is_numconst r then REFL tm
                else fail()
            with Failure _ -> failwith "REAL_RAT_NEG_CONV";;

let REAL_RAT_ABS_CONV =
  let pth = prove
   (`(abs(&n) = &n) /\
     (abs(--(&n)) = &n) /\
     (abs(&m / &n) = &m / &n) /\
     (abs(--(&m) / &n) = &m / &n)`,
    REWRITE_TAC[REAL_ABS_DIV; REAL_ABS_NEG; REAL_ABS_NUM]) in
  GEN_REWRITE_CONV I [pth];;

let REAL_RAT_INV_CONV =
  let pth1 = prove
   (`(inv(&0) = &0) /\
     (inv(&1) = &1) /\
     (inv(--(&1)) = --(&1))`,
    REWRITE_TAC[REAL_INV_0; REAL_INV_1; REAL_INV_NEG])
  and pth2 = prove
   (`(inv(&n) = &1 / &n) /\
     (inv(--(&n)) = --(&1) / &n) /\
     (inv(&m / &n) = &n / &m) /\
     (inv(--(&m) / &n) = --(&n) / &m)`,
    REWRITE_TAC[REAL_INV_DIV] THEN
    REWRITE_TAC[REAL_INV_NEG; real_div; REAL_MUL_RNEG;
      REAL_MUL_LID; REAL_MUL_LNEG]) in
  GEN_REWRITE_CONV I [pth1] ORELSEC
  GEN_REWRITE_CONV I [pth2];;

(* ------------------------------------------------------------------------- *)
(* Addition.                                                                 *)
(* ------------------------------------------------------------------------- *)

let REAL_RAT_ADD_CONV =
  let pth = prove
   (`&0 < y1 ==> &0 < y2 ==> &0 < y3 ==>
     ((x1 * y2 + x2 * y1) * y3 = x3 * y1 * y2)
     ==> (x1 / y1 + x2 / y2 = x3 / y3)`,
    REPEAT DISCH_TAC THEN
    MP_TAC RAT_LEMMA2 THEN
    ASM_REWRITE_TAC[] THEN
    DISCH_THEN SUBST1_TAC THEN
    REWRITE_TAC[GSYM REAL_INV_MUL; GSYM real_div] THEN
    SUBGOAL_THEN `&0 < y1 * y2 /\ &0 < y3` MP_TAC THENL
     [ASM_REWRITE_TAC[] THEN MATCH_MP_TAC REAL_LT_MUL THEN
      ASM_REWRITE_TAC[];
      DISCH_THEN(fun th -> ASM_REWRITE_TAC[MATCH_MP RAT_LEMMA5 th])])
  and dest_divop = dest_binop `$/`
  and dest_addop = dest_binop `$+`
  and x1 = `x1:real` and x2 = `x2:real` and x3 = `x3:real`
  and y1 = `y1:real` and y2 = `y2:real` and y3 = `y3:real` in
  let RAW_REAL_RAT_ADD_CONV tm =
    let r1,r2 = dest_addop tm in
    let x1',y1' = dest_divop r1
    and x2',y2' = dest_divop r2 in
    let x1n = dest_intconst x1' and y1n = dest_intconst y1'
    and x2n = dest_intconst x2' and y2n = dest_intconst y2' in
    let x3n = x1n */ y2n +/ x2n */ y1n
    and y3n = y1n */ y2n in
    let d = gcd_num x3n y3n in
    let x3n' = quo_num x3n d and y3n' = quo_num y3n d in
    let x3n'',y3n'' = if y3n' >/ Int 0 then x3n',y3n'
                      else minus_num x3n',minus_num y3n' in
    let x3' = mk_intconst x3n'' and y3' = mk_intconst y3n'' in
    let th0 = INST [x1',x1; y1',y1; x2',x2; y2',y2; x3',x3; y3',y3] pth in
    let th1 = REAL_INT_POS_PROVE (REAL_INT_POS_PROVE
               (REAL_INT_POS_PROVE th0)) in
    let tm2,tm3 = dest_eq(fst(dest_imp(concl th1))) in
    let th2 = (LAND_CONV (COMB2_CONV (RAND_CONV REAL_INT_MUL_CONV)
                                     REAL_INT_MUL_CONV THENC
                          REAL_INT_ADD_CONV) THENC
               REAL_INT_MUL_CONV) tm2
    and th3 = (RAND_CONV REAL_INT_MUL_CONV THENC REAL_INT_MUL_CONV) tm3 in
    MP th1 (TRANS th2 (SYM th3)) in
  REAL_INT_RAT_BINOP_CONV THENC RAW_REAL_RAT_ADD_CONV THENC REAL_RAT_INT_CONV;;

(* ------------------------------------------------------------------------- *)
(* Subtraction.                                                              *)
(* ------------------------------------------------------------------------- *)

let REAL_RAT_SUB_CONV =
  let pth = prove
   (`x - y = x + --y`,
    REWRITE_TAC[real_sub]) in
  GEN_REWRITE_CONV I [pth] THENC
  RAND_CONV REAL_RAT_NEG_CONV THENC REAL_RAT_ADD_CONV;;

(* ------------------------------------------------------------------------- *)
(* Multiplication.                                                           *)
(* ------------------------------------------------------------------------- *)

let REAL_RAT_MUL_CONV =
  let pth = prove
   (`&0 < y1 ==> &0 < y2 ==> &0 < y3 ==>
     (x1 * x2 * y3 = y1 * y2 * x3)
     ==> ((x1 / y1) * (x2 / y2) = x3 / y3)`,
    REPEAT DISCH_TAC THEN REWRITE_TAC[real_div] THEN
    ONCE_REWRITE_TAC[AC REAL_MUL_AC
      `(a * b) * (c * d) = (a * c) * (b * d)`] THEN
    REWRITE_TAC[GSYM REAL_INV_MUL] THEN
    REWRITE_TAC[GSYM real_div] THEN
    SUBGOAL_THEN `&0 < y1 * y2 /\ &0 < y3` MP_TAC THENL
     [ASM_REWRITE_TAC[] THEN MATCH_MP_TAC REAL_LT_MUL THEN
      ASM_REWRITE_TAC[];
      DISCH_THEN(fun th -> ASM_REWRITE_TAC[MATCH_MP RAT_LEMMA5 th]) THEN
      ASM_REWRITE_TAC[GSYM REAL_MUL_ASSOC; REAL_MUL_AC]])
  and dest_divop = dest_binop `$/`
  and dest_mulop = dest_binop `$*`
  and x1 = `x1:real` and x2 = `x2:real` and x3 = `x3:real`
  and y1 = `y1:real` and y2 = `y2:real` and y3 = `y3:real` in
  let RAW_REAL_RAT_MUL_CONV tm =
    let r1,r2 = dest_mulop tm in
    let x1',y1' = dest_divop r1
    and x2',y2' = dest_divop r2 in
    let x1n = dest_intconst x1' and y1n = dest_intconst y1'
    and x2n = dest_intconst x2' and y2n = dest_intconst y2' in
    let x3n = x1n */ x2n
    and y3n = y1n */ y2n in
    let d = gcd_num x3n y3n in
    let x3n' = quo_num x3n d and y3n' = quo_num y3n d in
    let x3n'',y3n'' = if y3n' >/ Int 0 then x3n',y3n'
                      else minus_num x3n',minus_num y3n' in
    let x3' = mk_intconst x3n'' and y3' = mk_intconst y3n'' in
    let th0 = INST [x1',x1; y1',y1; x2',x2; y2',y2; x3',x3; y3',y3] pth in
    let th1 = REAL_INT_POS_PROVE (REAL_INT_POS_PROVE
               (REAL_INT_POS_PROVE th0)) in
    let tm2,tm3 = dest_eq(fst(dest_imp(concl th1))) in
    let th2 = (RAND_CONV REAL_INT_MUL_CONV THENC REAL_INT_MUL_CONV) tm2
    and th3 = (RAND_CONV REAL_INT_MUL_CONV THENC REAL_INT_MUL_CONV) tm3 in
    MP th1 (TRANS th2 (SYM th3)) in
  REAL_INT_RAT_BINOP_CONV THENC RAW_REAL_RAT_MUL_CONV THENC REAL_RAT_INT_CONV;;

(* ------------------------------------------------------------------------- *)
(* Division.                                                                 *)
(* ------------------------------------------------------------------------- *)

let REAL_RAT_DIV_CONV =
  let pth = prove
   (`x / y = x * inv(y)`,
    REWRITE_TAC[real_div]) in
  let rawconv = GEN_REWRITE_CONV I [pth] THENC
                RAND_CONV REAL_RAT_INV_CONV THENC REAL_RAT_MUL_CONV in
  let div_tm = `$/` in
  fun tm ->
    let lop,r = dest_comb tm in
    let op,l = dest_comb lop in
    if op = div_tm & is_intconst l & is_numconst r
    then failwith "REAL_RAT_DIV_CONV: No change"
    else rawconv tm;;

(* ------------------------------------------------------------------------- *)
(* Powers.                                                                   *)
(* ------------------------------------------------------------------------- *)

let REAL_RAT_POW_CONV =
  let pth = prove
   (`(x / y) pow n = (x pow n) / (y pow n)`,
    REWRITE_TAC[REAL_POW_DIV]) in
  REAL_INT_POW_CONV ORELSEC
  (GEN_REWRITE_CONV I [pth] THENC
   COMB2_CONV (RAND_CONV REAL_INT_POW_CONV) REAL_INT_POW_CONV);;

(* ------------------------------------------------------------------------- *)
(* Everything.                                                               *)
(* ------------------------------------------------------------------------- *)

let REAL_RAT_RED_CONV = FIRST_CONV
  [REAL_RAT_LE_CONV; REAL_RAT_LT_CONV; REAL_RAT_GE_CONV;
   REAL_RAT_GT_CONV; REAL_RAT_EQ_CONV;
   CHANGED_CONV REAL_RAT_NEG_CONV; REAL_RAT_ABS_CONV; REAL_RAT_INV_CONV;
   REAL_RAT_ADD_CONV; REAL_RAT_SUB_CONV;
   REAL_RAT_MUL_CONV; REAL_RAT_DIV_CONV; REAL_RAT_POW_CONV];;

let REAL_RAT_REDUCE_CONV =
  let conv = GEN_REWRITE_CONV I (basic_rewrites()) in
  DEPTH_CONV (conv ORELSEC REAL_RAT_RED_CONV);;

(* ------------------------------------------------------------------------- *)
(* A conversion to evaluate summations (not clear it belongs here...)        *)
(* ------------------------------------------------------------------------- *)

let REAL_SUM_CONV =
  let Sum_tm = `Sum` in 
  let pth = prove
   (`Sum(0,1) f = f 0`,
    REWRITE_TAC[num_CONV `1`; Sum; REAL_ADD_LID; ADD_CLAUSES]) in
  let conv0 = GEN_REWRITE_CONV I [CONJUNCT1 Sum; pth]
  and conv1 = REWR_CONV(CONJUNCT2 Sum) in
  let rec sum_conv tm =
    try conv0 tm
    with Failure _ ->
      (LAND_CONV(RAND_CONV num_CONV) THENC 
       conv1 THENC LAND_CONV sum_conv) tm in
  fun tm ->
    let sn,bod = dest_comb tm in
    let s,ntm = dest_comb sn in
    let _,htm = dest_pair ntm in
    if s = Sum_tm & is_numeral htm
    then sum_conv tm
    else failwith "REAL_SUM0_CONV";;
