(* ========================================================================= *)
(* Calculation with naturals.                                                *)
(* ========================================================================= *)

let mangle = GEN_REWRITE_RULE DEPTH_CONV [NUMERAL];;

(* ------------------------------------------------------------------------- *)
(* Big collection of rewrites to do trivial arithmetic.                      *)
(*                                                                           *)
(* Note that we have none for DIV and MOD, and that PRE and SUB are a bit    *)
(* inefficient; log(n)^2 instead of log(n).                                  *)
(* ------------------------------------------------------------------------- *)

let ARITH_ZERO = prove
 (`(NUMERAL 0 = 0) /\
   (BIT0 _0 = _0)`,
  REWRITE_TAC[NUMERAL; BIT0; mangle ADD_CLAUSES]);;

let ARITH_SUC = prove
 (`(!n. SUC(NUMERAL n) = NUMERAL(SUC n)) /\
   (SUC _0 = BIT1 _0) /\
   (!n. SUC (BIT0 n) = BIT1 n) /\
   (!n. SUC (BIT1 n) = BIT0 (SUC n))`,
  REWRITE_TAC[NUMERAL; BIT0; BIT1; mangle ADD_CLAUSES]);;

let ARITH_PRE = prove
 (`(!n. PRE(NUMERAL n) = NUMERAL(PRE n)) /\
   (PRE _0 = _0) /\
   (!n. PRE(BIT0 n) = if n = _0 then _0 else BIT1 (PRE n)) /\
   (!n. PRE(BIT1 n) = BIT0 n)`,
  REWRITE_TAC[NUMERAL; BIT1; BIT0; mangle PRE] THEN
  INDUCT_TAC THEN REWRITE_TAC
    [NUMERAL; mangle PRE; mangle ADD_CLAUSES; mangle NOT_SUC; ARITH_ZERO]);;

let ARITH_ADD = prove
 (`(!m n. NUMERAL(m) + NUMERAL(n) = NUMERAL(m + n)) /\
   (_0 + _0 = _0) /\
   (!n. _0 + BIT0 n = BIT0 n) /\
   (!n.        _0 + BIT1 n = BIT1 n) /\
   (!n.   BIT0 n + _0 = BIT0 n) /\
   (!n.   BIT1 n + _0 = BIT1 n) /\
   (!m n. BIT0 m + BIT0 n = BIT0 (m + n)) /\
   (!m n. BIT0 m + BIT1 n = BIT1 (m + n)) /\
   (!m n. BIT1 m + BIT0 n = BIT1 (m + n)) /\
   (!m n. BIT1 m + BIT1 n = BIT0 (SUC(m + n)))`,
  PURE_REWRITE_TAC[NUMERAL; BIT0; BIT1; mangle ADD_CLAUSES; SUC_INJ] THEN
  REWRITE_TAC[ADD_AC]);;

let ARITH_MULT = prove
 (`(!m n. NUMERAL(m) * NUMERAL(n) = NUMERAL(m * n)) /\
   (_0 * _0 = _0) /\
   (!n. _0 * BIT0 n = _0) /\
   (!n. _0 * BIT1 n = _0) /\
   (!n. BIT0 n * _0 = _0) /\
   (!n. BIT1 n * _0 = _0) /\
   (!m n. BIT0 m * BIT0 n = BIT0 (BIT0 (m * n))) /\
   (!m n. BIT0 m * BIT1 n = BIT0 m + BIT0 (BIT0 (m * n))) /\
   (!m n. BIT1 m * BIT0 n = BIT0 n + BIT0 (BIT0 (m * n))) /\
   (!m n. BIT1 m * BIT1 n = BIT1 m + BIT0 n + BIT0 (BIT0 (m * n)))`,
  PURE_REWRITE_TAC
   [NUMERAL; BIT0; BIT1; mangle MULT_CLAUSES; mangle ADD_CLAUSES; SUC_INJ] THEN
  REWRITE_TAC[LEFT_ADD_DISTRIB; RIGHT_ADD_DISTRIB; ADD_AC]);;

let ARITH_EXP = prove
 (`(!m n. (NUMERAL m) EXP (NUMERAL n) = NUMERAL(m EXP n)) /\
   (_0 EXP _0 = BIT1 _0) /\
   (!m. (BIT0 m) EXP _0 = BIT1 _0) /\
   (!m. (BIT1 m) EXP _0 = BIT1 _0) /\
   (!n. _0 EXP (BIT0 n) = (_0 EXP n) * (_0 EXP n)) /\
   (!m n. (BIT0 m) EXP (BIT0 n) = ((BIT0 m) EXP n) * ((BIT0 m) EXP n)) /\
   (!m n. (BIT1 m) EXP (BIT0 n) = ((BIT1 m) EXP n) * ((BIT1 m) EXP n)) /\
   (!n. _0 EXP (BIT1 n) = _0) /\
   (!m n. (BIT0 m) EXP (BIT1 n) =
        BIT0 m * ((BIT0 m) EXP n) * ((BIT0 m) EXP n)) /\
   (!m n. (BIT1 m) EXP (BIT1 n) =
        BIT1 m * ((BIT1 m) EXP n) * ((BIT1 m) EXP n))`,
  REWRITE_TAC[NUMERAL] THEN REPEAT STRIP_TAC THEN
  TRY(GEN_REWRITE_TAC (LAND_CONV o RAND_CONV) [BIT0; BIT1]) THEN
  REWRITE_TAC[mangle EXP; mangle MULT_CLAUSES; EXP_ADD]);;

let ARITH_EVEN = prove
 (`(!n. EVEN(NUMERAL n) = EVEN n) /\
   (EVEN _0 = T) /\
   (!n. EVEN(BIT0 n) = T) /\
   (!n. EVEN(BIT1 n) = F)`,
  REWRITE_TAC[NUMERAL; BIT1; BIT0; mangle EVEN; EVEN_ADD]);;

let ARITH_ODD = prove
 (`(!n. ODD(NUMERAL n) = ODD n) /\
   (ODD _0 = F) /\
   (!n. ODD(BIT0 n) = F) /\
   (!n. ODD(BIT1 n) = T)`,
  REWRITE_TAC[NUMERAL; BIT1; BIT0; mangle ODD; ODD_ADD]);;

let ARITH_LE = prove
 (`(!m n. NUMERAL m <= NUMERAL n = m <= n) /\
   ((_0 <= _0) = T) /\
   (!n. (BIT0 n <= _0) = (n = _0)) /\
   (!n. (BIT1 n <= _0) = F) /\
   (!n. (_0 <= BIT0 n) = T) /\
   (!n. (_0 <= BIT1 n) = T) /\
   (!m n. (BIT0 m <= BIT0 n) = m <= n) /\
   (!m n. (BIT0 m <= BIT1 n) = m <= n) /\
   (!m n. (BIT1 m <= BIT0 n) = m < n) /\
   (!m n. (BIT1 m <= BIT1 n) = m <= n)`,
  REWRITE_TAC[NUMERAL; BIT1; BIT0; mangle NOT_SUC;
      mangle(GSYM NOT_SUC); SUC_INJ] THEN
  REWRITE_TAC[mangle LE_0] THEN REWRITE_TAC[mangle LE; GSYM MULT_2] THEN
  REWRITE_TAC[LE_MULT_LCANCEL; SUC_INJ; mangle MULT_EQ_0; mangle NOT_SUC] THEN
  REWRITE_TAC[mangle NOT_SUC] THEN REWRITE_TAC[LE_SUC_LT] THEN
  REWRITE_TAC[LT_MULT_LCANCEL] THEN
  SUBGOAL_THEN `2 = SUC 1` (fun th -> REWRITE_TAC[th]) THENL
   [REWRITE_TAC[NUMERAL; BIT0; BIT1; mangle ADD_CLAUSES];
    REWRITE_TAC[mangle NOT_SUC; NOT_SUC; EQ_MULT_LCANCEL] THEN
    REWRITE_TAC[ONCE_REWRITE_RULE[DISJ_SYM] LE_LT] THEN
    MAP_EVERY X_GEN_TAC [`m:num`; `n:num`] THEN
    SUBGOAL_THEN `~(SUC 1 * m = SUC (SUC 1 * n))`
      (fun th -> REWRITE_TAC[th]) THEN
    DISCH_THEN(MP_TAC o AP_TERM `EVEN`) THEN
    REWRITE_TAC[EVEN_MULT; EVEN_ADD; NUMERAL; BIT1; EVEN]]);;

let ARITH_LT = prove
 (`(!m n. NUMERAL m < NUMERAL n = m < n) /\
   ((_0 < _0) = F) /\
   (!n. (BIT0 n < _0) = F) /\
   (!n. (BIT1 n < _0) = F) /\
   (!n. (_0 < BIT0 n) = _0 < n) /\
   (!n. (_0 < BIT1 n) = T) /\
   (!m n. (BIT0 m < BIT0 n) = m < n) /\
   (!m n. (BIT0 m < BIT1 n) = m <= n) /\
   (!m n. (BIT1 m < BIT0 n) = m < n) /\
   (!m n. (BIT1 m < BIT1 n) = m < n)`,
  REWRITE_TAC[NUMERAL; GSYM NOT_LE; ARITH_LE] THEN
  REWRITE_TAC[mangle LE]);;

let ARITH_GE = REWRITE_RULE[GSYM GE; GSYM GT] ARITH_LE;;

let ARITH_GT = REWRITE_RULE[GSYM GE; GSYM GT] ARITH_LT;;

let ARITH_EQ = prove
 (`(!m n. (NUMERAL m = NUMERAL n) = (m = n)) /\
   ((_0 = _0) = T) /\
   (!n. (BIT0 n = _0) = (n = _0)) /\
   (!n. (BIT1 n = _0) = F) /\
   (!n. (_0 = BIT0 n) = (_0 = n)) /\
   (!n. (_0 = BIT1 n) = F) /\
   (!m n. (BIT0 m = BIT0 n) = (m = n)) /\
   (!m n. (BIT0 m = BIT1 n) = F) /\
   (!m n. (BIT1 m = BIT0 n) = F) /\
   (!m n. (BIT1 m = BIT1 n) = (m = n))`,
  REWRITE_TAC[NUMERAL; GSYM LE_ANTISYM; ARITH_LE] THEN
  REWRITE_TAC[LET_ANTISYM; LTE_ANTISYM; mangle LE_0]);;

let ARITH_SUB = prove
 (`(!m n. NUMERAL m - NUMERAL n = NUMERAL(m - n)) /\
   (_0 - _0 = _0) /\
   (!n. _0 - BIT0 n = _0) /\
   (!n. _0 - BIT1 n = _0) /\
   (!n. BIT0 n - _0 = BIT0 n) /\
   (!n. BIT1 n - _0 = BIT1 n) /\
   (!m n. BIT0 m - BIT0 n = BIT0 (m - n)) /\
   (!m n. BIT0 m - BIT1 n = PRE(BIT0 (m - n))) /\
   (!m n. BIT1 m - BIT0 n = if n <= m then BIT1 (m - n) else _0) /\
   (!m n. BIT1 m - BIT1 n = BIT0 (m - n))`,
  REWRITE_TAC[NUMERAL; mangle SUB_0] THEN PURE_REWRITE_TAC[BIT0; BIT1] THEN
  REWRITE_TAC[GSYM MULT_2; SUB_SUC; LEFT_SUB_DISTRIB] THEN
  REWRITE_TAC[SUB] THEN REPEAT GEN_TAC THEN COND_CASES_TAC THEN
  REWRITE_TAC[mangle SUB_EQ_0] THEN RULE_ASSUM_TAC(REWRITE_RULE[NOT_LE]) THEN
  ASM_REWRITE_TAC[LE_SUC_LT; LT_MULT_LCANCEL; ARITH_EQ] THEN
  POP_ASSUM(CHOOSE_THEN SUBST1_TAC o REWRITE_RULE[LE_EXISTS]) THEN
  REWRITE_TAC[ADD1; LEFT_ADD_DISTRIB] THEN
  REWRITE_TAC[ADD_SUB2; GSYM ADD_ASSOC]);;

let ARITH = end_itlist CONJ
  [ARITH_ZERO; ARITH_SUC; ARITH_PRE;
   ARITH_ADD; ARITH_MULT; ARITH_EXP;
   ARITH_EVEN; ARITH_ODD;
   ARITH_EQ; ARITH_LE; ARITH_LT; ARITH_GE; ARITH_GT;
   ARITH_SUB];;

(* ------------------------------------------------------------------------- *)
(* Now more delicate conversions for situations where efficiency matters.    *)
(* ------------------------------------------------------------------------- *)

let NUM_SUC_CONV',NUM_SUC_CONV =
  let [tth;bth1;bth2;rth] = CONJUNCTS ARITH_SUC in
  let tconv = GEN_REWRITE_CONV I [tth]
  and bconv = GEN_REWRITE_CONV I [bth1; bth2]
  and rconv = GEN_REWRITE_CONV I [rth] in
  let rec NUM_SUC_CONV tm =
    try bconv tm
    with Failure _ -> (rconv THENC RAND_CONV NUM_SUC_CONV) tm in
  NUM_SUC_CONV,
  (tconv THENC RAND_CONV NUM_SUC_CONV);;

let NUM_PRE_CONV =
  let tth = prove
   (`PRE 0 = 0`,
    REWRITE_TAC[PRE]) in
  let pth = prove
   (`(SUC m = n) ==> (PRE n = m)`,
    DISCH_THEN(SUBST1_TAC o SYM) THEN REWRITE_TAC[PRE])
  and m = `m:num` and n = `n:num` in
  let suc = `SUC` in
  let pre = `PRE` in
  fun tm -> try let l,r = dest_comb tm in
                if not l = pre then fail() else
                let x = dest_numeral r in
                if x =/ Int 0 then tth else
                let tm' = mk_numeral (x -/ Int 1) in
                let th1 = NUM_SUC_CONV (mk_comb(suc,tm')) in
                MP (INST [tm',m; r,n] pth) th1
            with Failure _ -> failwith "NUM_PRE_CONV";;

(* This one is hardly "delicate" --- all the relations rolled into
   one and implemented by rewriting. However it's not likely to
   be speed-critical. *)

let NUM_REL_CONV',NUM_REL_CONV =
  let pth = prove
   (`(NUMERAL m >= NUMERAL n = n <= m) /\
     (NUMERAL m > NUMERAL n = n < m)`,
    REWRITE_TAC[GE; GT; NUMERAL]) in
  let eq_tth,eq_oths = CONJ_PAIR ARITH_EQ
  and le_tth,le_oths = CONJ_PAIR ARITH_LE
  and lt_tth,lt_oths = CONJ_PAIR ARITH_LT in
  let tconv = GEN_REWRITE_CONV I [pth; eq_tth; le_tth; lt_tth]
  and oconv = REPEATC(GEN_REWRITE_CONV I [eq_oths; le_oths; lt_oths]) in
  oconv,(tconv THENC oconv);;

let NUM_EQ_CONV =
  let pth = CONJUNCT1 ARITH_EQ in
  GEN_REWRITE_CONV I [pth] THENC NUM_REL_CONV';;

let NUM_LE_CONV =
  let pth = CONJUNCT1 ARITH_LE in
  GEN_REWRITE_CONV I [pth] THENC NUM_REL_CONV';;

let NUM_LT_CONV =
  let pth = CONJUNCT1 ARITH_LT in
  GEN_REWRITE_CONV I [pth] THENC NUM_REL_CONV';;

let NUM_GE_CONV =
  let pth = prove
   (`NUMERAL m >= NUMERAL n = n <= m`,
    REWRITE_TAC[NUMERAL; GE]) in
  GEN_REWRITE_CONV I [pth] THENC NUM_REL_CONV';;

let NUM_GT_CONV =
  let pth = prove
   (`NUMERAL m > NUMERAL n = n < m`,
    REWRITE_TAC[NUMERAL; GT]) in
  GEN_REWRITE_CONV I [pth] THENC NUM_REL_CONV';;

let NUM_EVEN_CONV =
  let tth,rths = CONJ_PAIR ARITH_EVEN in
  GEN_REWRITE_CONV I [tth] THENC
  GEN_REWRITE_CONV I [rths];;

let NUM_ODD_CONV =
  let tth,rths = CONJ_PAIR ARITH_ODD in
  GEN_REWRITE_CONV I [tth] THENC
  GEN_REWRITE_CONV I [rths];;

let NUM_ADD_CONV',NUM_ADD_CONV =
  let [tth; bth1; bth2; bth3; bth4; bth5; rth1; rth2; rth3; xth] =
    CONJUNCTS ARITH_ADD in
  let bths = prove
   (`(!n. n + _0 = n) /\
     (!n. _0 + n = n)`,
    REWRITE_TAC[mangle ADD_CLAUSES]) in
  let tconv = GEN_REWRITE_CONV I [tth]
  and bconv = GEN_REWRITE_CONV I [bths]
  and rconv = GEN_REWRITE_CONV I [rth1; rth2; rth3]
  and xconv = GEN_REWRITE_CONV I [xth] in
  let rec NUM_ADD_CONV tm =
    try bconv tm
    with Failure _ -> try
        (rconv THENC RAND_CONV NUM_ADD_CONV) tm
    with Failure _ ->
        (xconv THENC
         RAND_CONV (RAND_CONV NUM_ADD_CONV THENC NUM_SUC_CONV')) tm in
  NUM_ADD_CONV,
  (tconv THENC RAND_CONV NUM_ADD_CONV);;

let NUM_SUB_CONV =
  let pth0 = prove
   (`p <= n ==> (p - n = 0)`,
    REWRITE_TAC[SUB_EQ_0])
  and pth1 = prove
   (`(m + n = p) ==> (p - n = m)`,
    DISCH_THEN(SUBST1_TAC o SYM) THEN
    REWRITE_TAC[ADD_SUB])
  and m = `m:num` and n = `n:num` and p = `p:num`
  and minus = `$-`
  and plus = `$+`
  and le = `$<=` in
  fun tm -> try let l,r = dest_binop minus tm in
                let ln = dest_numeral l
                and rn = dest_numeral r in
                if  ln <=/ rn then
                  let pth = INST [l,p; r,n] pth0
                  and th0 = EQT_ELIM(NUM_REL_CONV (mk_binop le l r)) in
                  MP pth th0
                else
                  let kn = ln -/ rn in
                  let k = mk_numeral kn in
                  let pth = INST [k,m; l,p; r,n] pth1
                  and th0 = NUM_ADD_CONV (mk_binop plus k r) in
                  MP pth th0
            with Failure _ -> failwith "NUM_SUB_CONV";;

let NUM_MULT_CONV' =
  let p_tm  = `p:num`
  and x_tm  = `x:num`
  and y_tm  = `y:num`
  and u_tm  = `u:num`
  and v_tm  = `v:num`
  and w_tm  = `w:num`
  and z_tm  = `z:num`
  and u_tm' = `u':num`
  and w_tm' = `w':num`
  and a_tm  = `a:num`
  and b_tm  = `b:num`
  and c_tm  = `c:num`
  and d_tm  = `d:num`
  and e_tm  = `e:num`
  and c_tm' = `c':num`
  and d_tm' = `d':num`
  and e_tm' = `e':num`
  and s_tm  = `s:num`
  and t_tm  = `t:num`
  and q_tm  = `q:num`
  and r_tm  = `r:num` in
  let pth = prove
   (`(u' + v = x) ==>
     (w' + z = y) ==>
     (p * u = u') ==>
     (p * w = w') ==>
     (u + v = a) ==>
     (w + z = b) ==>
     (a * b = c) ==>
     (u' * w = d) ==>
     (v * z = e) ==>
     (p * e = e') ==>
     (p * d = d') ==>
     (p * c = c') ==>
     (d' + e = s) ==>
     (d + e' = t) ==>
     (s + c' = q) ==>
     (r + t = q) ==> (x * y = r)`,
    MAP_EVERY (K (DISCH_THEN(SUBST1_TAC o SYM))) (upto 14) THEN
    REWRITE_TAC[LEFT_ADD_DISTRIB; RIGHT_ADD_DISTRIB] THEN
    REWRITE_TAC[MULT_AC; ADD_AC] THEN
    CONV_TAC(LAND_CONV NUM_CANCEL_CONV) THEN
    DISCH_THEN SUBST1_TAC THEN
    REWRITE_TAC[ADD_AC]) in
  let dest_mul = dest_binop `$*` in
  let mk_raw_numeral =
    let Z = mk_const("_0",[])
    and BIT0 = mk_const("BIT0",[])
    and BIT1 = mk_const("BIT1",[]) in
    let rec mk_num n =
      if n =/ Int 0 then Z else
      mk_comb((if mod_num n (Int 2) =/ Int 0 then BIT0 else BIT1),
              mk_num(quo_num n (Int 2))) in
    mk_num in
  let rec dest_raw_numeral tm =
    if try fst(dest_const tm) = "_0" with Failure _ -> false then Int 0 else
    let l,r = dest_comb tm in
    let n = Int 2 */ dest_raw_numeral r in
    let cn = fst(dest_const l) in
    if cn = "BIT0" then n
    else if cn = "BIT1" then n +/ Int 1
    else failwith "dest_raw_numeral" in
  let rec sizeof_rawnumeral tm =
    if is_const tm then 0 else
    1 + sizeof_rawnumeral(rand tm) in
  let MULTIPLICATION_TABLE =
    let pth = prove
     (`(_0 * x = _0) /\
       (x * _0 = _0) /\
       (BIT1 _0 * x = x) /\
       (x * BIT1 _0 = x)`,
      REWRITE_TAC[BIT1; mangle MULT_CLAUSES]) in
    let mk_mul = mk_binop `$*` in
    let odds = map (fun x -> 2 * x + 1) (upto 7) in
    let nums = map (mk_raw_numeral o Int) odds in
    let pairs = allpairs mk_mul nums nums in
    let ths = map (REWRITE_CONV[ARITH]) pairs in
    GEN_REWRITE_CONV I (pth::ths) in
  let NUM_MULT_EVEN_CONV' =
    let pth = prove
     (`(BIT0 x * y = BIT0(x * y)) /\
       (x * BIT0 y = BIT0(x * y))`,
      REWRITE_TAC[BIT0; BIT1; mangle MULT_CLAUSES; mangle ADD_CLAUSES] THEN
      REWRITE_TAC[LEFT_ADD_DISTRIB; RIGHT_ADD_DISTRIB; GSYM ADD_ASSOC]) in
    GEN_REWRITE_CONV I [pth] in
  let right_th = prove
   (`s * BIT1 x = s + BIT0 (s * x)`,
    REWRITE_TAC[BIT0; BIT1; mangle ADD_CLAUSES; mangle MULT_CLAUSES] THEN
    REWRITE_TAC[LEFT_ADD_DISTRIB; ADD_ASSOC])
  and left_th = prove
   (`BIT1 x * s = s + BIT0 (x * s)`,
    REWRITE_TAC[BIT0; BIT1; mangle ADD_CLAUSES; mangle MULT_CLAUSES] THEN
    REWRITE_TAC[RIGHT_ADD_DISTRIB; ADD_AC]) in
  let LEFT_REWR_CONV = REWR_CONV left_th
  and RIGHT_REWR_CONV = REWR_CONV right_th in
  let rec NUM_MULT_CONV' tm =
    try MULTIPLICATION_TABLE tm
    with Failure _ -> try
        let th1 = NUM_MULT_EVEN_CONV' tm in
        let l,r = dest_comb(rand(concl th1)) in
        TRANS th1 (AP_TERM l (NUM_MULT_CONV' r))
    with Failure _ ->
      let xtm,ytm = dest_mul tm in
      let x = dest_raw_numeral xtm
      and y = dest_raw_numeral ytm in
      let Nx = sizeof_rawnumeral xtm
      and Ny = sizeof_rawnumeral ytm in
      let N2 = max Nx Ny in
      let N = (N2 + 1) / 2 in
      if Nx < N or (N < 32 & Nx < Ny) then
        NUM_MULT_RIGHT_CONV' tm
      else if Ny < N or N < 32 then
        NUM_MULT_LEFT_CONV' tm
      else
      let p = power_num (Int 2) (Int N) in
      let u = quo_num x p
      and w = quo_num y p in
      let u' = p */ u
      and w' = p */ w in
      let v = x -/ u'
      and z = y -/ w' in
      let a = u +/ v
      and b = w +/ z in
      let c = a */ b in
      let d = u' */ w
      and e = v */ z in
      let e' = p */ e
      and d' = p */ d
      and c' = p */ c in
      let s = d' +/ e
      and t = d +/ e' in
      let q = s +/ c' in
      let r = x */ y in
      let ptm  = mk_raw_numeral p
      and xtm  = mk_raw_numeral x
      and ytm  = mk_raw_numeral y
      and utm  = mk_raw_numeral u
      and vtm  = mk_raw_numeral v
      and wtm  = mk_raw_numeral w
      and ztm  = mk_raw_numeral z
      and utm' = mk_raw_numeral u'
      and wtm' = mk_raw_numeral w'
      and atm  = mk_raw_numeral a
      and btm  = mk_raw_numeral b
      and ctm  = mk_raw_numeral c
      and dtm  = mk_raw_numeral d
      and etm  = mk_raw_numeral e
      and ctm' = mk_raw_numeral c'
      and dtm' = mk_raw_numeral d'
      and etm' = mk_raw_numeral e'
      and stm  = mk_raw_numeral s
      and ttm  = mk_raw_numeral t
      and qtm  = mk_raw_numeral q
      and rtm  = mk_raw_numeral r in
      let th0 = INST
       [ptm,p_tm; xtm,x_tm; ytm,y_tm; utm,u_tm; vtm,v_tm; wtm,w_tm;
        ztm,z_tm; utm',u_tm'; wtm',w_tm'; atm,a_tm; btm,b_tm; ctm,c_tm;
        dtm,d_tm; etm,e_tm; ctm',c_tm'; dtm',d_tm'; etm',e_tm'; stm,s_tm;
        ttm,t_tm; qtm,q_tm; rtm,r_tm] pth in
      let th1 = MP th0 (NUM_ADD_CONV' (lhand(lhand(concl th0)))) in
      let th2 = MP th1 (NUM_ADD_CONV' (lhand(lhand(concl th1)))) in
      let th3 = MP th2 (NUM_MULT_CONV' (lhand(lhand(concl th2)))) in
      let th4 = MP th3 (NUM_MULT_CONV' (lhand(lhand(concl th3)))) in
      let th5 = MP th4 (NUM_ADD_CONV' (lhand(lhand(concl th4)))) in
      let th6 = MP th5 (NUM_ADD_CONV' (lhand(lhand(concl th5)))) in
      let th7 = MP th6 (NUM_MULT_CONV' (lhand(lhand(concl th6)))) in
      let th8 = MP th7 (NUM_MULT_CONV' (lhand(lhand(concl th7)))) in
      let th9 = MP th8 (NUM_MULT_CONV' (lhand(lhand(concl th8)))) in
      let tha = MP th9 (NUM_MULT_CONV' (lhand(lhand(concl th9)))) in
      let thb = MP tha (NUM_MULT_CONV' (lhand(lhand(concl tha)))) in
      let thc = MP thb (NUM_MULT_CONV' (lhand(lhand(concl thb)))) in
      let thd = MP thc (NUM_ADD_CONV' (lhand(lhand(concl thc)))) in
      let the = MP thd (NUM_ADD_CONV' (lhand(lhand(concl thd)))) in
      let thf = MP the (NUM_ADD_CONV' (lhand(lhand(concl the)))) in
      MP thf (NUM_ADD_CONV' (lhand(lhand(concl thf))))
    and NUM_MULT_RIGHT_CONV' tm =
     (RIGHT_REWR_CONV THENC
      (RAND_CONV(RAND_CONV NUM_MULT_CONV')) THENC
      NUM_ADD_CONV') tm
    and NUM_MULT_LEFT_CONV' tm =
     (LEFT_REWR_CONV THENC
      (RAND_CONV(RAND_CONV NUM_MULT_CONV')) THENC
      NUM_ADD_CONV') tm in
  NUM_MULT_CONV';;

let NUM_MULT_CONV =
  let tconv = REWR_CONV(CONJUNCT1 ARITH_MULT) in
  tconv THENC RAND_CONV NUM_MULT_CONV';;

let NUM_EXP_CONV =
  let pth0 = prove
   (`(x EXP n = y) ==> (y * y = z) ==> (x EXP (BIT0 n) = z)`,
     REPEAT(DISCH_THEN(SUBST1_TAC o SYM)) THEN
     REWRITE_TAC[BIT0; EXP_ADD])
  and pth1 = prove
   (`(x EXP n = y) ==> (y * y = w) ==> (x * w = z) ==> (x EXP (BIT1 n) = z)`,
    REPEAT(DISCH_THEN(SUBST1_TAC o SYM)) THEN
    REWRITE_TAC[BIT1; EXP_ADD; EXP])
  and pth = prove
   (`x EXP _0 = BIT1 _0`,
    MP_TAC (CONJUNCT1 EXP) THEN REWRITE_TAC[NUMERAL; BIT1] THEN
    DISCH_THEN MATCH_ACCEPT_TAC)
  and tth = prove
   (`(NUMERAL x) EXP (NUMERAL n) = x EXP n`,
    REWRITE_TAC[NUMERAL])
  and fth = prove
   (`x = NUMERAL x`,
    REWRITE_TAC[NUMERAL])
  and n = `n:num` and w = `w:num` and x = `x:num`
  and y = `y:num` and z = `z:num`
  and Z = `_0` and BIT0 = `BIT0` and BIT1 = `BIT1`
  and mul = `$*` in
  let tconv = GEN_REWRITE_CONV I [tth] in
  let rec NUM_EXP_CONV l r =
    if r = Z then INST [l,x] pth else
    let b,r' = dest_comb r in
    if b = BIT0 then
      let th1 = NUM_EXP_CONV l r' in
      let tm1 = rand(concl th1) in
      let th2 = NUM_MULT_CONV' (mk_binop mul tm1 tm1) in
      let tm2 = rand(concl th2) in
      MP (MP (INST [l,x; r',n; tm1,y; tm2,z] pth0) th1) th2
    else
      let th1 = NUM_EXP_CONV l r' in
      let tm1 = rand(concl th1) in
      let th2 = NUM_MULT_CONV' (mk_binop mul tm1 tm1) in
      let tm2 = rand(concl th2) in
      let th3 = NUM_MULT_CONV' (mk_binop mul l tm2) in
      let tm3 = rand(concl th3) in
      MP (MP (MP (INST [l,x; r',n; tm1,y; tm2,w; tm3,z] pth1) th1) th2) th3 in
  fun tm -> try let th = tconv tm in
                let lop,r = dest_comb (rand(concl th)) in
                let _,l = dest_comb lop in
                let th' = NUM_EXP_CONV l r in
                let tm' = rand(concl th') in
                TRANS (TRANS th th') (INST [tm',x] fth)
            with Failure _ -> failwith "NUM_EXP_CONV";;

let NUM_DIVMOD_CONV,NUM_DIV_CONV,NUM_MOD_CONV =
  let pth = prove
   (`(q * n + r = m) ==> r < n ==> (m DIV n = q) /\ (m MOD n = r)`,
    MESON_TAC[DIVMOD_UNIQ])
  and m = `m:num` and n = `n:num` and q = `q:num` and r = `r:num`
  and dtm = `$DIV` and mtm = `$MOD` in
  let NUM_DIVMOD_CONV x y =
    let k = quo_num x y
    and l = mod_num x y in
    let th0 = INST [mk_numeral x,m; mk_numeral y,n;
                    mk_numeral k,q; mk_numeral l,r] pth in
    let tm0 = lhand(lhand(concl th0)) in
    let th1 = (LAND_CONV NUM_MULT_CONV THENC NUM_ADD_CONV) tm0 in
    let th2 = MP th0 th1 in
    let tm2 = lhand(concl th2) in
    MP th2 (EQT_ELIM(NUM_REL_CONV tm2)) in
  (fun tm -> try NUM_DIVMOD_CONV tm
             with Failure _ -> failwith "NUM_DIVMOD_CONV"),
  (fun tm -> try let xt,yt = dest_binop dtm tm in
                 CONJUNCT1(NUM_DIVMOD_CONV (dest_numeral xt) (dest_numeral yt))
             with Failure _ -> failwith "NUM_DIV_CONV"),
  (fun tm -> try let xt,yt = dest_binop mtm tm in
                 CONJUNCT2(NUM_DIVMOD_CONV (dest_numeral xt) (dest_numeral yt))
             with Failure _ -> failwith "NUM_MOD_CONV");;

let NUM_FACT_CONV =
  let suc = `SUC`
  and mul = `$*` in
  let pth_0 = prove
   (`FACT 0 = 1`,
    REWRITE_TAC[FACT])
  and pth_suc = prove
   (`(SUC x = y) ==> (FACT x = w) ==> (y * w = z) ==> (FACT y = z)`,
    REPEAT (DISCH_THEN(SUBST1_TAC o SYM)) THEN
    REWRITE_TAC[FACT])
  and w = `w:num` and x = `x:num` and y = `y:num` and z = `z:num` in
  let mksuc n =
    let n' = n -/ (Int 1) in
    NUM_SUC_CONV (mk_comb(suc,mk_numeral n')) in
  let rec NUM_FACT_CONV n =
    if n =/ Int 0 then pth_0 else
    let th0 = mksuc n in
    let tmx = rand(lhand(concl th0)) in
    let tm0 = rand(concl th0) in
    let th1 = NUM_FACT_CONV (n -/ Int 1) in
    let tm1 = rand(concl th1) in
    let th2 = NUM_MULT_CONV (mk_binop mul tm0 tm1) in
    let tm2 = rand(concl th2) in
    let pth = INST [tmx,x; tm0, y; tm1,w; tm2,z] pth_suc in
    MP (MP (MP pth th0) th1) th2 in
  fun tm ->
    try let l,r = dest_comb tm in
        if fst(dest_const l) = "FACT"
        then NUM_FACT_CONV (dest_numeral r)
        else fail()
    with Failure _ -> failwith "NUM_FACT_CONV";;

(* ------------------------------------------------------------------------- *)
(* Final hack-together.                                                      *)
(* ------------------------------------------------------------------------- *)

let NUM_RED_CONV =
  let gconv_net = itlist (uncurry net_of_conv)
    [`SUC(NUMERAL n)`,NUM_SUC_CONV;
     `PRE(NUMERAL n)`,NUM_PRE_CONV;
     `NUMERAL m < NUMERAL n`,NUM_REL_CONV;
     `NUMERAL m <= NUMERAL n`,NUM_REL_CONV;
     `NUMERAL m > NUMERAL n`,NUM_REL_CONV;
     `NUMERAL m >= NUMERAL n`,NUM_REL_CONV;
     `NUMERAL m = NUMERAL n`,NUM_REL_CONV;
     `EVEN(NUMERAL n)`,NUM_EVEN_CONV;
     `ODD(NUMERAL n)`,NUM_ODD_CONV;
     `NUMERAL m + NUMERAL n`,NUM_ADD_CONV;
     `NUMERAL m - NUMERAL n`,NUM_SUB_CONV;
     `NUMERAL m * NUMERAL n`,NUM_MULT_CONV;
     `(NUMERAL m) EXP (NUMERAL n)`,NUM_EXP_CONV;
     `(NUMERAL m) DIV (NUMERAL n)`,NUM_DIV_CONV;
     `(NUMERAL m) MOD (NUMERAL n)`,NUM_MOD_CONV]
    (basic_net()) in
  REWRITES_CONV gconv_net;;

let NUM_REDUCE_CONV = DEPTH_CONV NUM_RED_CONV;;

let NUM_REDUCE_TAC = CONV_TAC NUM_REDUCE_CONV;;

(* ------------------------------------------------------------------------- *)
(* I do like this after all...                                               *)
(* ------------------------------------------------------------------------- *)

let num_CONV =
  let SUC_tm = `SUC` in
  fun tm ->
    let n = dest_numeral tm -/ Int 1 in
    if n </ Int 0 then failwith "num_CONV" else
    let tm' = mk_numeral n in
    SYM(NUM_SUC_CONV (mk_comb(SUC_tm,tm')));;
