(* ========================================================================= *)
(* Boolean theory including (intuitionistic) defs of logical connectives.    *)
(* ========================================================================= *)

(* ------------------------------------------------------------------------- *)
(* Set up parse status of basic and derived logical constants.               *)
(* ------------------------------------------------------------------------- *)

parse_as_prefix "~";;

map parse_as_binder ["\\"; "!"; "?"; "?!"];;

map parse_as_infix ["=",(2,"right");
                    "==>",(4,"right");
                    "\\/",(6,"right");
                    "/\\",(8,"right")];;

(* ------------------------------------------------------------------------- *)
(* Special syntax for Boolean equations (IFF).                               *)
(* ------------------------------------------------------------------------- *)

let is_beq tm =
  try let s,ty = dest_const(rator(rator tm)) in
      s = "=" & hd(snd(dest_type ty)) = bool_ty
  with Failure _ -> false;;

(* ------------------------------------------------------------------------- *)
(* Some more useful rules for implication etc.                               *)
(* ------------------------------------------------------------------------- *)

let rec DISCH_ALL th =
  try DISCH_ALL (DISCH (hd (hyp th)) th)
  with Failure _ -> th;;

let UNDISCH th =
  try MP th (ASSUME(rand(rator(concl th))))
  with Failure _ -> failwith "UNDISCH";;

let rec UNDISCH_ALL th =
  if is_imp (concl th) then UNDISCH_ALL (UNDISCH th)
  else th;;

let ADD_ASSUM tm th = MP (DISCH tm th) (ASSUME tm);;

let EQ_IMP_RULE th =
  try let l,r = dest_eq(concl th) in
      DISCH l (EQ_MP th (ASSUME l)), DISCH r (EQ_MP(SYM th)(ASSUME r))
  with Failure _ -> failwith "EQ_IMP_RULE";;

let PROVE_HYP ath bth = MP (DISCH (concl ath) bth) ath;;

let IMP_TRANS th1 th2 =
  try let ant = rand(rator(concl th1)) in
      DISCH ant (MP th2 (MP th1 (ASSUME ant)))
  with Failure _ -> failwith "IMP_TRANS";;

(* ------------------------------------------------------------------------- *)
(* Basic derived principle of definition justifying |- c = t for closed "t". *)
(* ------------------------------------------------------------------------- *)

let new_definition tm =
  let l,r = dest_eq tm in
  let name = fst(dest_cvar l) in
  SYM(new_basic_definition name (REFL r));;

(* ------------------------------------------------------------------------- *)
(* Rules for T                                                               *)
(* ------------------------------------------------------------------------- *)

let T_DEF = new_definition
 `T = ((\x:bool. x) = (\x:bool. x))`;;

let TRUTH = EQ_MP (SYM T_DEF) (REFL `\x:bool .x`);;

let EQT_ELIM th =
  try EQ_MP (SYM th) TRUTH
  with Failure _ -> failwith "EQT_ELIM";;

let EQT_INTRO =
  let t = `t:bool` and T = `T` in
  let pth =
    COMPACT(DISCH t (IMP_ANTISYM_RULE (DISCH t TRUTH) (DISCH T (ASSUME t)))) in
  fun th -> MP (INST[concl th,t] pth) th;;

(* ------------------------------------------------------------------------- *)
(* Rules for !                                                               *)
(* ------------------------------------------------------------------------- *)

let FORALL_DEF = new_definition
 `$! = \P:A->bool. P = \x. T`;;

let is_forall = is_binder "!";;
let dest_forall = dest_binder "!";;
let mk_forall = mk_binder "!";;
let list_mk_forall(vs,bod) = itlist (curry mk_forall) vs bod;;
let strip_forall = splitlist dest_forall;;

let SPEC =
  let P = `P:A->bool`
  and x = `x:A` in
  let pth =
    let th1 = EQ_MP(AP_THM FORALL_DEF `P:A->bool`) (ASSUME `$! (P:A->bool)`) in
    let th2 = AP_THM (CONV_RULE BETA_CONV th1) `x:A` in
    let th3 = CONV_RULE (RAND_CONV BETA_CONV) th2 in
    COMPACT(DISCH_ALL (EQT_ELIM th3)) in
  fun tm th ->
    try let abs = rand(concl th) in
        CONV_RULE BETA_CONV
         (MP (PINST [snd(dest_var(bndvar abs)),aty] [abs,P; tm,x] pth) th)
    with Failure _ -> failwith "SPEC";;

let SPECL tms th =
  try rev_itlist SPEC tms th
  with Failure _ -> failwith "SPECL";;

let SPEC_VAR th =
  let bv = variant (thm_frees th) (bndvar(rand(concl th))) in
  bv,SPEC bv th;;

let rec SPEC_ALL th =
  if is_forall(concl th) then SPEC_ALL(snd(SPEC_VAR th)) else th;;

let ISPEC t th =
  let x,_ = try dest_forall(concl th) with Failure _ ->
    failwith "ISPEC: input theorem not universally quantified" in
  let tyins = try type_match (snd(dest_var x)) (type_of t) [] with Failure _ ->
    failwith "ISPEC can't type-instantiate input theorem" in
  try SPEC t (INST_TYPE tyins th)
  with Failure _ -> failwith "ISPEC: type variable(s) free in assumptions";;

let ISPECL tms th =
  try if tms = [] then th else
      let avs = fst (chop_list (length tms) (fst(strip_forall(concl th)))) in
      let tyins = itlist2 type_match (map (snd o dest_var) avs)
                          (map type_of tms) [] in
      SPECL tms (INST_TYPE tyins th)
  with Failure _ -> failwith "ISPECL";;

let GEN =
  let P = `P:A->bool` in
  let pth =
    let th1 = ASSUME `P = \x:A. T` in
    let th2 = AP_THM FORALL_DEF `P:A->bool` in
    COMPACT(DISCH_ALL (EQ_MP (SYM(CONV_RULE(RAND_CONV BETA_CONV) th2)) th1)) in
  fun x th ->
    try let th1 = ABS x (EQT_INTRO th) in
        let th2 = PINST [snd(dest_var x),aty] [rand(rator(concl th1)),P] pth in
        MP th2 th1
    with Failure _ -> failwith "GEN";;

let GENL = itlist GEN;;

let GEN_ALL th =
  let asl,c = dest_thm th in
  let vars = subtract (frees c) (freesl asl) in
  GENL vars th;;

(* ------------------------------------------------------------------------- *)
(* Rules for ?                                                               *)
(* ------------------------------------------------------------------------- *)

let EXISTS_DEF = new_definition
 `$? = \P:A->bool. !Q. (!x. P x ==> Q) ==> Q`;;

let is_exists = is_binder "?";;
let dest_exists = dest_binder "?";;
let mk_exists =  mk_binder "?";;
let list_mk_exists(vs,bod) =  itlist (curry mk_exists) vs bod;;
let strip_exists = splitlist dest_exists;;

let EXISTS =
  let P = `P:A->bool` and x = `x:A` and Px = `(P:A->bool) x` in
  let pth =
    let th1 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM EXISTS_DEF P) in
    let th2 = SPEC `x:A` (ASSUME `!x:A. P x ==> Q`) in
    let th3 = DISCH `!x:A. P x ==> Q` (MP th2 (ASSUME `(P:A->bool) x`)) in
    COMPACT(DISCH_ALL (EQ_MP (SYM th1) (GEN `Q:bool` th3))) in
  fun (etm,stm) th ->
    try let qf,abs = dest_comb etm in
        let bth = BETA_CONV(mk_comb(abs,stm)) in
        let cth = PINST [type_of stm,aty] [abs,P; stm,x] pth in
        MP cth (EQ_MP (SYM bth) th)
    with Failure _ -> failwith "EXISTS";;

let SIMPLE_EXISTS v th =
  EXISTS (mk_exists(v,concl th),v) th;;

let CHOOSE =
  let P = `P:A->bool` and Q = `Q:bool` in
  let pth =
    let th1 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM EXISTS_DEF P) in
    let th2 = SPEC `Q:bool` (UNDISCH(fst(EQ_IMP_RULE th1))) in
    COMPACT(DISCH_ALL (DISCH `$? (P:A->bool)` (UNDISCH th2))) in
  fun (v,th1) th2 ->
    try let abs = rand(concl th1) in
        let bv,bod = dest_abs abs in
        let cmb = mk_comb(abs,v) in
        let pat = vsubst[v,bv] bod in
        let th3 = CONV_RULE BETA_CONV (ASSUME cmb) in
        let th4 = GEN v (DISCH cmb (MP (DISCH pat th2) th3)) in
        let th5 = PINST [snd(dest_var v),aty] [abs,P; concl th2,Q] pth in
        MP (MP th5 th4) th1
    with Failure _ -> failwith "CHOOSE";;

let SIMPLE_CHOOSE v th =
  CHOOSE(v,ASSUME (mk_exists(v,hd(hyp th)))) th;;

(* ------------------------------------------------------------------------- *)
(* Rules for /\                                                              *)
(* ------------------------------------------------------------------------- *)

let AND_DEF = new_definition
 `$/\ = \t1 t2. !t. (t1 ==> t2 ==> t) ==> t`;;

let is_conj = is_binary "/\\";;
let dest_conj = dest_binary "/\\";;
let mk_conj = mk_binary "/\\";;
let list_mk_conj = end_itlist (curry mk_conj);;
let conjuncts = striplist dest_conj;;

let CONJ =
  let P = `P:bool` and Q = `Q:bool` in
  let pth =
    let th1 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM AND_DEF `P:bool`) in
    let th2 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM th1 `Q:bool`) in
    let th3 = ASSUME `P ==> Q ==> t` in
    let th4 = DISCH `P ==> Q ==> t` (UNDISCH (UNDISCH th3)) in
    COMPACT(DISCH P (DISCH Q (EQ_MP (SYM th2) (GEN `t:bool` th4)))) in
  fun th1 th2 ->
    let th = INST [concl th1,P; concl th2,Q] pth in
    MP (MP th th1) th2;;

let CONJUNCT1 =
  let P = `P:bool` and Q = `Q:bool` in
  let pth =
    let th1 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM AND_DEF `P:bool`) in
    let th2 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM th1 `Q:bool`) in
    let th3 = SPEC P (EQ_MP th2 (ASSUME `P /\ Q`)) in
    let th4 = DISCH P (DISCH Q (ASSUME P)) in
    COMPACT(DISCH_ALL (MP th3 th4)) in
  fun th ->
    try let l,r = dest_conj(concl th) in
        MP (INST [l,P; r,Q] pth) th
    with Failure _ -> failwith "CONJUNCT1";;

let CONJUNCT2 =
  let P = `P:bool` and Q = `Q:bool` in
  let pth =
    let th1 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM AND_DEF `P:bool`) in
    let th2 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM th1 `Q:bool`) in
    let th3 = SPEC Q (EQ_MP th2 (ASSUME `P /\ Q`)) in
    let th4 = DISCH P (DISCH Q (ASSUME Q)) in
    COMPACT(DISCH_ALL (MP th3 th4)) in
  fun th ->
    try let l,r = dest_conj(concl th) in
        MP (INST [l,P; r,Q] pth) th
    with Failure _ -> failwith "CONJUNCT2";;

let CONJ_PAIR th =
  try CONJUNCT1 th,CONJUNCT2 th
  with Failure _ -> failwith "CONJ_PAIR: Not a conjunction";;

let CONJUNCTS = striplist CONJ_PAIR;;

(* ------------------------------------------------------------------------- *)
(* Rules for \/                                                              *)
(* ------------------------------------------------------------------------- *)

let OR_DEF = new_definition
 `$\/ = \t1 t2. !t. (t1 ==> t) ==> (t2 ==> t) ==> t`;;

let is_disj = is_binary "\\/";;
let dest_disj = dest_binary "\\/";;
let mk_disj = mk_binary "\\/";;
let list_mk_disj = end_itlist (curry mk_disj);;
let disjuncts = striplist dest_disj;;

let DISJ1 =
  let P = `P:bool` and Q = `Q:bool` in
  let pth =
    let th1 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM OR_DEF `P:bool`) in
    let th2 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM th1 `Q:bool`) in
    let th3 = MP (ASSUME `P ==> t`) (ASSUME `P:bool`) in
    let th4 = GEN `t:bool` (DISCH `P ==> t` (DISCH `Q ==> t` th3)) in
    COMPACT(DISCH_ALL (EQ_MP (SYM th2) th4)) in
  fun th tm ->
    try MP (INST [concl th,P; tm,Q] pth) th
    with Failure _ -> failwith "DISJ1";;

let DISJ2 =
  let P = `P:bool` and Q = `Q:bool` in
  let pth =
    let th1 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM OR_DEF `P:bool`) in
    let th2 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM th1 `Q:bool`) in
    let th3 = MP (ASSUME `Q ==> t`) (ASSUME `Q:bool`) in
    let th4 = GEN `t:bool` (DISCH `P ==> t` (DISCH `Q ==> t` th3)) in
    COMPACT(DISCH_ALL (EQ_MP (SYM th2) th4)) in
  fun tm th ->
    try MP (INST [tm,P; concl th,Q] pth) th
    with Failure _ -> failwith "DISJ1";;

let DISJ_CASES =
  let P = `P:bool` and Q = `Q:bool` and R = `R:bool` in
  let pth =
    let th1 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM OR_DEF `P:bool`) in
    let th2 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM th1 `Q:bool`) in
    let th3 = SPEC `R:bool` (EQ_MP th2 (ASSUME `P \/ Q`)) in
    let th4 = DISCH `P \/ Q` (UNDISCH (UNDISCH th3)) in
    COMPACT(DISCH `P ==> R` (DISCH `Q ==> R` th4)) in
  fun th0 th1 th2 ->
    try let l,r = dest_disj (concl th0) in
        let th = INST [l,P; r,Q; concl th1,R] pth in
        MP (MP (MP th (DISCH l th1)) (DISCH r th2)) th0
    with Failure _ -> failwith "DISJ_CASES";;

let SIMPLE_DISJ_CASES th1 th2 =
  DISJ_CASES (ASSUME(mk_disj(hd(hyp th1),hd(hyp th2)))) th1 th2;;

(* ------------------------------------------------------------------------- *)
(* Rules for negation and falsity.                                           *)
(* ------------------------------------------------------------------------- *)

let F_DEF = new_definition
 `F = !t:bool. t`;;

let NOT_DEF = new_definition
 `$~ = \t. t ==> F`;;

let is_neg tm =
  try fst(dest_const(rator tm)) = "~"
  with Failure _ -> false;;

let dest_neg tm =
  try let n,p = dest_comb tm in
      if fst(dest_const n) = "~" then p else fail()
  with Failure _ -> failwith "dest_neg";;

let mk_neg =
  let neg_tm = `$~` in
  fun tm -> try mk_comb(neg_tm,tm)
            with Failure _ -> failwith "mk_neg";;

let NOT_ELIM =
  let P = `P:bool` in
  let pth = COMPACT(CONV_RULE(RAND_CONV BETA_CONV) (AP_THM NOT_DEF P)) in
  fun th ->
    try EQ_MP (INST [rand(concl th),P] pth) th
    with Failure _ -> failwith "NOT_ELIM";;

let NOT_INTRO =
  let P = `P:bool` in
  let pth = COMPACT(SYM(CONV_RULE(RAND_CONV BETA_CONV) (AP_THM NOT_DEF P))) in
  fun th ->
    try EQ_MP (INST [rand(rator(concl th)),P] pth) th
    with Failure _ -> failwith "NOT_ELIM";;

let EQF_INTRO =
  let P = `P:bool` in
  let pth =
    let th1 = NOT_ELIM (ASSUME `~ P`)
    and th2 = DISCH `F` (SPEC P (EQ_MP F_DEF (ASSUME `F`))) in
    COMPACT(DISCH_ALL (IMP_ANTISYM_RULE th1 th2)) in
  fun th ->
    try MP (INST [rand(concl th),P] pth) th
    with Failure _ -> failwith "EQF_INTRO";;

let EQF_ELIM =
  let P = `P:bool` in
  let pth =
    let th1 = EQ_MP (ASSUME `P = F`) (ASSUME `P:bool`) in
    let th2 = DISCH P (SPEC `F` (EQ_MP F_DEF th1)) in
    COMPACT(DISCH_ALL (NOT_INTRO th2)) in
  fun th ->
    try MP (INST [rand(rator(concl th)),P] pth) th
    with Failure _ -> failwith "EQF_ELIM";;

let NEG_DISCH =
  let falsity = `F` in
  fun t th ->
    try if concl th = falsity then NOT_INTRO(DISCH t th) else DISCH t th
    with Failure _ -> failwith "NEG_DISCH";;

let CONTR =
  let P = `P:bool` in
  let pth = COMPACT(DISCH_ALL (SPEC P (EQ_MP F_DEF (ASSUME `F`)))) in
  fun tm th ->
    try MP (INST [tm,P] pth) th
    with Failure _ -> failwith "CONTR";;

(* ------------------------------------------------------------------------- *)
(* Rules for unique existence.                                               *)
(* ------------------------------------------------------------------------- *)

let EXISTS_UNIQUE_DEF = new_definition
 `$?! = \P:A->bool. ($? P) /\ (!x y. ((P x) /\ (P y)) ==> (x = y))`;;

let EXISTENCE =
  let P = `P:A->bool` in
  let pth =
    let th1 = CONV_RULE (RAND_CONV BETA_CONV) (AP_THM EXISTS_UNIQUE_DEF P) in
    let th2 = UNDISCH (fst(EQ_IMP_RULE th1)) in
    COMPACT(DISCH_ALL (CONJUNCT1 th2)) in
  fun th ->
    try let abs = rand(concl th) in
        let ty = snd(dest_var(bndvar abs)) in
        MP (PINST [ty,aty] [abs,P] pth) th
    with Failure _ -> failwith "EXISTENCE";;
