 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File:	progress.tcl
#
# Description:
#	Management of a widget containing a panel representing progress
#	of a long-running operation.
#
#
# Global variables:
#c	progress_priv(variable,$w)
#		Variable name that holds the fraction of progress made
#		on the operation displayed in window $w.

 # progress.tcl,v 1.1.1.1 1994/12/07 10:17:30 donald Exp
 # /homes/drs1004/repository/tkaux/progress.tcl,v
 # progress.tcl,v
# Revision 1.1.1.1  1994/12/07  10:17:30  donald
# First import after some small chanegs.
#
 # Revision 1.11  1994/10/27  18:29:42  kennykb
 # Release 2.0 -- 10-27-94.  To be uploaded to archive sites.
 #
 # Revision 1.10  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.9  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.8  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.7  1993/10/20  19:04:11  kennykb
 # Repaired copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.6  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.5  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.4  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.3  1993/07/21  19:44:36  kennykb
 # Finished cleaning up structured commentary.
 #
 # Revision 1.2  1993/07/16  15:58:00  kennykb
 # Renamed all commands that start with `wiget.' to either `widget_' or
 # `widget:'.  Added the code for creating composite widgets.
 #
 # Revision 1.1  1993/06/03  15:31:00  kennykb
 # Initial revision
 #

# Procedure:	progress
#
# Synopsis:
#	Widget that displays a panel representing the progress of a
#	long-running operation.
#
# Usage:
#c	progress pathName ?-option value?...
#
# Parameters:
#c	pathName
#		Path name of the window that will hold the progress indication.
#
# Options:
#	Name:			quitCommand
#	Class:			QuitCommand
#	Command-line string:	-quit, -quitcommand
#	Default:		"destroy <pathName>"
#		Command to be invoked when the user presses the QUIT button.
#
#	Name:			text
#	Class:			Text
#	Command-line string:	-text
#	Default:		The empty string.
#		Text to be displayed in the `progress' dialog box.
#
#	Name:			variable
#	Class:			Variable
#	Command-line string:	-var, -variable
#	Default:		progress<pathName>
#		Global variable that holds the fraction of the operation that
#		has completed.
#
#	Other options are as for `frame.'
#
# Return value:
#	The path name of the widget is returned.
#
# Description:
#	The `progress' command constructs a simple dialog box consisting of:
#	- A message
#	- A bar showing the fraction of progress made in an operation
#	- A `quit' button.
#
#	The action taken by the `quit' button, and the message are
#	customizable.  The bar showing the amount of progress is tied to a
#	user-specified global variable.
#
# Bugs:
#	The `progress' widget is not a first-class widget.  It should at least
#	respond correctly to the `config' command. 

	# Default look and feel:

option add *Progress.quitCommand {} widgetDefault
option add *Progress.text {} widgetDefault
option add *Progress.variable {} widgetDefault

proc progress {w args} {

	frame $w -class Progress

	set quit [option get $w quitCommand QuitCommand]
	if {$quit == ""} {
		set quit "destroy $w"
	}
	set text [option get $w text Text]
	set variable [option get $w variable Variable]
	if {$variable == ""} {
		set variable progress$w
	}
	set fa {}

	# Parse command line

	while {[llength $args] > 0} {
		set string [lindex $args 0]
		set args [lrange $args 1 end]
		case $string in {
			{-quit -quitcommand} {
				set quit [lindex $args 0]
				set args [lrange $args 1 end]
			}
			-text {
				set text [lindex $args 0]
				set args [lrange $args 1 end]
			}
			{-v -var -variable} {
				set variable [lindex $args 0]
				set args [lrange $args 1 end]
			}
			default {
				lappend fa $string
				lappend fa [lindex $args 0]
				set args [lrange $args 1 end]
			}
		}
	}

	eval [list $w config] $fa

	if {$text != ""} {
		pack append $w \
			[label $w.l -text $text] \
				{top padx 10 frame center}
	}
	pack append $w \
		[progress:2 $w.p $variable] \
				{top frame w} \
		[eval fraction [list $w.f -variable $variable] $args] \
				{top expand padx 10 pady 10 frame center} \
		[focusable button $w.b -text "Quit" \
			-command "after 1 [list $quit]"] \
				{top expand pady 5 frame n}

	return $w
}

# Procedure:	progress:2
#
# Synopsis:
#	Internal procedure to aid in creating a `progress' widget.
#
# Usage:
#c	progress:2 pathName variable
#
# Parameters:
#c	pathName
#		Path name of a window
#c	variable
#		Name of a global variable containing the fraction of
#		progress to be displayed.
#
# Return value:
#	pathName is returned.
#
# Description:
#	progress:2 is an internal procedure that creates a label widget that
#	gets packed into the frame created by a `progress' window, and
#	establishes a trace on the variable to update the label.

proc progress:2 {w var} {
	global progress_priv
	set progress_priv(variable,$w) $var
	label $w -text " 0.0% complete"
	proc update$w {v1 v2 op} [format {
		trace_action $v1 $v2 $op {
			progress:2:update %s $value
		}
	} $w]
	uplevel #0 trace variable $var w update$w
	widget_addBinding $w Destroy "progress:2:destroy $w"
	return $w
}

# Procedure:	progress:2:update
#
# Synopsis:
#	Internal procedure to update part of a progress widget.
#
# Usage:
#	progress:2:update pathName fraction
#
# Parameters:
#c	pathName
#		Path name of a label widget.
#c	fraction
#		Fraction of the operation that has completed.
#
# Return value:
#	None.
#
# Description:
#	progress:2:update is called by a `trace' operation any time the value
#	on display in a `progress' window changes.  It updates the label widget
#	and synchronizes the display.
#
# Possible improvement:
#	Bunch the updates by use of an `after' handler.  This might be a big
#	performance win if a task runs quickly.

proc progress:2:update {w value} {
	$w config -text [format "%4.1f%% complete" [expr 100.0*$value]]
	update
}

# Procedure:	progress:2:destroy
#
# Synopsis:
#	Internal procedure to clean up when a progress widget is destroyed.
#
# Usage:
#c	progress:2:destroy pathName
#
# Parameters:
#c	pathName
#		Path name of a label widget.
#
# Return value:
#	None.
#
# Description:
#	`progress:2:destroy' is an internal procedure that is called when
#	a `progress' widget is destroyed.

proc progress:2:destroy w {
	global progress_priv
	uplevel #0 trace vdelete $progress_priv(variable,$w) w update$w
	rename update$w {}
	unset progress_priv(variable,$w)
}
