#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




# jbindings.tcl - support for Entry and Text bindings
# 
# Copyright 1992-1994 by Jay Sekora.  All rights reserved, except 
# that this file may be freely redistributed in whole or in part 
# for nonprofit, noncommercial use.
# 
# MODIFICATIONS:
#       This file has been modified extensively by Donald Syme as part
# of TkHol.  The Entry bindings have been removed (a good set being available)
# in the tkauxlib library).  The Text bindings have been extended to
# include normal Microsft-Windows standard movement commands.
#
# 
# Essentially, call
#   j:tb:emacs_bind Text    ---   for Emacs-like Text bindings
# (the emacs_bind procedures include the basic_bind procedures; you
# don't have to call both.)
# You can check $j_tb(modified,$WIDGET) to see if a text widget has been
# modified.  (Set it to false when you save the text widget.)
# j:tb:is_dirty w

# TO DO:
# ^L
# sentence-manipulation stuff
# case change commands, transposition commands
# commands to do with mark?
# word deletion - fix to use buffer
# generalise movement to copying-to-cutbuffer and deletion
# IMPROVE ENTRY BINDINGS
# literal-insert for entry


######################################################################
# j:selection_if_any - return selection if it exists, else {}
#   this is from kjx@comp.vuw.ac.nz (R. James Noble)
#   defined elsewhere, but copied here for portability
######################################################################

proc j:selection_if_any {} {
  if {[catch {selection get} s]} {return ""} {return $s}
}


######################################################################
####
###  GENERAL BINDING ROUTINES
####
######################################################################

######################################################################
# j:bindings:init - initialise info for bindings
######################################################################

proc j:bindings:init {} {
  global j_teb
  set j_teb(repeat_count) 1
  set j_teb(cutbuffer) {}
  set j_teb(dragscroll,txnd) 0
  set j_teb(dragscroll,delay) 50
  set j_teb(prefix) 0
  set j_teb(scanpaste_time) 0
  set j_teb(scanpaste_paste) 1
}

######################################################################
# j:bindings:debug - watch some of the variables
######################################################################

proc j:bindings:debug {} {
  global j_teb
  toplevel .foo
  wm geometry .foo +0+0
  label .foo.filename -width 15 -textvariable FILENAME
  label .foo.count -width 15 -textvariable j_teb(repeat_count)
  label .foo.prefix -width 15 -textvariable j_teb(prefix)
  pack .foo.filename -in .foo
  pack .foo.count -in .foo
  pack .foo.prefix -in .foo
}

######################################################################
# j:bindings:repeatable tclcode - execute tclcode one or more times
######################################################################

proc j:bindings:repeatable { tclcode } {
  global j_teb
  
  set j_teb(prefix) 0                           ;# no longer collectig digits
  if {$j_teb(repeat_count) > 1} {
    for {set jri 0} {$jri < $j_teb(repeat_count)} {incr jri} {
      uplevel 1 "eval [list $tclcode]"          ;# variables in caller
    }
    set j_teb(repeat_count) 1
    } else {
    uplevel 1 "eval [list $tclcode]"            ;# variables in caller
  }
}

######################################################################
# j:bindings:clear_count - clear argument count
######################################################################

proc j:bindings:clear_count {} {
  global j_teb
  set j_teb(repeat_count) 1
}

######################################################################
# j:bindings:repeat4 - set repeat count to four, or multiply by four
######################################################################

proc j:bindings:repeat4 {} {
  global j_teb
  
  if {$j_teb(repeat_count) < 1} {
    set j_teb(repeat_count) 1
  }
  set j_teb(repeat_count) [expr {$j_teb(repeat_count)*4}]
}

######################################################################
# j:bindings:start_number digit - start a numeric argument
#   invalid if not bound to (a sequence ending in) a digit key
######################################################################

proc j:bindings:start_number { digit } {
  global j_teb
  set j_teb(prefix) 1                   ;# collecting # prefix
  set j_teb(repeat_count) [expr "$digit"]
}

######################################################################
# j:bindings:continue_number digit - continue a numeric argument
#   invalid if not bound to a digit key
######################################################################

proc j:bindings:continue_number { digit } {
  global j_teb
  set j_teb(repeat_count) [expr {($j_teb(repeat_count)*10)+$digit}]
}
  

######################################################################
####
###  GENERAL TEXT BINDING ROUTINES
####
######################################################################


######################################################################
# j:tb:mark_dirty W - mark widget W as dirty (modified)
######################################################################

proc j:tb:mark_dirty { W } {
  global j_teb
  set j_teb(modified,$W) 1
}

######################################################################
# j:tb:mark_clean W - mark widget W as clean (unmodified)
######################################################################

proc j:tb:mark_dirty { W } {
  global j_teb
  set j_teb(modified,$W) 0
}

######################################################################
# j:tb:is_dirty W - return 1 if W is dirty (modified) else 0
######################################################################

proc j:tb:is_dirty { W } {
  global j_teb
  return j_teb(modified,$W)
}

######################################################################
# j:tb:literal_bind W prefix - set prefix for literal insertion
#   use AFTER creating other bindings.  W can be Text or widget
######################################################################
# the stuff with j_teb(literal-odd) is to make, eg, ^Q^Q insert "^Q",
####   but not ^Q^Q^Q, or ^Q^F^Q  DOESN'T WORK

proc j:tb:literal_bind { W prefix } {
  foreach binding [bind $W] {
    bind $W ${prefix}${binding}         {j:tb:insert_glyph "%A" %W}
  }
  bind $W ${prefix}<space>              {j:tb:insert_glyph " " %W}
  bind $W ${prefix}<Any-space>          {j:tb:insert_glyph " " %W}
  bind $W ${prefix}<Return>             {j:tb:insert_glyph "\n" %W}
  bind $W ${prefix}<Any-Return>         {j:tb:insert_glyph "\n" %W}
  bind $W ${prefix}${prefix}            {j:tb:insert_glyph "%A" %W}
  
  bind $W ${prefix}                     {catch {unset j_teb(literal-odd)}}
  bind $W ${prefix}{$prefix}            {
    if {[lsearch -exact [array names j_teb] literal-odd] != -1} {
      unset j_teb(literal-odd)
    } else {
      {j:tb:insert_glyph "%A" %W}
      set j_teb(literal-odd) 1
    }
  }

}

######################################################################
# j:tb:insert_glyph A W - insert glyph A into text widget W
######################################################################

proc j:tb:insert_glyph { A W } {
  global j_teb
  
  if {"$A" != ""} {
    catch {$W delete sel.first sel.last}
    j:bindings:repeatable {
      $W insert insert $A
      $W yview -pickplace insert
      set j_teb(modified,$W) 1
    }
  }
}

######################################################################
# j:tb:insert_nondigit A W - insert A into text widget W, clear arg flag
######################################################################

proc j:tb:insert_nondigit { A W } {
  global j_teb

  if {"$A" != ""} {
     catch {$W delete sel.first sel.last}
    j:bindings:repeatable {
      $W insert insert $A
      $W yview -pickplace insert
      set j_teb(modified,$W) 1
    }
  }
}


######################################################################
# j:tb:insert_digit A W - insert digit A into W, unless collecting arg
######################################################################

proc j:tb:insert_digit { A W } {
  global j_teb
    
  if $j_teb(prefix) {
    j:bindings:continue_number $A
    return 0
  } else {
    if {"$A" != ""} {
      catch {$W delete sel.first sel.last}
      j:bindings:repeatable {
        $W insert insert $A
        $W yview -pickplace insert
        set j_teb(modified,$W) 1
      }
    }
  }
}

######################################################################
###  TEXT MOUSE COMMANDS
######################################################################

# j:tb:paste_selection W - insert selection into W
proc j:tb:paste_selection { W args } {
  set selection [j:selection_if_any]
  
  if {[string length $selection] != 0} {
    global j_teb
    set j_teb(modified,$W) 1
  
    $W insert insert $selection
    $W yview -pickplace insert
  }
}

######################################################################
# routines to let scanning and pasting to double-up on the same button
# based on code by Tom Phelps <phelps@cs.berkeley.edu>
######################################################################

# j:tb:start_scan_or_paste W x y t - start a drag or paste, recording
#   current location and time so we can later decide whether to paste or drag
# BIND TO <ButtonPress-N>
proc j:tb:start_scan_or_paste { W x y t } {
  global j_teb
  $W scan mark $y
  set j_teb(scanpaste_time) $t
  set j_teb(scanpaste_paste) 1
}

# j:tb:continue_scan W x y t - scan the entry, and mark the fact that
#   we're scanning, not pasting.
# BIND TO <BN-Motion>
proc j:tb:continue_scan { W x y t } {
  global j_teb
  $W scan dragto $y
  set j_teb(scanpaste_paste) 0
}

# j:tb:end_scan_or_paste W x y t - if we haven't been scanning, and it's
#   been less than 500ms since button-down, paste selection
# BIND TO <ButtonRelease-N>
proc j:tb:end_scan_or_paste { W x y t } {
  global j_teb
  if {[expr {$t-$j_teb(scanpaste_time)}] < 500} {
    # j:tb:paste_selection $W
    catch {
      $W insert insert [selection get]
      $W yview -pickplace insert
    }
  }
}

######################################################################
# routines to allow scrolling during text selection
# from raines@cgibm1.slac.stanford.edu (Paul E. Raines)
######################################################################

proc j:tb:start_sel { W x y } {
    global j_teb
    if {[selection own]==$W} {
       catch {set j_teb(cutbuffer) [selection get]
        # puts "j_teb(cutbuffer) = $j_teb(cutbuffer)"
       $W tag remove sel 1.0 end
       j:tb:setup_hidden_selection $W}
    }
    text.press $W $x $y
}

# j:tb:drag_sel W loc - begin dragging out selection
proc j:tb:drag_sel { W loc } {
  global j_teb

  set ypos [lindex [split $loc ","] 1]
  if {$ypos > [winfo height $W]} {
    if {!$j_teb(dragscroll,txnd)} {
      after $j_teb(dragscroll,delay) j:tb:extend_sel $W
    }
    set j_teb(dragscroll,txnd) 1
    set j_teb(dragscroll,direction) down
  } else {
    if {$ypos < 0} {
      if {!$j_teb(dragscroll,txnd)} {
        after $j_teb(dragscroll,delay) j:tb:extend_sel $W
      }
      set j_teb(dragscroll,txnd) 1
      set j_teb(dragscroll,direction) up
    } else {
      set j_teb(dragscroll,txnd) 0
      set j_teb(dragscroll,direction) 0
    }
  }

   if {!$j_teb(dragscroll,txnd)} {
        tk_textSelectTo $W $loc
  }
}

# j:tb:extend_sel W - drag out a selection, scrolling if necessary
proc j:tb:extend_sel { W } {
  global j_teb

  if {$j_teb(dragscroll,txnd)} {
    if {$j_teb(dragscroll,direction) == "down"} {
      tk_textSelectTo $W sel.last+1l
      $W yview -pickplace sel.last+1l
    } else {
      if {$j_teb(dragscroll,direction) == "up"} {
        tk_textSelectTo $W sel.first-1l
        $W yview -pickplace sel.first-1l
      } else { return }
    }
    after $j_teb(dragscroll,delay) j:tb:extend_sel $W
  }
}


# j:tb:end_sel W - finish a selection
proc j:tb:end_sel { W } {
  global j_teb
  set j_teb(dragscroll,txnd) 0
  if {[llength [$W tag ranges sel]]!=0} {
      j:tb:setup_visible_selection $W
  }
}

######################################################################
###  TEXT SCROLLING COMMANDS - fragile - assume widget has a scrollbar
######################################################################
# fragile---assumes first word of yscrollcommand is name of scrollbar!
# should catch case of no yscrollcommand!
# ALSO---should handle arguments (scroll by line rather than windowful)

proc j:tb:extend_sel_with_move { command W } {
    if [catch {$W index sel.first} selfirst_prior] {
        set selfirst_prior [$W index insert]
    }
    if [catch {$W index sel.last} sellast_prior] {
        set sellast_prior [$W index insert]
    }
    set insert_prior [$W index insert]
    j:tb:[set command] $W
    if {[$W index insert] == $insert_prior} { return }
    if {[$W index insert] < $selfirst_prior} {
        $W tag add sel insert $sellast_prior
        return
    }
    if {[$W index insert] > $sellast_prior} {
        $W tag add sel $selfirst_prior insert
        return
    }
    $W tag remove sel $selfirst_prior $sellast_prior
    if {[$W index insert] < $insert_prior} {
        $W tag add sel $selfirst_prior insert
        return
    }
    if {[$W index insert] > $insert_prior} {
        $W tag add sel insert $sellast_prior
        return
    }
}


proc j:tb:unselect_sel_then_move { command W } {
    catch {$W tag remove sel sel.first sel.last}
    j:tb:[set command] $W
}


proc j:tb:scroll_down { W } {
  global j_teb
  j:bindings:clear_count
  
  set yscrollcommand [lindex [$W configure -yscrollcommand] 4]
  set scrollbar [lindex $yscrollcommand 0]      ;# cross fingers and hope!
  
  $W mark set insert "[lindex [$scrollbar get] 3].0"
  $W yview insert
}

proc j:tb:scroll_up { W } {
  global j_teb
  j:bindings:clear_count
  
  set yscrollcommand [lindex [$W configure -yscrollcommand] 4]
  set scrollbar [lindex $yscrollcommand 0]      ;# cross fingers and hope!

  set currentstate [$scrollbar get]
  # following is buggy if lines wrap:
  set newlinepos [expr {[lindex $currentstate 2] - [lindex $currentstate 1]}]
  $W mark set insert "$newlinepos.0-2lines"
  $W yview insert
}


######################################################################
###  TEXT MOVEMENT COMMANDS
######################################################################

# j:tb:bol W - move to start of line (ignores count)
proc j:tb:bol { W } {
  j:bindings:repeatable {$W mark set insert {insert linestart}}
  $W yview -pickplace insert
}

# j:tb:eol W - move to end of line (ignores count)
proc j:tb:eol { W } {
  j:bindings:repeatable {$W mark set insert {insert lineend}}
  $W yview -pickplace insert
}

# j:tb:up W - move up
proc j:tb:up { W } {
  j:bindings:repeatable {$W mark set insert {insert - 1 line}}
  $W yview -pickplace insert
}

# j:tb:down W - move down
proc j:tb:down { W } {
  j:bindings:repeatable {$W mark set insert {insert + 1 line}}
  $W yview -pickplace insert
}

# j:tb:left W - move left
proc j:tb:left { W } {
  j:bindings:repeatable {$W mark set insert {insert - 1 char}}
  $W yview -pickplace insert
}

# j:tb:right W - move right
proc j:tb:right { W } {
  j:bindings:repeatable {$W mark set insert {insert + 1 char}}
  $W yview -pickplace insert
}

# j:tb:bof W - move to beginning of file (widget)
proc j:tb:bof { W } {
  j:bindings:repeatable {
    $W mark set insert 0.0
    $W yview -pickplace 0.0
  }
}

# j:tb:eof W - move to end of file (widget)
proc j:tb:eof { W } {
  j:bindings:repeatable {
    $W mark set insert end
    $W yview -pickplace end
  }
}

# j:tb:word_left W - move back one word
proc j:tb:word_left { W } {
  j:bindings:repeatable {
    while {[$W compare insert != 1.0] &&
           [string match "\[ \t\n\]" [$W get {insert - 1 char}]]} {
      $W mark set insert {insert - 1 char}
    }
    while {[$W compare insert != 1.0] &&
           ![string match "\[ \t\n\]" [$W get {insert - 1 char}]]} {
      $W mark set insert {insert - 1 char}
    }
    $W yview -pickplace insert
  }
}

# j:tb:word_right W - move forward one word
proc j:tb:word_right { W } {
  j:bindings:repeatable {
    while {[$W compare insert != end] &&
           [string match "\[ \t\n\]" [$W get insert]]} {
      $W mark set insert {insert + 1 char}
    }
    while {[$W compare insert != end] &&
           ![string match "\[ \t\n\]" [$W get insert]]} {
      $W mark set insert {insert + 1 char}
    }
    $W yview -pickplace insert
  }
}

######################################################################
###  TEXT DELETION COMMANDS
######################################################################

# j:tb:delete_selection_or_right W - delete character at insert
proc j:tb:delete_selection_or_right { W } {
  if [catch {              
      global j_teb
      set j_teb(modified,$W) 1
  
      set j_teb(cutbuffer) [$W get sel.first sel.last]
      $W delete sel.first sel.last
  }] {
    if [$W compare insert != end] {
      global j_teb
      set j_teb(modified,$W) 1
  
      j:bindings:repeatable {
        $W delete insert
      }
    }
  }
}

# j:tb:delete_selection_or_left W - delete character before insert
proc j:tb:delete_selection_or_left { W } {
  if [catch {              
      global j_teb
      set j_teb(modified,$W) 1
  
      set j_teb(cutbuffer) [$W get sel.first sel.last]
      $W delete sel.first sel.last
  }] {
    if [$W compare insert != 1.0] {
      global j_teb
      set j_teb(modified,$W) 1
  
      j:bindings:repeatable {
          $W delete {insert - 1 char}
          $W yview -pickplace insert
      }
    }
  }
}

#### FOLLOWING TWO SHOULD BE INTEGRATED WITH MOTION COMMANDS.
#### FOLLOWING TWO NEED TO HANDLE CUTBUFFER!

# j:tb:delete_left_word W - move back one word
proc j:tb:delete_left_word { W } {
  j:bindings:repeatable {
    $W mark set del_to insert
    while {[$W compare insert != 1.0] &&
           [string match "\[ \t\n\]" [$W get {insert - 1 char}]]} {
      $W mark set insert {insert - 1 char}
    }
    while {[$W compare insert != 1.0] &&
           ![string match "\[ \t\n\]" [$W get {insert - 1 char}]]} {
      $W mark set insert {insert - 1 char}
    }
    $W delete insert del_to
    $W yview -pickplace insert
  }
}

# j:tb:delete_right_word W - move forward one word
proc j:tb:delete_right_word { W } {
  j:bindings:repeatable {
    $W mark set del_from insert
    while {[$W compare insert != end] &&
           [string match "\[ \t\n\]" [$W get insert]]} {
      $W mark set insert {insert + 1 char}
    }
    while {[$W compare insert != end] &&
           ![string match "\[ \t\n\]" [$W get insert]]} {
      $W mark set insert {insert + 1 char}
    }
    $W delete del_from insert
    $W yview -pickplace insert
  }
}

######################################################################
###  TEXT EMACS DELETION COMMANDS
######################################################################

# j:tb:cut-line_again W - delete insert to end-of-line, appending cutbuffer
proc j:tb:cut-line_again { W } {
  global j_teb
  set j_teb(modified,$W) 1
  
  # if no argument, DON'T cut "\n" unless it's only thing at insert
  if {$j_teb(repeat_count) < 2} {
    j:bindings:clear_count                              ;# in case it's eg -1
    if {[$W index insert] == [$W index {insert lineend}]} then {
      append j_teb(cutbuffer) [$W get insert]
      $W delete insert
    } else {
      append j_teb(cutbuffer) [$W get insert {insert lineend}]
      $W delete insert {insert lineend}
    }
  } else {
    # with argument, cut that many lines (including "\n")
      append j_teb(cutbuffer) [$W get insert {insert lineend + 1 char}]
      $W delete insert {insert lineend + 1 char}
  }
  
  set j_teb(repeat_count) 1
}

# j:tb:cut-line W - delete insert to end-of-line, setting cutbuffer
#   (arg handled by called procedure)
proc j:tb:cut-line { W } {
  global j_teb
  set j_teb(cutbuffer) {}
  j:tb:cut-line_again $W
}

# j:tb:cut_selection W - delete selected region, setting cutbuffer
# and owning selection
proc j:tb:cut_selection { W } {
  global j_teb
  set j_teb(modified,$W) 1

  j:bindings:clear_count

  set j_teb(cutbuffer) {}
  catch {
    set j_teb(cutbuffer) [$W get sel.first sel.last]
    $W delete sel.first sel.last
    j:tb:setup_hidden_selection $W
  }
}


######################################################################
###  TEXT EMACS MARK COMMANDS
######################################################################

# j:tb:set-mark-command W - set emacs mark at current insert point
proc j:tb:set-mark-command { W } {
  $W mark set emacs_mark insert
}

# j:tb:exchange-point-and-mark W - swap insert point and emacs mark
proc j:tb:exchange-point-and-mark { W } {
  if {[lsearch [$W mark names] emacs_mark] != -1} {
    set mark [$W index emacs_mark]
    $W mark set emacs_mark insert
    $W mark set insert $mark
    $W yview -pickplace insert
  } else {
    error "The mark is not set in text widget $W."
  }
}

######################################################################
###  Hidden selection after cutting
######################################################################

proc j:tb:setup_hidden_selection { W } {
    # puts "In :tb:setup_hidden_selection"
    selection own $W "unset j_teb(cutbuffer)"
    selection handle $W j:tb:get_selection STRING
}

proc j:tb:get_hidden_selection { offset maxBytes } {
        global j_teb
        return [string range $j_teb(cutbuffer) $offset [expr $offset+$maxBytes-1]]
}

proc j:tb:setup_visible_selection { W } {
    selection own $W "$W tag remove sel 1.0 end"
    selection handle $W "j:tb:get_visible_selection $W" STRING
}


proc j:tb:get_visible_selection { W offset maxBytes } {
        global j_teb
        set text [$W get sel.first sel.last]
        return [string range $text $offset [expr $offset+$maxBytes-1]]
}


######################################################################
######################################################################
# ROUTINES TO SET UP TEXT BINDINGS
######################################################################
######################################################################

proc j:tb:emacs_bind { W } {
  bind $W <Escape>              { }
  
  # mouse bindings (for scrolling selections)
  bind $W <ButtonPress-1>       {j:tb:start_sel %W %x %y}
  bind $W <B1-Motion>           {j:tb:drag_sel %W @%x,%y}
  bind $W <ButtonRelease-1>     {j:tb:end_sel %W}
  
  # mouse bindings (for scanning and pasting)
  bind $W <2>                   {j:tb:start_scan_or_paste %W %x %y %t}
  bind $W <B2-Motion>           {j:tb:continue_scan %W %x %y %t}
  bind $W <ButtonRelease-2>     {j:tb:end_scan_or_paste %W %x %y %t}

  # Microsft style movement keys
  bind $W <Up>                  {j:tb:unselect_sel_then_move up %W}
  bind $W <Down>                {j:tb:unselect_sel_then_move down %W}
  bind $W <Left>                {j:tb:unselect_sel_then_move left %W}
  bind $W <Right>               {j:tb:unselect_sel_then_move right %W}
  bind $W <Home>                {j:tb:unselect_sel_then_move bol %W}
  bind $W <End>                 {j:tb:unselect_sel_then_move eol %W}
  bind $W <Control-Home>        {j:tb:unselect_sel_then_move bof %W}
  bind $W <Control-End>         {j:tb:unselect_sel_then_move eof %W}
  bind $W <Control-Left>        {j:tb:unselect_sel_then_move word_left %W}
  bind $W <Control-Right>       {j:tb:unselect_sel_then_move word_right %W}
  bind $W <Shift-Up>            {j:tb:extend_sel_with_move up %W}
  bind $W <Shift-Down>          {j:tb:extend_sel_with_move down %W}
  bind $W <Shift-Left>          {j:tb:extend_sel_with_move left %W}
  bind $W <Shift-Right>         {j:tb:extend_sel_with_move right %W}
  bind $W <Shift-Home>          {j:tb:extend_sel_with_move bol %W}
  bind $W <Shift-End>           {j:tb:extend_sel_with_move eol %W}
  bind $W <Shift-Control-Home>  {j:tb:extend_sel_with_move bof %W}
  bind $W <Shift-Control-End>   {j:tb:extend_sel_with_move eof %W}
  bind $W <Shift-Control-Left>  {j:tb:extend_sel_with_move word_left %W}
  bind $W <Shift-Control-Right> {j:tb:extend_sel_with_move word_right %W}

  # Microsoft style cut & paste
  bind $W <Control-Insert>      {j:tb:copy_selection %W}
  bind $W <Shift-Insert>        {j:tb:paste_selection %W}
  bind $W <Shift-Delete>        {j:tb:cut_selection %W}
  
  # scrolling keys
  bind $W <Next>                {j:tb:unselect_sel_then_move scroll_down %W}
  bind $W <Prior>               {j:tb:unselect_sel_then_move scroll_up %W}
  bind $W <Shift-Next>          {j:tb:extend_sel_with_move scroll_down %W}
  bind $W <Shift-Prior>         {j:tb:extend_sel_with_move scroll_up %W}

  # Emacs control and function keys
  bind $W <Control-Key-i>       {j:tb:insert_nondigit "\t" %W}
  bind $W <Control-Key-j>       {j:tb:insert_nondigit "\n" %W}
  
  # Regular deletion keys
  bind $W <Control-Key-h>       {j:tb:delete_selection_or_left %W}
  bind $W <Delete>              {j:tb:delete_selection_or_left %W}
  bind $W <BackSpace>           {j:tb:delete_selection_or_left %W}
  
  # Emacs deletion keys
  bind $W <Any-Mod1-Delete>     {j:tb:delete_left_word %W}
  bind $W <Any-Mod1-Key-d>      {j:tb:delete_right_word %W}
  bind $W <Any-Mod1-Key-b>      {j:tb:unselect_sel_then_move word_left %W}
  bind $W <Any-Mod1-Key-f>      {j:tb:unselect_sel_then_move word_right %W}
  bind $W <Control-d>           {j:tb:delete_selection_or_right %W}

  bind $W <Return>              {j:tb:insert_nondigit "\n" %W}
  bind $W <Any-Key>             {j:tb:insert_nondigit %A %W}
  
  # Emacs argument count
  bind $W <Control-Key-u>               {j:bindings:repeat4}
  foreach d {0 1 2 3 4 5 6 7 8 9} {
    bind $W <Control-Key-u><Key-$d>     {j:bindings:start_number %A}
    bind $W <Escape><Key-$d>            {j:bindings:start_number %A}
    bind $W <Key-$d>                    {j:tb:insert_digit %A %W}
  }
  
  # Emacs movement keys
  bind $W <Control-p>                   {j:tb:unselect_sel_then_move up %W}
  bind $W <Control-n>                   {j:tb:unselect_sel_then_move down %W}
  bind $W <Control-b>                   {j:tb:unselect_sel_then_move left %W}
  bind $W <Control-f>                   {j:tb:unselect_sel_then_move right %W}
  bind $W <Control-a>                   {j:tb:unselect_sel_then_move bol %W}
  bind $W <Control-e>                   {j:tb:unselect_sel_then_move eol %W}
  bind $W <Escape><less>                {j:tb:unselect_sel_then_move bof %W}
  bind $W <Escape><greater>             {j:tb:unselect_sel_then_move eof %W}
  bind $W <Escape><b>                   {j:tb:unselect_sel_then_move word_left %W}
  bind $W <Escape><f>                   {j:tb:unselect_sel_then_move word_right %W}
  
  # Emacs scrolling keys
  bind $W <Control-v>                   {j:tb:unselect_sel_then_move scroll_down %W}
  bind $W <Escape><v>                   {j:tb:unselect_sel_then_move scroll_up %W}

  # Emacs deletion keys
  bind $W <Control-k>                   {j:tb:cut-line %W}
  bind $W <Control-k><Control-k>        {j:tb:cut-line_again %W}
  bind $W <Control-w>                   {j:tb:cut_selection %W}
  bind $W <Control-y>                   {j:tb:paste_selection %W}
  
  # Emacs mark and point
  bind $W <Control-space>               {j:tb:set-mark-command %W}
  bind $W <Control-at>                  {j:tb:set-mark-command %W}
          
  # Emacs keys
  bind $W <Control-x>                   {catch {unset j_teb(C-X-odd)}}
  bind $W <Control-x><Control-x>        {
    if {[lsearch -exact [array names j_teb] C-X-odd] != -1} {
      unset j_teb(C-X-odd)
    } else {
      j:tb:exchange-point-and-mark %W
      set j_teb(C-X-odd) 1
    }
  }
  
  # literal insert
  bind $W <Control-q><Tab> {j:tb:insert_nondigit \t %W}
#  j:tb:literal_bind $W <Control-q>
}

j:bindings:init                         ;# set up global variables

















