#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




# File: buttonAux.tcl
#
# Description:
#       Auxiliary procedures needed to support keyboard traversal
#       of buttons using Tab and Shift Tab.
#
# Global variables used:
#c      button_priv(current,$d)
#               Current button under the mouse for display $d.
#c      button_priv(default,$w)
#               Default button for toplevel window $w.
#c      button_priv(relief,$w)
#               Saved relief of button $w while the button is pressed.


# Procedure:    button_bindForTraversal
#
# Synopsis:
#       Set up a widget or class for the Button feel.
#
# Usage:
#c      button_bindForTraversal arg arg...
#
# Parameters:
#       Any number of window or class names that are to be bound with the
#       Button feel and traversal via <Tab> and <Shift-Tab>
#
# Return value:
#       None.
#
# Description:
#       button_bindForTraversal causes a set of widgets or classes to
#       acquire bindings for keyboard traversal as buttons.
#
#       The bindings are as follows.
#
#       - The <space>, <Return>, and <Enter> keys are bound so that
#       pressing any of them when the button has the keyboard focus
#       has the same effect as pressing the button.
#
#       - The <Tab> and <Shift-Tab> keys are bound to transfer keyboard
#       focus to the next and previous application, respectively.
#
#       - The <F10> and <Alt> keys are bound as for tk_bindForTraversal
#
#       - <Any-Enter>, <Any-Leave>, <1> and <ButtonRelease-1> are
#       rebound to versions that are compatible with keyboard
#       traversal.
#
#       - <FocusIn> and <FocusOut> events are bound to procedures that
#       highlight the focused button.

proc button_bindForTraversal args {
        foreach w $args {
                focus_bindForTraversal $w
                tk_bindForTraversal $w
                bind $w <Any-Enter> "button:enter %W"
                bind $w <Any-Leave> "button:leave %W"
                bind $w <1> "button_press %W"
                bind $w <ButtonRelease-1> "button_release %W"
                bind $w <Key-space> "button_invoke %W -default"
                catch { bind $w <Control-Key-Linefeed> {
                                button_invoke %W -default
                        }}
                catch { bind $w <Control-Key-Return> {
                                button_invoke %W -default
                        }}
                catch { bind $w <Control-Key-KP_Enter> {
                                button_invoke %W -default
                        }}
                catch { bind $w <Control-Key-Enter> {
                                button_invoke %W -default
                        }}
                catch { bind $w <Key-Linefeed> {
                                button_invoke %W -default
                        }}
                catch { bind $w <Key-Return> {
                                button_invoke %W -default
                        }}
                catch { bind $w <Key-KP_Enter> {
                                button_invoke %W -default
                        }}
                catch { bind $w <Key-Enter> {
                                button_invoke %W -default
                        }}
                widget_bind $w GainFocus button:gainFocus
                widget_bind $w LoseFocus button:loseFocus
        }
}

# Procedure:    button:enter
#
# Synopsis:
#       Process mouse entry into a button.
#
# Usage:
#c      button:enter pathName
#
# Parameters:
#c      pathName
#               Path name of a button.
#
# Return value:
#       None.
#
# Description:
#       button:enter handles having the mouse enter a button.  It makes the
#       button's colors `active', providing that the button is not disabled.

proc button:enter w {
        global button_priv
        if {[lindex [$w config -state] 4] != "disabled"} {
                set screen [winfo screen $w]
                set button_priv(current,$screen) $w
                $w config -state active
        }
}

# Procedure:    button:leave
#
# Synopsis:
#       Process mouse exit from a button
#
# Usage:
#c      button:leave pathName
#
# Parameters:
#c      pathName
#               Path name of a button
#
# Return value:
#       None.
#
# Description:
#       button:leave handles having the mouse leave a button.  It makes
#       the button's state `normal' if it wasn't `disabled'.

proc button:leave w {
        global button_priv
        if {[lindex [$w config -state] 4] != "disabled"} {
                $w config -state normal
        }
        set screen [winfo screen $w]
        catch {unset button_priv(current,$screen)}
}

# Procedure:    button:gainFocus
#
# Synopsis:
#       Process keyboard focus into a button.
#
# Usage:
#c      button:gainFocus pathName
#
# Parameters:
#c      pathName
#               Path name of a button
#
# Return value:
#       None.
#
# Description:
#       button:gainFocus is called when a button receives the keyboard focus.
#       If the button is `focusable', it is marked to have the focus by
#       changing its frame's background to equal its own active background.

proc button:gainFocus w {
            $w config -background [lindex [$w config -activebackground] 4]
#set frame [winfo parent $w]
#       if {[winfo class $frame] == "Focusable"} {
#               set abg [lindex [$w configure -activebackground] 4]
#               if {$abg != [lindex [$frame configure -background] 4]} {
#                       $frame configure -background $abg
#               }
#       }
}

# Procedure:    button:loseFocus
#
# Synopsis:
#       Process keyboard focus out of a button.
#
# Usage:
#c      button:loseFocus pathName
#
# Parameters:
#c      pathName
#               Path name of a button
#
# Return value:
#       None.
#
# Description:
#       button:loseFocus is called when a button loses the keyboard focus.
#       If the button has the `focusable' look, its frame changes to have
#       the `inactive' colours.

proc button:loseFocus w {
            $w config -background [option get $w background Background]
#       if {[winfo exists $w] && [info commands $w] == $w} {
#           $w config -relief raised
#       }
#       if {[winfo exists $w] && [info commands $w] == $w} {
#               set frame [winfo parent $w]
#               if {[winfo class $frame] == "Focusable"} {
#                       set ibg [lindex [$w configure -background] 4]
#                       if {$ibg != [lindex \
#                                       [$frame configure -background] \
#                                       4]} {
#                               $frame configure -background $ibg
#                       }
#               }
#       }
}

# Procedure:    button_invoke
#
# Synopsis:
#       Process keyboard event to invoke a button.
#
# Usage:
#c      button_invoke pathName ?-default?
#
# Parameters:
#c      pathName
#               Path name of a button
#
# Options:
#       -default
#               Specifies that the button should become the default for its
#               top-level window.
#
# Return value:
#       None.
#
# Description:
#       button_invoke handles the case where the user invokes a button
#       by means of a key press.  It flashes the button if it's a push button,
#       then invokes the button as if the mouse had been pressed and released
#       over the button.  It also makes the button the default for its
#       toplevel window, if requested

proc button_invoke {w {default {-nodefault}}} {
        global button_priv
        if {$default == [string range "-default" 0 \
                                [expr [string length $default]-1]]} {
                button_makeDefault $w
        }
        if {[winfo class $w] == "Button"} {
                set screen [winfo screen $w]
                if {![info exists button_priv(current,$screen)] 
                    || $button_priv(current,$screen) != $w} {
                        uplevel #0 [list $w flash]
                }
        }
        uplevel #0 [list $w invoke]
}

# Procedure:    button_press
#
# Synopsis:
#       Process mouse press within a button.
#
# Usage:
#c      button_press pathName
#
# Parameters:
#c      pathName
#               Path name of a button
#
# Return value:
#       None
#
# Description:
#       button_press handles having the user press a button.  It checks
#       it the button is disabled.  If it isn't, it makes the button's relief
#       `sunken' (to simulate pressing it), and transfers the focus to it.

proc button_press w {
        global button_priv
        global button_strictMotif
        if {[lindex [$w config -state] 4] != "disabled"} {
                set button_priv(relief,$w) [lindex [$w config -relief] 4]
                $w config -relief sunken
                if {[info exists button_strictMotif]} {
                        focus_goTo $w
                }
        }
}

# Procedure:    button_release
#
# Synopsis:
#       Process mouse release within a button.
#
# Usage:
#c      button_release pathName
#
# Parameters:
#c      pathName
#               Path name of a button.
#
# Return value:
#       None.
#
# Description:
#       button_release handles having the user release the mouse over a
#       button.  It checks that the button is still the `current window'
#       (defined by button_priv(current)), and invokes it if it is.

proc button_release w {
        global button_priv
        if {[info exists button_priv(relief,$w)]} {
                $w config -relief $button_priv(relief,$w)
                unset button_priv(relief,$w)
        }
        set screen [winfo screen $w]
        if {[info exists button_priv(current,$screen)]} {
                if {$w == $button_priv(current,$screen)} {
                        button_invoke $w
                }
        }
}

# Procedure:    button_makeDefault
#
# Synopsis:
#       Designate the `default button' for a top-level window.
#
# Usage:
#c      button_makeDefault pathName
#
# Parameters:
#c      pathName
#               Path name of a button.
#
# Return value:
#       None.
#
# Description:
#       button_makeDefault makes a button the `default button' for its
#       top-level window.

proc button_makeDefault w {
        global button_priv
        if {[lindex [$w configure -state] 4] == "disabled"} return
        set t [winfo toplevel $w]
        if [info exists button_priv(default,$t)] {
                set d $button_priv(default,$t)
                if {$d != $w} {
                        if {[winfo exists $d] \
                            && [info commands $d] == $d} {
                                set frame [winfo parent $d]
                                if {[winfo class $frame] == "Focusable"} {
                                        $frame config -relief flat
                                }
                        }
                }
        }
        set button_priv(default,$t) $w
        widget_addBinding $t Destroy "catch \"unset button_priv(default,$t)\""
        set frame [winfo parent $w]
        if {[winfo class $frame] == "Focusable"} {
                $frame config -relief sunken
        }
}

# Procedure:    button_invokeDefault
#
# Synopsis:
#       Invoke the default button for a widget's top-level window
#
# Usage:
#c      button_invokeDefault pathName
#
# Parameters:
#c      pathName
#               Path name of any window
#
# Return value:
#       None.
#
# Description:
#       button_invokeDefault invokes the `default button' for any widget
#       in an application.

proc button_invokeDefault w {
        global button_priv
        set t [winfo toplevel $w]
        if {[info exists button_priv(default,$t)]} {
                set b $button_priv(default,$t)
                if {![winfo exists $b] \
                    || [info commands $b] != $b \
                    || [lindex [$b config -state] 4] == "disabled"} {
                        unset b
                }
        }
        if {![info exists b]} {
                set b [button:findDefault $t]
        }
        if {$b == ""} {
                error "No command to invoke"
        }
        button_invoke $b
}

# Procedure:    button:findDefault
#
# Synopsis:
#       Establish a default button for a top-level window where none
#       has been specified.
#
# Usage:
#c      button:findDefault pathName
#
# Parameters:
#c      pathName
#               Path name of a window
#
# Return value:
#       Name of the first non-disabled button among the window's children
#
# Description:
#       button:findDefault is used to locate a default button where none
#       has been defined.

proc button:findDefault w {
        global tk_version
        case [winfo class $w] in {
                {Button Checkbutton Radiobutton} {
                        if {[lindex [$w config -state] 4] != "disabled"} {
                                return $w
                        }
                }
        }
        set kids [winfo children $w]
        if {$tk_version >= 3.3} {
                set init {set i 0}
                set cond {$i < [llength $kids]}
                set reinit {incr i}
        } else {
                set init {set i [expr [llength $kids]-1]}
                set cond {$i >= 0}
                set reinit {incr i -1}
        }
        for $init $cond $reinit {
                set kid [lindex $kids $i]
                if {$kid != [winfo toplevel $kid]} {
                        set d [button:findDefault $kid]
                        if {$d != ""} {
                                return $d
                        }
                }
        }
        return ""
}

# Procedure: button_setDefault
#
# Synopsis:
#       Find the default button of a top-level window, and return its
#       identity.  Establish a default if none has been specified.
#
# Usage:
#c      button_setDefault pathName
#
# Parameters:
#c      pathName
#               Path name of a window, generally a toplevel.
#
# Return value:
#       Name of the default button of the window
#
# Description:
#       button_setDefault finds the default button of a window, and
#       returns its identity.  If the window has no default defined,
#       a suitable button is located and made the default.

proc button_setDefault w {
        global button_priv
        if [info exists button_priv(default,[winfo toplevel $w])] {
                return $button_priv(default,[winfo toplevel $w])
        }
        set db [button:findDefault $w]
        if {$db != ""} {
                button_makeDefault $db
        }
        return $db
}

# Procedure: focusable
#
# Synopsis:
#       Display the Motif `focus frame' around a button, checkbutton,
#       or radiobutton.  The `focus frame' indicates the current position
#       of the keyboard focus and the current default button for the
#       application.
#
# Usage:
#c      focusable buttonCommand pathName ?args?
#
# Parameters:
#c      buttonCommand
#               one of `button,' `checkbutton,' or `radiobutton.'
#c      pathName
#               Path name of the `focus frame' widget to be created
#c      args
#               Arguments for the button command
#
# Return value:
#       Path name of the `focus frame' widget
#
# Description:
#       Motif-style button actions require a frame that decorates each
#       button to indicate the default action and the position of the
#       keyboard focus.  Preceding the command that creates the button
#       with the `focusable' command greates this indication.

option add *Focusable.borderWidth 2 widgetDefault

proc focusable {type w args} {
        global button_priv
        global focusable_priv
        if [winfo exists $w] { error "$w already exists"; return }
        frame $w -class Focusable
        pack append $w \
                [frame $w.strut1 -geom 0x1] {top} \
                [frame $w.strut2 -geom 0x1] {bottom} \
                [frame $w.strut3 -geom 1x0] {left} \
                [frame $w.strut4 -geom 1x0] {right} \
                [uplevel 1 [list $type $w.b] $args] \
                        {expand fill}
        widget_addBinding $w.b Destroy "
                after 1 catch \"widget_checkAndDestroy $w\"
        "
        $w config -background [lindex [$w.b config -background] 4]
        rename $w [set w]_renamed
        proc $w { args } "return \[eval [list focusable:command $w] \$args\]"
        widget_addBinding $w Destroy "rename [set w]_renamed {}"
        set focusable_priv($w,type) $type
        set t [winfo toplevel $w]
        if {[info exists button_priv(default,$t)] \
            && "$w.b" == $button_priv(default,$t)} {
                button_makeDefault $w.b
        }
        return $w
}


proc focusable:command { w command args } {
        global focusable_priv
        switch -- $command $focusable_priv($w,type) {
            if {[llength $args]==0} { 
                return $w.b
            } else {
                eval [list $w.b] $args
            }
        } default {
            eval [list [set w]_renamed $command] $args
        }
}

# Procedure: sensitive
#
# Synopsis:
#       Make a button's state sensitive to changes in a variable.
#
# Usage:
#c      sensitive ?command/args?
#
# Parameters:
#c      command/args
#               together forms a command which will create a button.
#               Any arguments of the form -sensitivevar <var> and
#               -sensitiveexpr <var> are stripped out before the
#               command is called.
#
# Return value:
#       Path name of the button
#
# Description:
#
# Arguments:
#       -sensitivevar
#       -sensitiveexpr
#               A global Tcl variable for the button to be sensitive to.
#               When the variable changes value, <sensitiveexpr> is
#               reevaluated and should return 1 (for an active button)
#               or 0 (for a disabled button).

proc sensitive {args} {
        set argc [llength $args]
        set passon_args ""
        for {set i 0} {$i < $argc} {incr i} {
            switch -- [lindex $args $i] -sensitivevar {
                incr i
                lappend sensitivevars [lindex $args $i]
            } -sensitivevars {
                incr i
                eval lappend sensitivevars [lindex $args $i]
            } -sensitiveexpr {
                incr i
                set sensitiveexpr [lindex $args $i]
            } default {
                lappend passon_args [lindex $args $i]
            }
        }
        set w [eval $passon_args]
        eval [list make_sensitive $w] $args
}

proc make_sensitive {w args} {
        set argc [llength $args]
        set passon_args ""
        for {set i 0} {$i < $argc} {incr i} {
            switch -- [lindex $args $i] -sensitivevar {
                incr i
                lappend sensitivevars [lindex $args $i]
            } -sensitivevars {
                incr i
                eval lappend sensitivevars [lindex $args $i]
            } -menuentry {
                incr i
                set menuentry [lindex $args $i]
            } -sensitiveexpr {
                incr i
                set sensitiveexpr [lindex $args $i]
            } default {
                lappend passon_args [lindex $args $i]
            }
        }
        # puts "passon_args = $passon_args"
        if ![info exists sensitivevars] { error "sensitive vars must be specified" }
        if ![info exists sensitiveexpr] { set sensitiveexpr \$[lindex $sensitivevars 0] }
        if ![info exists menuentry] { set menuentry "" }

        foreach sensitivevar $sensitivevars {
            upvar #0 $sensitivevar sensitive
            set command [list sensitive:change $w $sensitiveexpr $menuentry]
            trace variable sensitive w $command
            widget_addBinding $w Destroy "trace vdelete $sensitivevar w [list $command]"
        }
        if {$menuentry == ""} { 
            set config config 
        } else {
            set config [list entryconfig $menuentry]
        }
        if [uplevel #0 "expr $sensitiveexpr"] {
            if [catch {eval $w.b $config -state normal}] {
                eval $w $config -state normal
            }
        } else { 
            if [catch {eval $w.b $config -state disabled}] {
                eval $w $config -state disabled
            }
        }
        return $w
}


proc sensitive:change { w sensitiveexpr menuentry args } {
        global errorInfo
        set status [catch {uplevel #0 "expr $sensitiveexpr"} res]
        if $status { puts "warning (sensitive:change) : $res, $errorInfo"; return}
        if {$menuentry == ""} { 
            set config config
        } else {
            set config [list entryconfig $menuentry]
        }
        if $res {
            if [catch {eval $w $config -state normal}] {
                eval $w.b $config -state normal
            }
        } else { 
            if [catch {eval $w $config -state disabled}] {
                eval $w.b $config -state disabled
            }
        }
}    
    
    
