#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




#----------------------------------------------------------------------------

option add *Heirarchy.font "-Adobe-Helvetica-Bold-R-Normal--*-100-*" widgetDefault
option add *Heirarchy.textFill black widgetDefault
option add *Heirarchy.selectTextFill black widgetDefault
option add *Heirarchy.bitmapForeground black widgetDefault
option add *Heirarchy.selectBitmapForeground black widgetDefault
option add *Heirarchy.selectBackground #b2dfee widgetDefault
option add *Heirarchy.itemBackground #ffe4c4 widgetDefault
option add *Heirarchy.command "" widgetDefault
option add *Heirarchy.nodechildren hierarchy::dircontents widgetDefault
option add *Heirarchy.nodelook hierarchy::filelook widgetDefault
option add *Heirarchy.rootnode "/." widgetDefault
option add *Heirarchy.rootanchor nw widgetDefault
option add *Heirarchy.paddepth 20 widgetDefault
option add *Heirarchy.padstack 6 widgetDefault
option add *Heirarchy.padtextbitmap 3 widgetDefault
option add *Heirarchy.expand 1 widgetDefault
option add *Heirarchy.debug 0 widgetDefault
option add *Heirarchy.multiselect 1 widgetDefault
option add *Heirarchy.commonselect 0 widgetDefault

proc hierarchy { w args} {
        global TkHolShell_flags
        global vals

        frame $w -class Heirarchy

        # Look up the default values for the widget options

        set vals($w,font) [option get $w font Font]
        set vals($w,command) [option get $w command Command]
        set vals($w,rootnode) [option get $w rootnode RootNode]
        set vals($w,rootanchor) [option get $w rootanchor RootAnchor]
        set vals($w,nodechildren) [option get $w nodechildren NodeRetriever]
        set vals($w,nodelook) [option get $w nodelook NodeLook]
        set vals($w,expand) [option get $w expand Expand]
        set vals($w,paddepth) [option get $w paddepth PadDepth]
        set vals($w,textFill) [option get $w textFill TextFill]
        set vals($w,selectTextFill) [option get $w selectTextFill SelectTextFill]
        set vals($w,bitmapForeground) [option get $w bitmapForeground BitmpForeground]
        set vals($w,selectBitmapForeground) [option get $w selectBitmapForeground SelectBitmapForeground]
        set vals($w,itemBackground) [option get $w itemBackground ItemBackground]
        set vals($w,selectBackground) [option get $w selectBackground SelectBackground]
        set vals($w,padstack) [option get $w padstack PadStack]
        set vals($w,padtextbitmap) [option get $w padtextbitmap PadTextBitmap]
        set vals($w,debug) [option get $w debug Debug]
        set vals($w,multiselect) [option get $w multiselect MultiSelect]
        set vals($w,commonselect) [option get $w commonselect CommonSelect]

        set vals($w,selection) ""
        set vals($w,bindings) ""

        canvas $w.canv
        pack $w.canv -fill both -expand yes
        eval [list hierarchy::configure $w] $args

        hierarchy::bindnodes $w <Double-Button-1> "hierarchy::togglenode $w %n"
        hierarchy::bindnodes $w <Button-2> "hierarchy::togglenode $w %n"

        bind $w <Destroy> "hierarchy::upon_destroy $w"
        return $w
}



#------------------------------------------------------------
# PUBLIC hierarchy::configure
#
# Configure the widget.  Does not respond well to *all* configuration
# options.
#
#------------------------------------------------------------


proc hierarchy::configure { w args} {
        global vals
        set passon_args ""
        set newvals ""
        set redraw 0
        set remake 0
        for {set i 0} {$i<[llength $args]} {incr i} {
            case [lindex $args $i]  -font {
                incr i
                set redraw 1
                lappend newvals [list font [lindex $args $i]]
            } -selectTextFill {
                incr i
                set redraw 1
                lappend newvals [list selectTextFill [lindex $args $i]]
            } -bitmapForeground    {
                incr i
                set redraw 1
                lappend newvals [list bitmapForeground [lindex $args $i]]
            } -selectBitmapForeground {
                incr i
                set redraw 1
                lappend newvals [list selectBitmapForeground [lindex $args $i]]
            } -itemBackground {
                incr i
                set redraw 1
                lappend newvals [list itemBackground [lindex $args $i]]
            } -selectBackground {
                incr i
                set redraw 1
                lappend newvals [list selectBackground [lindex $args $i]]
            } -padstack {
                incr i
                set redraw 1
                lappend newvals [list padstack [lindex $args $i]]
            } -expand {
                incr i
                set redraw 1
                lappend newvals [list expand [lindex $args $i]]
            } -textFill {
                incr i
                set redraw 1
                lappend newvals [list textFill [lindex $args $i]]
            } -command {
                incr i
                set redraw 1
                set remake 1
                lappend newvals [list command [lindex $args $i]]
            } -rootnode {
                set redraw 1
                set remake 1
                incr i
                lappend newvals [list rootnode [lindex $args $i]]
            } -rootanchor {
                incr i
                set redraw 1
                lappend newvals [list rootanchor [lindex $args $i]]
            } -nodechildren {
                incr i
                set redraw 1
                set remake 1
                lappend newvals [list nodechildren [lindex $args $i]]
            } -nodelook {
                incr i
                set redraw 1
                set remake 1
                lappend newvals [list nodelook [lindex $args $i]]
            } -debug {
                incr i
                lappend newvals [list debug [lindex $args $i]]
            } -paddepth {
                incr i
                set redraw 1
                lappend newvals [list paddepth [lindex $args $i]]
            } -padtextbitmap {
                incr i
                set redraw 1
                lappend newvals [list padtextbitmap [lindex $args $i]]
            } -multiselect {
                incr i
                lappend newvals [list multiselect [lindex $args $i]]
            } -commonselect {
                incr i
                set redraw 1
                lappend newvals [list commonselect [lindex $args $i]]
            } default { 
                set arg1 [lindex $args $i]
                incr i
                set arg2 [lindex $args $i]
                lappend passon_args $arg1 $arg2
            }
        }
        if {$redraw || $remake} {
            $w.canv delete all
        }
        if {$remake} {

            # remove all "$w,*,select" node selection values in "vals"
            # This is crucial or else the traces remain.

            set searchid [array startsearch vals]
            set toremove ""
            while {[array anymore vals $searchid]} {
                set elem [array nextelement vals $searchid]
                if [string match $w,*,select $elem] {
                    lappend toremove $elem
                }
                if [string match $w,*,traced $elem] {
                    lappend toremove $elem
                }
            }
            array donesearch vals $searchid
            foreach elem $toremove {
                unset vals($elem)
            }
        }
        eval [list $w.canv config] $passon_args
        foreach newval $newvals {
            set vals($w,[lindex $newval 0]) [lindex $newval 1]
        }
        if {$redraw && !$remake} {
            hierarchy::redraw_after_node_change $w $vals($w,rootnode)
        }
        if $remake {
            set vals($w,$vals($w,rootnode),showkids) 0
            case $vals($w,expand) all {
                hierarchy::expandnodeall $w $vals($w,rootnode)
            } default {
                hierarchy::expandnoden $w $vals($w,rootnode) $vals($w,expand)
            }
            if [info exists vals($w,configured)] {
                hierarchy::choose_initial_viewport $w
            }
        }

        if {$vals($w,command)==""} {
            hierarchy::bindnodes $w <Button-1> "hierarchy::setselection $w %n"
            if $vals($w,multiselect) {
                hierarchy::bindnodes $w <Shift-Button-1> "hierarchy::toggleselection $w %n"
                hierarchy::bindnodes $w <Control-Button-1> "hierarchy::toggleselection $w %n"
            }
        } else {
            hierarchy::bindnodes $w <Button-1> "
                hierarchy::setselection $w %n
                $vals($w,command) \[hierarchy::selection $w\]"
            if $vals($w,multiselect) {
                hierarchy::bindnodes $w <Shift-Button-1> "hierarchy::toggleselection $w %n ; $vals($w,command) \[hierarchy::selection $w\]"
                hierarchy::bindnodes $w <Control-Button-1> "hierarchy::toggleselection $w %n ; $vals($w,command) \[hierarchy::selection $w\]"
            }
        }
        bind $w.canv <Configure> "hierarchy::configure_notify $w"

}

#------------------------------------------------------------
# PUBLIC hierarchy::expandnodeall
#
# Expand all nodes from the given node then redraw, pruning at the
# any nodes given as arguments.
#
#------------------------------------------------------------

proc hierarchy::expandnodeall { w node_path args } {
        global vals
        eval [list hierarchy::expandnoden $w $node_path 99999] $args
}
                                  
                                       
#------------------------------------------------------------
# PUBLIC hierarchy::collapsenodeall
#
# Collapse the given node and all children then redraw.
#
#------------------------------------------------------------

proc hierarchy::collapsenodeall { w node_path } {
        if [hierarchy::collapsenode $w $node_path] {
            hierarchy::redraw_after_node_change $w $node_path
            hierarchy::discard_children $w $node_path
        }
}
                                  
#------------------------------------------------------------
# PUBLIC hierarchy::expandnoden
#
# Expand the given node to the given depth.  The following arguments
# are recognized AT THE END of the arguments supplied.
#       -prune <list of node names/paths>
#               Prune the expansion at the given nodes.  The nodes
#               should be specified by names if -commonselect is true,
#               and by node paths if commonselect is false.
#------------------------------------------------------------

proc hierarchy::expandnoden { w node_path n args } {
        global vals
        if {$n<=0} { return 0 }
        global busy
        incr busy
        global feedback
        set feedback [list $w "Adjusting tree..."]
        set prune ""
        for {set i 0} {$i<[llength $args]} {incr i} {
            case [lindex $args $i]  -prune {
                incr i
                set redraw 1
                set prune [lindex $args $i]
            } default { 
                error "unrecognized argument [lindex $args $i]"
            }
        }
        if {[hierarchy::expandnoden_aux $w $node_path $n $prune]} {
            hierarchy::redraw_after_node_change $w $node_path
        }
        set feedback [list $w "Please wait..."]
        incr busy -1
}

                                  
proc hierarchy::expandnoden_aux { w node_path n prune } {
        global vals
        if $vals($w,commonselect) {
            set nodename [lindex $node_path [expr [llength $node_path]-1]]
        } else {
            set nodename $node_path
        }
#       puts "prune = $prune, nodename = $nodename, n = $n, noe_path = $node_path"
        if {[lsearch $prune $nodename]!=-1} { return 0 }
        set returnval [hierarchy::expandnode $w $node_path]
        if {$returnval==0 && $n==1} { return 0 }
        if {$n==1} { return 1 }
        incr n -1
#       puts "doing kids for node_path = $node_path"
        foreach kid $vals($w,$node_path,kids) {
            set kid_node_path $node_path                   
            lappend kid_node_path $kid
            hierarchy::expandnoden_aux $w $kid_node_path $n $prune
        }
        return 1
}


#------------------------------------------------------------
# Utility routines to collapse and expand a single node
# without redrawing.
#
# Nb. doesn't reposition
#
# Routines often return 0/1 to indicate if any change
# has occurred in the tree.
#------------------------------------------------------------


proc hierarchy::expandnode { w node_path } {
        global vals
        if {$vals($w,$node_path,showkids)} { return 0 }
        set vals($w,$node_path,showkids) 1
        set vals($w,$node_path,kids) [eval $vals($w,nodechildren) [list $node_path]]
        if {[llength $vals($w,$node_path,kids)]==0} { 
            if ![info exists vals($w,$node_path,look)] {
                set vals($w,$node_path,look) [eval $vals($w,nodelook) [list $node_path] 0]
                hierarchy::setup_select_variable $w $node_path
            }
            return 0 
        }
        set vals($w,$node_path,look) [eval $vals($w,nodelook) [list $node_path] 1]
        hierarchy::setup_select_variable $w $node_path
        
        foreach kid $vals($w,$node_path,kids) {
            set kid_node_path $node_path                   
            lappend kid_node_path $kid
            set vals($w,$kid_node_path,look) [eval $vals($w,nodelook) [list $kid_node_path] 0]
            hierarchy::setup_select_variable $w $kid_node_path
            set vals($w,$kid_node_path,showkids) 0
        }
        return 1
}

proc hierarchy::collapsenode { w node_path } {
        global vals
        if {!$vals($w,$node_path,showkids)} { return 0 }
        set vals($w,$node_path,showkids) 0
        if {[llength $vals($w,$node_path,kids)]==0} { return 0}
        set vals($w,$node_path,look) [eval $vals($w,nodelook) [list $node_path] 0]
        foreach kid $vals($w,$node_path,kids) {
            set kid_node_path $node_path                   
            lappend kid_node_path $kid
            hierarchy::collapsenode $w $kid_node_path
        }
        return 1
}


proc hierarchy::discardnode { w node_path } {
        global vals
        if $vals($w,commonselect) {
            set varname [lindex $node_path [expr [llength $node_path]-1]]
        } else {
            set varname $node_path
        }
        trace vdelete vals($w,$varname,select) w "hierarchy::select_change $w [list $node_path]"
        if {[llength [trace vinfo vals($w,$varname,select)]]==0} {
            catch {unset vals($w,$varname,select)}
        }
        catch {$w.canv delete bitmap:$node_path}
        catch {$w.canv delete text:$node_path}
        catch {$w.canv delete rect:$node_path}
        catch {unset vals($w,$node_path,showkids)}
        catch {unset vals($w,$node_path,look)}
        catch {unset vals($w,$node_path,usages)}
        catch {unset vals($w,$node_path,my_depth_usage)}
        catch {unset vals($w,$node_path,traced)}
}


proc hierarchy::discard_children { w node_path } {
        global vals
        if [info exists vals($w,$node_path,kids)] {
            foreach kid $vals($w,$node_path,kids) {
                set kid_node_path $node_path               
                lappend kid_node_path $kid
                hierarchy::discardnode $w $kid_node_path
                hierarchy::discard_children $w $kid_node_path
            }
            unset vals($w,$node_path,kids) 
        }
}

                                  
#------------------------------------------------------------
# hierarchy::togglenode
#
# Toggle the given node then redraw.
#
#------------------------------------------------------------

proc hierarchy::togglenode { w node_path } {
        global vals
        if $vals($w,$node_path,showkids) {
            hierarchy::collapsenodeall $w $node_path
        } else {
            hierarchy::expandnode1 $w $node_path
        }
}


#------------------------------------------------------------
# PUBLIC hierarchy::setselection
#
#
#------------------------------------------------------------

proc hierarchy::setselection { w args} {
        global vals
        foreach varname $vals($w,selection) {
            set vals($w,$varname,select) 0
        }
        set vals($w,selection) ""
        foreach node_path $args {
            if $vals($w,commonselect) {
                set varname [lindex $node_path [expr [llength $node_path]-1]]
            } else {
                set varname $node_path
            }
            if {[info exists vals($w,$varname,select)]} {
                if {!$vals($w,$varname,select)} {
                    set vals($w,$varname,select) 1
                }
            }
            lappend vals($w,selection) $varname
        }
}

#------------------------------------------------------------
# PUBLIC hierarchy::addtoselection
#
#
#------------------------------------------------------------

proc hierarchy::addtoselection { w args} {
        global vals
        if $vals($w,debug) { puts "before addtoselection args = $args, vals($w,selection) = vals($w,selection)" }
        foreach node_path $args {
            if $vals($w,commonselect) {
                set varname [lindex $node_path [expr [llength $node_path]-1]]
            } else {
                set varname $node_path
            }
            if {!$vals($w,$varname,select)} {
                set vals($w,$varname,select) 1
                lappend vals($w,selection) $varname
            }
        }
        if $vals($w,debug) { puts "after addtoselection, vals($w,selection) = vals($w,selection)" }
}

#------------------------------------------------------------
# PUBLIC hierarchy::removefromselection
#
#
#------------------------------------------------------------

proc hierarchy::removefromselection { w args} {
        global vals
        if $vals($w,debug) { puts "before removefromselection args = $args, vals($w,selection) = vals($w,selection)" }
        foreach node_path $args {
            if $vals($w,commonselect) {
                set varname [lindex $node_path [expr [llength $node_path]-1]]
            } else {
                set varname $node_path
            }
            if {$vals($w,$varname,select)} {
                set vals($w,$varname,select) 0
                set index [lsearch $vals($w,selection) $varname]
                set vals($w,selection) [lreplace $vals($w,selection) $index $index]
            }
        }
        if $vals($w,debug) { puts "after removefromselection, vals($w,selection) = vals($w,selection)" }
}


#------------------------------------------------------------
# PUBLIC hierarchy::toggleselection
#
#
#------------------------------------------------------------

proc hierarchy::toggleselection { w args} {
        global vals
        foreach node_path $args {
            if $vals($w,commonselect) {
                set varname [lindex $node_path [expr [llength $node_path]-1]]
            } else {
                set varname $node_path
            }
            if {$vals($w,$varname,select)} {
                hierarchy::removefromselection $w $node_path
            } else {
                hierarchy::addtoselection $w $node_path
            }
        }
}


#------------------------------------------------------------
# PUBLIC hierarchy::selection
#
#------------------------------------------------------------

proc hierarchy::selection { w } {
        global vals
        return $vals($w,selection)
}

#------------------------------------------------------------
# PUBLIC hierarchy::bindnodes
#
#------------------------------------------------------------

proc hierarchy::bindnodes { w event command } {
        global vals
        hierarchy::bindnode_aux $w $vals($w,rootnode) $event $command
        lappend vals($w,bindings) [list $event $command]
}

proc hierarchy::bindnode_aux { w node_path event command } {
        global vals

        # this is dodgy - see regsub subspec info.

        regsub %n $command [list $node_path] subst_command
        if $vals($w,debug) { puts "command = $command, node_path = $node_path, subst_command = $subst_command" }
#       regsub %W $subst_command $w subst_command
        catch {$w.canv bind text:$node_path $event $subst_command}
        catch {$w.canv bind bitmap:$node_path $event $subst_command}
        if $vals($w,$node_path,showkids) {
            foreach kid $vals($w,$node_path,kids) {
                set kid_node_path $node_path               
                lappend kid_node_path $kid
                hierarchy::bindnode_aux $w $kid_node_path $event $command
            }
        }
}



proc hierarchy::bindelem { w node_path elem } {
        global vals
        foreach binding $vals($w,bindings) {
            regsub %n [lindex $binding 1] [list $node_path] subst_command
            $w.canv bind [set elem]:$node_path [lindex $binding 0] $subst_command
        }
}


#------------------------------------------------------------
# UTILITY ROUTINES
#
# The remainder of these functions are utility functions
# only.
#------------------------------------------------------------

#------------------------------------------------------------
# hierarchy::upon_destroy
#
# Called just when the "hierarchy" widget is destroyed.
#------------------------------------------------------------

proc hierarchy::upon_destroy { w } {
        global vals
        hierarchy::collapsenode $w $vals($w,rootnode)
        hierarchy::discard_children $w $vals($w,rootnode)
        hierarchy::discardnode $w $vals($w,rootnode)

        # remove all values in "vals" indexed by $w
        set searchid [array startsearch vals]
        set toremove ""
        while {[array anymore vals $searchid]} {
            set elem [array nextelement vals $searchid]
            if [string match $w,* $elem] {
                lappend toremove $elem
            }
        }
        array donesearch vals $searchid
        foreach elem $toremove {
           unset vals($elem)
        }
}

#------------------------------------------------------------
#
#------------------------------------------------------------
proc hierarchy::setup_select_variable { w node_path } {
        global vals
        if $vals($w,commonselect) {
            set varname [lindex $node_path [expr [llength $node_path]-1]]
        } else {
            set varname $node_path
        }
        if ![info exists vals($w,$varname,select)] {
            set vals($w,$varname,select) 0
        }
        if ![info exists vals($w,$node_path,traced)] {
            set vals($w,$node_path,traced) 1
            trace variable vals($w,$varname,select) w "hierarchy::select_change $w [list $node_path]"
        }
}

proc hierarchy::select_change { w node_path arg1 arg2 op } {
        global vals
        hierarchy::adjust_look $w $node_path
        hierarchy::remake_selection_box $w $node_path
}


                                       
#------------------------------------------------------------
#
#
#------------------------------------------------------------

proc hierarchy::reevaluatenode { w node_path } {
        global vals
        hierarchy::expandnode $w $node_path
        if {$n==1} return
        if $vals($w,$node_path,showkids) {
            foreach kid $vals($w,$node_path,kids) {
                set kid_node_path $node_path               
                lappend kid_node_path $kid
                hierarchy::reevaluatenode $w $kid_node_path
            }
        }
}
                                       


#------------------------------------------------------------
#
#
#------------------------------------------------------------

proc hierarchy::expandnode1 { w node_path } {
        global vals
        hierarchy::expandnoden $w $node_path 1
}



                                  
                                  
#------------------------------------------------------------
# Reevaluate all expanded theories and redraw everything.
# Useful if something in the tree has changed.
#
#------------------------------------------------------------

proc hierarchy::complete_redraw { w } {
        global vals
        hierarchy::reevaluateenode $w $vals($w,rootnode)
        hierarchy::redraw_after_node_change $w $node_path
}


proc hierarchy::remake_selection_box { w node_path } {
        global vals
        if ![info exists vals($w,configured)] return
        catch {$w.canv delete rect:$node_path}
        if $vals($w,commonselect) {
            set varname [lindex $node_path [expr [llength $node_path]-1]]
        } else {
            set varname $node_path
        }
        if {$vals($w,$varname,select)} {
            set textBackground $vals($w,selectBackground)
            # remake the selection box for the text if it has one
            set text_dimensions [$w.canv bbox text:$node_path] 
            eval [list $w.canv create rectangle] $text_dimensions [list -fill $textBackground -width 1 -tags [list rect:$node_path]]
            $w.canv lower rect:$node_path text:$node_path
        }
}

proc hierarchy::adjust_look { w node_path } {
        global vals
        if $vals($w,commonselect) {
            set varname [lindex $node_path [expr [llength $node_path]-1]]
        } else {
            set varname $node_path
        }
        set text [lindex $vals($w,$node_path,look) 0]
        set bitmap [lindex $vals($w,$node_path,look) 3]
        if {$bitmap!=""} {
            if $vals($w,$varname,select) {
                set bitmapForeground $vals($w,selectBitmapForeground)
                set bitmapBackground $vals($w,selectBackground)
            } else {
                set bitmapForeground [lindex $vals($w,$node_path,look) 4]
                set bitmapBackground $vals($w,itemBackground)
                if {$bitmapForeground==""} { set bitmapForeground $vals($w,bitmapForeground) }
            }
            $w.canv itemconfigure bitmap:$node_path -bitmap $bitmap -foreground $bitmapForeground  -background $bitmapBackground
        }
        if {$text!=""} {
            set textFont [lindex $vals($w,$node_path,look) 2]
            if $vals($w,$varname,select) {
                set textFill $vals($w,selectTextFill)
            } else {
                set textFill [lindex $vals($w,$node_path,look) 1]
                if {$textFill==""} { set textFill $vals($w,textFill) }
            }
            if {$textFont==""} { set textFont $vals($w,font) }
            $w.canv itemconfigure text:$node_path -text $text -fill $textFill -font $textFont 
        }               

}


#------------------------------------------------------------
# Redrawing apparatus
#
# recompute_positions recurses through the tree wokring
# out the relative offsets of children from their parents
# in terms of depth/stack(width) values.  
#
# "changed_node" is either empty or a node name which indicates
# where the only changes have occured in the hierarchy since the last
# call to rcompute_positions.  This is used because when a node is toggled
# on/off deep in the hierarchy then not all the positions of items
# need to be recomputed.  The only ones that do are everything below
# the changed node (of ocurse), and also everything which might depend on
# the stack usage of that node (i.e. everything above it).  Specifically
# the usages of the changed node's siblings do *not* need to be recomputed.
#------------------------------------------------------------

proc hierarchy::min {args} {
    set min 99999
    foreach a $args { if $a<$min {set min $a} } 
    return $min
}
proc hierarchy::max {args} {
    set max -99999
    foreach a $args { if $a>$max {set max $a} } 
    return $max
}


proc hierarchy::recompute_positions { w changed_node_path } {
        global vals
        return [hierarchy::recompute_positions_aux $w $vals($w,rootnode) $changed_node_path]
}

proc hierarchy::recompute_positions_aux { w node_path changed_node_path } {
        global vals
                                                     
        # If the changed_node_path now has only one element then
        # it must be one of the children of the current node.
        # We do not need to recompute the
        # usages of its siblings if it is.
        
        set changed_node_is_child [expr [llength $changed_node_path]==1]
        if $changed_node_is_child {
            set changed_node [lindex $changed_node_path 0]
        } else {
            set remaining_changed_node_path [lrange $changed_node_path 1 end]
        }
        if $vals($w,debug) { puts "$node_path: changed_node_path = $changed_node_path, changed_node_is_child = $changed_node_is_child" } 
        
        
        # Run through the children, recursively calculating their usage
        # of stack-depth real-estate, and allocating an intial placement
        # for each child in vals$w,$kid,
        #
        # Values do not need to be recompted for siblings of the changed
        # node and their descendants.  For the changed_node itself, in the
        # recursive call we set the value of changed_node to {} to prevent
        # any further changed_node checks.
        
        set children_stack_usage 0
        set children_depth_usage 0
        if $vals($w,$node_path,showkids) { 
            foreach kid $vals($w,$node_path,kids) {
                set kid_node_path $node_path               
                lappend kid_node_path $kid
                set vals($w,$kid_node_path,offset) $children_stack_usage
                if {$changed_node_is_child && $changed_node==$kid} {
                    set vals($w,$kid_node_path,usages) [hierarchy::recompute_positions_aux $w $kid_node_path {}]
                } else {
                    if {!$changed_node_is_child} {
                        set vals($w,$kid_node_path,usages) [hierarchy::recompute_positions_aux $w $kid_node_path $remaining_changed_node_path]
                    }
                }
                set child_stack_usage [lindex $vals($w,$kid_node_path,usages) 0]
                if [regexp ^(n|s|e|w)$ $vals($w,rootanchor)] {
                    # these anchors are stacked centrally, and so
                    # we adjust each child back by half its stack
                    # usage to account for the centering.
                    incr vals($w,$kid_node_path,offset) [expr $child_stack_usage/2]
                }
                incr children_stack_usage $child_stack_usage
                set children_depth_usage [hierarchy::max $children_depth_usage [lindex $vals($w,$kid_node_path,usages) 1]]
                incr children_stack_usage $vals($w,padstack)
            }
        }
        incr children_stack_usage -$vals($w,padstack)
                       
        # Make the items (if they do no already exist)
        # and place them any old place.  Adjust their look also.
        # The items get repositioned later.  
        
        set text [lindex $vals($w,$node_path,look) 0]
        set bitmap [lindex $vals($w,$node_path,look) 3]
        if {$bitmap!=""} {
            if [llength [$w.canv find withtag bitmap:$node_path]]==0 {
                $w.canv create bitmap 0 0 -anchor $vals($w,rootanchor) -tags [list bitmap:$node_path]
                hierarchy::bindelem $w $node_path bitmap
            }
        }
        if {$text!=""} {
            if [llength [$w.canv find withtag text:$node_path]]==0 {
                $w.canv create text 0 0 -anchor $vals($w,rootanchor) -tags [list text:$node_path]
                hierarchy::bindelem $w $node_path text
            }
        }               
        hierarchy::adjust_look $w $node_path


        # Now calculate the stack usage of our little piece
        # of the world.
        # We have to create the bitmap and text itams to get an idea
        # of their size
        
        set bitmap_height 0
        set bitmap_width 0
        set text_width 0                                     
        set text_height 0                                    
        
        if {$bitmap!=""} {
            set bitmap_dimensions [$w.canv bbox bitmap:$node_path]
            if $vals($w,debug) { puts "$node_path: bitmap_dimensions = $bitmap_dimensions" }
            set bitmap_height [expr [lindex $bitmap_dimensions 3]-[lindex $bitmap_dimensions 1]]
            set bitmap_width [expr [lindex $bitmap_dimensions 2]-[lindex $bitmap_dimensions 0]]
            if $vals($w,debug) { puts "$node_path: bitmap_height     = $bitmap_height" }
            if $vals($w,debug) { puts "$node_path: bitmap_width      = $bitmap_width" }
        }
        if {$text!=""} {
            set text_dimensions [$w.canv bbox text:$node_path]
            if $vals($w,debug) { puts "$node_path: text_dimensions = $text_dimensions" }
            set text_height [expr [lindex $text_dimensions 3]-[lindex $text_dimensions 1]]
            set text_width [expr [lindex $text_dimensions 2]-[lindex $text_dimensions 0]]
            if $vals($w,debug) { puts "$node_path: text_height     = $text_height" }
            if $vals($w,debug) { puts "$node_path: text_width      = $text_width" }
        }
        
        if [regexp ^(ne|se|e|nw|sw|w)$ $vals($w,rootanchor)] {
            # these anchors are stacked vertically
            set my_stack_usage [hierarchy::max $text_height $bitmap_height]
            set my_depth_usage [expr {$text_width+$bitmap_width+$vals($w,padtextbitmap)}]
        } else {
            # these anchors are stacked horizontally
            set my_stack_usage [hierarchy::max $text_width $bitmap_width]
            set my_depth_usage [expr {$text_height+$bitmap_height+$vals($w,padtextbitmap)}]
        }
        
        if $vals($w,debug) { puts "$node_path: my_stack_usage = $my_stack_usage, my_depth_usage = $my_depth_usage" }
        
        # Now reposition the children in the case of the centre
        # positioned items by half of $usage.  In the case
        # of the others position them downward by "my_stack_usage"
                        
        if [regexp ^(n|s|e|w)$ $vals($w,rootanchor)] {
            # these anchors are stacked centrally
            set overall_stack_usage [hierarchy::max $children_stack_usage $my_stack_usage]
            set overall_depth_usage [expr $children_depth_usage+$vals($w,paddepth)+$my_depth_usage]
        } else {
            # these anchors are stacked on one side only
            # the depth of the item itself does not effect the overall depth
            # unless it is greater than all it children (e.g. if it has no children)
            set overall_stack_usage [expr $children_stack_usage+$my_stack_usage+$vals($w,padstack)]
            set overall_depth_usage [hierarchy::max [expr $children_depth_usage+$vals($w,paddepth)] $my_depth_usage]
        }
        if $vals($w,debug) { puts "$node_path: overall_stack_usage = $overall_stack_usage, overall_depth_usage = $overall_depth_usage" }
        if $vals($w,$node_path,showkids) { 
            foreach kid $vals($w,$node_path,kids) {
                set kid_node_path $node_path               
                lappend kid_node_path $kid
                if [regexp ^(n|s|e|w)$ $vals($w,rootanchor)] {
                    # these anchors are stacked centrally
                    incr vals($w,$kid_node_path,offset) [expr -$children_stack_usage/2]
                    # we need this to stop lone-children looking silly
                    if {abs($vals($w,$kid_node_path,offset)) < 2} {
                        set vals($w,$kid_node_path,offset) 0
                    }
                } else {
                    # these anchors are stacked on one side only
                    
                    incr vals($w,$kid_node_path,offset) [expr $my_stack_usage+$vals($w,padstack)]
                }       
                if $vals($w,debug) { puts "$node_path: vals($w,$kid_node_path,offset) = $vals($w,$kid_node_path,offset)" }
            }
        }
        # remember some facts for locating the bitmap
        # and also for drawing decorations
        set vals($w,$node_path,my_stack_usage) $my_stack_usage 
        set vals($w,$node_path,my_depth_usage) $my_depth_usage 
        set vals($w,$node_path,bitmap_width) $bitmap_width
        set vals($w,$node_path,bitmap_height) $bitmap_height
        
        return [list $overall_stack_usage $overall_depth_usage]
}

proc hierarchy::configure_notify { w } {
        global vals         
        if $vals($w,debug) { puts "hierarchy::configure_notify, w = $w" }
        if ![info exists vals($w,configured)] {
            set vals($w,configured) 1
            hierarchy::redraw_after_node_change $w {}
        }
        hierarchy::choose_initial_viewport $w
}                                                       

proc hierarchy::choose_initial_viewport { w } {
        global vals
        set height [winfo height $w.canv]
        set width [winfo width $w.canv]
        case $vals($w,rootanchor) {
            nw {
                set xview 0
                set yview 0
            }
            n {
                set xview [expr ($vals($w,stack_usage)/2-$width/2)/10]
                set yview 0
            }
            ne {
                set xview [expr ($vals($w,depth_usage)-$width)/10]
                set yview 0
            }
            e {
                set xview [expr ($vals($w,depth_usage)-$width)/10]
                set yview [expr ($vals($w,stack_usage)/2-$height/2)/10]
            }
            se {
                set xview [expr ($vals($w,depth_usage)-$width)/10]
                set yview [expr ($vals($w,stack_usage)-$height)/10+1]
            }
            s {
                set xview [expr ($vals($w,stack_usage)/2-$width/2)/10]
                set yview [expr ($vals($w,depth_usage)-$height)/10+1]
            }
            sw {
                set xview 0
                set yview [expr ($vals($w,stack_usage)-$height)/10+1]
            }
            w {
                set xview 0
                set yview [expr ($vals($w,stack_usage)/2-$height/2)/10]
            }
        }
        $w.canv xview $xview
        $w.canv yview $yview

}


        # When a node changes, the positions of a whole lot of things
        # change.  The size of the scroll region also changes.
        # The heuristic we use to reposition the viewport is to
        # try to keep the node that was adjusted in the same
        # position on the screen.

        # Calculate the screen location of the "still node".  This is
        # used later to relocate the still node back to the same
        # screen location
        
proc hierarchy::redraw_after_node_change { w changed_node_path } {
        global vals
        
        set remaining_changed_node_path [lrange $changed_node_path 1 end]
        
        $w.canv delete decorations
                
        if ![info exists vals($w,configured)] return
        
        if {$changed_node_path==""} { 
            set still_node_path $vals($w,rootnode) 
        } else { 
            set still_node_path $changed_node_path
         }
        set still_spot [$w.canv coords text:$still_node_path]
        set still_spot_x [expr [lindex $still_spot 0]-[$w.canv canvasx 0]]
        set still_spot_y [expr [lindex $still_spot 1]-[$w.canv canvasy 0]]
                           
        # Now calculate the new offset locations of everything
        
        set usages [hierarchy::recompute_positions $w $remaining_changed_node_path]
        set vals($w,stack_usage) [lindex $usages 0]
        set vals($w,depth_usage) [lindex $usages 1]
        
        # Now set the new scroll region
        
        if [regexp ^(ne|se|e|nw|sw|w)$ $vals($w,rootanchor)] {
            # these anchors are stacked vertically
            set width [expr $vals($w,depth_usage)+10]
            set height [expr $vals($w,stack_usage)+10]
        } else {
            set width [expr $vals($w,stack_usage)+10]
            set height [expr $vals($w,depth_usage)+10]
        }
        $w.canv config -scrollregion [list 0 0 $width $height]
        
        # Next recursively move all the bits around to 
        # their correct positions.
        # We choose a point (start_depthpos,start_stackpos) to begin at.
        
        case $vals($w,rootanchor) {
            nw {
                set start_depthpos 0
                set start_stackpos 0
            }
            n {
                set start_depthpos 0
                set start_stackpos [expr $vals($w,stack_usage)/2]
            }
            ne {
                set start_depthpos [expr $vals($w,depth_usage)]
                set start_stackpos 0
            }
            e {
                set start_depthpos $vals($w,depth_usage)
                set start_stackpos [expr $vals($w,stack_usage)/2]
            }
            se {
                set start_depthpos $vals($w,depth_usage)
                set start_stackpos $vals($w,stack_usage)
            }
            s {
                set start_depthpos $vals($w,depth_usage)
                set start_stackpos [expr $vals($w,stack_usage)/2]
            }
            sw {
                set start_depthpos 0
                set start_stackpos $vals($w,stack_usage)
                # in this case we do not need to fiddle with the xview/yview
            }
            w {
                set start_depthpos 0
                set start_stackpos [expr $vals($w,stack_usage)/2]
                # in this case we do not need to fiddle with the xview/yview
            }
        }
        hierarchy::redraw_aux $w $vals($w,rootnode) $start_depthpos $start_stackpos
        
        # Calculate where the still node is now on the canvas and adjust
        # the viewport so it is located on the screen at the
        # same screen location.
        
        set new_still_spot [$w.canv coords text:$still_node_path]
        # Nb. 10 is the scroll increment.  it should be calculated.
        
        $w.canv xview [expr ([lindex $new_still_spot 0]-$still_spot_x)/10]
        $w.canv yview [expr ([lindex $new_still_spot 1]-$still_spot_y)/10]
        
}

proc hierarchy::redraw_aux { w node_path depthpos stackpos } {
        global vals
        
        # now draw the bar line on which each child line sits,
        # and each child line as well.
        
        set iscentered [regexp ^(n|s|e|w)$ $vals($w,rootanchor)]
        set isvertical [regexp ^(ne|se|e|nw|sw|w)$ $vals($w,rootanchor)]
        case $vals($w,rootanchor) {
            nw { set depthop +; set stackop + }
            n { set depthop +; set stackop + }
            ne { set depthop -; set stackop + }
            e { set depthop -; set stackop + }
            se { set depthop -; set stackop - }
            s { set depthop -; set stackop + }
            sw { set depthop +; set stackop - }
            w { set depthop +; set stackop + }
        }
        # position the text and bitmap
        
        if $isvertical {
            set bitmapx $depthpos
            set bitmapy $stackpos
            set textx [expr "$depthpos [set depthop] $vals($w,$node_path,bitmap_width) [set depthop] $vals($w,padtextbitmap)"]
            set texty $stackpos
        } else {
            set bitmapx $stackpos
            set bitmapy [expr "$depthpos [set depthop] $vals($w,$node_path,bitmap_height) [set depthop] $vals($w,padtextbitmap)"]
            set textx $stackpos
            set texty $depthpos
        }
        if {[lindex $vals($w,$node_path,look) 3]!=""} {
            $w.canv coords bitmap:$node_path $bitmapx $bitmapy
        }
        if {[lindex $vals($w,$node_path,look) 0]!=""} {
            $w.canv coords text:$node_path $textx $texty
            hierarchy::remake_selection_box $w $node_path
        }
        
#       puts "in redraw_aux, node_path = $node_path, showkids = $vals($w,$node_path,showkids), kids = [catch {set $vals($w,$node_path,kids)}]"
        if {!$vals($w,$node_path,showkids)} return
        if {[llength $vals($w,$node_path,kids)]==0} return

                               
        # Note that below x/y's get swapped around for non-vertical stacking
                
            set minkid_stackpos 99999
            set maxkid_stackpos -99999
            if $iscentered {         
                set top_depthpos [expr "$depthpos [set depthop] $vals($w,$node_path,my_depth_usage)"]
                set bar_depthpos [expr "$depthpos [set depthop] $vals($w,$node_path,my_depth_usage) [set depthop] $vals($w,paddepth)/2"]
                set kid_depthpos [expr "$bar_depthpos [set depthop] $vals($w,paddepth)/2"]
                if $isvertical {
                    $w.canv create line $top_depthpos $stackpos $bar_depthpos $stackpos -width 1 -tags decorations
                } else {
                    $w.canv create line $stackpos $top_depthpos $stackpos $bar_depthpos -width 1 -tags decorations
                }
            } else {
                set bar_depthpos [expr "$depthpos [set depthop] $vals($w,paddepth)/2"]
                set kid_depthpos [expr "$bar_depthpos [set depthop] $vals($w,paddepth)/2"]
            }
            foreach kid $vals($w,$node_path,kids) {
                set kid_node_path $node_path               
                lappend kid_node_path $kid
                set kid_stackpos [expr "$stackpos [set stackop] $vals($w,$kid_node_path,offset)"]
                hierarchy::redraw_aux $w $kid_node_path $kid_depthpos $kid_stackpos

                # adjust the bar to the kid to be in the center of the kid
                # this is a result of us using the root anchoring to anchor
                # the texts and bitmaps as well, which works well except
                # for here.
                #
                # Note this doesn't effect where the child is drawn - it only
                # effects the drawing of decorations.
                #
                # Checking for the maximum/minimum extents is used
                # to know how to draw the bar that connects all the little
                # bars together.  Doing the "is-it-greater/is-it-less"
                # checks both before and after the adjustment to kid_stackpos
                # is an effective way of getting the small extra part of
                # the bar needed to connect the parent to the rest.
                
                if {$kid_stackpos<$minkid_stackpos} { set minkid_stackpos $kid_stackpos}
                if {$kid_stackpos>$maxkid_stackpos} { set maxkid_stackpos $kid_stackpos}
                if {!$iscentered} {
                    set kid_stackpos [expr "$kid_stackpos [set stackop] ($vals($w,$kid_node_path,my_stack_usage)/2)"]
                }
                if {$kid_stackpos<$minkid_stackpos} { set minkid_stackpos $kid_stackpos}
                if {$kid_stackpos>$maxkid_stackpos} { set maxkid_stackpos $kid_stackpos}
                
                if $isvertical {
                    $w.canv create line $bar_depthpos $kid_stackpos $kid_depthpos $kid_stackpos -width 1 -tags decorations
                } else {
                    $w.canv create line $kid_stackpos $bar_depthpos $kid_stackpos $kid_depthpos -width 1 -tags decorations
                }
            }
            if $isvertical {
                $w.canv create line $bar_depthpos $minkid_stackpos $bar_depthpos $maxkid_stackpos -width 1 -tags decorations
            } else {
                $w.canv create line $minkid_stackpos $bar_depthpos $maxkid_stackpos $bar_depthpos -width 1 -tags decorations
            }
}



#------------------------------------------------------------
#
#
#------------------------------------------------------------

proc hierarchy::filelook { node_path showing_kids } {
        global hierarchy_library 
        set path [join $node_path /]
        set file [lindex $node_path [expr [llength $node_path]-1]]
        if [file readable $path] { 
            if [hierarchy::dir_has_subdirs $path] {
                if $showing_kids {
                    set bitmap @$hierarchy_library/src/folder_minus.xbm
                } else {
                    set bitmap @$hierarchy_library/src/folder_plus.xbm
                }
            } else {
                set bitmap @$hierarchy_library/src/folder.xbm
            }
            set textFill black
            set bitmapColor black
        } else {
            set textFill grey75
            set bitmapColor grey75
            set bitmap @$hierarchy_library/src/folder.xbm
        }
        return [list $file $textFill {} $bitmap $bitmapColor] 
}


proc hierarchy::dircontents { node_path } {
        set path [join $node_path /]
        set files [glob -nocomplain $path/*]
        set dirs ""
        foreach file $files {
            if [file isdirectory $file] { 
                set pieces [split $file /]
                lappend dirs [lindex $pieces [expr [llength $pieces]-1]]
            }
        }
        return $dirs
}

proc hierarchy::dir_has_subdirs { path } {
        set files [glob -nocomplain $path/*]
        set dirs ""
        foreach file $files {
            if [file isdirectory $file] { return 1 }
        }
        return 0
}



