#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




proc TkGoalProof::delayedLoad { } {
        global TkGoalProof_flags TkGoalProof_library
        global feedback
        if ![info exists TkGoalProof_flags(loaded)] {
            if [hol90] {
                set feedback [list {} "Loading the replacement hol90 Goalstack package..."]
                hol_load_library "goaltrees"
            }
            set TkGoalProof_flags(loaded) 1
        }
}

proc TkGoalProof::delayedLoadPattTacs { } {
        global TkGoalProof_flags 

        if ![info exists TkGoalProof_flags(termpath_tactics_loaded)] { 
            hol_load_library termpath_tactics
            ML -toplevel 1 "open HolTermPatternTactics"
            set TkGoalProof_flags(termpath_tactics_loaded) 1
        }
}


option add *TkGoalProof.initial "" widgetDefault

proc lcontains { l x } {
    expr "[lsearch -exact $l $x] != -1"
}


proc TkGoalProof { w args} {
        global TkGoalProof_version
        global TkGoalProof_flags
        global gui_flags
        global vals
        global busy
        incr busy
        TkGoalProof::delayedLoad
        if [hol90] {
            if {[ML -type int "version_number"] < 7} {
                errormessage $w "Sorry, TkGoalProof requires hol88 or hol90.7 or greater"
                incr busy -1; return
            } 
        }


        for {set i 0} {$i<[llength $args]} {incr i} {
            case [lindex $args $i] -initial {
                incr i
                set initial [lindex $args $i]
            } default {
                error "unrecognized arg [lindex $args $i]"
            }
        }
        toplevel $w -class TkGoalProof
        wm withdraw $w
        if ![info exists initial] { set initial [option get $w initial Initial] }

        regsub -all {\.} $w "" tmp1
        set vals($w,goalstack) gdp[set tmp1]_goalstack
        set vals($w,focus) gdp[set tmp1]_focus

        ML -toplevel 1 "val $vals($w,goalstack) = GoalTrees.empty_goalI ();
                        val $vals($w,focus) = GoalTrees.first_focusI $vals($w,goalstack)"

        # 2. Do some windowing grunge
        #
        global feedback
        
        wm title $w "Proof Window"

        set feedback [list $w "Creating TkGoalProof display..."]
        set vals($w,name) ""
        set vals($w,savable) 0

        # 
        # 3. Create the buttons and stuff for controlling and displaying
        # the goals.
        #

        pack [collapsible frame $w.goalsf \
                -title "Current Goal(s)" \
                -visible 1 \
                -collapseCommand "pack config $w.goalsf -expand "] \
         -fill both -expand yes
        pack [buttonbar $w.goalsf.b.controls] -fill x
        pack [richtextobjects $w.goals \
                -selectioncommand "TkGoalProof::new_goal_subset_selected $w" \
                -exportselection 0 \
                -mlbindbuttons 0] \
         -fill both -expand yes -in $w.goalsf.b
            
        $w.goals richtext text config \
                -height 10 \
                -width 55
        $w.goals listbox config \
                -geometry 7x6
        $w.goals richtext configure \
                -editingvar vals($w,editing) \
                -selectiontypesvar vals($w,selectiontypes)
        

        pack [collapsible frame $w.nexttac \
                -title "Next Tactic" \
                -visible 1] \
                -side top -fill both
        pack [buttonbar $w.nexttac.b.controls] -fill x
        pack [fontcheck scrollable text $w.nexttac.b.tac  \
                -borderwidth 2 \
                -relief sunken \
                -height 6 \
                -font $gui_flags(font,codeentry)] \
            -fill both 

        set vals($w,button,setgoal) [$w.goalsf.b.controls addbutton sensitive button \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "\$vals($w,editing)" \
                -text "Set Goal" \
                -command "TkGoalProof::set_goal_from_widget $w"]
        $w.goalsf.b.controls addbutton sensitive button \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -text "New Goal/Restart" \
                -command "TkGoalProof::newgoal $w"

        $w.nexttac.b.controls addbutton sensitive button \
                -text "Clear" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -command "$w.nexttac.b.tac text delete 1.0 end; focus_goTo [$w.nexttac.b.tac text]" \
                -font $gui_flags(font,bbar_buttons)

        set vals($w,button,applytactic) [$w.nexttac.b.controls addbutton sensitive button \
                -text "Apply Tactic" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -command "TkGoalProof::ApplyML $w \[$w.nexttac.b.tac text get 1.0 end\]"]

        $w.nexttac.b.controls addbutton sensitive button \
                -text "Animate Forward" \
                -sensitivevars vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -command "TkGoalProof::fast_forward $w 1"
        $w.nexttac.b.controls addbutton sensitive button \
                -text "Fast Forward" \
                -sensitivevars vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -command "TkGoalProof::fast_forward $w 0"
        $w.nexttac.b.controls addbutton sensitive button \
                -text "Backup" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -command "TkGoalProof::Backup $w retain"
        $w.nexttac.b.controls addbutton sensitive button \
                -text "Backup & Discard Attempt" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -command "TkGoalProof::Backup $w discard"
        bind $w.nexttac.b.tac.b <Meta-p> "TkGoalProof::history_tac_shift $w 1"
        bind $w.nexttac.b.tac.b <Meta-n> "TkGoalProof::history_tac_shift $w -1"
        
        pack [collapsible frame $w.entiretac \
                -title "Entire Tactic" \
                -visible 1] \
                -side top -fill both -pady 5
        pack [buttonbar $w.entiretac.b.controls] -fill x
        pack [richtext $w.entiretac.b.tac \
                -focusvar vals($w,subtactic_focus) \
                -cursorvar vals($w,cursor_in_tactic)] \
                -fill both
        
        $w.entiretac.b.tac text config -height 6

        $w.entiretac.b.controls addbutton sensitive button \
                -text "Trim" \
                -sensitivevars vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -command "TkGoalProof::trim $w"
        $w.entiretac.b.controls addbutton sensitive button \
                -text "Delete Tactic" \
                -sensitivevars vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -command "TkGoalProof::delete_at_focus $w"
        $w.entiretac.b.controls addbutton sensitive button \
                -text "Insert Tactic" \
                -sensitivevars vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)" \
                -command "TkGoalProof::insert_space_before_focus $w"
        
        #
        # 4. now set up the menus for the TkGoalProof window. Disable the "Save"
        # menus until the theorem is actually proven.
        #

        TkGoalProof::make_menus $w
        TkGoalProof::remake_tactic_history_menu $w


        # 
        # 6. Register the upon-destroy handler.
        #

        widget_addBinding $w Destroy "TkGoalProof::upon_destroy $w"
        wm protocol $w WM_DELETE_WINDOW "TkGoalProof::verify_destroy $w"

        trace variable vals($w,subtactic_focus) w "TkGoalProof::subtactic_focus_change $w"

        update
        wm deiconify $w 
        tkwait visibility $w
        update
        focus_goToFirst $w.goals
        update

        if {$initial == ""} {
            TkGoalProof::enter_goal_entry_mode $w
        } else {
            TkGoalProof::reload_proof $w $initial
        }
        
        incr busy -1
        return $w
}


#----------------------------------------------------------------------------
# TkGoalProof::verify_destroy 
#
# Called when the user tries to destroy the window via
# WM_DELETE_WINDOW window manager action or by "Close" menu option.
#----------------------------------------------------------------------------

proc TkGoalProof::verify_destroy { w } {
        global vals
        global verify
        if {$vals($w,savable)
            && [info exists vals($w,saved)] 
            && !$vals($w,saved)} {
            set choice [choice $w.verify -text "You have not saved the theorem.\nDo you want to do this now?"]
            case $choice Yes {
                TkGoalProof::save $w
            } No {
                destroy $w
            } 
        } else {
            destroy $w
        }
}

#----------------------------------------------------------------------------
# TkGoalProof::upon_destroy 
#
# Empty the goal stacks when the window is destroyed.
#
#----------------------------------------------------------------------------

proc TkGoalProof::upon_destroy { w } {
        global vals
        ML -toplevel 1  "val $vals($w,goalstack) = ()"
        unset_vals_for_widget $w
        trace vdelete vals($w,subtactic_focus) w "TkGoalProof::subtactic_focus_change $w"

}


#----------------------------------------------------------------------------
# make_menus
# remake_user_menus
# remake_tactic_history_menu
#
# remake_user_menus is called automatically via tracing 
# whenever a change happens in the TkGoalProof_flags array 
# (this seems efficient since that is where most of the menus are specified,
# and not much else is specified there)
#
# remake_tactic_history_menu is called manually whenever we add/subtract
# a tactic.
#
# Using MakeMenus doesn't result in much saving here..., and really just
# obscures what's going on.  Menu code always seems to long and
# routine in Tk.
#----------------------------------------------------------------------------

proc TkGoalProof::make_menus { w } {
        global vals
        global TkGoalProof_tactic_history
        global TkGoalProof_flags
        set menus ""
        lappend menus \
            [list proof "Proof" 0 left                                  \
                [list                                                   \
                    [list command "Set Goal" "TkGoalProof::set_goal_from_widget $w" 4]  \
                    [list command "New Goal/Restart" "TkGoalProof::restart $w" 0] \
                    [list command "Apply Tactic"                        \
                        "TkGoalProof::ApplyML $w                        \
                                \[$w.nexttac.b.tac text get 1.0 end\]" 0]       \
                    [list command "Animate Forward" "TkGoalProof::fast_forward $w 1" 0]  \
                    [list command "Fast Forward" "TkGoalProof::fast_forward $w 0" 0]  \
                    [list command "Backup" "TkGoalProof::Backup $w retain" 0]  \
                    [list command "Backup & Discard Attempted Proof" "TkGoalProof::Backup $w discard" 9]  \
                    [list sep]                                          \
                    [list command "Save..." "TkGoalProof::save $w" 0]   \
                    [list command "New Proof Window..." "newwin TkGoalProof" 10]        \
                    [list sep]                                          \
                    [list command "Close" "TkGoalProof::verify_destroy $w" 0]           \
                ]                                                       \
            ]

        lappend menus [list tactic_history $TkGoalProof_flags(tactic_history,title)  0 left {}]

        for {set n 1} {[info exists TkGoalProof_flags(tactics$n)]} {incr n} {
            if {[llength $TkGoalProof_flags(tactics$n)] == 0} { incr n; continue }
            lappend menus [list tactics$n $TkGoalProof_flags(tactics$n,title) 0 left {}]
        }
                      
        for {set n 1} {[info exists TkGoalProof_flags(conversions$n)]} {incr n} {
            if {[llength $TkGoalProof_flags(conversions$n)] == 0} { incr n; continue }
            lappend menus [list conversions$n $TkGoalProof_flags(conversions$n,title) 0 left {}]
        }
                      
        set entries ""
        foreach preference_group [preferences::groups] {
            lappend entries [list command "$preference_group..." "preferences -group [list $preference_group]" {}]
        }
        lappend menus [list preferences "Options" 0 left $entries]

        lappend menus [list help "Help" 0 left [Help::menu_entries TkGoalProof]]
        MakeMenus $w $menus


        make_sensitive $w.menu.proof.m -menuentry "Fast Forward" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)"
        make_sensitive $w.menu.proof.m -menuentry "Animate Forward" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)"
        make_sensitive $w.menu.proof.m -menuentry "Backup*" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)"
        make_sensitive $w.menu.proof.m -menuentry "Backup & Discard*" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)"
        make_sensitive $w.menu.proof.m -menuentry "Apply*" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)"
        make_sensitive $w.menu.proof.m -menuentry "*Restart*" \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)"
        make_sensitive $w.menu.proof.m -menuentry "Set Goal*" \
                -sensitivevar vals($w,editing)
        make_sensitive $w.menu.proof.m -menuentry "Save*" \
                -sensitivevar vals($w,savable)

        make_sensitive $w.menu.tactic_history \
            -sensitivevar vals($w,editing) \
            -sensitiveexpr "!\$vals($w,editing)"
        for {set n 1} {[info exists TkGoalProof_flags(tactics$n)]} {incr n} {
            if {[llength $TkGoalProof_flags(tactics$n)] == 0} { incr n; continue }
            trace variable TkGoalProof_flags(tactics$n) w "TkGoalProof::remake_user_menus $w"
            make_sensitive $w.menu.tactics$n \
                -sensitivevar vals($w,editing) \
                -sensitiveexpr "!\$vals($w,editing)"
        }
        for {set n 1} {[info exists TkGoalProof_flags(conversions$n)]} {incr n} {
            if {[llength $TkGoalProof_flags(conversions$n)] == 0} { incr n; continue }
            trace variable TkGoalProof_flags(conversions$n) w "TkGoalProof::remake_user_menus $w"
            make_sensitive $w.menu.conversions$n \
                -sensitivevars [list vals($w,selectiontypes) vals($w,editing)] \
                -sensitiveexpr "!\$vals($w,editing) && \[lcontains \$vals($w,selectiontypes) HOL_TERMPATTERN\]"
        }
        TkGoalProof::remake_user_menus $w
}       


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc TkGoalProof::remake_user_menus { w args } {

        if ![info exists w] { return }
        global TkGoalProof_flags

        for {set n 1} {[info exists TkGoalProof_flags(tactics$n)]} {incr n} {
            if {[llength $TkGoalProof_flags(tactics$n)] == 0} { incr n; continue }

            $w.menu.tactics$n.m delete 0 last
            foreach tactic $TkGoalProof_flags(tactics$n) {
                $w.menu.tactics$n.m add command \
                        -label $tactic \
                        -command [list TkGoalProof::place_in_buffer $w $tactic]
            }
        }
                      
        for {set n 1} {[info exists TkGoalProof_flags(conversions$n)]} {incr n} {
            if {[llength $TkGoalProof_flags(conversions$n)] == 0} { incr n; continue }

            $w.menu.conversions$n.m delete 0 last
            foreach conversion $TkGoalProof_flags(conversions$n) {
                $w.menu.conversions$n.m add command \
                        -label $conversion \
                        -command [list TkGoalProof::ConvertSubterm $w $conversion]
            }
        }
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc TkGoalProof::remake_tactic_history_menu { w } {
        global TkGoalProof_tactic_history
        global gui_flags
        catch {$w.menu.tactic_history.m delete 0 last}
        catch {menu $w.menu.tactic_history.m}
        foreach history_tactic $TkGoalProof_tactic_history {
                set command [list TkGoalProof::place_in_buffer $w $history_tactic -replace]
                $w.menu.tactic_history.m add command \
                    -command $command \
                    -font $gui_flags(font,menus) \
                    -label $history_tactic
        }
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

# called when a change in focus is signalled by user action in the
# "entire tactic" window.
proc TkGoalProof::subtactic_focus_change { w args } {
        global errorInfo
        global vals
        set subtactic_path [lindex $vals($w,subtactic_focus) 1]
        set subtactic_path_length [llength $subtactic_path]
        # chop "L" off end of subtactic path to get new focus
        set new_focus [lrange $subtactic_path 0 [expr $subtactic_path_length-2]]
        set new_focus2 ""
        for {set i 0} {$i < [llength $new_focus]} {incr i 2} {
            lappend new_focus2 "([list [lindex $new_focus $i],[lindex $new_focus [expr $i+1]]])"
        }
        if {![ML -type bool "(GoalTrees.focus_is_activeI $vals($w,goalstack) (ref [ml_list $new_focus2])) 
                             orelse (GoalTrees.focus_is_openedI $vals($w,goalstack) (ref [ml_list $new_focus2]))"]} {
            errormessage $w.error \
"Presently you may only focus on tactics \
which have just been applied or which can be applied next.  \
Fast forwarding to a particular location is not yet available."
            error "invalid focus"
        } else {
            ML -type any "$vals($w,focus) := [ml_list $new_focus2]"
            TkGoalProof::DisplayGoalAtFocus $w
            TkGoalProof::DisplayNextTac $w
            focus_goTo [$w.nexttac.b.tac text]
            
            # no need to update the entire tactic display.
        }    
}


#--------------------------------------------------------------------
# Called when the selection changes in the listbox.
# If we are not at a THENL branch and not all the goals
# are selected we have to split the THEN into cases.  This can be
# undone using backup.  If at a THENL branch we just switch the focus.
#--------------------------------------------------------------------

proc TkGoalProof::new_goal_subset_selected { w args } {
        global busy
        global vals
        incr busy
        set sel [$w.goals listbox cursel]
        set numgoals [$w.goals listbox size]
        if {[llength $sel]!=$numgoals} {

            # take the first in the selection.  add 1 since listbox indexes are one out.
            set newgoal [expr [lindex $sel 0]+1]
            $w.goals listbox select from [lindex $sel 0]
            if $vals($w,at_THENL) {
              if {$newgoal!=$vals($w,focused_case)} {
                ML -type any "GoalTrees.refocus_at_THENL_startI $newgoal $vals($w,goalstack) $vals($w,focus)"
                TkGoalProof::goal_and_focus_change $w
              } 
            } else {
                ML -type any "GoalTrees.top_modify_proofI (GTrees.split_at_focus (!$vals($w,focus))) $vals($w,goalstack)"
                TkGoalProof::NextFocus $w
                TkGoalProof::goal_and_focus_change $w
            }
        } 
        incr busy -1
}


#----------------------------------------------------------
# called when the focus changes by automatic action following the
# (a) start of a goal or 
# (b) application of a tactic or 
# (c) backing up.
#----------------------------------------------------------

proc TkGoalProof::goal_and_focus_change { w } {
        TkGoalProof::DisplayGoalAtFocus $w
        update
        TkGoalProof::tactic_change $w
}

#----------------------------------------------------------
# called when the entire-tactic changes but the goal/focus does not.
#----------------------------------------------------------

proc TkGoalProof::tactic_change { w } {
        global vals
        TkGoalProof::DisplayNextTac $w
        update
        TkGoalProof::DisplayEntireTac $w
        update
        # suspend our tracing on vals($w,subtactic_focus) temporarily.
        # The richtext widget still traces it, updating the tactic
        # display appropriately.
        # "L" on the subtactic focus indicates the leaf (tactic) node at the
        # given focus.  The focus path itself is suficient but using L markings
        # makes it easier for RichText to tell what tags are leaf nodes
        # and what aren't.
        trace vdelete vals($w,subtactic_focus) w "TkGoalProof::subtactic_focus_change $w"

        # transform the goaltree style focus to a RichText style focus in the entire-tactic
        # the "eval eval" this does a flatten!
        set latest_focus [ML -type intpairlist "!$vals($w,focus)"]
        eval eval lappend temp $latest_focus
        lappend temp L
        set vals($w,subtactic_focus) [list $vals($w,entiretac_objectid) $temp]
        trace variable vals($w,subtactic_focus) w "TkGoalProof::subtactic_focus_change $w"
}




#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc TkGoalProof::save { w args } {
        global vals

        modalDialog transient TheoremSaveBox $w.save \
                -savename $vals($w,name) \
                -header "Goal Proved!" \
                -resultsvariable vals($w,waiter) \
                -withfeedback 0
        widget_waitVariable vals($w,waiter)
        modalDialog.end $w.save
        switch -- [lindex $vals($w,waiter) 0] Ok {
            set save [lindex $vals($w,waiter) 1]
            set savename [lindex $vals($w,waiter) 2]
            set bind [lindex $vals($w,waiter) 3]
            set view_script [lindex $vals($w,waiter) 4]
            set append [lindex $vals($w,waiter) 5]
            set bindname $savename

            global feedback
            set feedback [list $w "Saving theorem..."]
            set tactic [ML "layout_proof (GoalTrees.executed_proofI $vals($w,goalstack))"]
            set tactic [string trim $tactic]
            set script "prove([hol_term $vals($w,goal)],\n    [join [split $tactic \n] "\n    "]\n)"
            
            set top_thm_code "GoalTrees.final_thmI $vals($w,goalstack)"
            if {$bind && $save} {
                ML -toplevel 1 -check 1 "[ml_val] $bindname = save_thm ([ml_string $savename], $top_thm_code)"
                set script "[ml_val] $bindname = \nsave_thm([ml_string $savename], $script)[ml_end_dec]\n"
                set title "Proof: \[$bindname\]"
            } elseif {$bind} {
                ML -toplevel 1 -check 1 "[ml_val] $bindname = $top_thm_code"
                set script "[ml_val] $bindname = $script[ml_end_dec]\n"
                set title "Proof: \[$bindname\]"
            } elseif {$save} {
                ML -toplevel 1 -check 1 "save_thm ( [ml_string $savename], $top_thm_code)"
                set script "save_thm ([ml_string $savename], $script)[ml_end_dec]\n"
                set title "Proof: \[$savename\]"
            } else {
                set script "$script[ml_end_dec]\n"
                set title "Proof"
            }
            if $append {
                Scripts::add_to_script $script
            }
            set vals($w,saved) 1
            if $view_script {                  
                global gui_flags
                set newwin [newwin toplevel -withfeedback 0 -title $title]
                pack [fontcheck scrollable text $newwin.script \
                        -font $gui_flags(font,codeentry) \
                        -borderwidth 2 \
                        -relief sunken \
                        -height 10 \
                        -setgrid 1] \
                -side top \
                -expand yes \
                -fill both -padx 10 -pady 10
                $newwin.script text insert 1.0 $script
                pack [frame $newwin.controls] -pady 5 -side bottom
                pack [fontcheck focusable button $newwin.controls.selectall \
                        -command "$newwin.script text tag add sel 1.0 end" \
                        -text "Select All" \
                        -width 10 \
                        -font $gui_flags(font,buttons)] -side left
                pack [fontcheck focusable button $newwin.controls.close \
                        -command "destroy $newwin" \
                        -text "Close" \
                        -width 10 \
                        -font $gui_flags(font,buttons)] -side left

            }
        }
}


#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc TkGoalProof::restart { w } {
        global vals
        TkGoalProof::enter_goal_entry_mode $w
        $w.goals editpreterm [list TEXT $vals($w,goal)]
}

proc TkGoalProof::newgoal { w } {
        TkGoalProof::restart $w
#       global vals
#       ML -toplevel 1 "val $vals($w,goalstack) = GoalTrees.empty_goalI ();
#                        val $vals($w,focus) = GoalTrees.first_focusI $vals($w,goalstack)"
#       TkGoalProof::enter_goal_entry_mode $w
#       $w.goals editpreterm [list TEXT ""]
}

#----------------------------------------------------------------------------
# reload_proof
#
# Called if an intital goal/proof is specified.
#----------------------------------------------------------------------------

proc TkGoalProof::reload_proof { w argscode } {
        global vals
        global busy
        incr busy
        global feedback
        set feedback [list $w "Setting goal..."]
        set goalcode "#goal (ScriptFragments.data_for_bwdproof ($argscode))"
        set namecode "#name (ScriptFragments.data_for_bwdproof ($argscode))"
        set goal [ML $goalcode]
        set proofcode "#proof (ScriptFragments.data_for_bwdproof ($argscode))"
        set status [catch {ML -type void -check 1 "$vals($w,goalstack) := !(GoalTrees.set_goalsI (\[([ml_list ""],$goal)\], BwdProofs.BWDTHENL \[$proofcode\]))"} err]

        if $status {
            incr busy -1
            errormessage $w.error $err
            return
        }
        set vals($w,name) [ML $namecode]
        set vals($w,goal) [hol_determ $goal]

        TkGoalProof::enter_proof_mode $w

        incr busy -1
}

#----------------------------------------------------------------------------
# set_goal_from_widget
#
# Called when the user wants to set the goal from the text in the goal widget
# by pressing the "SetGoal" button.  The other way to set the
# goal is via an initial goal/proof to replay
#----------------------------------------------------------------------------

proc TkGoalProof::set_goal_from_widget { w } {
        global vals
        global busy
        incr busy
        global feedback
        set feedback [list $w "Setting goal..."]
        set goal [string trim [$w.goals getpreterm]]
        set status [catch {ML -type void -check 1 "$vals($w,goalstack) := !(GoalTrees.set_goalsI (\[([ml_list ""],[hol_term $goal])\], GoalTrees.entire_proofI $vals($w,goalstack)))"} err]

        if $status {
            incr busy -1
            errormessage $w.error $err
            return
        }
        set vals($w,goal) $goal

        # add the goal to the goal history 
        global TkGoalProof_flags
        global TkGoalProof_goal_history
        TkGoalProof::add_to_history TkGoalProof_goal_history $TkGoalProof_flags(goal_history,depth) $goal
        TkGoalProof::enter_proof_mode $w

        incr busy -1
}

#----------------------------------------------------------------
# Called when switching from proof mode to goal entry mode, or
# when entering the initial mode.
#----------------------------------------------------------------

proc TkGoalProof::enter_goal_entry_mode { w } {
        global vals
        global busy
        incr busy

        $w.goals configure \
                -showlist 0 \
                -showcontrols 0
        $w.goals editpreterm [list TEXT ""]
        set vals($w,savable) 0
        $w.nexttac.b.tac text config -state disabled
#       $w.entiretac.b.tac deleteall
        RichText::bind_for_normal_selection $w.entiretac.b.tac
        button_makeDefault $vals($w,button,setgoal)
        update
        focus_goTo [$w.goals richtext text]
        set vals($w,next_goal_num) -1
        incr busy -1
}
#----------------------------------------------------------------
# Called when switching from goal entry mode to proof mode or
# when entering the initial mode.  The
# initial goal should be available in the goalstack.  The text
# for the initial goal should be available in [[vals($w,goal)]].
#
# Causes the goal to be redisplayed via TkGoalProof::goal_and_focus_change.
#----------------------------------------------------------------

proc TkGoalProof::enter_proof_mode { w } {
        global vals
        global busy
        incr busy
        set status [catch {ML -type void -check 1 "$vals($w,focus) := !(GoalTrees.first_focusI $vals($w,goalstack))"} err]
        $w.goals configure \
                -showlist 1 \
                -showcontrols 1
        TkGoalProof::goal_and_focus_change $w
        RichText::bind_for_subobject_focus $w.entiretac.b.tac
        set vals($w,saved) 0
        set vals($w,savable) 0
        $w.nexttac.b.tac text config -state normal
        button_makeDefault $vals($w,button,applytactic)
        update
        focus_goTo [$w.nexttac.b.tac text]

        incr busy -1
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc TkGoalProof::DisplayTheorem { w } {
        global vals
        global feedback
        global busy
        incr busy

        set feedback [list $w "Displaying theorem..."]
        set top_thm_code "GoalTrees.final_thmI $vals($w,goalstack)"
        $w.goals setobjects [list "Goal+" [list THM [list CODE $top_thm_code]]]
        $w.goals expandall
        set vals($w,entiretac_objectid) [$w.entiretac.b.tac insertobject [list PBWDOP [list CODE "(GoalTrees.entire_partial_proofI $vals($w,goalstack))"]]]
        incr busy -1
}


proc TkGoalProof::DisplayGoalAtFocus { w } {
        global vals
        global feedback
        global busy
        global TkGoalProof_flags
        incr busy

        set vals($w,at_THENL) [ML -type bool "GoalTrees.at_THENL_startI $vals($w,goalstack) $vals($w,focus)"]
        if $vals($w,at_THENL) {
            set propositions_code "GoalTrees.propositions_at_THENL_startI $vals($w,goalstack) $vals($w,focus)"
            set vals($w,focused_case) [ML -type int "GoalTrees.focused_case_at_THENL_startI $vals($w,goalstack) $vals($w,focus)"]
        } else {
            set propositions_code "GoalTrees.propositions_at_focusI $vals($w,goalstack) $vals($w,focus)"
        }
        set feedback [list $w "Displaying goals..."]
        set objspecs ""
        set is_proveds [ML -type boollist "map GoalTrees.GTrees.is_proved ($propositions_code)"]
        # puts "is_proveds = $is_proveds"
        for {set goalnum 1} {$goalnum <= [llength $is_proveds]} {incr goalnum} {
            # puts "is_proved = [lindex $is_proveds [expr $goalnum-1]], goalnum = $goalnum"
            if {[lindex $is_proveds [expr $goalnum-1]]} {
                lappend objspecs [list "Goal $goalnum+" [list GOAL [list CODE "(Thm.dest_thm (GoalTrees.GTrees.dest_proved (el $goalnum ($propositions_code))))"]]]
            } else {
                lappend objspecs [list "Goal $goalnum" [list GOAL [list CODE "(GoalTrees.GTrees.dest_posed (el $goalnum ($propositions_code)))"]]]
            }
        }
        # puts "objspecs = $objspecs"
        eval $w.goals setobjects $objspecs
        if $vals($w,at_THENL) {
            catch {$w.goals select "Goal $vals($w,focused_case)"}
            catch {$w.goals select "Goal $vals($w,focused_case)+"}
            switch -- $TkGoalProof_flags(expandgoals) All {
                $w.goals expandall
                $w.goals yview 
            } First {
                catch {$w.goals expand "Goal $vals($w,focused_case)"}
                catch {$w.goals expand "Goal $vals($w,focused_case)+"}
            }
        } else {
            $w.goals select all
            $w.goals expandall
        }
        
        incr busy -1
}

proc TkGoalProof::DisplayEntireTac { w } {
        global vals
        $w.entiretac.b.tac deleteall
        set vals($w,entiretac_objectid) [$w.entiretac.b.tac insertobject [list PBWDOP [list CODE "(GoalTrees.entire_partial_proofI $vals($w,goalstack))"]]]
}

proc TkGoalProof::DisplayNextTac { w } {
        global vals
        $w.nexttac.b.tac text config -state normal
        $w.nexttac.b.tac text delete 1.0 end
        set text [ML "GoalTrees.current_tacI $vals($w,goalstack) $vals($w,focus)"]
        if {$text!="........"} {
            $w.nexttac.b.tac text insert 1.0 $text
        }
        set vals($w,next_tac_num) -1
}

#----------------------------------------------------------------------------
#----------------------------------------------------------------------------

proc TkGoalProof::NextFocus { w } {
        global vals
        if [ML -type bool "GoalTrees.THENL_mismatchI $vals($w,goalstack)"] {
            errormessage $w.error "Warning: The structure of the proof has changed (THENL mismtach)"
        }
        ML -type any "GoalTrees.top_modify_proofI GTrees.adjust_THENLs $vals($w,goalstack)"
        ML -type void "GoalTrees.next_focusI $vals($w,goalstack) $vals($w,focus)"
}

#----------------------------------------------------------------------------
# TkGoalProof::Backup
#
#----------------------------------------------------------------------------

proc TkGoalProof::Backup { w action } {
        global vals
        global busy
        incr busy 
        set status [catch {
            ML -type void "GoalTrees.backup_and_[set action]_attemptI $vals($w,goalstack) $vals($w,focus)"
            TkGoalProof::goal_and_focus_change $w
        } err]
        if $status { 
            puts stderr "warning: $err"
            TkGoalProof::restart $w 
        }
        set vals($w,savable) 0

        incr busy -1
}

#----------------------------------------------------------------------------
# TkGoalProof::trim
#
#----------------------------------------------------------------------------

proc TkGoalProof::trim { w } {
        global vals
        global busy
        incr busy 
        ML -type void "GoalTrees.closeI $vals($w,focus) $vals($w,goalstack)"
        ML -type void "GoalTrees.modify_proofI GTrees.trim $vals($w,goalstack)"
        TkGoalProof::NextFocus $w
        TkGoalProof::tactic_change $w
        incr busy -1
}

proc TkGoalProof::delete_at_focus { w } {
        global vals
        global busy
        incr busy 
        ML -type void "GoalTrees.closeI $vals($w,focus) $vals($w,goalstack)"
        ML -type void "GoalTrees.modify_proofI (GTrees.delete_at_focus (!$vals($w,focus))) $vals($w,goalstack)"
        TkGoalProof::NextFocus $w
        TkGoalProof::tactic_change $w
        incr busy -1
}

proc TkGoalProof::insert_space_before_focus { w } {
        global vals
        global busy
        incr busy 
        ML -type void "GoalTrees.closeI $vals($w,focus) $vals($w,goalstack)"
        ML -type void "GoalTrees.modify_proofI (GTrees.insert_space_before_focus (!$vals($w,focus))) $vals($w,goalstack)"
        TkGoalProof::NextFocus $w
        TkGoalProof::tactic_change $w
        incr busy -1
}
#----------------------------------------------------------------------------
# TkGoalProof::fast_forward
#
#----------------------------------------------------------------------------

proc TkGoalProof::fast_forward { w update } {
        global vals
        if $vals($w,savable) {
            errormessage $w.error "The goal is already proved"
            return
        }
        global busy
        incr busy 
        while {!$vals($w,savable)} {
            set nexttac [$w.nexttac.b.tac text get 1.0 end]
            if {$nexttac=="........"} {
                break
            }
            set status [TkGoalProof::ApplyML $w $nexttac -update $update]
            TkGoalProof::DisplayNextTac $w
            if {!$status} { break }
        }
        if !$vals($w,savable) {
            TkGoalProof::goal_and_focus_change $w
        }
        incr busy -1
}



#----------------------------------------------------------------------------
# Functions to display the next/previous tactic/goal in response
# to keyboard shortcuts Meta-N/P
#----------------------------------------------------------------------------

proc TkGoalProof::history_tac_shift { w inc } {
        global vals
        global TkGoalProof_tactic_history
        incr vals($w,next_tac_num) $inc
        set text [lindex $TkGoalProof_tactic_history $vals($w,next_tac_num)]
        if {$vals($w,next_tac_num) == -2} {
            incr vals($w,next_tac_num) [expr -$inc] 
            $w.nexttac.b.tac text delete 1.0 end
        } elseif {$vals($w,next_tac_num) == [llength $TkGoalProof_tactic_history]} {
            incr vals($w,next_tac_num) [expr -$inc] 
        } else {
            $w.nexttac.b.tac text delete 1.0 end
            $w.nexttac.b.tac text insert 1.0 $text
        }
}

proc TkGoalProof::history_goal_shift { w inc } {
        global vals
        global TkGoalProof_goal_history
        incr vals($w,next_goal_num) $inc
        set text [lindex $TkGoalProof_goal_history $vals($w,next_goal_num)]
        if {$vals($w,next_goal_num) == -2} {
            incr vals($w,next_goal_num) [expr -$inc] 
            $w.goals editpreterm [list TEXT ""]
        } elseif {$vals($w,next_goal_num) == [llength $TkGoalProof_goal_history]} {
            incr vals($w,next_goal_num) [expr -$inc] 
        } else {
            $w.goals editpreterm [list TEXT $text]
        }
}

proc TkGoalProof::add_to_history { history_var depth text } {
        upvar #0 $history_var history
        set where [lsearch -exact $history $text]
        if {$where==-1} {
            set history [lrange $history 0 [expr {$depth - 2}]]
        } else {
            set history [lreplace $history $where $where]
        }
        set history [linsert $history 0 $text]
}

#----------------------------------------------------------------------------
# TkGoalProof::ApplyML
#
# Expands the given ml code as a tactic.  
#----------------------------------------------------------------------------

proc TkGoalProof::ApplyML { w ml {updateflag -update} {update 1} } {

        global vals
        global TkGoalProof_flags
        global TkGoalProof_tactic_history
        global feedback
        global busy
        incr busy
        
        # 1. Alter the focus to reflect the focus selected in the listbox
        # widget.

#       set focus [lindex [$w.goals listbox curselection] 0]
#        ML -type void "GoalTrees.switch_subgoalI ($focus+1) $vals($w,goalstack) $vals($w,focus)"

        # 2. Adjust the tactic history list
        # Only keep the list to depth $TkGoalProof_flags(tactic_history,depth).
        # Also remake the tactic history menu

        TkGoalProof::add_to_history  TkGoalProof_tactic_history $TkGoalProof_flags(tactic_history,depth) $ml
        
        # 3. Apply the tactic.  Don't do anything permanent before
        # here (except adding to the history) because evaluating 
        # the tactic may cause an exception.

        regsub -all \n $ml " " tac
        set realml "GoalTrees.expandI [ml_bool $TkGoalProof_flags(detectNoChange)] ($tac, [ml_string $tac]) $vals($w,focus) $vals($w,goalstack)"
        set feedback [list $w "Applying tactic..."]
        if [catch {ML -type void -check 1  $realml} message] {
            errormessage $w.error $message
            incr busy -1
            return 0
        }
        
        # 4. The tactic application has done something.
        # Calculate the new focus and display the new goals.

        if $update { TkGoalProof::remake_tactic_history_menu $w }
        set vals($w,savable) [ML -type bool "GoalTrees.top_is_provedI $vals($w,goalstack)"]
        if $vals($w,savable) {
            set vals($w,saved) 0
            ML -type void "GoalTrees.top_modify_proofI GTrees.adjust_THENLs $vals($w,goalstack)"
            TkGoalProof::DisplayTheorem $w
            TkGoalProof::DisplayEntireTac $w
        } else {
            TkGoalProof::NextFocus $w
            if $update { TkGoalProof::goal_and_focus_change $w }
        }

        # 6. If the theorem is proved display the theorem and
        # pop up the save box.

        if $vals($w,savable) {
            TkGoalProof::save $w
        }
        incr busy -1
        return 1
}

#----------------------------------------------------------------------------
# TkGoalProof::place_in_buffer
#
#----------------------------------------------------------------------------

proc TkGoalProof::place_in_buffer { w ml {replace {}} } {
        global vals
        global busy
        incr busy

        set origml $ml  
        set isdelayed [regexp {<([^ ]*)>} $ml dummy type_needed]
        if $isdelayed { 
            switch -- $type_needed term { 
                set real_type_needed HOL_TERM
            } default { 
                set real_type_needed $type_needed
            }
            regsub <(.*)> $ml "" ml
            set adjust [expr {[string length $ml] - [string first < $origml]}]
        } else {
            set adjust 0
        }
        if {$replace=="-replace"} {
            $w.nexttac.b.tac text delete 1.0 end
        }
        $w.nexttac.b.tac text insert insert $ml
        $w.nexttac.b.tac text mark set insert "insert - $adjust chars"
        focus_goTo [$w.nexttac.b.tac text]
        incr busy -1    
}

proc TkGoalProof::ConvertSubterm { w conv } {
        global vals
        global busy
        incr busy
        TkGoalProof::delayedLoadPattTacs
        set termpattern [selection get HOL_TERMPATTERN]
        set tac "PATT_CONV_TAC $termpattern [ml_opt_paren $conv]"
        TkGoalProof::place_in_buffer $w $tac -replace
        incr busy -1    
}

#----------------------------------------------------------------------------
#
#----------------------------------------------------------------------------

proc gdp { } {
   after 1 {
        source $TkGoalProof_library/src/TkGoalProof.tcl
        newwin TkGoalProof
   }
}

