(*--------------------------------------------------------------------------*)
(*                  Copyright (c) Donald Syme 1992                          *)
(*                  All rights reserved                                     *)
(*                                                                          *)
(* Donald Syme, hereafter referred to as `the Author', retains the copyright*)
(* and all other legal rights to the Software contained in this file,       *)
(* hereafter referred to as `the Software'.                                 *)
(*                                                                          *)
(* The Software is made available free of charge on an `as is' basis. No    *)
(* guarantee, either express or implied, of maintenance, reliability,       *)
(* merchantability or suitability for any purpose is made by the Author.    *)
(*                                                                          *)
(* The user is granted the right to make personal or internal use of the    *)
(* Software provided that both:                                             *)
(* 1. The Software is not used for commercial gain.                         *)
(* 2. The user shall not hold the Author liable for any consequences        *)
(*    arising from use of the Software.                                     *)
(*                                                                          *)
(* The user is granted the right to further distribute the Software         *)
(* provided that both:                                                      *)
(* 1. The Software and this statement of rights are not modified.           *)
(* 2. The Software does not form part or the whole of a system distributed  *)
(*    for commercial gain.                                                  *)
(*                                                                          *)
(* The user is granted the right to modify the Software for personal or     *)
(* internal use provided that all of the following conditions are observed: *)
(* 1. The user does not distribute the modified software.                   *)
(* 2. The modified software is not used for commercial gain.                *)
(* 3. The Author retains all rights to the modified software.               *)
(*                                                                          *)
(* Anyone seeking a licence to use this software for commercial purposes is *)
(* invited to contact the Author.                                           *)
(*--------------------------------------------------------------------------*)



(* 
 * use "hol90_replay/src/script_fragments.sig"; 
 * use "/homes/drs1004/projects/tkhol/hol90/replay/src/script_fragments.sml"; 
 * val s = "load_theory \"prim_rec\"; new_theory \"arithmetic\";";
 * val s = "load_theory \"...\";";
 * val s = "[]";
 * val s = "[1,2]";
 * 
 * val f = "/homes/drs1004/hol90/theories/src/mk_arithmetic.sml";
 * val f = "/homes/drs1004/projects/tkhol/hol90_replay/tests/t1.sml";
 * val x = rev (ScriptFragments.analyse_script(open_in f));
 * map (fn s => (#describe s) ()) x;
 * map (fn s => (#edit_tool s) ()) x;
 * map (fn s => (#pseudonyms s) ()) x;
 * map (fn s => (#data s) ()) x;
*)

structure ScriptFragments : ScriptFragments_sig  =
struct

structure Ast = System.Ast;
structure BwdProofs = BwdProofs;

local
open SMLPP;
open SmlToStrings;
open System.Ast;
open System.Symbol;
open System.Compile;
open System.Env;
in
type data = exn;
type fragment = {
          origin : (string * (int * int)) option,
          describe : string,
          edit_tool : string,
          data : data,
          pseudonyms : string list
        }
fun mk_fragment {describe,edit_tool,data,pseudonyms} = 
    {origin=NONE,describe=describe,edit_tool=edit_tool,data=data,pseudonyms=pseudonyms}
val dest_fragment = I
fun set_origin origin {describe,edit_tool,data,pseudonyms,origin=_} = 
    {origin=SOME origin,describe=describe,edit_tool=edit_tool,data=data,pseudonyms=pseudonyms}

infix |>
fun a |> b = b a

type script = (fragment * (int * int)) list
val mk_script = I
val dest_script = I

fun unstring_exp (StringExp e) = e
  | unstring_exp x = exp_to_string x;
fun unvar_pat (VarPat [e]) = System.Symbol.name e
  | unvar_pat p = pat_to_string p;
fun concatl l = itlist (curry (op ^)) l ""
fun join [] c = ""
  | join l c = end_itlist (fn s1 => fn s2 => s1 ^ c ^ s2) l
fun join2 [] c1 c2 = ""
  | join2 [x] c1 c2 = x
  | join2 [x,y] c1 c2 = x ^ c2 ^ y
  | join2 (h::t) c1 c2 = h ^ c1 ^ (join2 t c1 c2)
fun trim (n,s) = String.substring (s,0,n) ^ "..." handle _ => s

fun quote s = "\"" ^ s ^ "\""

open BwdProofs;

fun dest_BWDPROOF (BWDPROOF x) = x
exception NOT_RECOGNISED
fun bwdproof_of_exp exp = 
   (case exp of
     AppExp {argument=TupleExp [exp1,exp2],function=VarExp [sym]} =>
        (case (System.Symbol.name sym) of
             "THEN" => 
                 let val lhs = dest_BWDPROOF (bwdproof_of_exp exp1)
                     val rhs = dest_BWDPROOF (bwdproof_of_exp exp2)
                 in
                     BWDPROOF (lhs@rhs)
                 end
           | "THENL" => 
                 let val cases = map bwdproof_of_exp (SMLDestructors.DList exp2)
                     val lhs = dest_BWDPROOF (bwdproof_of_exp exp1)
                 in
               BWDPROOF (lhs@[BWDTHENL cases])
                 end
           | _ => raise NOT_RECOGNISED)
    | _ => raise NOT_RECOGNISED)
  handle NOT_RECOGNISED => BWDPROOF [BWDTAC (exp_to_string exp)]

type fragment_matcher = dec -> fragment;
val mk_fragment_matcher = I
val dest_fragment_matcher = I

(*-----------------------------------------------------------------------
 * The Matcher, with fallback to "unknown" fragment.
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_UNKNOWN
fun unknown_decl_frag dec = 
mk_fragment {
   describe= 
      case dec of
         System.Ast.ValDec vbl => 
            "Declare " ^ (join2 (map (pat_to_string o (fn (Vb {pat,exp}) => pat)) vbl) "," " and ")
       | dec =>
            "Declaration: " ^ trim(40,dec_to_string dec),
   edit_tool="",
   data= DATA_IS_FOR_UNKNOWN,
   pseudonyms =  []
}

val matchers = ref ([]:fragment_matcher list);
fun new_matcher fragment_matcher =
    matchers := fragment_matcher::(!matchers);

fun match x =
   tryfind (fn f => f x) (!matchers)
   handle _ => unknown_decl_frag x;


(*-----------------------------------------------------------------
 * File analysis
 *-----------------------------------------------------------------*)

val collapseLocals = ref false;
fun analyse_dec file (MarkDec (dec,from,to)) =
       (match (SMLDestructors.unmark_dec dec)) |> set_origin (file,(from,to))

exception NOT_SPECIAL;
fun collapse_dec file dec = 
    (case (case dec of 
              MarkDec (dec,_,_) => dec 
            | dec => dec) of
        SeqDec decs =>
           flatten (map (collapse_dec file) decs)
      | LocalDec (dec1,dec2) =>
           if (!collapseLocals)
           then (collapse_dec file dec1)@(collapse_dec file dec2)
           else raise NOT_SPECIAL
      | _ => raise NOT_SPECIAL)
    handle NOT_SPECIAL =>
       [analyse_dec file dec]

datatype analysis = ERRORS of string | FRAGMENTS of fragment list
fun errorfile () = 
   let val pid = System.Unsafe.CInterface.getpid()
   in "/tmp/sml.errors." ^ Integer.makestring(pid)
   end

fun fetch_errors () =
   let val error_in = open_in(errorfile())
       val errors = input (error_in,10000)
       val _ = close_in(error_in)
       val _ = System.Unsafe.SysIO.unlink (errorfile())
   in
       errors
   end;

fun analyse_script input_file =
   let val error_out = open_out(errorfile())
       val input_in = open_in input_file
       val error_consumer = {
            consumer= curry output error_out,
            flush= fn () => (),
            linewidth=65
           }
       val res = 
          let val source = makeSource(input_file, 1, input_in, false, error_consumer)
              val (SeqDec decs,newenv) = parse(source,staticPart(layerEnv (!topLevelEnvRef, !pervasiveEnvRef)))
          in
              FRAGMENTS (collapse_dec input_file (SeqDec (rev decs)))
          end
          handle err => (close_out error_out; close_in input_in; ERRORS (fetch_errors()))
       val _ = close_out error_out
       val _ = close_in input_in
   in
       res
   end;

(*-----------------------------------------------------------------------
 * Matcher utilities 
 *---------------------------------------------------------------------*)


fun lookup_record symname (RecordExp components) =
    let fun find_it (sym,ast) =
        if (System.Symbol.name sym = symname) then ast
        else raise Match 
    in
        tryfind find_it components
    end


(*-----------------------------------------------------------------------
 * Arbitrary (presumably side-effecting) ML executions
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_EXECUTION
fun execution_frag {exp} = mk_fragment {
   describe= "Execute " ^ trim(15,exp_to_string exp),
   edit_tool= "",
   data= DATA_IS_FOR_EXECUTION,
   pseudonyms =  []
};

fun match_execution (ValDec [Vb {exp=exp,pat=VarPat [sym]}]) =
    if (System.Symbol.name sym = "it")
    then execution_frag {exp=exp}
    else raise Match
  | match_execution _ = raise Match
val _ = new_matcher match_execution;


(*-----------------------------------------------------------------------
 * HOL-specific matchers...
 *---------------------------------------------------------------------*)

(*-----------------------------------------------------------------------
 * load_theory
 *---------------------------------------------------------------------*)


exception DATA_IS_FOR_LOAD_THEORY
fun load_theory_frag {theory: System.Ast.exp} = mk_fragment {
   describe= "Load the theory " ^ quote (unstring_exp theory),
   edit_tool= "",
   data= DATA_IS_FOR_LOAD_THEORY,
   pseudonyms =  []
};


fun match_load_theory 
       (ValDec
          [Vb
             {exp=AppExp
                    {argument=theory,
                     function=VarExp [funsym]},
              pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "load_theory")
    then load_theory_frag {theory=theory}
    else raise Match
  | match_load_theory _ = raise Match
val _ = new_matcher match_load_theory;

(*-----------------------------------------------------------------------
 * extend_theory
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_EXTEND_THEORY
fun extend_theory_frag {theory: System.Ast.exp} = mk_fragment {
   describe= "Extend the theory " ^ quote (unstring_exp theory),
   edit_tool= "",
   data= DATA_IS_FOR_EXTEND_THEORY,
   pseudonyms =  []
};

fun match_extend_theory 
       (ValDec
          [Vb
             {exp=AppExp
                    {argument=theory,
                     function=VarExp [funsym]},
              pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "extend_theory")
    then load_theory_frag {theory=theory}
    else raise Match
  | match_extend_theory _ = raise Match
val _ = new_matcher match_extend_theory;

(*-----------------------------------------------------------------------
 * new_theory
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_NEW_THEORY
fun new_theory_frag {theory: System.Ast.exp} = mk_fragment {
   describe= "Create the theory " ^ quote (unstring_exp theory),
   edit_tool= "",
   data=DATA_IS_FOR_NEW_THEORY,
   pseudonyms =  []

};

fun match_new_theory 
       (ValDec
          [Vb
             {exp=AppExp
                    {argument=theory,
                     function=VarExp [funsym]},
              pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "new_theory")
    then new_theory_frag {theory=theory}
    else raise Match
  | match_new_theory _ = raise Match
val _ = new_matcher match_new_theory;

(*-----------------------------------------------------------------------
 * close_theory
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_CLOSE_THEORY
val close_theory_frag = mk_fragment {
   describe= "Switch to proof mode",
   edit_tool= "",
   data= DATA_IS_FOR_CLOSE_THEORY,
   pseudonyms =  []
};

fun match_close_theory 
       (ValDec
          [Vb
             {exp=AppExp
                    {argument=RecordExp [],
                     function=VarExp [funsym]},
              pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "close_theory")
    then close_theory_frag
    else raise Match
  | match_close_theory _ = raise Match
val _ = new_matcher match_close_theory;

(*-----------------------------------------------------------------------
 * export_theory
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_EXPORT_THEORY
val export_theory_frag = mk_fragment {
   describe= "Export the current theory",
   edit_tool= "export_theory",
   data= DATA_IS_FOR_EXPORT_THEORY,
   pseudonyms =  []
};

fun match_export_theory 
       (ValDec
          [Vb
             {exp=AppExp
                    {argument=RecordExp [],
                     function=VarExp [funsym]},
              pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "export_theory")
    then export_theory_frag
    else raise Match
  | match_export_theory _ = raise Match
val _ = new_matcher match_export_theory;


(*-----------------------------------------------------------------------
 * load_library
 *---------------------------------------------------------------------*)


exception DATA_IS_FOR_LOAD_LIBRARY
fun load_library_frag {lib: System.Ast.exp, theory: System.Ast.exp} = mk_fragment {
   describe= "Load the library " ^ quote (unstring_exp lib),
   edit_tool= "",
   data= DATA_IS_FOR_LOAD_LIBRARY,
   pseudonyms =  []

};

fun match_load_library
 (ValDec
    [Vb
       {exp=AppExp
              {argument=record_ast,
               function=VarExp [funsym]},
        pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "load_library")
    then 
       let val lib = lookup_record "lib" record_ast
           val theory = lookup_record "theory" record_ast
       in
           load_library_frag {lib=lib,theory=theory}
       end
    else raise Match
  | match_load_library _ = raise Match
val _ = new_matcher match_load_library;

   
(*-----------------------------------------------------------------------
 * new_type_definition
 *---------------------------------------------------------------------*)


exception DATA_IS_FOR_NEW_TYPE_DEFINITION of {name:string,pred:string,inhab_thm:string}
fun data_for_new_type_definition e = 
    (raise e) handle DATA_IS_FOR_NEW_TYPE_DEFINITION data => data

fun new_type_definition_frag (data as {name,pred,inhab_thm}) = mk_fragment {
   describe= "Introduce type " ^ quote name ^ " (new_type_definition)",
   edit_tool= "NewTypeIntroduction",
   data= DATA_IS_FOR_NEW_TYPE_DEFINITION data,
   pseudonyms = [name]
};

fun match_new_type_definition
       (ValDec [Vb {exp=AppExp {argument=record_ast,
                                function=VarExp [funsym]}, 
                    pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "new_type_definition")
    then 
       let val name = lookup_record "name" record_ast
           val pred = lookup_record "pred" record_ast
           val inhab_thm = lookup_record "inhab_thm" record_ast
       in
           new_type_definition_frag {name=unstring_exp name,pred=exp_to_string pred,inhab_thm=exp_to_string inhab_thm}
       end
    else raise Match
  | match_new_type_definition _ = raise Match
val _ = new_matcher match_new_type_definition;

(*-----------------------------------------------------------------------
 * define_type
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_DEFINE_TYPE of {name:string,fixities:string list,type_spec:string} 
fun data_for_define_type e = 
    (raise e) handle DATA_IS_FOR_DEFINE_TYPE data => data

exception TRIM_END
fun trim_end' (h::t) l2 =
   if (h::t)=l2
   then []
   else h::(trim_end' t l2)
  | trim_end' _ _ = raise TRIM_END 
fun trim_end s1 s2 = implode (trim_end' (explode s1) (explode s2))

fun define_type_frag (data as {name,fixities,type_spec}) =
  let val type_name = 
           (trim_end name "_Axiom" 
            handle TRIM_END => trim_end name "_AXIOM"
            handle TRIM_END => trim_end name "_axiom"
            handle _ => name)
   in mk_fragment {     
     describe= "Define recursive type - " ^ quote (type_name),
     edit_tool= "TkDefineType",
     data= DATA_IS_FOR_DEFINE_TYPE data,
     pseudonyms =  [
         name, 
         type_name ^ "_Induct", 
         type_name ^ "_INDUCT", 
         type_name ^ "_11", 
         type_name ^ "_CASES",
         type_name ^ "_Cases", 
         type_name ^ "_cases", 
         type_name ^ "_DISTINCT", 
         type_name ^ "_Distinct", 
         type_name ^ "_distinct"
     ]
   }
   end;

fun match_define_type
       (ValDec [Vb {exp=AppExp {argument=record_ast,
                                function=VarExp [funsym]},
                    pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "define_type")
    then 
       let val name = lookup_record "name" record_ast
           val fixities = lookup_record "fixities" record_ast
           val type_spec = lookup_record "type_spec" record_ast
       in
           define_type_frag {name=unstring_exp name,fixities=[exp_to_string fixities],type_spec=exp_to_string type_spec}
       end
    else raise Match
  | match_define_type _ = raise Match
val _ = new_matcher match_define_type;

(*-----------------------------------------------------------------------
 * new_recursive_definition
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_NEW_RECURSIVE_DEFINITION of {def:string, fixity:string, name:string, rec_axiom:string} 
fun data_for_new_recursive_definition e = 
    (raise e) handle DATA_IS_FOR_NEW_RECURSIVE_DEFINITION data => data

fun new_recursive_definition_frag (data as {def,fixity,name,rec_axiom}) = 
mk_fragment {
   describe= "Define " ^ quote name ^ " (new_recursive_definition)",
   edit_tool= "NewRecDef",
   data= DATA_IS_FOR_NEW_RECURSIVE_DEFINITION data,
   pseudonyms = [name]
};


fun match_new_recursive_definition
       (ValDec [Vb {exp=AppExp {argument=record_ast,
                               function=VarExp [funsym]},
                    pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "new_recursive_definition")
    then 
       let val def = lookup_record "def" record_ast
           val fixity = lookup_record "fixity" record_ast
           val name = lookup_record "name" record_ast
           val rec_axiom = lookup_record "rec_axiom" record_ast
       in
           new_recursive_definition_frag {def=exp_to_string def,fixity=exp_to_string fixity,name=unstring_exp name,rec_axiom=exp_to_string rec_axiom}
       end
    else raise Match
  | match_new_recursive_definition _ = raise Match
val _ = new_matcher match_new_recursive_definition;

(*-----------------------------------------------------------------------
 * new_definition
 * new_infix_definition
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_NEW_DEFINITION of {fixity: string, def:string, name:string}
fun data_for_new_definition e = 
    (raise e) handle DATA_IS_FOR_NEW_DEFINITION data => data


fun new_definition_frag (data as {def,name,fixity}) = mk_fragment {
   describe= "Define " ^ quote name,
   edit_tool= "NewDef",
   data= DATA_IS_FOR_NEW_DEFINITION data,
   pseudonyms =  [name]
};

fun match_new_definition
       (ValDec [Vb {exp=AppExp {argument=TupleExp [name',def,precedence],
                                function=VarExp [funsym]},
                    pat=VarPat [patsym]}]) =
    if (System.Symbol.name funsym = "new_infix_definition")
    then new_definition_frag {fixity="Infix " ^ exp_to_string precedence,name=unstring_exp name',def=exp_to_string def}
    else raise Match

  | match_new_definition
       (ValDec [Vb {pat=VarPat [patsym],
                    exp=AppExp {function=VarExp [funsym],
                                argument=TupleExp [name',def]}}]) =
    if (System.Symbol.name funsym = "new_definition")
    then new_definition_frag {fixity="Prefix",name=unstring_exp name',def=exp_to_string def}
    else raise Match

  | match_new_definition
       (ValDec [Vb {pat=VarPat [patsym],
                    exp=AppExp {function=VarExp [funsym],
                                argument=TupleExp [name',def]}}]) =
    if (System.Symbol.name funsym = "new_binder_definition")
    then new_definition_frag {fixity="Binder",name=unstring_exp name',def=exp_to_string def}
    else raise Match

  | match_new_definition _ = raise Match
val _ = new_matcher match_new_definition;


(*-----------------------------------------------------------------------
 * Forward Proofs
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_FWDPROOF of {save: bool, name: string,derivation: System.Ast.exp}
fun data_for_fwdproof e = 
    (raise e) handle DATA_IS_FOR_FWDPROOF data => data


fun fwdproof_frag (data as {save,name,derivation}) = mk_fragment {
   describe= if save 
             then "Prove theorem " ^ quote name ^ " (forward)"
             else "Prove lemma " ^ quote name ^ " (forward)",
   edit_tool= "TkFwdProof",
   data= DATA_IS_FOR_FWDPROOF data,
   pseudonyms =  [name]
};

fun match_fwdproof 
       (ValDec [Vb {pat=VarPat [patsym],
                    exp=AppExp {function=VarExp [funsym],
                                argument=TupleExp [name',derivation]}}]) =
    if (System.Symbol.name funsym = "save_thm")
    then fwdproof_frag {save=true,name=unstring_exp name',derivation=derivation}
    else raise Match
  | match_fwdproof _ = raise Match
val _ = new_matcher match_fwdproof;


(*----------------------------------------------------------------------
 * Backward Proofs
 *      val x = store_thm(_,_,_)
 *      val x = save_thm(_,prove(_,_)
 *      val x = save_thm(_,TAC_PROOF(_,_)
 *---------------------------------------------------------------------*)

exception DATA_IS_FOR_BWDPROOF of {theory: string option,name: string,goal: string, proof: BwdProofs.bwdproof}
fun data_for_bwdproof e = 
    (raise e) handle DATA_IS_FOR_BWDPROOF data => data

fun bwdproof_frag (data as {theory,name,goal,proof}) = mk_fragment {
   describe= case theory of
                SOME theory => "Prove theorem " ^ quote name ^ " (backward)"
              | NONE => "Prove lemma " ^ quote name ^ " (backward)",
   edit_tool= "TkGoalProof",
   data= DATA_IS_FOR_BWDPROOF data,
   pseudonyms = [name]
};

fun match_bwdproof 
       (ValDec [Vb {pat=mlbinding,
                    exp=AppExp {function=VarExp [funsym],
                                argument=args}}]) =
   (if (System.Symbol.name funsym = "store_thm" orelse System.Symbol.name funsym = "prove_thm")
    then case args of
       TupleExp [name',goal,proof] =>
             bwdproof_frag {theory=SOME "-", name=unstring_exp name',goal=exp_to_string goal,proof=bwdproof_of_exp proof}
     | _ => (raise Match)
    else (raise Match)

    handle Match =>
    if (System.Symbol.name funsym = "save_thm")
    then case args of 
       TupleExp [name',AppExp {function=VarExp [funsym2],argument=TupleExp [goal,proof]}] =>
          if (System.Symbol.name funsym2 = "prove" orelse System.Symbol.name funsym = "TAC_PROOF")
          then bwdproof_frag {theory=SOME "-", name=unstring_exp name',goal=exp_to_string goal,proof=bwdproof_of_exp proof}
          else (raise Match)
     | _ => (raise Match)
    else (raise Match)

    handle Match =>
    if (System.Symbol.name funsym = "prove" orelse System.Symbol.name funsym = "TAC_PROOF")
    then case args of
       TupleExp [goal,proof] =>
           bwdproof_frag {theory=NONE, name=unvar_pat mlbinding,goal=exp_to_string goal,proof=bwdproof_of_exp proof}
     | _ => (raise Match)
    else (raise Match))

  | match_bwdproof _ = (raise Match)

   
val _ = new_matcher match_bwdproof;

end; (* local *)

end;


