(*--------------------------------------------------------------------------*)
(*                  Copyright (c) Donald Syme 1992                          *)
(*                  All rights reserved                                     *)
(*                                                                          *)
(* Donald Syme, hereafter referred to as `the Author', retains the copyright*)
(* and all other legal rights to the Software contained in this file,       *)
(* hereafter referred to as `the Software'.                                 *)
(*                                                                          *)
(* The Software is made available free of charge on an `as is' basis. No    *)
(* guarantee, either express or implied, of maintenance, reliability,       *)
(* merchantability or suitability for any purpose is made by the Author.    *)
(*                                                                          *)
(* The user is granted the right to make personal or internal use of the    *)
(* Software provided that both:                                             *)
(* 1. The Software is not used for commercial gain.                         *)
(* 2. The user shall not hold the Author liable for any consequences        *)
(*    arising from use of the Software.                                     *)
(*                                                                          *)
(* The user is granted the right to further distribute the Software         *)
(* provided that both:                                                      *)
(* 1. The Software and this statement of rights are not modified.           *)
(* 2. The Software does not form part or the whole of a system distributed  *)
(*    for commercial gain.                                                  *)
(*                                                                          *)
(* The user is granted the right to modify the Software for personal or     *)
(* internal use provided that all of the following conditions are observed: *)
(* 1. The user does not distribute the modified software.                   *)
(* 2. The modified software is not used for commercial gain.                *)
(* 3. The Author retains all rights to the modified software.               *)
(*                                                                          *)
(* Anyone seeking a licence to use this software for commercial purposes is *)
(* invited to contact the Author.                                           *)
(*--------------------------------------------------------------------------*)



structure GoalTrees : GoalTrees_sig = 
struct
structure Thm = Thm;
structure GTrees = GTrees;

(* open GTrees; *)

fun GOALTREES_ERR(func,mesg) =
     HOL_ERR{origin_structure = "GoalTrees",
             origin_function = func,
             message = mesg}

fun ref_apply f r = (r := (f (!r)));
fun ref_project f r = f (!r);

(* -------------------------------------------------------------------------
 * Undo/History/Proof Fragment Memory
 *
 * History provides "registers-with-bounded-history". You can read (via
 * "project"), write (via "apply"), and undo. This is a general purpose
 * type.
 *
 *--------------------------------------------------------------------------*)

(*abstype*) datatype 'a history = HISTORY of {obj :'a, past :'a list, limit :int}
(*with*)

   exception CANT_BACKUP_ANYMORE;

   fun new_history {obj, limit} = HISTORY{obj = obj, past = [], limit = limit}
   
   local
   fun chop n alist = fst (Lib.split_after n alist) handle _ => alist
   in
   fun apply f (HISTORY{obj, past, limit}) = 
         HISTORY{obj = f obj, past = chop limit (obj :: past), limit = limit}
   fun topapply f (HISTORY{obj, past, limit}) = 
         HISTORY{obj = f obj, past = past, limit = limit}
   fun set_limit (HISTORY{obj,past,limit}) n =
      if (n<0) then raise GOALTREES_ERR("set_limit","negative limit")
      else HISTORY{obj = obj, past = chop n past, limit = n}
   end;
   
   fun project f (HISTORY{obj, ...}) = f obj;
   
   fun undo f (HISTORY{past = h::rst, limit,obj=obj}) = 
             HISTORY{obj = f {popped=obj,recovered=h}, past = rst, limit = limit}
     | undo _ (HISTORY{past = [], ...}) = raise CANT_BACKUP_ANYMORE
   
(*end; *)


(* -------------------------------------------------------------------------
 * Goalstacks with history and tricky stickiness in backup.
 *
 * The trick here is to modify
 * previous states with attempted proofs so that when backups
 * happen the last attempted proof is there waiting to be re-edited.
 *--------------------------------------------------------------------------*)

type proofstate = (GTrees.pstate * GTrees.focus) history;
fun get_pstate (a,b) = a
fun get_focus (a,b) = b

exception NO_EXTANT_BWDPROOFS;

fun set_goals g = new_history{obj = (GTrees.set_goals g,([]:GTrees.focus)), limit = 15};
val empty_goal = new_history{obj = (GTrees.empty_goal,([]:GTrees.focus)), limit = 15};

(* We need to get a new focus in the old proof - next_focus
finds the focus "closest' to the old focus. *)
fun backup_and_retain_attempt s = 
    let val new_history =
        undo (fn {popped=((attempt,_),_),recovered=((_,gtree),oldfocus)} => 
               ((attempt,gtree),oldfocus)) s
    in (new_history,GTrees.next_focus (project get_pstate new_history) (project get_focus s)) 
    end
fun backup_and_discard_attempt s = 
    (undo (fn {popped,recovered} => recovered) s,
     project get_focus s) 

fun set_backup i s = set_limit s i;

fun expand args (tactic,tac) focus s = 
    apply (fn (pstate,_) => (GTrees.expand args (tactic,tac) focus pstate, focus)) s
fun close focus s = 
    topapply (fn (pstate,f) => (GTrees.close focus pstate, f)) s
fun modify_proof func s = 
    apply (fn (pstate,f) => (GTrees.modify_proof func pstate,f)) s
fun top_modify_proof func s = 
    topapply (fn (pstate,f) => (GTrees.modify_proof func pstate,f)) s

val first_focus = project (GTrees.first_focus o get_pstate);
val active_foci = project (GTrees.active_foci o get_pstate);
val next_focus = project (GTrees.next_focus o get_pstate);

val focus_is_active = project (GTrees.focus_is_active o get_pstate);
val focus_is_opened = project (GTrees.focus_is_opened o get_pstate);
val THENL_mismatch = project (GTrees.THENL_mismatch o get_pstate);
val current_tac = project (GTrees.current_tac o get_pstate);
val propositions_at_focus = project (GTrees.propositions_at_focus o get_pstate);
val propositions_at_THENL_start = project (GTrees.propositions_at_THENL_start o get_pstate);
val at_THENL_start = project (GTrees.at_THENL_start o get_pstate);
val focused_case_at_THENL_start = project (GTrees.focused_case_at_THENL_start o get_pstate);
fun refocus_at_THENL_start i = project (GTrees.refocus_at_THENL_start i o get_pstate);

val top_is_proved = project (GTrees.top_is_proved o get_pstate);
val final_thm = project (GTrees.final_thm o get_pstate);
val initial_goal = project (GTrees.initial_goal o get_pstate);
val entire_partial_proof = project (GTrees.entire_partial_proof o get_pstate);
val executed_proof = project (GTrees.executed_proof o get_pstate);
val entire_proof = project (GTrees.entire_proof o get_pstate);

(* -------------------------------------------------------------------------
 * Imperative versions, side effecting on a focus/goaltree reference cell.
 *--------------------------------------------------------------------------*)

fun empty_goalI () = ref empty_goal;
fun set_goalsI gp = ref (set_goals gp);
fun backup_and_retain_attemptI pstate_ref focus_ref = 
   let val (pstate,focus) = backup_and_retain_attempt (!pstate_ref) 
       val _ = (pstate_ref := pstate)
       val _ = (focus_ref := focus)
   in ()
   end;
fun backup_and_discard_attemptI pstate_ref focus_ref = 
   let val (pstate,focus) = backup_and_discard_attempt (!pstate_ref) 
       val _ = (pstate_ref := pstate)
       val _ = (focus_ref := focus)
   in ()
   end;
val set_backupI = ref_apply o set_backup;

fun modify_proofI f = ref_apply (modify_proof f)
fun top_modify_proofI f = ref_apply (top_modify_proof f)
fun expandI args tacs = ref_apply o (ref_project (expand args tacs))
val closeI = ref_apply o (ref_project close)

val first_focusI = ref o (ref_project first_focus)
val active_fociI = ref_project active_foci
val next_focusI = ref_apply o ref_project next_focus
fun refocus_at_THENL_startI i = ref_apply o (ref_project (refocus_at_THENL_start i))

val focus_is_activeI = ref_project o (ref_project focus_is_active)
val focus_is_openedI = ref_project o (ref_project focus_is_opened)
val THENL_mismatchI = ref_project THENL_mismatch
val current_tacI = ref_project o (ref_project current_tac)
val propositions_at_focusI = ref_project o (ref_project propositions_at_focus)
val propositions_at_THENL_startI = ref_project o (ref_project propositions_at_THENL_start)
val at_THENL_startI = ref_project o (ref_project at_THENL_start)
val focused_case_at_THENL_startI = ref_project o (ref_project focused_case_at_THENL_start)


val top_is_provedI = ref_project top_is_proved;
val final_thmI = ref_project final_thm;
val initial_goalI = ref_project initial_goal;
val entire_partial_proofI = ref_project entire_partial_proof;
val executed_proofI = ref_project executed_proof;
val entire_proofI = ref_project entire_proof;

end; (* GoalTrees *)

