#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




proc tksml_slave::ProcessArgs { argc argv } {
        global tksml_slave_flags 
        global gui_flags
        global HolHelp_flags

        if [info exists tksml_slave_flags(default_ml)] {
            set tksml_slave_flags(ml) $tksml_slave_flags(default_ml)
        } else {
            set tksml_slave_flags(ml) "sml"
        }
        set tksml_slave_flags(mltype_strategy) infer
        set tksml_slave_flags(interrupt_handlers) ""
        auto_load Preferences_Add
        if {[llength [info commands Preferences_Add]]==1} {
            Preferences_Add "Communicating with ML" "Preferences related to how the underlying MLL process is run are set here.  Most changes to these options will only take effect after you restart $gui_flags(title). " \
                [list \
                    [list tksml_slave_flags(prompt_strategy) slavePrompt STRING "" "ML prompt" "The (primary) prompt to use for the ML session.  This string should be one unlikely to appear in the output from a normal ML session as $gui_flags(title) needs to detect it in the ML output.  If the value is empty then $gui_flags(title) will choose a prompt for you."] \
                    [list tksml_slave_flags(max_heap) maxHeap INT 70000 "Maximum Heap" "$gui_flags(title) offers some protection for detecting process which are growing too big.  Such NJ-SML processes will exit if they are not stopped.  $gui_flags(title) does this by looking for \[Increasing heap to...\] messages on the output stream.  When the figure in the message is greater than the figure given here, $gui_flags(title) will try to send an interrupt signal to the SML process."] \
                    [list tksml_slave_flags(secondaryprompt_strategy) slaveSecondaryPrompt STRING "" "ML secondary prompt" "The (secondary) prompt to use for the ML session when running ML90.  This string should be one unlikely to appear in the output from a normal ML session as $gui_flags(title) needs to detect it in the ML output.  If the value is empty then $gui_flags(title) will choose a secondary prompt for you."] \
                    [list tksml_slave_flags(trace) slaveTrace BOOL 0 "ML tracing" "Whether to trace the hidden input/output to and from the ML session."] \
                    [list tksml_slave_flags(stdio) slaveStdio BOOL 1 "Mimic ML stdio" "Whether to mimic the stdio of a ML session from the stdin/stdout of\nthis program."] \
                    [list tksml_slave_flags(compatMarker) compatMarker BOOL 1 "Compatability Marker" "Internal to TkHol but may be need adjustment if this go wrong."] \
                    [list tksml_slave_flags(tmpdir) tmpdir STRING "/tmp" "\"tmp\" directory" "Directory for temporary files.  Changes take effect upon restart."] \
             ]
            Preferences_Add "Logging Interactions" "Preferences related to logging your interactions with ML are set here." \
                [list \
                    [list tksml_slave_flags(log) tkSmlSlaveLog BOOL 0 "Log interactions" "Whether to log interactions with the ML session which change the state of the session.  This is similar to constructing a script of the activity within a session.  Only interactions from the interface which do not cause errors are recorded.  All input from the stdin is recorded."] \
                    [list tksml_slave_flags(logfile) tkSmlSlaveLogFile FILE "./log.ml" "Log File" "The file in which to record interactions should logging be turned on.  A change to this value takes effect immediately."] \
                ]                                
                
        }

        set tksml_slave_flags(ml_argument_name) "-ml"
        set tksml_slave_flags(mltype_argument_name) "-mltype"
        set tksml_slave_flags(mltype_equivs) [list "sml sml" "cml cml"]
        set tksml_slave_flags(mltype_infer_patts) [list "sml sml"]

        for {set arg 0} {$arg < $argc} {incr arg} {
            switch -- [lindex $argv $arg] -trace {
                set tksml_slave_flags(trace) 1 
            } -initialprompt {
                set tksml_slave_flags(initialprompt) 1 
            } -prompt {
                incr arg
                set tksml_slave_flags(prompt_strategy) [lindex $argv $arg]
            } -ml_argument_name {
                incr arg
                set tksml_slave_flags(ml_argument_name) [lindex $argv $arg]
            } -mltype_argument_name {
                incr arg
                set tksml_slave_flags(mltype_argument_name) [lindex $argv $arg]
            } -mltype_infer_patt {
                incr arg
                lappend tksml_slave_flags(mltype_infer_patts) [lindex $argv $arg]
            } -mltype_equiv {
                incr arg
                lappend tksml_slave_flags(mltype_equivs) [lindex $argv $arg]
            } $tksml_slave_flags(ml_argument_name) {
                incr arg
                set tksml_slave_flags(ml) [lindex $argv $arg] 
            } $tksml_slave_flags(mltype_argument_name) {
                incr arg
                set tksml_slave_flags(mltype_strategy) [lindex $argv $arg] 
            } -stdio {
                incr arg
                set tksml_slave_flags(stdio) [lindex $argv $arg] 
            } -tmpdir {
                incr arg
                set tksml_slave_flags(tmpdir) [lindex $argv $arg] 
            } -theory {
                incr arg
                set tksml_slave_flags(initial_theory) [lindex $argv $arg] 
            } -library {
                incr arg
                lappend tksml_slave_flags(libraries) [lindex $argv $arg] 
            }
        }
        if {$tksml_slave_flags(mltype_strategy)=="infer"} {
            foreach patt $tksml_slave_flags(mltype_infer_patts) {
                if [regexp [lindex $patt 1] $tksml_slave_flags(ml)] {
                    set tksml_slave_flags(mltype) [lindex $patt 0]
                }
            }
            if ![info exists tksml_slave_flags(mltype)] {
                puts stderr "\nNote: $gui_flags(title) couldn't automatically determine the ML type of  $tksml_slave_flags(ml) so $gui_flags(title) is assuming it's New Jersey Standard ML.  Use \"-mltype cml\" if this is wrong.\n"
                set tksml_slave_flags(mltype) sml
            }
        } else {
            foreach equiv $tksml_slave_flags(mltype_equivs) {
                if {$tksml_slave_flags(mltype_strategy)==[lindex $equiv 1]} {
                    set tksml_slave_flags(mltype) [lindex $equiv 0]
                }
            }
            if ![info exists tksml_slave_flags(mltype)] {
                puts stderr "\n$gui_flags(title) internal error - strange value for $tksml_slave_flags(mltype_strategy).  Assuming mltype is sml."
                set tksml_slave_flags(mltype) sml
            }
        }
        set tksml_slave_flags(cml) 0
        set tksml_slave_flags(sml) 0
#       puts "mltype = $tksml_slave_flags(mltype)"
#       puts "ml = $tksml_slave_flags(ml)"
#       puts "mltype_strategy = $tksml_slave_flags(mltype_strategy)"
        switch -- $tksml_slave_flags(mltype) {
           cml {
                set tksml_slave_flags(cml) 1
           } 
           sml {
                set tksml_slave_flags(sml) 1
           } 
        }
        
        if {[sml] && $tksml_slave_flags(logfile)=="./log.ml"} {
            set tksml_slave_flags(logfile) ./log.sml
        }
        
        if [cml] {
            set tksml_slave_flags(error_strings) [list \
                {unbound or non.*} \
                {syntax error.*} \
                {operator and operand don't agree.*} \
                {Console interrupt.*} \
                {ill\-typed phrase.*} \
                {unclosed string.*} \
                {illegal token.*} \
                {evaluation failed.*} \
                {skipping.*} \
            ]
            set tksml_slave_flags(feedback_strings) ""
        }
        if [sml] {
            set tksml_slave_flags(error_strings) [list \
                {Error: .*} \
                {Interrupt.*} \
                {uncaught .*} \
                {use failed:.*} \
            ]
            set tksml_slave_flags(feedback_strings) [list \
                {\[opening [^\n]*\]} \
                {\[Major collection...} \
                {[0-9][0-9]% used[^\n]*\]} \
            ]
        }
        set tksml_slave_flags(exception_filters) ""

#{Error: syntax error found.*} \
#{Error: operator and operand don't agree.*} \
#{Error: unbound variable or constructor.*} \
#{Error: operator is not a function.*} \
#{Error: syntax error found at.*} \
#{Error: unbound structure.*} \
#{Error: expression and handler don't.*} \
#{Error: tycon arity for label.*} \
#{Error: case object and rules don't agree.*} \
#{Error: illegal token.*} \

}


proc tksml_slave::InitialisePackage { args } {
        global tksml_slave_flags 
        global gui_flags

        # log_user is an Expect command
        exp_log_user $tksml_slave_flags(stdio)

        # Create the slave session.

        tksml_slave::Create $tksml_slave_flags(ml)

}


proc tksml_slave::establish_stdin_processor { } {
       global user_spawn_id

        # ``interact'' never returns, yet sets things to raw mode.
        # Blah.  We don't want raw mode, since this application isn't
        # doing lots of key translations.
        # Hence we have to use ``after'' to get the mode established.

        # Also, every now and then interact drops out with a strange error
        # which I can't work out.  Hence the while loop.
   
        trap tksml_slave::SIGINT SIGINT
        while 1 {
           after 1000 {
             stty -raw echo
           }
           set status [catch {
               interact eof exit 
           } err] 
           if $status { puts stderr "warning: $err" }
         }
}


proc tksml_slave::ShutdownPackage { } {
        tksml_slave::Destroy
}


proc tksml_slave::register_exception_filter { filter } {
        global tksml_slave_flags 
        set tksml_slave_flags(exception_filters) "$tksml_slave_flags(exception_filters)\n$filter"
}


proc tksml_slave::register_error_string { error } {
        global tksml_slave_flags 
        lappend tksml_slave_flags(error_strings) $error
}

proc tksml_slave::register_feedback_string { feedback_pattern } {
        global tksml_slave_flags 
        lappend tksml_slave_flags(feedback_strings) $feedback_pattern
}

proc tksml_slave::Create { exe } {
        global tksml_slave_flags 
        global gui_flags 
        global tksml_slave_library
        global feedback

        global spawn_id

        global ml_prompt
        global ml_secondary_prompt

        # create a named pipe to send the result along


        # Echoing and return/new-line processing is set to the emacs
        # shell default to allow compatibility of output with what
        # emacs expects.  This does not seem to stuff up output
        # when TkHol is run from a shell.
        #
        # 8 bit character processing is important as the output stream
        # contains 8 bit characters for special symbol-font
        # characters like "/\".

        set feedback [list {} "Starting $exe..."]
        set stty_init "cs8 -echo -onlcr"
        set old_log_user [log_user]
        log_user 0
        spawn $exe
        match_max 20000 
        log_user $old_log_user


        # Wait for the first prompt and set up the initial prompt.
        # Beware that the executable may have some other initial prompt
        # - we account for the possibility of it having the 
        # user specified prompt.

        if {$tksml_slave_flags(prompt_strategy)==""} {
            set ml_prompt "$gui_flags(title) [ml_initial_prompt]"
        } else {
            set ml_prompt $tksml_slave_flags(prompt_strategy)
        }
        if {$tksml_slave_flags(secondaryprompt_strategy)==""} {
            set ml_secondary_prompt "$gui_flags(title) [ml_initial_secondary_prompt]"
        } else {
            set ml_secondary_prompt $tksml_slave_flags(secondary_prompt_strategy)
        }

        set timeout 30
        expect -- "\n[ml_initial_prompt]" { } "\n[ml_prompt]" { }
        if [catch {exp_send "\n"}] { 
                puts "\n\"$exe\" could not be executed.  Perhaps it is not a valid executable or is not on your path.  Please specify an executable using the $tksml_slave_flags(ml_argument_name) flag.\n\nExiting...\n\n"
                exit 
        }


        if [cml] {
            ML -toplevel 1 "set_prompt [ml_string $ml_prompt]"
        }
        if [sml] {
            ML -toplevel 1 "
                System.Control.primaryPrompt := [ml_string $ml_prompt];
                System.Control.secondaryPrompt := [ml_string $ml_secondary_prompt]
            "
        }

        # Write out the new prompt so user's aren't confused by
        # the new prompt only appearing when they type something
        # for the first time.  It also encourages users to believe that things
        # are now ready for them to type away.

        if [log_user] {
            puts -nonewline "\n$ml_prompt"
        }

        # - Define the "send_result" function to allow ML calls
        # that return results to be made.

        ML -toplevel 1 "use [ml_string $tksml_slave_library/src/tclresult.sml]"

        # - Define the "tcl" function to allow callbacks from the ML session.
        # This works by writing the sending signals back to the Tk
        # process, and information via a temporary file.
        #
        # - Redefine the quit() function in cml.
        # - Redefine the exit() function in sml.
        #
        # We check if this is already done by looking at the value
        # of Tcl.tksml_compat_marker.  If this variable doesn't exist then we
        # compile the Tcl_Merge code and load the Tcl pid storage location.
        #
        # It is important not to reload the "tcl" routine into an executable
        # if it already exists.  This is because code which uses it
        # may exist in the executable already and we can't force a reload
        # easily.  

        set status [catch {set compat [ML -type int "(!Tcl.tksml_compat_marker)"]}]
        if {$status} {
            ML -toplevel 1 "
fun compile file =
   let val old_interp = !System.Control.interp
   in System.Control.interp := false;
      use file handle e => (System.Control.interp := old_interp; raise e);
      System.Control.interp := old_interp
   end
"
            ML -toplevel 1 "compile [ml_string $tksml_slave_library/src/tcl.sml];
                            Tcl.tcl_pid := [pid];
                            Tcl.tcl_tmpdir := [ml_string $tksml_slave_flags(tmpdir)]"
        }

        # Set up the trap-catch for SIGUSR1's to implement callbacks.
        # Gross but effective.

        trap {
                  # puts "Got a SIGUSR1"
                  global tksml_slave_flags
                  set files [glob -nocomplain $tksml_slave_flags(tmpdir)/tksml_tcl[pid].*]
                  foreach file $files {
                      # puts "Reading tcl from $file"
                      set f [open $file]
                      set command [read $f]
                      close $f
                      exec rm -f $file
                      tksml_slave::external_execution $command
                      if $gui_flags(debug) { puts "Got a SIGUSR1, command = $command" }
                  }
                  # puts "Done with SIGUSR1"
        } SIGUSR1

}

proc tksml_slave::Destroy { } {
        global tksml_slave_flags 
        if [cml] {
           exp_send "really_quit();;\n"
        }
        exec rm -f "$tksml_slave_flags(tmpdir)/tksml*[pid]*"
        exp_close
}


proc tksml_slave::SIGINT { } {
        global gui_flags

        exec kill -INT [exp_pid]
        if [cml] {
            exp_send "\n:q\n"
        }


        catch {grab release [grab current]}
        global busy

        if ![info exists gui_flags(title)] { set gui_flags(title) TkSML }
        if {[info exists busy] && $busy>0} {
            set busy 0
#           puts "$gui_flags(title) was in a hung state.  This has now been cleared."
        }

#
#       puts "\nPress Ctrl-C again to kill $gui_flags(title)."
#       puts "  (NOTE: You may also use Ctrl-C to interrupt ML computations )"
#       puts "  ($gui_flags(title) just wants to make sure you aren't trying to do that)"
#
#       trap {global busy ; set busy 1 ; exit } SIGINT
}


proc tksml_slave::external_execution { command } {
        global tksml_slave_flags
        lappend tksml_slave_flags(commands) $command
        tksml_slave::try_flush_external_executions
}

proc tksml_slave::try_flush_external_executions {} {
        global busy
        global tksml_slave_flags
        if {$busy && [info exists tksml_slave_flags(commands)]} { 
            after 1000 tksml_slave::try_flush_external_executions
            return 
        }
        if {[info exists tksml_slave_flags(commands)]} { 
            set buffer $tksml_slave_flags(commands)
            unset tksml_slave_flags(commands)
            foreach command $buffer {
                eval $command
            }
        }
}



proc EXPECT_FEEDBACK {} {
        global tksml_slave_flags
        list -re "[join $tksml_slave_flags(feedback_strings) {|}]" {
                global feedback
                set feedback [list {} $expect_out(0,string)]
        }
}

proc EXPECT_MLDONE { } { 
        list "\nTCLML_DONE" break
}

proc EXPECT_MLERROR { } {
        global tksml_slave_flags
        list -re "([join $tksml_slave_flags(error_strings) {|}])[ml_re_prompt]" {
                        set error 1
                        set errval $expect_out(1,string)
                  }
}

proc EXPECT_CAPTURE_ML_PROMPT { } {
        list [ml_prompt] { 
                    if ![info exists retval] { set retval "" }
                    set retval [set retval]$expect_out(0,string)
                    set return 1
                }
}


proc EXPECT_CAPTURE_LINE { } {
    list -re "(\[^\n\]*\n)" { 
        if ![info exists retval] { set retval "" }
        set retval [set retval]$expect_out(1,string)

    }
}

proc EXPECT_PROCESS_ML_PROMPT { } {
        list [ml_prompt] {
                    set output $expect_out(buffer)
                    set error [catch {eval $output_processor [list $output]} errval]
                    set return [expr !$error]
                    set retval ""
        }
}

proc EXPECT_PROCESS_LINE { } {
        list -re (\[^\n\]*)\n { 
                    set output $expect_out(buffer)
                    set error [catch {eval $output_processor [list $output]} errval]
                    continue
        }
}

proc EXPECT_ML_RESULT { } {
    list "TCLML_RESULT" {
        global tksml_slave_flags 
        set f [open "$tksml_slave_flags(tmpdir)/tksml_result[pid]" r]
        set retval [read $f]
        close $f
        set return 1
        expect -- [ml_prompt]
    }
}



proc EXPECT_FULL_BUFFER { } {
    list full_buffer {
        puts stderr "full buffer - increasing buffer size.  Repeat the operation..."
        match_max [expr [match_max]*2]
    }
}

proc EXPECT_BIG_HEAP { } {
    
    list -re {\[Increasing heap to ([^\n]*)k\]} {
        global tksml_slave_flags
        global feedback
        catch [if {$expect_out(1,string) > $tksml_slave_flags(max_heap)} {
            puts stderr "WARNING!!! SML process exceeding heap limit of $tksml_slave_flags(max_heap).  $gui_flags(title) has a facility to automaticaly detect SML processes which are running out of control and due to crash because of lack of memory.  You may change the heap limit under the \"Communicating with ML\" options menu.

Trying to send interrupt..."
            tksml_slave::SIGINT
        }] err
#       puts stderr "Heap Increase detected, size = $expect_out(1,string), result = $err"
        set feedback [list {} $expect_out(0,string)]
    }
}



proc ML {args} {
        global tksml_slave_flags
        global tksml_slave_library
        global busy

        set direct 0
        set toplevel 0
        set check 0
        set run 1
        set log 0
        set trace 0
        set type string
        set output_processor ""
        for {set i 0} {$i<[llength $args]} {incr i} {
            switch -- [lindex $args $i] -type {
                incr i
                set type [lindex $args $i]
            } -toplevel {
                incr i
                set toplevel [lindex $args $i]
            } -run {
                incr i
                set run [lindex $args $i]
            } -trace {
                incr i
                set trace [lindex $args $i]
            } -direct {
                incr i
                set direct [lindex $args $i]
            } -check {
                incr i
                set check [lindex $args $i]
            } -log {
                incr i
                set log [lindex $args $i]
            } -output_processor {
                incr i
                set output_processor [lindex $args $i]
            } default {
                if [info exists mlcode] { error "too many arguments to ML" }
                set mlcode [lindex $args $i]
            }
        }
        if {!$direct && !$toplevel} {
            switch -- $type bool { 
                set val_code "(if ($mlcode) then [ml_string 1] else [ml_string 0])"
            } string {
                set val_code "($mlcode)"
            } int {
                set val_code "([ml_int_to_string] ($mlcode))"
            } void {
                set val_code "(($mlcode); [ml_string ""])"
            } stringlist {
                set val_code "(Tcl_Merge ($mlcode))"
            } boollist {
                set val_code "(Tcl_Merge (map (fn b => if b then [ml_string 1] else [ml_string 0]) ($mlcode)))"
            } intlist {
                set val_code "(Tcl_Merge (map [ml_int_to_string] ($mlcode)))"
            } intpairlist {
                set val_code "(Tcl_Merge (map (fn (l,r) => Tcl_Merge \[[ml_int_to_string] l,[ml_int_to_string] r\]) ($mlcode)))"
            } intpair {
                set val_code "(Tcl_Merge (let val (l,r) = ($mlcode) in \[[ml_int_to_string] l,[ml_int_to_string] r\] end))"
            } stringlistlist {
                set val_code "(Tcl_Merge (map Tcl_Merge ($mlcode)))"
            } any {
                set val_code "(($mlcode); [ml_string ""])"
            } default {
                error "unknown type $type"
            }

            set logged_code "$mlcode[ml_end_dec]"
            set full_code \
"[ml_letval] res = 
  $val_code
  $tksml_slave_flags(exception_filters) 
[ml_letin] ([ml_output] [ml_string "\nTCLML_DONE\n"]; 
TclResult.send_result ([ml_string $tksml_slave_flags(tmpdir)/tksml_result[pid]], res); 
[ml_output] [ml_string TCLML_RESULT]) [ml_letend] [ml_end_dec]"
        }

        set logged_code "$mlcode[ml_end_dec]"

        # Now do some checks on the code to see if it looks likely to
        # stuff to ML session around.
        # The code checked is exactly what would go in the log file
        # since if that's OK then what we construct should be OK.

        if $check {
            set checker $tksml_slave_library/../../cmldecls/smldecls
            if [catch {exec $checker << $logged_code} error] { 
                error "\nsyntax error: $error\n\n" 
            }
        }

        # only wrap exception handlers around toplevel decls. not 
        # involving structres/signatures
        if $toplevel {
            if {[regexp local|exception|open|structure|signature|functor|infix $mlcode]} {
                set full_code "$mlcode\n[ml_end_dec]\n [ml_output] [ml_string "\nTCLML_DONE\n"][ml_end_dec]"
            } else {
                set full_code "$mlcode\n$tksml_slave_flags(exception_filters)[ml_end_dec]\n [ml_output] [ml_string "\nTCLML_DONE\n"][ml_end_dec]"
            }
        }


        if $direct {
            set full_code $mlcode
            set logged_code $mlcode
        }

        # Part 2.  Run the code.

        set delayed_executions ""

        set retval ""
        incr busy

        if {$run} {
            global expect_out
            set return 0
            set error 0
            set old_log_user [log_user]
            set actual_trace [expr $tksml_slave_flags(trace) || $trace]
            log_user $actual_trace
            if $actual_trace {
                puts $full_code
            }
        
            # perform a simple check - individual lines longer than 256
            # characters sieze up the expect input stream/pty.  This
            # is incredibly annoying.
            # 
            
            set use_file 0
            set pieces [split $full_code \n]
            foreach piece $pieces {
                if {[string length $piece]>200} {
                    set use_file 1
                break
                }
            } 
            if $use_file {
                set f [open $tksml_slave_flags(tmpdir)/tksml_command[pid] w]
                puts $f $full_code
                close $f
                exp_send "use [ml_string $tksml_slave_flags(tmpdir)/tksml_command[pid]];\n"
            } else {
                exp_send -raw "$full_code\n"
            }
            

            # Now anlayze the output

            set EXPECT_FEEDBACK [EXPECT_FEEDBACK]
            set EXPECT_BIG_HEAP [EXPECT_BIG_HEAP]
            if $direct {
                if {$output_processor==""} {
                  while {!$return && !$error} {
                    eval expect $EXPECT_BIG_HEAP [EXPECT_CAPTURE_ML_PROMPT] [EXPECT_CAPTURE_LINE] $EXPECT_FEEDBACK
                  }
                } else {
                  while {!$return && !$error} {
                    eval expect $EXPECT_BIG_HEAP [EXPECT_PROCESS_ML_PROMPT] [EXPECT_PROCESS_LINE] $EXPECT_FEEDBACK
                  }
                }
            } else {
               set EXPECT_MLERROR [EXPECT_MLERROR]
               set EXPECT_MLDONE [EXPECT_MLDONE]
               set EXPECT_ML_RESULT [EXPECT_ML_RESULT]
#              set EXPECT_FULL_BUFFER [EXPECT_FULL_BUFFER]
               while {!$return && !$error} {
                  eval expect $EXPECT_BIG_HEAP $EXPECT_MLDONE $EXPECT_MLERROR $EXPECT_FEEDBACK
               }
               if {!$toplevel} {
                  while {!$return && !$error} {
                eval expect $EXPECT_BIG_HEAP $EXPECT_ML_RESULT
                  }
               }
            }
            
            log_user $old_log_user
            
            if {$error} { 
                incr busy -1
                error $errval
            }
        }

        if {$tksml_slave_flags(log) && $log} {
            catch {
                set f [open $tksml_slave_flags(logfile) a+]
                puts $f $logged_code
                close $f
            }
        }

        incr busy -1
        return $retval

}


proc ML_bind { args } {
        global tksml_slave_flags
        set passon_args ""
        for {set i 0} {$i<[llength $args]} {incr i} {
            switch -glob -- [lindex $args $i] -* {
                lappend passon_args [lindex $args $i]
                incr i
                lappend passon_args [lindex $args $i]
            } default {
                if [info exists val] { error "too many arguments to ML_bind" }
                if [info exists name] { 
                    set val [lindex $args $i] 
                } else {
                    set name [lindex $args $i] 
                }
            }
        }
        if ![info exists val] { error "not enough arguments to ML_bind" }
        set code "[ml_val] $name = ($val) $tksml_slave_flags(exception_filters)"
        eval ML $passon_args -toplevel 1 [list $code]

}

