#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------




proc composite_define {classes config initproc} {

        set class [lindex $classes 0]

        upvar #0 ${class}_commands commands
        upvar #0 ${class}_configResource configResource
        upvar #0 ${class}_configResourceClass configResourceClass
        upvar #0 ${class}_configDefault configDefault
        upvar #0 ${class}_configSwitch configSwitch
        upvar #0 ${class}_configAction configAction
        upvar #0 ${class}_initProc initProc

        catch {unset commands}
        catch {unset configResource}
        catch {unset configResourceClass}
        catch {unset configDefault}
        catch {unset configSwitch}
        catch {unset configAction}
        catch {unset initProc}

        if {$class != "Composite"} {
                require [string tolower composite]
                composite:inherit $class Composite
        }
        foreach parent [lrange $classes 1 end] {
                require [string tolower $parent]
                composite:inherit $class $parent
        }

        foreach item $config {
                set flag [lindex $item 0]
                if {[llength $item] == 1} {
                        set resource $configResource($flag)
                        unset configResource($flag)
                        unset configResourceClass($resource)
                        unset configDefault($resource)
                        catch {unset configSwitch($resource)}
                        unset configAction($resource)
                }

                if {[llength $item] >= 2} {
                        set resource [lindex $item 1]
                        set configResource($flag) $resource
                }
                if {[llength $item] >= 3} {
                        set resourceClass [lindex $item 2]
                        if {[llength $item] >= 4} {
                                set default [lindex $item 3]
                        } else {
                                set default {}
                        }
                        if {[llength $item] >= 5} {
                                set action [lindex $item 4]
                        } else {
                                set action composite_config
                        }
                        set configResourceClass($resource) $resourceClass
                        set configDefault($resource) $default
                        set configAction($resource) $action
                        set configSwitch($resource) $flag
                }
        }

        set initProc $initproc

        composite:createClassProc $class
}

# Procedure: composite:inherit
#
# Synopsis:
#       Internal procedure to establish an inheritance relationship
#       between two composite widget classes.
#
# Usage:
#c      composite:inherit childClass parentClass
#
# Parameters:
#c      childClass
#               Name of the child class (the one that is inheriting
#               widget commands and configuration switches)
#c      parentClass
#               Name of the parent class (the one from which
#               widget commands and configuration switches are being
#               inherited
#
# Return value:
#       None specified.
#
# Description:
#       composite:ingerit arranges the state of the global variables so that
#       widgets of class `childClass' accept the same configuration switches
#       and widget commands as do those of `parentClass'.

proc composite:inherit {class parent} {
        # puts "composite:inherit, class = $class, parent = $parent"
        upvar #0 ${class}_commands commands
        upvar #0 ${class}_configResource configResource
        upvar #0 ${class}_configResourceClass configResourceClass
        upvar #0 ${class}_configDefault configDefault
        upvar #0 ${class}_configSwitch configSwitch
        upvar #0 ${class}_configAction configAction
        upvar #0 ${parent}_commands parentCommands
        upvar #0 ${parent}_configResource parentConfigResource
        upvar #0 ${parent}_configResourceClass parentConfigResourceClass
        upvar #0 ${parent}_configDefault parentConfigDefault
        upvar #0 ${parent}_configSwitch parentConfigSwitch
        upvar #0 ${parent}_configAction parentConfigAction

        foreach key [array names parentCommands] {
                set commands($key) $parentCommands($key)
        }
        foreach key [array names parentConfigResource] {
                set configResource($key) $parentConfigResource($key)
        }
        foreach key [array names parentConfigResourceClass] {
                set configResourceClass($key) $parentConfigResourceClass($key)
        }
        foreach key [array names parentConfigDefault] {
                set configDefault($key) $parentConfigDefault($key)
        }
        foreach key [array names parentConfigSwitch] {
                set configSwitch($key) $parentConfigSwitch($key)
        }
        foreach key [array names parentConfigAction] {
                set configAction($key) $parentConfigAction($key)
        }
}

# Procedure:    composite:createClassProc
#
# Synopsis:
#       Create the procedure that is named after a composite widget class
#       and in turn is used to create widgets of that class.
#
# Usage:
#c      composite:createClassProc className
#
# Parameters:
#c      className
#               Name of the widget class
#
# Return value:
#       None specified.

proc composite:createClassProc class {
        proc [string tolower $class] {w args} "
                composite:create \$w $class \$args
        "
}

#
                    #############################
                    # Composite widget creation #
                    #############################

# Procedure:    composite:create
#
# Synopsis:
#       Internal procedure to create a composite widget defined by
#       composite_define.
#
# Usage:
#c      composite:create pathName class params
#
# Parameters:
#c      pathName
#               Path name of the widget being created.
#c      class
#               Class of the widget being created.
#c      params
#               Configuration options supplied at widget creation time.
#
# Return value:
#       Name of the widget created.
#
# Description: 
#       composite:create does the dirty work of a widget command
#       defined by composite_define.  It accepts the widget name, the
#       widget's class, and the configuration parameters.  It creates
#       the frame in which the widget will be constructed, parses
#       configuration parameters (and the corresponding options from
#       the option database), calls the widget initialization
#       procedure, renames the frame command, and puts in its place
#       the composite widget command.  
#
#       Old note: (Finally, it calls
#       composite:applyInitialConfig to make the configuration options
#       take effect.)  
#
#       This is no longer done as it is assumed the initial config
#       is ``applied'' during creation.  The creation routine should
#       look up relevant values in the config$w array.
#
#       

proc composite:create {w class params} {
        # puts "composite:create, w = $w, class = $class, params = $params"
        upvar #0 ${class}_initProc initproc
        frame $w -class $class
        set status [catch {
                composite:getInitialConfig $w $params
                widget_addBinding $w Destroy "composite:deleteConfig $w"
                $initproc $w
                rename $w ${class}_alias$w
                proc $w {command args} "
                        eval \[list composite:command $w \$command\] \$args
                "
                widget_addBinding $w Destroy "composite:deleteAlias $w"
                update idletasks
                composite:applyInitialConfig $w
        } message]
        if {$status != 0} {
                global errorInfo
                global errorCode
                set info $errorInfo
                set code $errorCode
                catch {destroy $w}
                # puts "error in composite:create, status = $status, info = $info, code = $code"
                error $message $info $code
        }
        # puts "leaving composite:create, w = $w, class = $class, params = $params"
        return $w
}

# Procedure: composite:getInitialConfig
#
# Synopsis:
#       Internal procedure to determine a composite widget's
#       initial configuration parameters.
#
# Usage:
#c      composite_getInitialConfig pathName ?-flag value?...
#
# Parameters:
#c      pathName
#               Path name of the widget being created.
#c      flag, value
#               Configuration flags that apply to the widget.
#
# Description:
#       composite:getInitialConfig determines the full set of configuration
#       parameters when creating a composite widget.  It scans the defaults
#       for the widget, and overrides any that are specified with X resources
#       or the `option' command.  It then further overrides any parameters
#       than are specified on the command line.  The resulting set of
#       parameters is stashed in the `config$pathName' array
#       for further processing in composite:applyInitialConfig.

proc composite:getInitialConfig {w params} {
        set class [winfo class $w]
        upvar #0 ${class}_configDefault default
        upvar #0 ${class}_configResourceClass resourceClass
        upvar #0 config$w config
        foreach r [array names default] {
                set value [option get $w $r $resourceClass($r)]
                if {$value == ""} {
                        set value $default($r)
                }
                set config($r) $value
        }
        while {[llength $params] >= 2} {
                set r [composite:findConfigFlag $w [lindex $params 0]]
                set value [lindex $params 1]
                set params [lrange $params 2 end]
                set config($r) $value
        }
}

# Procedure:    composite:applyInitialConfig
#
# Synopsis:
#       Apply the initial configuration to a composite widget.
#
# Usage:
#c      composite:applyInitialConfig pathName
#
# Parameters:
#c      pathName
#               Path name of the widget being created
#
# Description:
#       `composite:applyInitialConfig' constructs a `pathName configure'
#       command that contains all the initial options for a composite widget.
#       It then executes that command in the caller's lexical scope to
#       configure the widget.

proc composite:applyInitialConfig w {
        set class [winfo class $w]
        upvar #0 config$w config
        upvar #0 ${class}_configSwitch configSwitch
        set command [list $w configure]
        foreach r [lsort [array names config]] {
                lappend command $configSwitch($r) $config($r)
        }
        uplevel 1 $command
}

#
                   ################################
                   # Composite widget destruction #
                   ################################

# Procedure:    composite:deleteConfig
#
# Synopsis:
#       Delete the global array describing a composite widget's
#       configuration as the widget is being destroyed.
#
#       Also, delete the global array allocated for a widget's private
#       data as the widget is being destroyed.
# Usage:
#c      composite:deleteConfig pathName
#
# Parameters:
#       pathName
#               Path name of the widget being destroyed.
#
# Return value:
#       None specified.

proc composite:deleteConfig w {
        upvar #0 config$w config
        catch { unset config }
        upvar #0 $w data
        catch { unset data }
}


# Procedure:    composite:deleteAlias
#
# Synopsis:
#       Delete the alias procedure that handles the widget command
#       of a composite widget.
#
# Usage:
#c      composite:deleteAlias pathName
#
# Parameters:
#c      pathName
#               Path name of the widget being destroyed.
#
# Return value:
#       None specified.

proc composite:deleteAlias w {
        set class [winfo class $w]
        catch { rename ${class}_alias$w {} }
}

         ####################################################
         # Composite widget configuration option processing #
         ####################################################

# Procedure:    composite_configFlag
#
# Synopsis:
#       Define a configuration option for a composite widget.
#
# Usage:
#c      composite_configFlag className option paramList body
#
# Parameters:
#c      className
#               Widget class for which the option is being defined.
#c      option
#               Name of the option being defined.  This is NOT its
#               command-line switch, but rather its X resource name.
#c      paramList
#               Parameters to the function that handles the option.  The
#               function is expected to accept three parameters -- the name
#               of the widget, the name of the configuration option, and
#               the value of the configuration option.
#c      body
#               Body of the procedure that handles the option.  In addition
#               to doing any local option processing, the procedure may call
#               one of the following procedures to apply an option recursively:
#
#                       + composite_config, which causes the option to
#                       be applied to the Tk widget and all its descendants.
#
#                       + composite_configLocal, which causes the option to
#                       be applied to the Tk widget only, and
#
#                       + composite_configChild, which causes the option to
#                       be applied to the descendant widgets only.
#
# Return value:
#       The name of the function that handles the option.  The function will
#       always have the name
#
#c              ${className}_config_XXX
#
#       where _XXX is the configuration flag being processed.
#
# Description:
#       composite_configFlag defines the function that handles a single
#       configuration option for a widget.  It is responsible for taking
#       any necessary action when the configuration changes.
#
# Example:
#c      composite_configFlag myWidget textVariable {w flag vname} {
#
#c              $w.entry.box configure -textvariable $vname
#
#c      }

proc composite_configFlag {class resource params body} {
        proc ${class}_config_${resource} $params $body
        composite_configFlagProc $class $resource ${class}_config_${resource}
        # puts "composite_configFlag, class = $class, resource = $resource, defined proc = ${class}_config_${resource}"
}

# Procedure:    composite_configFlagProc
#
# Synopsis:
#       Define a configuration option for a composite widget.
#
# Usage:
#c      composite_configFlagProc className option procName
#
# Parameters:
#c      className
#               Widget class for which the option is being defined.
#c      option
#               Name of the option being defined.  This is NOT its
#               command-line switch, but rather its X resource name.
#c      procName
#               Name of the procedure that handles the option.  The
#               procedure is expected to accept three parameters:
#                       + the name of the widget.
#
#                       + the name of the option.
#
#                       + the value of the option.
#
#               Three choices for `procName' are provided by the system:
#
#                       + composite_config, which causes the option to
#                       be applied to the Tk widget and all its descendants.
#
#                       + composite_configLocal, which causes the option to
#                       be applied to the Tk widget only, and
#
#                       + composite_configChild, which causes the option to
#                       be applied to the descendant widgets only.
#
#               These procedures may also be called from within a user's
#               configuration procedure.
#
# Return value:
#       The name of the procedure that handles the option.
#
# Description:
#       composite_configFlag associates the procedure that handles a single
#       configuration option for a widget.  It is responsible for taking
#       any necessary action when the configuration changes.
#
# Example:
#c      composite_configFlag myWidget relief composite_configLocal

proc composite_configFlagProc {class resource procName} {
        upvar #0 ${class}_configAction configAction
        set configAction($resource) $procName
        # puts "composite_configFlagProc, class = $class, resource = $resource, procName = $procName"
}

# Procedure:    composite:applyConfig
#
# Synopsis:
#       Internal procedure that applies configuration flags to a composite
#       widget.
#
# Usage:
#c      composite:applyConfig pathName ?-flag value?...
#
# Parameters:
#c      pathName
#               Path name of the widget being configured
#c      -flag, value
#               Any number of flag-value pairs giving the widget's
#               desired configuration.  Flags may be abbreviated to any
#               unique prefix.  These flags are the command-line
#               switches, not the resource names.
#
# Return value:
#       Not specified.
#
# Description:
#       The `composite:applyConfig' procedure is called from a
#       composite widget's `configure' subcommand once it is determined that
#       the caller has supplied name-value pairs for the configuration
#       options.  It looks up each configuration option in the table
#       of flags that the widget accepts, and calls the action procedure
#       for each.
#
#       The action procedure is expected to call `composite_config',
#       `composite_configLocal', or `composite_configChild' for every
#       option it finds that must be applied recursively.  These procedures
#       import the FRAMEFLAGS and CHILDFLAGS arrays from the lexical
#       scope of composite:applyConfig, and install the appropriate
#       flags therein.
#
#       After all the configuration procedures have run, the
#       `composite:applyLocalConfig' procedure is called with all the
#       local configuration flags, passing the name of the alias.
#       command that actually enters Tk for the widget.  It applies all
#       the local configuration flags to the widget.
#
#       Finally, `composite:applyChildConfig' is applied to all the
#       children of the widget, passing the child configuration flags.
#       This call recursively applies all the propagated configuration options.

proc composite:applyConfig {w args} {

        # Determine widget class

        set class [winfo class $w]

        # Import the configuration action table for the class

        upvar #0 ${class}_configAction action
        upvar #0 config$w values
        upvar #0 ${class}_configSwitch switches

        # Parse the arguments.  Call a configuration procedure for each.

        set argv $args
        while {[llength $argv] >= 2} {
                set key [composite:findConfigFlag $w [lindex $argv 0]]
                set switch $switches($key)
                set value [lindex $argv 1]
                set argv [lrange $argv 2 end]
                # puts "action($key) = $action($key)"
                switch -- [$action($key) $w $key $value] {
                    frame { set FRAMEFLAGS($switch) $value }
                    child { set CHILDFLAGS($switch) $value }
                    both { set FRAMEFLAGS($switch) $value; set CHILDFLAGS($switch) $value }
                    private { }
                    default { error "Bad return code from $action($key) - should be frame, child, both or private" }
                }
                set values($key) $value
        }


#       catch {puts "FRAMEFLAGS = [array names FRAMEFLAGS]"}
#       catch {puts "CHILDFLAGS = [array names CHILDFLAGS]"}

        # Apply any local configuration flags

        set command "composite:applyLocalConfig ${class}_alias$w"
        set status [catch {array names FRAMEFLAGS} keys]
        if {$status == 0} {
                foreach key $keys {
                        lappend command $key $FRAMEFLAGS($key)
                }
                eval $command
        }

# Propagate flags into the children.

#       set propagate {}
#       set status [catch {array names CHILDFLAGS} keys]
#       if {$status == 0} {
#               foreach key $keys {
#                       lappend propagate $key $CHILDFLAGS($key)
#               }
#               foreach child [winfo children $w] {
#                       eval [list composite:applyChildConfig $child] \
#                               $propagate
#               }
#       }
        return ""
}

# Procedure:    composite_config
#
# Synopsis:
#       Cause a configuration option to be applied to a widget and all of
#       its descendants.
#
# Usage:
#c      composite_configPrivateNop pathName optionName value
#
# Parameters:
#c      pathName
#               Path name of the composite widget.
#c      optionName
#               Name of the option (NOT its command-line switch).
#c      value
#               Value of the option.
#
# Return value:
#       None specified.
#
# Description:
#       The `composite_configPrivateNop' procedure is for configuration
#       options which don't need any particular special action performed
#       when set.

proc composite_configPrivateNop {w flag value} {
        return private
#       composite_configLocal $w $flag $value
#       composite_configChild $w $flag $value
}

# Procedure:    composite_config
#
# Synopsis:
#       Cause a configuration option to be applied to a widget and all of
#       its descendants.
#
# Usage:
#c      composite_config pathName optionName value
#
# Parameters:
#c      pathName
#               Path name of the composite widget.
#c      optionName
#               Name of the option (NOT its command-line switch).
#c      value
#               Value of the option.
#
# Return value:
#       None specified.
#
# Description:
#       The `composite_config' procedure is the configuration
#       procedure for configuration options that have none.  It simply
#       returns "both" indicating that the option should be propogated
#       to the frame and any children interested in the flag.

proc composite_config {w flag value} {
        return both
}

# Procedure:    composite_configLocal
#
# Synopsis:
#       Cause a configuration option to be applied to a widget but not
#       its descendants.
#
# Usage:
#c      composite_configLocal pathName optionName value
#
# Parameters:
#c      pathName
#               Path name of the composite widget.
#c      optionName
#               Name of the option (NOT its command-line switch).
#c      value
#               Value of the option.
#
# Return value:
#       None specified.
#
# Description: 
#       The `composite_configLocal' procedure marks an option for
#       application to the frame of the composite widget, by 
#       returning "frame".

proc composite_configLocal {w flag value} {
        return frame
}

# Procedure:    composite_configChild
#
# Synopsis:
#       Cause a configuration option to be applied to a widget's
#       descendants.
#
# Usage:
#c      composite_configChild pathName optionName value
#
# Parameters:
#c      pathName
#               Path name of the composite widget.
#c      optionName
#               Name of the option (NOT its command-line switch).
#c      value
#               Value of the option.
#
# Return value:
#       None specified.
#
# Description: 
#       The `composite_configChildren' procedure marks an option for
#       propogation to a widget's descendants, by returning 
#       "child".

proc composite_configChild {w flag value} {
        return child
}

# Procedure:    composite:applyLocalConfig
#
# Synopsis:
#       Internal procedure to apply configuration options to the
#       frame of the composite widget.
#
# Usage:
#c      composite:applyLocalConfig pathName ?-flag value?...
#
# Parameters:
#c      pathName
#               Path name of the widget being configured, or an alias
#               procedure that identifies the widget.
#c      -flag, value
#               Switch-value pairs that give the widget's configuration.
#               Switches that the widget does not accept may be included,
#               and will be deleted.  The flags may not be abbreviated
#               at this point.
#
# Return value:
#       None specified.
#
# Description:
#       composite:applyLocalConfig selects the flag-value pairs from its
#       command line that apply to the the specified widget, 
#       and applies them.
#
#       This routine is used with the `w' argument  being either 
#               - a child widget (when propogating options to children)
#            or - the frame of the composite widget.

proc composite:applyLocalConfig {w args} {
        while {[llength $args] >= 2} {
                set key [lindex $args 0]
                set value [lindex $args 1]
                set args [lrange $args 2 end]
                set flag($key) $value
        }
        set command [list $w configure]
        foreach entry [$w config] {
                set key [lindex $entry 0]
                if [info exists flag($key)] {
                        lappend command $key $flag($key)
                }
        }
        eval $command
}

# Procedure:    composite:applyChildConfig
#
# Synopsis:
#       Internal procedure to apply configuration options to a descendant
#       of the local widget.
#
# Usage:
#c      composite:applyChildConfig pathName ?-flag value?...
#
# Parameters:
#c      pathName
#               Path name of the child widget being configured.
#c      -flag, value
#               Switch-value pairs that give the widget's configuration.
#               Switches that the widget does not accept may be included,
#               and will be deleted.  The flags may not be abbreviated
#               at this point.
#
# Return value:
#       None specified.
#
# Description:
#       composite:applyChildConfig selects the flag-value pairs from its
#       command line that apply to the specified widget, and applies them.
#       Configuration is not propagated into top-level widgets.
#       If a widget is not a composite widget, the flag-value pairs are
#       also applied recursively to all its children, allowing the
#       use of `frame' widgets as artifacts of geometry management without
#       needing to define widget classes for all of them.

proc composite:applyChildConfig {w args} {

        # Don't propagate configuration into toplevel widgets.

        if {$w == [winfo toplevel $w]} {
                return
        }

        # Apply any local configuration options that apply

        eval [list composite:applyLocalConfig $w] $args

        # Let managed widgets handle their own configuration.

        if [composite_isManaged $w] {
                return
        }

        # Apply flags to children of unmanaged widgets.

        foreach child [winfo children $w] {
                composite:applyChildConfig $child
        }
}

# Procedure:    composite:findConfigFlag
#
# Synopsis:
#       Internal procedure to process unique-prefix abbreviation of
#       configuration switches.
#
# Usage:
#c      composite:findConfigFlag pathName -flag
#
# Parameters:
#c      pathName
#               Path name of the widget being configured.
#c      -flag
#               Flag to locate.
#
# Return value:
#       X resource name corresponding to the specified flag.
#
# Description:
#       composite:findConfigFlag calls composite:matchName to select
#       the configuration option that matches the given switch, and
#       returns the switch's resource name.

proc composite:findConfigFlag {w flag} {
        set class [winfo class $w]
        upvar #0 ${class}_configResource configResource
        set status [catch {composite:matchName $flag configResource} result]
        if {$status == 0} {
                return $configResource($result)
        } else {
                error "$class does not support a $flag flag" \
                        "$class does not support a $flag flag\n\
\tavailable flags are [lsort [array names configResource]]"
        }
}

#
               ########################################
               # Composite widget command processing. #
               ########################################

# Procedure:    composite_subcommand
#
# Synopsis:
#       Define a subcommand for a composite widget's widget command.
#
# Usage:
#c      composite_subcommand className command paramList body
#
# Parameters:
#c      className
#               The class of widget for which the subcommand is being defined.
#c      command
#               The name of the widget command being defined.
#c      paramList
#               The parameters that the widget command expects.  The
#               first parameter is always the widget path name; the remaining
#               parameters are taken from the call to the widget command.
#c      body
#               The body of the procedure that executes the command.
#
# Return value:
#       The name of the procedure that executes the command.  It will have
#       the form
#
#c              ${className}_command_XXX
#
#       where XXX is the name of the command.
#
# Description:
#       composite_subcommand defines a new widget command for a widget.
#
# Example:
#c      composite_subcommand myWidget show {w string} {
#
#c              $w.label config -text $string
#
#c      }
#
#       mywidget foo
#
#c      .foo show "Hi there"

proc composite_subcommand {class command params body} {
        proc ${class}_command_${command} $params $body
        composite_subcommandProc $class $command ${class}_command_${command}
}

# Procedure:    composite_subcommand
#
# Synopsis:
#       Define a subcommand for a composite widget's widget command.
#
# Usage:
#c      composite_subcommand className command procName
#
# Parameters:
#c      type
#               The type of widget for which the subcommand is being defined.
#c      command
#               The name of the widget command being defined.
#c      procName
#               The procedure that handles the widget command.  The procedure's
#               first parameter is always the widget path name; the remaining
#               parameters are taken from the call to the widget command.
#
# Return value:
#       The name of the procedure that executes the command.
#
# Description:
#       composite_subcommand defines a new widget command for a widget.

proc composite_subcommandProc {class command procName} {
        upvar #0 ${class}_commands commands
        set commands($command) ${class}_command_${command}
}

# Procedure:    composite:command
#
# Synopsis:
#       Internal procedure to evaluate a widget command on a composite widget.
#
# Usage:
#c      composite:command pathName command args
#
# Parameters:
#c      pathName
#               Path name of a composite widget.
#c      command
#               Name of the command to execute.
#c      args
#               Parameters to the command
#
# Return value:
#       Specified by the user.
#
# Description:
#       The composite:command procedure executes a widget command on a
#       composite widget.  It includes unique-prefix matching for
#       subcommand names.

proc composite:command {w command args} {
        set class [winfo class $w]
        upvar #0 ${class}_commands commands
        set status [catch {composite:matchName $command commands} result]
        if {$status != 0} {
                error "$w: invalid command $command" \
"$w: invalid command $command; available commands are:
        [lsort [array names commands]]"
        }
        eval [list $commands($result) $w] $args
}

#
             ############################################
             # Service procedures for composite widgets #
             ############################################

# Procedure:    composite:matchName
#
# Synopsis:
#c      Name matching for widget commands
#
# Usage:
#c      composite:matchName name array
#
# Parameters:
#c      name
#               Name of a subcommand, configuration option, etc.
#c      array
#               Array whose indices are the names of available subcommands,
#               configuration options, etc.
#
# Return value:
#       Name extracted from the available set.
#
# Description:
#       composite:matchName accepts a name, and an array that contains the
#       possible values for the name.  It attempts to match the name to
#       one of the indices in the array.  If the name does not mach exactly,
#       but only one element of the array has the name as a prefix, that
#       array element is returned; this allows for the same unique-prefix
#       rules that Tk uses to match widget commands and command-line flags.

proc composite:matchName {name arrayname} {
        upvar 1 $arrayname array
        if [info exists array($name)] {
                return $name
        }
        set names [array names array]
        set index [lsearch $names $name*]
        if {$index >= 0} {
                set index2 [lsearch [lrange $names [expr $index+1] end] $name*]
                if {$index2 < 0} {
                        return [lindex $names $index]
                }
        }
        error "$name not found in $arrayname"
}

# Procedure:    composite_isManaged
#
# Synopsis:
#       Determine whether a widget is a managed composite widget.
#
# Usage:
#c      composite_isManaged pathName
#
# Parameters:
#c      pathName
#               Path name of a widget.
#
# Return value:
#       1 if the widget is a managed composite widget.
#
#       0 otherwise.

proc composite_isManaged {w} {
        set class [winfo class $w]
        expr {[info commands ${class}_alias$w] == "${class}_alias$w"}
}

