#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------





 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File: choicebox.tcl
#
# Description:
#       Tcl procedures to implement a `choice box' widget, that is,
#       a widget with a possible icon, a possible message, and a set of
#       buttons offering various choices.

# Procedure:    choicebox
#
# Synopsis:
#       Widget that constructs a dialog box offering the user a multiple
#       choice.
#
# Usage:
#c      choicebox pathName ?-option value...?
#
# Parameters:
#c      pathName
#               Path name of the choice box to create
#
# Options:
#c      -text TEXT
#               Uses TEXT as the message for the choice box.
#
#c      -icon ICON
#               Uses the bitmap in the file `ICON.xbm' on the current
#               auto_path as the icon for the choice box.
#
#c      -buttons {b1 b2 b3...}
#               Creates buttons with text `b1', `b2', `b3', ...
#               If any of the buttons is a two-element list, the
#               first element is used as the text, and the second as
#               a variable value
#
#c      -textvariable NAME
#               Name of a global variable that receives the name of the
#               button that was pressed.
#
#       Other options are as for `frame.'  The `-class' option is illegal;
#       the class name is always `choicebox'.
#
# Return value:
#       Name of the widget created.
#
# Example:
#c      choicebox $w    -text "Are you sure?" \

#c                      -icon think \

#c                      -buttons {{Yes 1} {No 0} {Help ""}} \

#c                      -textvariable isSure$w

# Bugs:
#       - The choicebox widget does not use the widget construction
#       primitives to make itself a first-class widget.  It does not
#       respond to the `configure' widget command properly.
#
#       - The options cannot be specified as X defaults.
#
#       - foreground, background, etc. are not supplied as options.

        # Default look:
option add *Choicebox*background #ffe4c4 widgetDefault
option add *Choicebox.m.foreground black widgetDefault
option add *Choicebox*activeBackground #eed5b7 widgetDefault
option add *Choicebox*activeForeground black widgetDefault

proc choicebox {w args} {
        set fargs {}
        set text {}
        set icon {}
        set buttons {}
        set textvariable $w
        while {[llength $args] >= 2} {
                set option [lindex $args 0]
                set value [lindex $args 1]
                set args [lrange $args 2 end]
                case $option in {
                        -text { set text $value }
                        -icon { set icon $value }
                        -buttons { set buttons $value }
                        -textvariable { set textvariable $value }
                        default {
                                lappend fargs $option $value
                        }
                }
        }
        if {$args != ""} {
                error "choicebox: extraneous argument $args"
        }
        eval frame $w -class Choicebox $fargs
        pack append $w \
                [choicebox:buttons $w.b \
                        -buttons $buttons \
                        -textvariable $textvariable] \
                                {bottom expand fill}
        if {$text != ""} {
                global gui_flags
                pack append $w \
                        [fontcheck message $w.m -text $text -width 400 -font $gui_flags(font,feedback)] \
                                {right expand fill}
        }
        if {$icon != ""} {
                pack append $w \
                        [icon $w.i -icon $icon] \
                                {fill}
        }
        return $w
}

# Procedure: choicebox:buttons
#
# Synopsis:
#       Make the push-button panel for a choicebox.
#
# Usage:
#c      choicebox:buttons pathName ?-option value...?
#
# Parameters:
#c      pathName
#               Path name of the button box widget
#
# Options:
#c      -buttons {b1 b2 b3}
#               List of button text, or button text - variable value pairs
#
#c      -textvariable v
#               Variable name in which to store the button value.
#
#       Other options as for frame.
#
# Return value:
#       Name of the button box widget.
#
# Description:
#       choicebox:buttons is an internal procedure used by the choicebox
#       procedure to fabricate the array of push buttons that appear at
#       the bottom of a choicebox.

proc choicebox:buttons {w args} {
        set fargs {}
        set textvariable $w
        set buttons {}
        while {[llength $args] >= 2} {
                set option [lindex $args 0]
                set value [lindex $args 1]
                set args [lrange $args 2 end]
                case $option in {
                        -buttons { set buttons $value }
                        -textvariable { set textvariable $value }
                        default {
                                lappend fargs $option $value
                        }
                }
        }
        if {$args != ""} {
                error "choicebox:buttons: extraneous argument $args"
        }
        eval frame $w $fargs
        set bnum 0
        foreach b $buttons {
                incr bnum
                if {[llength $b] > 1} {
                        set bval [lindex $b 1]
                } else {
                        set bval [lindex $b 0]
                }
                global gui_flags
                pack append $w \
                        [fontcheck focusable button $w.b$bnum \
                                -font $gui_flags(font,buttons) \
                                -text [lindex $b 0] \
                                -command "set $textvariable $bval"] \
                                        {left expand padx 10 pady 10}
        }
        return $w
}


proc choice { w args } {
        global choice
        eval [list modalDialog transient choicebox $w] \
            [list -withfeedback 0 \
            -icon think \
            -buttons [list Yes No Cancel]] \
            $args \
            [list -textvariable choice]
        wm protocol [winfo toplevel [winfo parent $w]] WM_DELETE_WINDOW "set choice {}"
        widget_waitVariable choice
        modalDialog.end $w
        return $choice
}

