;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                             HOL 88 Version 2.0                          ;;;
;;;                                                                         ;;;
;;;   FILE NAME:        hol-tcl-writ.l                                          ;;;
;;;                                                                         ;;;
;;;   DESCRIPTION:      Lisp functions for printing HOL terms               ;;;
;;;                                                                         ;;;
;;;   USES FILES:       f-franz.l (or f-cl.l), f-constants.l, f-macro.l,    ;;;
;;;                     f-ol-rec.l, genmacs.l                               ;;;
;;;                                                                         ;;;
;;;                     University of Cambridge                             ;;;
;;;                     Hardware Verification Group                         ;;;
;;;                     Computer Laboratory                                 ;;;
;;;                     New Museums Site                                    ;;;
;;;                     Pembroke Street                                     ;;;
;;;                     Cambridge  CB2 3QG                                  ;;;
;;;                     England                                             ;;;
;;;                                                                         ;;;
;;;   COPYRIGHT:        University of Edinburgh                             ;;;
;;;   COPYRIGHT:        University of Cambridge                             ;;;
;;;   COPYRIGHT:        INRIA                                               ;;;
;;;                                                                         ;;;
;;;   REVISION HISTORY: (none)                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; RJB 16.11.92 - All occurrences of <=> deleted.

(eval-when (compile)
   #+franz (include "lisp/f-franz")
   (include "lisp/f-constants")
   (include "lisp/f-macro")
   (include "lisp/f-ol-rec")
   (include "lisp/genmacs")
   (special ol-lam-sym hol-unops hol-binops binders
      %caught-output %catching-output
      |%show_types-flag|
      |%interface_print-flag| %turnstile |%print_top_types-flag|
      |%print_list-flag| |%print_cond-flag|
      |%print_quant-flag| |%print_restrict-flag|
      |%print_let-flag| |%print_uncurry-flag| |%print_infix-flag|
      |%print_lettypes-flag| |%print_top_val-flag|  |%print_set-flag|
      |%empty-set| |%finite-set-constructor| |%set-abstraction-constructor|
      |%pp_sexpr-flag| |%print_sexpr-flag| %pt1 |%print_parse_trees-flag|))

#+franz
(declare
   (localf subtract is-subset nargtys polytys uninferred-poly-remnant
      prep-ol-let
      print-ol-bnd
      print-ol-let
      prep-ol-uncurry
      is-ol-cons
      is-ol-list
      pre-prep-ol-list
      prep-ol-list
      print-ol-list
      is-ol-set-cons
      is-ol-finite-set
      pre-prep-ol-finite-set
      prep-ol-finite-set
      print-ol-finite-set
      prep-ol-set-abstraction
      print-ol-set-abstraction
      prep-ol-cond
      is-special-comb
      prep-ol-quant
      prep-ol-restrict
      prep-ol-unop
      prep-ol-binop
      print-ol-unop
      print-eq
      print-ol-binop
      print-neg
      print-ol-quant
      print-ol-restrict))


;;; DRS: Modified version of ptoken for printing a token to
;;;; the formatted tcl output with a tag, not counting the length
;;;; of the tag in the string printed.
;;;;

(defmacro ptcltoken (tag str)
   `(cond (%tcl-tagged-output
             (pstringlen (concat "{" ',tag " " ',str "} ") ,(flatc str)))
	  (t (pstringlen ',str ,(flatc str)))))   


(defmacro space-token ()
   `(cond (%tcl-tagged-output
             (pstringlen "{sp { }} " 1))
	  (t (pstringlen " " 1))))   


;;; DRS Modified version of pstring to add a tag, ignoring the
;;; DRS added text length of the tag and the surrounding tcl braces.
(defun ptclstring (tag str)
   (cond (%tcl-tagged-output
             (pstringlen (concat "{" tag " " str "} ") (flatsize2 str)))
         (t (pstringlen str (flatsize2 str)))))


;;; RJB 16.11.92: Pretty-printing with `show_types' true re-implemented.
;;;
;;; The basic idea is:
;;; (1) Type the bound variables of abstractions. Note that this ensures that
;;;     terms such as "@(x:*). T" are printed with enough type information.
;;; (2) Type each free variable exactly once.
;;; (3) Type polymorphic constants (including NIL) if they are not the operator
;;;     of an application.
;;; (4) Type an application (combination) if its operator is a constant whose
;;;     most general type contains a type variable which cannot be inferred
;;;     from the arguments (see uninferred-poly-remnant).
;;; This is achieved by passing around a list of variables (actually name/type
;;; pairs) that have already been adorned with type information during the
;;; term traversal. The only tricky bit is dealing with bound variables.
;;; As a binding is entered the bound variables are added to the list of typed
;;; variables coming down the term tree (The bound variables are typed at the
;;; binding so don't need to be done in the body.). As the traversal comes
;;; back up through the binding, the new variables added to the list (that is
;;; those variables that were adorned with type information inside the body)
;;; are extracted and added to the original list of typed variables. This
;;; allows the bound variables to be removed from the list so that free
;;; variables of the same name will still be typed, but without removing such
;;; variables from the list if they had already been typed.
;;;
;;; An additional optimisation is performed: A term is not printed with type
;;; information if it is a direct subterm of ~, /\, \/, or ==>. However, if
;;; the term is a variable it *is* added to the typed-variable list since we
;;; know its type can be inferred.
;;;
;;; This scheme certainly doesn't keep type information to a minimum but I
;;; don't think the amount will be excessive, and I think it will provide
;;; enough information for the type of the term to be inferred which was not
;;; the case for the old version.
;;;
;;; - RJB
;;;
;;; MJCG 01.02.94
;;; Printing of $ before infixed variables added
;;; MJCG 24/6/94 for HOL88.2.03
;;; Bugfix: special treatment of type printing for finite sets commented out

(defun print-tm (tm op1 typedvars)
   (let ((op2 (term-class tm))
         (tml (get-term-list tm))
         (ty (get-type tm)))
      (let ((tyflag               ; print type of this particular term?
               (and |%show_types-flag|
                  (case op2
                     (var (not (member (cdr tm) typedvars)))
                     (const (and (not (eq op1 'ratorofcomb))
                                 (opoly (constp (get-const-name tm)))))
                     ((listcomb infixcomb)
                        (let ((r (first tml))    ; find innermost operator
                              (n (- (length tml) 1)))
                           (if (eq (term-class r) 'infixcomb)
                              (setq n (+ n (- (length (get-term-list r)) 1))
                                    r (first (get-term-list r))))
                           (and (is-const r)
                              (uninferred-poly-remnant (get-const-name r) n))))
                     ((ol-list ol-finite-set) (eq tml nil))
                     (t nil))))
            (knownty (memq op1 '(|~| /\\ \\/ |==>| varstructrator))))
         (let ((printty (and tyflag (not knownty))))
         ; possibly one pair of parens for precedence, another for typing
         (let ((cl1 (closes op1 (if printty 'typed op2)))
               (cl2 (and printty (closes 'typed op2))))
            (if cl1 (ptcltoken |COMB| |(|))
            (if cl2 (ptcltoken |COMB| |(|))
            (pbegin 0)
            (setq typedvars
               (case op2
                  (var (progn (if (memq (get-var-name tm) hol-var-binops)
                                        (ptcltoken |VAR| |$|))
                              (ptclstring "VAR" (get-var-name tm))
                              (if tyflag (cons (cdr tm) typedvars) typedvars)))
                  (const (progn (print-const (get-const-name tm)) typedvars))
                  (cond (print-cond tml typedvars))
                  (listcomb (print-listcomb tml typedvars))
                  (infixcomb (print-infixcomb tml typedvars))
                  (restrict (print-ol-restrict tm typedvars))
                  (quant (print-ol-quant tm typedvars))
                  (|~| (print-ol-unop tm typedvars))
                  ((|=| /\\ \\/ |==>| |,|) (print-ol-binop tm typedvars))
                  (ol-let (print-ol-let tm typedvars))
                  (ol-list (print-ol-list tm typedvars))
                  (ol-finite-set (print-ol-finite-set tm typedvars))
                  (ol-set-abstraction (print-ol-set-abstraction tml typedvars))
                  (t (lcferror "print-tm"))))
            (cond (printty            ; print type
                  (if cl2 (ptcltoken |TYPE| |)|)
                     (ifn (memq op2 '(var const ol-list ol-finite-set))
                        (ptcltoken |TYPE| | |)))
                  (pbreak 0 0)
                  (ptcltoken |TYPE| |:|)
                  (print-ty (case op2 (ol-list (list '|list| ty))
;;;                                   (ol-finite-set (list '|set| ty))
                                      (t ty)) t)))
            (if cl1 (ptcltoken |COMB| |)|))
            (pend))
            typedvars))))

;;; DRS Modified version of pistring to pass through a tag to ptclstring
(defun pitclstring (tag str)
   (ptclstring tag (or (and |%interface_print-flag| (get str 'interface-print))
         str)))
								  
	 
;;; MJCG 19/10/88 for HOL88
;;; print a constant (may be a prefix, infix or binder standing alone)
;;; modified to invert interface-map
;;; Modified by DRS 1.11.94 - added tcl tags.
(defun print-const (name)
   (cond ((or (get name 'olinfix)
            (get name 'prefix)
            (get name 'binder))
         (ptcltoken |CONST| |$|)))
   (case name 
	(/\\ (cond 
	          (%tcl-tagged-output (ptcltoken |CONST| |/\\\\|))
	          (t (ptcltoken |CONST| |/\\|))))
	(\\/ (cond 
	          (%tcl-tagged-output (ptcltoken |CONST| |\\\\/|))
	          (t (ptcltoken |CONST| |\\/|))))
	(t (pitclstring "CONST" name))))




;;; MJCG 3/2/89 for HOL88
;;; Printing of let bindings
;;;    ((x) . u) --> x = u
;;;
;;;    ((f v1 ... vn) . u) --> f v1 ... vn = u
;;;
;;; Modified by RJB 16.11.92
;;;
;;; The variable being declared by the let binding is not printed with type
;;; information, but any occurrence of it within the body (as a recursive call)
;;; is printed with type information otherwise it may not be possible to infer
;;; the type of the body. If a structure is being declared (e.g. "(x,y)") all
;;; the variables in it *are* given types. This is because the `varstructrator'
;;; context gets lost once the structure is entered and I don't want to
;;; complicate the code with a more sophisticated technique.
;;;
;;; Note that the result returned by this function is not the usual list of
;;; variables that have already been typed but a cons of the variables declared
;;; by the let binding and that list.
;;; Modified by DRS 1.11.94 - added tcl tags.

(defun print-ol-bnd (b typedvars)
   (pibegin 0)
   (let ((letvars (print-tm (caar b) 'varstructrator nil))
         (boundvars nil))
      (pbreak 0 0)
      (mapc
         (function
            (lambda (y) (space-token) (pbreak 0 1)
               (setq boundvars
                  (append (print-tm y 'varstruct nil) boundvars))))
         (cdar b))
      (ptcltoken |COMB| | = |)
      (pbreak 0 2)
      (let ((bodyvars (append boundvars typedvars)))
         (let ((newlytypedvars (ldiff (print-tm (cdr b) 'let-rhs bodyvars)
                                  bodyvars)))
            (pend)
            (cons letvars
               (append (subtract newlytypedvars letvars) typedvars))))))

;;; MJCG 3/2/89 for HOL88
;;; Modified printing of let-terms
;;; Modified by RJB 16.11.92
;;; Modified by DRS 1.11.94 - added tcl tags.
(defun print-ol-let (tm typedvars)
   (let ((bnd (cadr tm))
         (body (caddr tm))
         (letvars nil))
      (pbegin 0)
      (ptcltoken |LETS| |let |)
      (let ((result (print-ol-bnd (car bnd) typedvars)))
         (setq letvars (append (car result) letvars))
         (setq typedvars (cdr result)))
      (mapc
         #'(lambda (y) (pbreak 1 0 ) (ptcltoken |LETS| |and |)
              (let ((result (print-ol-bnd y typedvars)))
                 (setq letvars (append (car result) letvars))
                 (setq typedvars (cdr result))))
         (cdr bnd))
      (pbreak 1 0)
      (ptcltoken |LETS| |in|)
      (pbreak 1 1)
      (let ((bodyvars (append letvars typedvars)))
         (setq typedvars
            (append
               (ldiff (print-tm body 'let-body bodyvars) bodyvars)
               typedvars)))
      (pend)
      typedvars))



;;; Modified by RJB 16.11.92
;;; Modified by DRS 1.11.94 - added tcl tags.

(defun print-ol-list (tm typedvars)
   (let ((termlist (get-term-list tm)))
      (pibegin 1)
      (ptcltoken |LIST| |[|)
      (cond (termlist
            (setq typedvars (print-tm (car termlist) t typedvars))
            (mapc
               #'(lambda (y) (ptcltoken |LIST| |;|) (pbreak 0 0)
                    (setq typedvars (print-tm y t typedvars)))
               (cdr termlist))))
      (ptcltoken |LIST| |]|)
      (pend)
      typedvars))

;;; code for printing "{t1, ... ,tn}"
;;; Duplicates code for lists -- would be more space-efficientr to fold set
;;; and list printing code into one set of routines.
;;; The current empty set and finite set constructor are held in the globals
;;; %empty-set, %finite-set-constructor.
;;; The current set abstraction constructor is held in the global
;;; %set-abstraction-constructor

;;; Modified by RJB 16.11.92
;;; Modified by DRS 1.11.94 - added tcl tags.

(defun print-ol-finite-set (tm typedvars)
   (let ((termlist (get-term-list tm)))
      (pibegin 1)
      (cond (termlist
             (cond 
	          (%tcl-tagged-output (ptcltoken |SET| |\\{|))
	          (t (ptcltoken |SET| |{|)))
             (setq typedvars (print-tm (car termlist) 'fin-set typedvars))
             (mapc
               #'(lambda (y) (ptcltoken |SET| |,|)
                             (pbreak 0 0)
                             (setq typedvars (print-tm y 'fin-set typedvars)))
               (cdr termlist))
             (cond 
	          (%tcl-tagged-output (ptcltoken |SET| |\\}|))
	          (t (ptcltoken |SET| |}|))))
	    (t
             (cond 
	          (%tcl-tagged-output (ptcltoken |CONST| |\\{\\}|))
	          (t (ptcltoken |CONST| |{}|)))))
      (pend)
      typedvars))

;;; Modified by RJB 16.11.92
;;; Modified by DRS 1.11.94 - added tcl tags.


(defun print-ol-set-abstraction (tml typedvars)
   (cond 
	(%tcl-tagged-output (ptcltoken |SET| |\\{|))
	(t (ptcltoken |SET| |{|)))
   (setq typedvars (print-tm (first tml) 'set-abs1 typedvars))
   (cond 
	(%tcl-tagged-output (space-token) (ptcltoken |SET| |\\|) (space-token) )
	(t (ptcltoken |SET| | \\|)))
   (pbreak 0 1)
   (setq typedvars (print-tm (second tml) 'set-abs2 typedvars))
   (cond 
	(%tcl-tagged-output (ptcltoken |SET| |\\}|))
	(t (ptcltoken |SET| |}|)))
   typedvars)

;;; print conditionals
;;; Modified by RJB 16.11.92
;;; Modified by DRS 1.11.94 - added tcl tags.

(defun print-cond (tml typedvars)
   (ptcltoken |COMB| |(|)
   (setq typedvars (print-tm (first tml) 'then typedvars))
   (space-token)
   (ptcltoken |COMB| |=>|)
   (space-token)
   (pbreak 0 1)
   (setq typedvars (print-tm (second tml) 'else typedvars))
   (space-token)
   (ptcltoken |COMB| "|")             ; vertical bar
   (space-token)
   (pbreak 0 1)
   (setq typedvars (print-tm (third tml) 'else typedvars))
   (ptcltoken |COMB| |)|)
   typedvars)


;;; print a long combination (f x1 ... xn)
;;; Copied from f-writol.l and modified by RJB 16.11.92
;;; Modified by DRS 1.11.94 - added tcl tags.
(defun print-listcomb (tml typedvars)
   (let ((y (pop tml)) (prev nil))
      (setq typedvars (print-tm y 'ratorofcomb typedvars))
      (while tml
         (setq prev y) (setq y (pop tml))
;         (if (and(memq (term-class prev) '(var const))
;               (memq (term-class y)    '(var const)))
            (space-token)
; )  ; space between two identifiers
         (pbreak 0 0)
         (setq typedvars (print-tm y 'listcomb typedvars)))
      typedvars))


;;; print a formula built from a unary operator
;;; Modified by RJB 16.11.92
;;; Modified by DRS 1.11.94 - added tcl tags.

(defun print-ol-unop (fm typedvars)
   (case (first fm)
      (|~| (print-neg fm typedvars))))

;;; print a formula built from a binary operator
;;; suppress parentheses using right-associativity (except for =)
;;; print tuples as an inconsistent block

;;; first an ad-hoc function for printing equations
;;; MJCG 20/10/88 for HOL88
;;; modified to use pitclstring

;;; Modified by RJB 16.11.92
;;; Modified by DRS 1.11.94 - added tcl tags.

(defun print-eq (fm typedvars)
   (setq typedvars (print-tm (second fm) '|=| typedvars))
   ;;;  (ptcltoken |COMB| | =|)              ; old code
   (space-token)(pitclstring "CONST" '|=|)
   (pbreak 1 0)
   (print-tm (third fm) '|=| typedvars))

;;; MJCG 19/10/88 for HOL88
;;; print a user-defined infix operator
;;; modified to invert interface-map
;;; Modified by RJB 16.11.92
;;; MJCG added comment on 31/01/94 for HOL88.2.02
;;; Modified by DRS 1.11.94 - added tcl tags.
(defun print-infixcomb (tml typedvars)
   (setq typedvars (print-tm (second tml) 'infixcomb typedvars))
   (space-token)
   (pitclstring "CONST" (get-const-name (first tml))) ;;; N.B. OK for infixed variables as
   (pbreak 1 0)                            ;;;      get-const-name = get-var-name (ugh!)
   (print-tm (third tml) 'infixcomb typedvars))  ; print-infixcomb

;;; MJCG 19/10/88 for HOL88
;;; print a binary operator
;;; modified to invert interface-map
;;; Modified by RJB 16.11.92
;;;
;;; Hcked to produce doubled backslashes for tcl output. (DRS)
;;; Modified by DRS 1.11.94 - added tcl tags.

(defun print-ol-binop (fm typedvars)
   (let ((op (first fm)))
      (case op
         (|=| (print-eq fm typedvars))
         (t (case op
               (|,| (pibegin 0))
               (t   (pbegin 0)))
            (while (eq op (first fm))
               (setq typedvars (print-tm (second fm) op typedvars))
               (case (first fm)
                  ;;;           (|,|   (ptcltoken |CONST| |,|)    (pbreak 0 0))
                  ;;;           (|=|   (ptcltoken |COMB| | =|)   (pbreak 1 0))
                  ;;;           (/\\  (ptcltoken |COMB| \ /\\)  (pbreak 1 0))
                  ;;;           (\\/  (ptcltoken |COMB| \  \\/   (pbreak 1 0))
                  ;;;           (|==>| (ptcltoken |COMB| | ==>|) (pbreak 1 0)))
                  (|,| (cond
                        ((and |%interface_print-flag|
                              (get '|,| 'interface-print))
                           (space-token)(pitclstring "CONST" '|,|) (pbreak 1 0))
                        (t (ptcltoken |CONST| |,|) (pbreak 0 0))))
                  (|/\\|   (space-token) (cond 
			    (%tcl-tagged-output (pitclstring "CONST" '|/\\\\|))
			    (t (pitclstring "CONST" '|/\\|)))
                         (pbreak 1 0))
                  (|\\/|   (space-token) (cond 
			    (%tcl-tagged-output (pitclstring "CONST" '|\\\\/|))
			    (t (pitclstring "CONST" '|\\/|)))
			 (pbreak 1 0))
                  (|==>| (space-token)(pitclstring "CONST" '|==>|) (pbreak 1 0)))
               (setq fm (third fm)))
            (setq typedvars (print-tm fm op typedvars))
            (pend)
            typedvars))))

;;; MJCG 20/10/88 for HOL88
;;; modified to use pistring
;;; print a negation
;;; Modified by RJB 16.11.92
;;; Modified by DRS 1.11.94 - added tcl tags.

(defun print-neg (fm typedvars)
   (pitclstring "CONST" '|~|) (print-tm (second fm) (first fm) typedvars))

;;; print Qx y z.w  instead of Qx. Qy. Qz. (where Q is a binder)
;;; this makes a big difference if the formula is broken over several lines
;;; "\" is treated as a quantifier for printing purposes
;;; Modified by DRS 1.11.94 - added tcl tags.


;;; MJCG 19/10/88 for HOL88
;;; print a quantifier
;;; modified to invert interface-map
;;; Modified by RJB 16.11.92
;;; Modified by DRS to add tcl tags.

(defun print-ol-quant (fm typedvars)
   (let ((quant (second fm))
         (vars (third fm))
         (body (fourth fm)))
      (pbegin 1)
      (case quant (|\\| (cond 
			    (%tcl-tagged-output (pitclstring "CONST" '|lambda|))
			    (t (pitclstring "CONST" '|\\|))))
                  (t (pitclstring "CONST" quant)))
      (if (not(memq quant binders)) (space-token))
      (pibegin 0)
      (let ((boundvars (print-tm vars 'quant nil)))
         (while (and (eq (first body) 'quant) (eq (second body) quant))
            (pbreak 1 0)
            (setq boundvars
               (append (print-tm (third body) 'quant nil) boundvars))
            (setq body (fourth body)))
         (pend)
         (ptcltoken |COMB| |.|)
         (pend)
         (pbreak 1 1)
         (let ((bodyvars (append boundvars typedvars)))
            (append (ldiff (print-tm body 'quant bodyvars) bodyvars)
               typedvars)))))

;;; MJCG 24.1.91
;;; Modified by RJB 16.11.92
;;; Modified by DRS to add tcl tags.
(defun print-ol-restrict (fm typedvars)
   (let ((quant (second fm))
         (restrict (third fm))
         (vars (fourth fm))
         (body (fifth fm)))
      (pbegin 1)
      (pitclstring "tm" quant)
      (if (not(memq quant binders)) (space-token))
      (pibegin 0)
      (let ((boundvars (print-tm vars 'restrict nil)))
         (while (and (eq (first body) 'restrict)
                     (eq (second body) quant)
                     (equal (third body) restrict))
            (pbreak 1 0)
            (setq boundvars
               (append (print-tm (fourth body) 'restrict nil) boundvars))
            (setq body (fifth body)))
         (pend)
         (ptcltoken |COMB| | ::|)
         (pbreak 1 1)
         (setq typedvars (print-tm restrict 'restrict typedvars))
         (ptcltoken |COMB| |.|)
         (pend)
         (pbreak 1 1)
         (let ((bodyvars (append boundvars typedvars)))
            (append
               (ldiff (print-tm body 'restrict bodyvars) bodyvars)
               typedvars)))))

;;; Change printing of predicate formulae to suppress HOL_ASSERT
;;; Modified by RJB 16.11.92
;;; Modified by DRS to add tcl tags.

(defun print-pred-form (fm)
   (cond ((not (eq (get-pred-sym fm) 'HOL_ASSERT))
         (ptclstring "COMB" (get-pred-sym fm))
         (pbreak 1 0)))
   (print-tm (get-pred-arg fm) t nil))

;;; RJB 1.7.92
;;; DRS 1.11.94
;;; Function to print a term without quotes (as a string)
;;; Function to print a term without quotes as a tcl list of tags and
;;; text.
(dml |term_to_plain_text| 1 ml-term_to_plain_text (|term| -> |string|))
(dml |term_to_hol_rich_text| 1 ml-term_to_hol_rich_text (|term| -> |string|))

;;; Modified by RJB 16.11.92
(defun ml-term_to_plain_text (tm)
 (setq %caught-output "")
 (setq %catching-output t)
 (cond (|%print_sexpr-flag| (sexpr-print(reshape-tm tm)))
       (t (print-tm (prep-tm tm) t nil)))

 (pnewline)
 (setq %catching-output nil)
 %caught-output
)
						 
(defun ml-term_to_hol_rich_text (tm)
  (setq %tcl-tagged-output t)			
  (ml-term_to_plain_text tm)
  (setq %tcl-tagged-output nil)			
  %caught-output
)

;;; Changes top-level printing of theorems to suppress quotes
;;; MJCG 10.12.90 for Centaur: modified to switch on |%print_sexpr-flag|
;;; TFM 92.07.08 : [DES] 8Jul92 line below installed.
;;; TFM 92.07.09 : previous change uninstalled, pending better solution.
;;; JRH 92.07.20 : better solution (from DES) put in; old code commented out

;;; (defun ml-print_thm (th)
;;;  (cond (|%print_sexpr-flag|
;;;         (sexpr-print
;;;           (list 'thm
;;;                 (mapcar (function reshape-thm) (car th))
;;;                 (reshape-thm (cdr th)))))
;;;        (t
;;;          (cond ((not(null(car th)))
;;;                 (mapc #'(lambda (x) (ptcltoken |COMB| |.|)) (car th))
;;;             (space-token)))  ;;; line below replaces this one.
;;; ;;;         (pbreak 1 2)))  ;;; allow a break if many hyps [DES] 8jul92
;;;          (ptclstring "THM" %turnstile)
;;;          (print-fm (prep-fm(cdr th)) t))))

(dml |thm_to_plain_text| 1 ml-thm_to_plain_text (|thm| -> |string|))
(dml |thm_to_hol_rich_text| 1 ml-thm_to_hol_rich_text (|thm| -> |string|))

(defun ml-thm_to_plain_text (th)
 (setq %caught-output "")
 (setq %catching-output t)
 (cond (|%print_sexpr-flag|
        (sexpr-print
          (list 'thm
                (mapcar (function reshape-thm) (car th))
                (reshape-thm (cdr th)))))
       (t
         (cond ((not(null(car th)))
                (pibegin 0)
                (mapc #'(lambda (x) (progn (ptcltoken |THM| |.|) (pbreak 0 0))) (car th))
                (pend)
                (cond ((> (length(car th)) (/ %margin 5)) (pbreak 1 2))
                      (t (space-token)))))
         (ptclstring "THM" %turnstile)
	 (space-token)
         (print-fm (prep-fm(cdr th)) t)))

(pnewline)
(setq %catching-output nil)
 %caught-output

)


(defun ml-thm_to_hol_rich_text (thm)
  (setq %tcl-tagged-output t)			
  (ml-thm_to_plain_text thm)
  (setq %tcl-tagged-output nil)
  %caught-output
)


;;; Printing a theorem and all its assumptions
(dml |all_thm_to_plain_text| 1 ml-all_thm_to_plain_text (|thm| -> |string|))
(dml |all_thm_to_hol_rich_text| 1 ml-all_thm_to_hol_rich_text (|thm| -> |string|))

(defun ml-all_thm_to_plain_text (th)
 (setq %caught-output "")
 (setq %catching-output t)
   (pibegin 0)
   (cond ((not(null(car th)))
         (print-fm(prep-fm(caar th))t)
         (mapc
            #'(lambda (x) (ptcltoken |THM| |, |) (pbreak 0 0) (print-fm(prep-fm x)t))
            (cdar th))
         (ptcltoken |THM| | |)
         (pbreak 0 0)))
   (ptclstring "THM" %turnstile)
   (print-fm (prep-fm(cdr th)) t)
   (pend)

(pnewline)
(setq %catching-output nil)
 %caught-output

)

(defun ml-all_thm_to_hol_rich_text (thm)
  (setq %tcl-tagged-output t)			
  (ml-all_thm_to_plain_text thm)
  (setq %tcl-tagged-output nil)
  %caught-output
)






















