%------------------------------------------------------------------------

  FILE		: mk_galois.ml                                                

  A theory of Galois connection.

  The theory is based on the definitions in CPO.th and CLa.th

  LOADS LIBRARY	: sets, tauts			       		        
  READS FILES	: CPO.th, CLa.th
  WRITES FILES  : galois.th					
 									
  AUTHOR	: Wishnu Prasetya
  AFFILIATION   : Dept. of Comp. Science, Utrecht University, 
                  the Netherlands
  DATE		: May 1995				

Let in the sequel:

    r : A->A->bool (ie, a relation on A)
    s : B->B->bool
    f : A->B
    g : B->A

Concepts/definitions supported:

  Galois r s f g   : f and g form a Galois connection wrt r and s
                     with f as the lower adjoint and g as the upper
                     adjoint. That is, they satisfy:

                     r (f x) y  =  s x (g y)

Galois connection usually assumes that r and s are partial orders and
are lattices.

Main results supported by this theory:

  - Plotkin's lemma on fix point fusion, given a lower adjoint of a
    galois connection.


May 1994
Wishnu Prasetya
wishnu@cs.ruu.nl

------------------------------------------------------------------------%

new_theory `galois`;;

library_loader (`galois`, [`sets`; `taut`],
                          [`CLa`;`CPO`],[],``,``,[]) ;;

let XRULE_ASSUM rule = RULE_ASSUM_TAC (\thm. rule thm ? thm) ;;

letrec UNDISCH_ALL_TAC (asml,g) =
   if (asml=[]) then ALL_TAC (asml,g)
   else (UNDISCH_TAC (hd asml) THEN UNDISCH_ALL_TAC) (asml,g) ;;

letrec DISCARD_ALL_TAC (asml,g) =
   if (asml=[]) then ALL_TAC (asml,g)
   else (POP_ASSUM (\thm. ALL_TAC) THEN DISCARD_ALL_TAC) (asml,g) ;;

%-------------------------------------------------------------------------
   Here is the definition of a galois connection.
-------------------------------------------------------------------------%

let GalCon = new_definition
  (`GalCon`,
   "GalCon (r:*->*->bool) (s:**->**->bool) f g = 
           (!a b. r (f a) b  =  s a (g b))") ;;
    
%-------------------------------------------------------------------------
   Here is the cancelation rule for Galois connection.
-------------------------------------------------------------------------%

let GC_lower_CANCEL = prove_thm
  (`GC_lower_CANCEL`,
   "!r s (f:**->*) g a. 
       PO s /\ GalCon r s f g ==> r ((f o g) a) a",
    REWRITE_TAC [PO; REFL; GalCon; o_THM]
    THEN REPEAT STRIP_TAC
    THEN ASM_REWRITE_TAC[]) ;;

let GC_upper_CANCEL = prove_thm
  (`GC_upper_CANCEL`,
   "!r s (f:**->*) g a. 
       PO r /\ GalCon r s f g ==> s a ((g o f) a)",
    REWRITE_TAC [PO; REFL; GalCon; o_THM]
    THEN REPEAT STRIP_TAC
    THEN XRULE_ASSUM (GEN_ALL o SYM o SPEC_ALL)
    THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
   In a diamond lattice, a lower adjoint distributes over CUP and upper
   adjoint over CAP.
-------------------------------------------------------------------------%

let GC_DISTR_CUP = prove_thm
  (`GC_DISTR_CUP`,
   "!r s (f:**->*) g.
       DLa r /\ DLa s /\ PO r /\ PO s /\ GalCon r s f g 
       ==> 
       CUP_Distr s r f",
    REWRITE_TAC [GalCon; CUP_Distr]
    THEN REPEAT STRIP_TAC
    THEN (IMP_RES_TAC o GEN_ALL o fst o EQ_IMP_RULE o SPEC_ALL) PO
    THEN SUBGOAL_THEN "!x y. (x = y) = (!z. (r:*->*->bool) x z = r y z)" 
         (\thm. REWRITE_TAC[thm])
    THENL [ IMP_RES_TAC High_EXTENSION; ALL_TAC]
    THEN GEN_TAC THEN ASM_REWRITE_TAC[]
    THEN IMP_RES_TAC (((CONV_RULE o ONCE_DEPTH_CONV) SYM_CONV) DLa_CUP_less)
    THEN ASM_REWRITE_TAC[IS_UB; IN_IMAGE]
    THEN XRULE_ASSUM (GEN_ALL o SYM o SPEC_ALL)
    THEN ASM_REWRITE_TAC []
    THEN DISCARD_ALL_TAC  
    THEN EQ_TAC THEN REPEAT STRIP_TAC
    THENL [ RES_TAC THEN ASM_REWRITE_TAC [] ;
            FIRST_ASSUM MATCH_MP_TAC
            THEN EXISTS_TAC "a:**" THEN ASM_REWRITE_TAC []
          ] ) ;;

let GC_DISTR_CAP = prove_thm
  (`GC_DISTR_CAP`,
   "!r s (f:**->*) g.
       DLa r /\ DLa s /\ PO r /\ PO s /\ GalCon r s f g 
       ==> 
       CAP_Distr r s g",
    REWRITE_TAC [GalCon; CAP_Distr]
    THEN REPEAT STRIP_TAC
    THEN (IMP_RES_TAC o GEN_ALL o fst o EQ_IMP_RULE o SPEC_ALL) PO
    THEN SUBGOAL_THEN "!x y. (x = y) = (!z. (s:**->**->bool) z x = s z y)" 
         (\thm. REWRITE_TAC[thm])
    THENL [ IMP_RES_TAC Low_EXTENSION; ALL_TAC]
    THEN XRULE_ASSUM (GEN_ALL o SYM o SPEC_ALL)
    THEN GEN_TAC THEN ASM_REWRITE_TAC[]
    THEN IMP_RES_TAC (((CONV_RULE o ONCE_DEPTH_CONV) SYM_CONV) DLa_CAP_greater)
    THEN ASM_REWRITE_TAC[IS_LB; IN_IMAGE]
    THEN XRULE_ASSUM (GEN_ALL o SYM o SPEC_ALL)
    THEN ASM_REWRITE_TAC []
    THEN DISCARD_ALL_TAC  
    THEN EQ_TAC THEN REPEAT STRIP_TAC
    THENL [ RES_TAC THEN ASM_REWRITE_TAC [] ;
            FIRST_ASSUM MATCH_MP_TAC
            THEN EXISTS_TAC "b:*" THEN ASM_REWRITE_TAC []
          ] ) ;;

%-------------------------------------------------------------------------
   A nicer Fix Point Fusion theorem can be obtained if we know that
   a lower adjoint of a galois connection is involved.
-------------------------------------------------------------------------%

let FP_upper_FUSION = prove_thm
  (`FP_upper_FUSION`,
   "!r s (f:*->**) (g:*->*) (h:**->**). 
        DLa r /\ PO r /\ Mono r r g /\ 
        DLa s /\ PO s /\ Mono s s h /\
        (?fu. GalCon s r f fu) /\
        (!x. s ((f o g) x) ((h o f) x)) 
        ==>
        s (f (LFP r g)) (LFP s h)",
   REWRITE_TAC [o_THM; GalCon] 
   THEN REPEAT STRIP_TAC
   THEN ASM_REWRITE_TAC[]
   THEN MATCH_MP_TAC TARSKI_MONO_less
   THEN XRULE_ASSUM (GEN_ALL o SYM o SPEC_ALL)
   THEN ASM_REWRITE_TAC[]
   THEN UNDISCH_TAC "PO (s:**->**->bool)"
   THEN REWRITE_TAC [PO; REFL; TRANS]
   THEN REPEAT STRIP_TAC
   THEN FIRST_ASSUM MATCH_MP_TAC
   THEN EXISTS_TAC "h((f:*->**)(fu (LFP s h)))"
   THEN ASM_REWRITE_TAC[]
   THEN IMP_RES_TAC TARSKI_MONO_equ
   THEN XRULE_ASSUM (REWRITE_RULE [Mono])
   THEN FIRST_ASSUM MATCH_MP_TAC
   THEN EXISTS_TAC "(h:**->**) (LFP s h)"
   THEN CONJ_TAC
   THENL 
   [ MATCH_MP_TAC (ASSUME "!(x:**) y. s x y ==> s(h x)(h y)")
     THEN MATCH_MP_TAC (REWRITE_RULE [o_THM] GC_lower_CANCEL)
     THEN REWRITE_TAC [GalCon]
     THEN EXISTS_TAC "r:*->*->bool"
     THEN ASM_REWRITE_TAC[] ;
     XRULE_ASSUM (REWRITE_RULE [PO; REFL; TRANS])
     THEN RES_TAC THEN ASM_REWRITE_TAC[] ]) ;;

     
let FP_FUSION = prove_thm
  (`FP_FUSION`,
   "!r s (f:*->**) (g:*->*) (h:**->**). 
        DLa r /\ PO r /\ Mono r r g /\ 
        DLa s /\ PO s /\ Mono s s h /\
        (?fu. GalCon s r f fu) /\
        (f o g = h o f)
        ==>
        (f (LFP r g) = LFP s h)",
    REPEAT STRIP_TAC
    THEN (MATCH_MP_TAC o CONJUNCT2 o CONJUNCT2 o 
          REWRITE_RULE [PO; ANTISYM] o ASSUME) "PO (s:**->**->bool)"
    THEN CONJ_TAC
    THENL
    [ MATCH_MP_TAC FP_upper_FUSION
      THEN ASM_REWRITE_TAC[]
      THEN CONJ_TAC
      THENL 
      [ EXISTS_TAC "fu:**->*" THEN ASM_REWRITE_TAC[] ; 
        REWRITE_TAC [(CONJUNCT1 o REWRITE_RULE [PO; REFL] o ASSUME) 
                      "PO (s:**->**->bool)"] ] ;
      MATCH_MP_TAC FP_lower_FUSION
      THEN ASM_REWRITE_TAC[]
      THEN REWRITE_TAC [(CONJUNCT1 o REWRITE_RULE [PO; REFL] o ASSUME) 
                         "PO (s:**->**->bool)"] 
    ] ) ;;

close_theory() ;;
