%------------------------------------------------------------------------

  FILE		: mk_CLa.ml                                                

  A theory of complete lattice. 

  The theory is based on the definitions in CPO.th.

  LOADS LIBRARY	: sets, tauts			       		        
  READS FILES	: CPO.th
  WRITES FILES  : CLa.th					
 									
  AUTHOR	: Wishnu Prasetya
  AFFILIATION   : Dept. of Comp. Science, Utrecht University, 
                  the Netherlands
  DATE		: May 1995				

  Let in the sequel r be a relation on type A, s a relation on type B, 
  f a function from A to B, and X a set on type A.
  Concepts/definitions supported:


Complete Lattice:

  CLa r        : r is a complete lattice
 
Distributivity of a function over CUP and CAP:

  CUP_Junc r s f   : f distributes over CUP (least upper bound) of
                     any non-empty set X
  CAP_Junc r s f   : f distributes over CAP (greatest lower bound) of
                     any non-empty set X
  CUP_Distr r s f  : f distributes over CUP of any set X
  CAP_Distr r s f  : f distributes over CAP of any set X

Main results supported by this theory:

  - Knaster-Tarski fix point theorem for monotonic functions.
  - Rolling Rule of fix point.
  - Plotkin's lemma for fix point fusion of monotonic functions.

May 1994
Wishnu Prasetya
wishnu@cs.ruu.nl

------------------------------------------------------------------------%

new_theory `CLa`;;

library_loader (`CLa`, [`sets`; `taut`],
                       [`CPO`],[],``,``,[]) ;;

let XRULE_ASSUM rule = RULE_ASSUM_TAC (\thm. rule thm ? thm) ;;

letrec UNDISCH_ALL_TAC (asml,g) =
   if (asml=[]) then ALL_TAC (asml,g)
   else (UNDISCH_TAC (hd asml) THEN UNDISCH_ALL_TAC) (asml,g) ;;

%-------------------------------------------------------------------------
  A complete lattice is a relation r such that for every non-empty set,
  the CUP and CAP of r exist.
-------------------------------------------------------------------------%

let CLa = new_definition
  (`CLa`,
   "CLa r = !X. ~(X={}) ==> (?a:*. isCUP r X a) /\ (?b. isCAP r X b)") ;;  

%-------------------------------------------------------------------------
  A diamond lattice is complete lattice with top and bottom element.
-------------------------------------------------------------------------%

let DLa = new_definition
  (`DLa`,
   "DLa r = !X. (?a:*. isCUP r X a) /\ (?b. isCAP r X b)") ;;  


let DLa_IMP_CLa = prove_thm
  (`DLa_IMP_CLa`,
   "!r:*->*->bool. DLa r ==> CLa r",
    REWRITE_TAC [DLa; CLa] THEN REPEAT STRIP_TAC
    THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
  In a complete lattice, the CUP of a non-empty X is indeed the least
  upper bound. Analog for CAP.
-------------------------------------------------------------------------%

let CLa_CUP = prove_thm
  (`CLa_CUP`,
   "!r (X:(*)set). CLa r /\ ~(X={}) ==> isCUP r X (CUP r X)",
    REWRITE_TAC [CLa] THEN REPEAT STRIP_TAC
    THEN RES_TAC
    THEN IMP_RES_TAC CUP) ;;

let CLa_CAP = prove_thm
  (`CLa_CAP`,
   "!r (X:(*)set). CLa r /\ ~(X={}) ==> isCAP r X (CAP r X)",
    REWRITE_TAC [CLa] THEN REPEAT STRIP_TAC
    THEN RES_TAC
    THEN IMP_RES_TAC CAP) ;;

%-------------------------------------------------------------------------
  In a diamond lattice, Bot and Top are indeed the bottom and top
  elements.
-------------------------------------------------------------------------%

let DLa_Bot_THM = prove_thm
  (`DLa_Bot_THM`,
   "!r (x:*). DLa r ==> r (Bot r) x",
    REWRITE_TAC [DLa] THEN REPEAT STRIP_TAC
    THEN MATCH_MP_TAC Bot_THM
    THEN ASM_REWRITE_TAC[]) ;;

let DLa_Top_THM = prove_thm
  (`DLa_Top_THM`,
   "!r (x:*). DLa r ==> r x (Top r)",
    REWRITE_TAC [DLa] THEN REPEAT STRIP_TAC
    THEN MATCH_MP_TAC Top_THM
    THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
  Here are some lemmas that may be handy in calculation.
-------------------------------------------------------------------------%

let CLa_CUP_less = prove_thm
  (`CLa_CUP_less`,
   "!r X (x:*). 
        CLa r /\ PO r /\ ~(X={}) ==> (IS_UB r X x = r (CUP r X) x)",
    REWRITE_TAC [CLa] THEN REPEAT STRIP_TAC 
    THEN RES_TAC THEN IMP_RES_TAC CUP_less 
    THEN ASM_REWRITE_TAC[]) ;;
      
let DLa_CUP_less = prove_thm
  (`DLa_CUP_less`,
   "!r X (x:*). 
        DLa r /\ PO r ==> (IS_UB r X x = r (CUP r X) x)",
    REWRITE_TAC [DLa] THEN REPEAT STRIP_TAC 
    THEN XRULE_ASSUM (SPEC "X:(*)set") THEN UNDISCH_ALL_TAC
    THEN REPEAT STRIP_TAC
    THEN IMP_RES_TAC CUP_less 
    THEN ASM_REWRITE_TAC[]) ;;

let CLa_CUP_greater = prove_thm
  (`CLa_CUP_greater`,
   "!r X (x:*). CLa r /\ x IN X ==> r x (CUP r X)",
    REWRITE_TAC [CLa] THEN REPEAT STRIP_TAC 
    THEN ASM_CASES_TAC "(X:(*)set)={}"
    THENL
    [  UNDISCH_TAC "(x:*) IN X"
       THEN ASM_REWRITE_TAC [NOT_IN_EMPTY] ;
       RES_TAC THEN IMP_RES_TAC CUP_greater ]) ;;

let CLa_CAP_greater = prove_thm
  (`CLa_CAP_greater`,
   "!r X (x:*). 
        CLa r /\ PO r /\ ~(X={}) ==> (IS_LB r X x = r x (CAP r X))",
    REWRITE_TAC [CLa] THEN REPEAT STRIP_TAC 
    THEN RES_TAC THEN IMP_RES_TAC CAP_greater
    THEN ASM_REWRITE_TAC[]) ;;

let DLa_CAP_greater = prove_thm
  (`DLa_CAP_greater`,
   "!r X (x:*). 
        DLa r /\ PO r ==> (IS_LB r X x = r x (CAP r X))",
    REWRITE_TAC [DLa] THEN REPEAT STRIP_TAC 
    THEN XRULE_ASSUM (SPEC "X:(*)set") THEN UNDISCH_ALL_TAC
    THEN REPEAT STRIP_TAC
    THEN IMP_RES_TAC CAP_greater
    THEN ASM_REWRITE_TAC[]) ;;
      
let CLa_CAP_less = prove_thm
  (`CLa_CAP_less`,
   "!r X (x:*). CLa r /\ x IN X ==> r (CAP r X) x",
    REWRITE_TAC [CLa] THEN REPEAT STRIP_TAC 
    THEN ASM_CASES_TAC "(X:(*)set)={}"
    THENL
    [  UNDISCH_TAC "(x:*) IN X"
       THEN ASM_REWRITE_TAC [NOT_IN_EMPTY] ;
       RES_TAC THEN IMP_RES_TAC CAP_less ]) ;;

%-------------------------------------------------------------------------
  A partial order that forms a complete lattice with bottom element is
  a CPO.
-------------------------------------------------------------------------%

let CLa_IMP_CPO = prove_thm
  (`CLa_IMP_CPO`,
   "!r. PO r /\ CLa r /\ (?a:*. isCUP r {} a) ==> CPO r",
    REWRITE_TAC [CLa; CPO]
    THEN REPEAT STRIP_TAC 
    THENL
    [ ASM_REWRITE_TAC[] ;
      EXISTS_TAC "a:*" THEN ASM_REWRITE_TAC[] ;
      ASM_CASES_TAC "(X:(*)set)={}"
      THENL
      [ EXISTS_TAC "a:*" THEN ASM_REWRITE_TAC[] ;
        RES_TAC THEN EXISTS_TAC "a':*" THEN ASM_REWRITE_TAC[]
      ] ]) ;;

%-------------------------------------------------------------------------
  Here is the definition of CAP and CUP distributive/junctive functions.
-------------------------------------------------------------------------%

let CUP_Distr = new_definition
  (`CUP_Distr`,
   "CUP_Distr r s (f:*->**) = (!X. f (CUP r X) = CUP s (IMAGE f X))") ;;

let CAP_Distr = new_definition
  (`CAP_Distr`,
   "CAP_Distr r s (f:*->**) = (!X. f (CAP r X) = CAP s (IMAGE f X))") ;;

let CUP_Junct = new_definition
  (`CUP_Junct`,
   "CUP_Junct r s (f:*->**) = 
        (!X. ~(X={}) ==> (f (CUP r X) = CUP s (IMAGE f X)))") ;;

let CAP_Junct = new_definition
  (`CAP_Junct`,
   "CAP_Junct r s (f:*->**) = 
        (!X. ~(X={}) ==> (f (CAP r X) = CAP s (IMAGE f X)))") ;;

%-------------------------------------------------------------------------
  Clearly a distributive function is also junctive.
-------------------------------------------------------------------------%

let CUP_Distr_IMP_Junct = prove_thm
  (`CUP_Distr_IMP_Junct`,
   "!r s (f:*->**). CUP_Distr r s f ==> CUP_Junct r s f",
    REWRITE_TAC [CUP_Distr; CUP_Junct] 
    THEN REPEAT STRIP_TAC THEN ASM_REWRITE_TAC[]) ;;

let CAP_Distr_IMP_Junct = prove_thm
  (`CAP_Distr_IMP_Junct`,
   "!r s (f:*->**). CAP_Distr r s f ==> CAP_Junct r s f",
    REWRITE_TAC [CAP_Distr; CAP_Junct] 
    THEN REPEAT STRIP_TAC THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
  For a CUP-distributive function f we have f Bot = Bot. (Consequently Bot 
  is the least fix point of f).
  Analog for CAP.
-------------------------------------------------------------------------%

let CUP_Distr_Bot = prove_thm
  (`CUP_Distr_Bot`,
   "!r s (f:*->**). CUP_Distr r s f ==> (f (Bot r) = (Bot s))",
    REWRITE_TAC [CUP_Distr; Bot]
    THEN REPEAT STRIP_TAC THEN ASM_REWRITE_TAC[IMAGE_EMPTY]) ;;

let CAP_Distr_Top = prove_thm
  (`CAP_Distr_Top`,
   "!r s (f:*->**). CAP_Distr r s f ==> (f (Top r) = (Top s))",
    REWRITE_TAC [CAP_Distr; Top]
    THEN REPEAT STRIP_TAC THEN ASM_REWRITE_TAC[IMAGE_EMPTY]) ;;

%-------------------------------------------------------------------------
  A CUP-junctive function that maps to a complete lattice is continous.
-------------------------------------------------------------------------%

let CUP_Junct_IMP_Cont = prove_thm
  (`CUP_Junct_IMP_Cont`,
   "!r s (f:*->**). CLa s /\ CUP_Junct r s f ==> Cont r s f",
    REWRITE_TAC [CUP_Junct; CLa; Cont] 
    THEN REPEAT STRIP_TAC
    THENL
    [ REWRITE_TAC [Directed] THEN REPEAT STRIP_TAC
      THEN SUBGOAL_THEN "~({a:**,b}={})" ASSUME_TAC
      THENL [ (CONV_TAC o ONCE_DEPTH_CONV) SYM_CONV
              THEN REWRITE_TAC [NOT_EMPTY_INSERT] ; 
              ALL_TAC]
      THEN RES_TAC
      THEN EXISTS_TAC "a':**" THEN ASM_REWRITE_TAC[] ;
      RES_TAC ]) ;;

%-------------------------------------------------------------------------
  Here begins the proof of Knaster-Tarski finx point theorem for 
  monotonic functions.
-------------------------------------------------------------------------%

let a = "CAP r {x:* | r (f x) x}" ;;
let A = "{x | r ((f:*->*) x) x}" ;;

%-------------------------------------------------------------------------
  We are going to show that a is a fix point. First, show that 
  f a is less than a.
-------------------------------------------------------------------------%

let lemma1 = prove
  ("!r (f:*->*). DLa r /\ PO r /\ Mono r r f ==> r (f ^a) ^a",
    REWRITE_TAC [Mono] THEN REPEAT STRIP_TAC
    THEN IMP_RES_TAC DLa_CAP_greater
    THEN EVERY_ASSUM 
         (\thm. REWRITE_TAC [SYM (SPEC_ALL thm)] ? ALL_TAC)
    THEN REWRITE_TAC[IS_LB]
    THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV   
    THEN REPEAT STRIP_TAC
    THEN MATCH_MP_TAC 
         ((CONJUNCT1 o CONJUNCT2 o REWRITE_RULE [TRANS; PO] o ASSUME)
           "PO (r:*->*->bool)")
    THEN EXISTS_TAC "(f:*->*) b" THEN ASM_REWRITE_TAC[]
    THEN FIRST_ASSUM MATCH_MP_TAC
    THEN MATCH_MP_TAC CLa_CAP_less
    THEN IMP_RES_TAC DLa_IMP_CLa 
    THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV 
    THEN ASM_REWRITE_TAC[]) ;;
    
%-------------------------------------------------------------------------
   Consequently f a is an element of A.
-------------------------------------------------------------------------%

let lemma2 = prove
  ("!r (f:*->*). DLa r /\ PO r /\ Mono r r f ==> (f ^a) IN ^A",
    (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV
    THEN REPEAT STRIP_TAC
    THEN IMP_RES_TAC lemma1
    THEN XRULE_ASSUM (REWRITE_RULE [Mono])
    THEN RES_TAC ) ;;

%-------------------------------------------------------------------------
   Since a is the CAP of A and f a IN A, it follows that a is less than
   f a.
-------------------------------------------------------------------------%

let lemma3 = prove
  ("!r (f:*->*). DLa r /\ PO r /\ Mono r r f ==> r ^a (f ^a)",
    REPEAT STRIP_TAC
    THEN IMP_RES_TAC lemma2
    THEN IMP_RES_TAC DLa_IMP_CLa
    THEN MATCH_MP_TAC CLa_CAP_less
    THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
   It follows from the anti-symmetry of r that f a = a.
-------------------------------------------------------------------------%

let lemma4 = prove
  ("!r (f:*->*). DLa r /\ PO r /\ Mono r r f ==> ((f ^a) = ^a)",
    REPEAT STRIP_TAC
    THEN IMP_RES_TAC lemma1
    THEN IMP_RES_TAC lemma3
    THEN UNDISCH_TAC "PO (r:*->*->bool)"
    THEN REWRITE_TAC [PO; ANTISYM]
    THEN REPEAT STRIP_TAC
    THEN RES_TAC) ;;

%-------------------------------------------------------------------------
   From the reflexivity of r it follows that a is also less than any
   fix point.
-------------------------------------------------------------------------%

let lemma5 = prove
  ("!r (f:*->*) x. DLa r /\ PO r /\ Mono r r f /\ (f x = x) 
                  ==> r ^a x",
    REWRITE_TAC [DLa; PO; REFL]
    THEN REPEAT STRIP_TAC
    THEN XRULE_ASSUM (SPEC A)
    THEN UNDISCH_ALL_TAC THEN REPEAT STRIP_TAC
    THEN MATCH_MP_TAC CAP_less
    THEN EXISTS_TAC "b:*" 
    THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV
    THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
   Here are the Knaster-Tarski theorem of fix point of a monotonic 
   function.
-------------------------------------------------------------------------%

let TARSKI_MONO_isLFP = prove_thm
  (`TARSKI_MONO_isLFP`,
   "!r (f:*->*). 
        DLa r /\ PO r /\ Mono r r f 
        ==>
        isLFP r f (CAP r {x | r (f x) x})",
   REWRITE_TAC [isLFP]
   THEN REPEAT STRIP_TAC
   THENL [ IMP_RES_TAC lemma4 ; IMP_RES_TAC lemma5 ]) ;;

let TARSKI_MONO_LFP = prove_thm
  (`TARSKI_MONO_LFP`,
   "!r (f:*->*). 
        DLa r /\ PO r /\ Mono r r f 
        ==>
        (LFP r f = CAP r {x | r (f x) x})",
    REPEAT STRIP_TAC
    THEN MATCH_MP_TAC UNIQUE_LFP
    THEN EXISTS_TAC "r:*->*->bool"
    THEN EXISTS_TAC "f:*->*"
    THEN IMP_RES_TAC TARSKI_MONO_isLFP
    THEN UNDISCH_TAC "PO (r:*->*->bool)"
    THEN REWRITE_TAC [PO] THEN REPEAT STRIP_TAC
    THEN ASM_REWRITE_TAC[]
    THEN IMP_RES_TAC LFP) ;;

let TARSKI_MONO_equ = prove_thm
  (`TARSKI_MONO_equ`,
   "!r (f:*->*). 
        DLa r /\ PO r /\ Mono r r f 
        ==>
        (f (LFP r f) = LFP r f)",
    REPEAT STRIP_TAC
    THEN IMP_RES_TAC TARSKI_MONO_LFP
    THEN IMP_RES_TAC (REWRITE_RULE [isLFP] TARSKI_MONO_isLFP)
    THEN ASM_REWRITE_TAC[]) ;;

let TARSKI_MONO_less = prove_thm
  (`TARSKI_MONO_less`,
   "!r (f:*->*) x. 
        DLa r /\ PO r /\ Mono r r f /\ (r (f x) x)
        ==>
        r (LFP r f) x",
    REPEAT STRIP_TAC
    THEN IMP_RES_TAC TARSKI_MONO_LFP
    THEN ASM_REWRITE_TAC[]
    THEN MATCH_MP_TAC CAP_less
    THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV
    THEN XRULE_ASSUM (REWRITE_RULE [DLa])
    THEN XRULE_ASSUM (SPEC A)
    THEN UNDISCH_ALL_TAC THEN REPEAT STRIP_TAC
    THEN EXISTS_TAC "b:*" THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
   Here is the Rolling Rule.
-------------------------------------------------------------------------%

let lemma1 = prove
  ("!r s (f:*->**) (g:**->*). 
        DLa r /\ PO r /\ Mono r s f /\ 
        DLa s /\ PO s /\ Mono s r g
        ==>
        s (f (LFP r (g o f))) (LFP s (f o g))",
    REPEAT STRIP_TAC 
    THEN SUBGOAL_THEN "Mono s s ((f:*->**) o (g:**->*))" ASSUME_TAC
    THENL [ IMP_RES_TAC Mono_o ; ALL_TAC ]
    THEN IMP_RES_TAC TARSKI_MONO_equ
    THEN EVERY_ASSUM (\thm. SUBST1_TAC (SYM thm) ? ALL_TAC)
    THEN REWRITE_TAC [o_THM]
    THEN MATCH_MP_TAC (REWRITE_RULE [Mono](ASSUME "Mono r s (f:*->**)"))
    THEN MATCH_MP_TAC TARSKI_MONO_less
    THEN XRULE_ASSUM (REWRITE_RULE [o_THM])
    THEN ASM_REWRITE_TAC [o_THM]
    THEN CONJ_TAC 
    THENL [ IMP_RES_TAC Mono_o ; ALL_TAC ]
    THEN UNDISCH_TAC "PO (r:*->*->bool)"
    THEN REWRITE_TAC [PO; REFL] THEN STRIP_TAC
    THEN ASM_REWRITE_TAC[]) ;;

let lemma2 = prove
  ("!r s (f:*->**) (g:**->*). 
        DLa r /\ PO r /\ Mono r s f /\ 
        DLa s /\ PO s /\ Mono s r g
        ==>
        s (LFP s (f o g)) (f (LFP r (g o f))) ",
    REPEAT STRIP_TAC 
    THEN SUBGOAL_THEN "Mono r r ((g:**->*) o (f:*->**))" ASSUME_TAC
    THENL [ IMP_RES_TAC Mono_o ; ALL_TAC ]
    THEN IMP_RES_TAC TARSKI_MONO_equ
    THEN SUBGOAL_THEN "Mono s s ((f:*->**) o (g:**->*))" ASSUME_TAC
    THENL [ IMP_RES_TAC Mono_o ; ALL_TAC ]
    THEN MATCH_MP_TAC TARSKI_MONO_less 
    THEN XRULE_ASSUM (REWRITE_RULE [o_THM])
    THEN ASM_REWRITE_TAC [o_THM]
    THEN UNDISCH_TAC "PO (s:**->**->bool)"
    THEN REWRITE_TAC [PO; REFL] THEN STRIP_TAC
    THEN ASM_REWRITE_TAC[]) ;;

let ROLING_THM = prove_thm
  (`ROLING_THM`,
   "!r s (f:*->**) (g:**->*). 
        DLa r /\ PO r /\ Mono r s f /\ 
        DLa s /\ PO s /\ Mono s r g
        ==>
        (f (LFP r (g o f)) = LFP s (f o g))",
   REPEAT STRIP_TAC 
   THEN SUBGOAL_THEN "ANTISYM (s:**->**->bool)" 
        (\thm. MATCH_MP_TAC (REWRITE_RULE [ANTISYM ]thm))
   THENL [ UNDISCH_TAC "PO (s:**->**->bool)"
           THEN REWRITE_TAC [PO] THEN TAUT_TAC ;
           ALL_TAC ]
   THEN CONJ_TAC
   THENL [ MATCH_MP_TAC lemma1 THEN ASM_REWRITE_TAC[] ;
           MATCH_MP_TAC lemma2 THEN ASM_REWRITE_TAC[]] ) ;;

%-------------------------------------------------------------------------
   Here is Plotkin's lemma of Fix Point Fusion.
-------------------------------------------------------------------------%

let FP_lower_FUSION = prove_thm
  (`FP_lower_FUSION`,
   "!r s (f:*->**) (g:*->*) (h:**->**). 
        DLa r /\ PO r /\ Mono r r g /\ 
        DLa s /\ PO s /\ Mono s s h /\
        (!x. s ((h o f) x) ((f o g) x))
        ==>
        s (LFP s h) (f (LFP r g))",
   REWRITE_TAC [o_THM] THEN REPEAT STRIP_TAC
   THEN MATCH_MP_TAC TARSKI_MONO_less
   THEN ASM_REWRITE_TAC[]
   THEN UNDISCH_TAC "PO (s:**->**->bool)"
   THEN REWRITE_TAC [PO; REFL; TRANS]
   THEN REPEAT STRIP_TAC
   THEN FIRST_ASSUM MATCH_MP_TAC
   THEN EXISTS_TAC "(f:*->**) (g (LFP r g))"
   THEN IMP_RES_TAC TARSKI_MONO_equ
   THEN ASM_REWRITE_TAC[]) ;;


close_theory() ;;






