(****************************************************************************)
(*                                                                          *)
(* Copyright 1997, 1998 University of Cambridge and University of Edinburgh *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : socket_server.sml                                        *)
(* DESCRIPTION   : Using a process as a server via sockets (SML/NJ 110).    *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 21st March 1997                                          *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 14th January 1998                                        *)
(****************************************************************************)

structure SocketServer =
struct

datatype socket
   = Unix of (UnixSock.unix,Socket.active Socket.stream) Socket.sock * string
   | INet of (INetSock.inet,Socket.active Socket.stream) Socket.sock;

exception Lookup;
fun lookup name =
   case (NetHostDB.getByName name)
   of SOME entry => (case (NetHostDB.addrs entry)
                     of [in_addr] => in_addr
                      | _ => raise Lookup)
    | _ => raise Lookup;

(* Required to force other connecting processes to hang *)
fun dummy_connect sock_addr =
   let val s = INetSock.TCP.socket ()
   in  Socket.connect (s,sock_addr); Socket.close s
   end;

exception StartServer of string;
fun start_server exec =
   let fun address filename =
          if OS.FileSys.access (filename,[])
          then let val sock_addr =
                      (OS.Process.system "sleep 1"; UnixSock.toAddr filename)
               in  OS.Process.system "sleep 1"; sock_addr
               end
          else address filename
       val pid =
          Word32.toString (Posix.Process.pidToWord (Posix.ProcEnv.getpid ()))
       val tmp_file = "/tmp/socket." ^ pid
   in  OS.Process.system (exec ^ " " ^ tmp_file ^ " &");
       let val sock_addr = address tmp_file
           val socket = UnixSock.Strm.socket ()
       in  Socket.connect (socket,sock_addr);
           Unix (socket,tmp_file)
       end
   end;

fun connect_to_local_server filename =
   let val sock_addr = UnixSock.toAddr filename
       val socket = UnixSock.Strm.socket ()
   in  Socket.connect (socket,sock_addr);
       Unix (socket,filename)
   end;

fun connect_to_server {host,port} =
   let val sock_addr = INetSock.toAddr (lookup host,port)
       val socket = INetSock.TCP.socket ()
   in  Socket.connect (socket,sock_addr);
       dummy_connect sock_addr; dummy_connect sock_addr;
       INet socket
   end;

fun close (Unix (socket,filename)) =
   (Socket.close socket; OS.Process.system ("rm -f " ^ filename); ())
  | close (INet socket) = Socket.close socket;

fun send_string (socket,s:string) =
   let val v = Word8Vector.tabulate
                  (String.size s,
                   fn i => Word8.fromInt (Char.ord (String.sub (s,i))))
       fun send socket = Socket.sendVec (socket,{buf = v,i = 0,sz = NONE})
   in  case socket of Unix (s,_) => send s | INet s => send s
   end;

fun receive_string socket =
   let val block_size = 1024
       fun receive socket = Socket.recvVec (socket,block_size)
       val v = case socket of Unix (s,_) => receive s | INet s => receive s
   in  String.implode
          (Word8Vector.foldr (fn (e,l) => Char.chr (Word8.toInt e) :: l) [] v)
   end;

end; (* SocketServer *)


(* For testing sockets interactively:
val host = "<machine name>" : string
and port = <socket number> : int;
val sock_addr = INetSock.toAddr (SocketServer.lookup host,port);
val socket = INetSock.TCP.socket () : Socket.active INetSock.stream_sock;
val _ = Socket.connect(socket,sock_addr);

SocketServer.send_string(socket,"this(is,a(test)).\n");
SocketServer.send_string(socket,"another(test).\n");
SocketServer.send_string(socket,"quit.\n");

Socket.close socket;
*)
