(****************************************************************************)
(*                                                                          *)
(* Copyright 1998-1999 University of Cambridge and University of Edinburgh  *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : schemes.sml                                              *)
(* DESCRIPTION   : Finding and sending induction schemes.                   *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 4th August 1998                                          *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 13th October 1999                                        *)
(****************************************************************************)

structure Schemes =
struct

local

open CLaReTPortable;

val built_in_types =
   ["fun","bool","prod","one","sum","ind"];

fun type_operators_in_type ops ty =
   let val {Args,Tyop} = dest_type ty
       val ops' = if (member Tyop ops) orelse (member Tyop built_in_types)
                  then ops
                  else Tyop :: ops
   in  type_operators_in_types ops' Args
   end
   handle HOL_ERR _ => ops
and type_operators_in_types ops [] = ops
  | type_operators_in_types ops (ty::tys) =
   type_operators_in_types (type_operators_in_type ops ty) tys;

fun type_operators_in_term types tm =
   (case (dest_term tm)
    of CONST {Name,Ty} => type_operators_in_type types Ty
     | VAR {Name,Ty} => type_operators_in_type types Ty
     | COMB {Rator,Rand} =>
          type_operators_in_term (type_operators_in_term types Rator) Rand
     | LAMB {Bvar,Body} => type_operators_in_term types Body);

fun type_operators_in_terms types [] = types
  | type_operators_in_terms types (tm::tms) =
   type_operators_in_terms (type_operators_in_term types tm) tms;

in

fun theory_of_type name =
   find (member name o map #Name o types)
      (current_theory () :: ancestry "-");

fun possible_schemes theory =
   let val induct = explode "induct"
       and Induct = explode "Induct"
       and INDUCT = explode "INDUCT"
       fun select name =
          let val chars = explode name
          in  (is_subsequence induct chars) orelse
              (is_subsequence Induct chars) orelse
              (is_subsequence INDUCT chars)
          end
   in  filter (select o fst) (theorems theory)
   end;

datatype theory_kind = Theory of string | BasicLogic | NoTheory;

fun subjects_of_scheme th =
   type_operators_in_types [] (CallClam.subjects_of_scheme (concl th));

fun subjects_and_schemes names theory =
   let fun sub [] = []
         | sub ((name,th)::schemes) =
          let fun filter [] = sub schemes
                | filter (s::ss) =
                 if (member s names)
                 then (s,(name,Theory theory,th)) :: filter ss
                 else filter ss
          in  filter (subjects_of_scheme th)
          end
   in  sub (possible_schemes theory)
   end;

fun schemes_for_type_operators names =
   let val names_and_theories = map (fn n => (n,theory_of_type n)) names
       val theories = setify (map snd names_and_theories)
       val names_and_schemes =
          flat (map (fn t => subjects_and_schemes
                                (map fst (filter (fn (n,t') => t' = t)
                                             names_and_theories)) t)
                   theories)
   in  map (fn n =>
              (n,map snd (filter (fn (name,_) => name = n) names_and_schemes)))
          names
   end;

fun send_schemes_for_terms ignore tms =
   let fun theory_string (Theory s) = "theory `" ^ s ^ "'"
         | theory_string BasicLogic = "basic logic"
         | theory_string NoTheory = "no theory"
       fun message kind (name,thy) tyop =
          Lib.say ("Sending " ^ kind ^ " `" ^ name ^ "' (" ^
                   theory_string thy ^ ") for type operator `" ^ tyop ^ "'\n")
       fun process_schemes (tyop,[]) =
          Lib.say ("No scheme found for type operator `" ^ tyop ^ "'\n")
         | process_schemes (tyop,schemes) =
          (map (fn (name,thy,th) =>
                   (message "scheme" (name,thy) tyop;
                    CallClam.send_scheme (name,th)
                    handle _ => Lib.say "... send failed\n")) schemes;
           ())
       fun process_rules (tyop,rules) =
          (map (fn (name,thy,th) =>
                   (message "rule" (name,thy) tyop;
                    CallClam.send_rule (name,th)
                    handle _ => Lib.say "... send failed\n")) rules;
           ())
       fun non_null [] = raise Empty
         | non_null x = x
       fun add_theory (c,th) = (c,BasicLogic,th)
       val known = type_operators_in_types [] (ClamDatabase.known_hol_types ())
       val tyops = type_operators_in_terms [] tms
       val unknown_tyops = subtract (subtract tyops known) ignore
       val basic_data =
          mapfilter (fn n => (n,HOLClamBasicLogic.data_for_type n))
             unknown_tyops
       val basic_schemes =
          mapfilter
             (fn (n,ds) =>
                 (n,map add_theory (non_null (HOLClamBasicLogic.schemes ds))))
             basic_data
       val basic_rules =
          mapfilter
             (fn (n,ds) => (n,map add_theory (HOLClamBasicLogic.rules ds)))
             basic_data
       (* Only consider tyops for which there is no data *)
       (* even if there are no schemes in the data.      *)
       val other_tyops = subtract unknown_tyops (map fst basic_data)
       val other_schemes = schemes_for_type_operators other_tyops
   in  map process_schemes (basic_schemes @ other_schemes);
       map process_rules basic_rules;
       ()
   end;

val auto_send = ref true;

local

val disabled = ref ([]:string list);

in

fun disable c = (disabled := c :: !disabled);

fun enable c =
   (disabled := CLaReTPortable.filter (fn s => not (s = c)) (!disabled));

fun send_schemes gl =
   if !auto_send
   then let val (tms,tm) = gl
        in  send_schemes_for_terms (!disabled) (tm :: tms)
        end
   else ();

val disabled = fn () => !disabled;

end;

end;

end; (* Schemes *)
