(****************************************************************************)
(*                                                                          *)
(* Copyright 1997-1999 University of Cambridge and University of Edinburgh  *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : hol_to_clam.sml                                          *)
(* DESCRIPTION   : Translation of HOL terms into Clam syntax.               *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 23rd January 1997                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 27th January 1999                                        *)
(****************************************************************************)

structure HOLtoClam =
struct

exception HOLtoClam of string;

local

fun error s = raise HOLtoClam s;

open Psyntax;

structure C = ClamAST;

val arg1 = rand o rator
and arg2 = rand;

val T = (--`T`--)
and F = (--`F`--);

fun is_digit c = "0" <= c andalso c <= "9";

fun is_uppercase c = "A" <= c andalso c <= "Z";

fun is_symbol c = " " <= c andalso c <= "/" orelse
                  ":" <= c andalso c <= "@" orelse
                  "[" <= c andalso c <= "`" orelse
                  "{" <= c andalso c <= "~";

fun symbol_name "!" = "PLING"
  | symbol_name "\"" = "DQUOTE"
  | symbol_name "#" = "SHARP"
  | symbol_name "$" = "DOLLAR"
  | symbol_name "%" = "PERCENT"
  | symbol_name "&" = "AMPERSAND"
  | symbol_name "*" = "TIMES"
  | symbol_name "+" = "PLUS"
  | symbol_name "," = "COMMA"
  | symbol_name "-" = "MINUS"
  | symbol_name "." = "DOT"
  | symbol_name "/" = "DIV"
  | symbol_name ":" = "COLON"
  | symbol_name ";" = "SEMICOLON"
  | symbol_name "<" = "LESS"
  | symbol_name "=" = "EQ"
  | symbol_name ">" = "GREAT"
  | symbol_name "?" = "QUESTION"
  | symbol_name "@" = "AT"
  | symbol_name "[" = "LBRACKET"
  | symbol_name "\\" = "BSLASH"
  | symbol_name "]" = "RBRACKET"
  | symbol_name "^" = "CARET"
  | symbol_name "{" = "LBRACE"
  | symbol_name "|" = "VBAR"
  | symbol_name "}" = "RBRACE"
  | symbol_name "~" = "TILDE"
  | symbol_name c = c;

fun symbol_sequence [] = (0,[])
  | symbol_sequence (cs as c::cs') =
   if (is_symbol c)
   then if (hd cs' = c handle Hd => false)
        then let val (n,cs'') = symbol_sequence cs' in (n + 1,cs'') end
        else (1,cs')
   else (0,cs);

fun replace_symbols s =
   let fun replace [] = ""
         | replace (cs as "'"::cs') = "_" ^ replace cs'
         | replace (cs as c::cs') =
          let val (n,cs'') = symbol_sequence cs
          in  if (n < 1)
              then c ^ (replace cs')
              else (symbol_name c) ^
                   (if (n = 1) then "" else int_to_string n) ^
                   (replace cs'')
          end
   in  replace (CLaReTPortable.explode s)
   end;

in

fun translate_metalanguage_name s = "hol_" ^ replace_symbols s;

fun translate_objectlanguage_name s =
   let val name = replace_symbols s
       val first = substring (name,0,1)
   in  if (is_uppercase first) orelse (is_digit first)
       then "hol" ^ name
       else name
   end;

type translation = (string * string) list;
type translations = translation * translation * translation * translation;

val null_translations = ([],[],[],[]);

fun merge_translations (tyvs1,tycs1,tmvs1,tmcs1) (tyvs2,tycs2,tmvs2,tmcs2) =
(*    (tyvs1 @ tyvs2,tycs1 @ tycs2,tmvs1 @ tmvs2,tmcs1 @ tmcs2); *)
 (union tyvs1 tyvs2, union tycs1 tycs2, union tmvs1 tmvs2, union tmcs1 tmcs2);


fun unitN x = (x,null_translations);

fun bindN (m,f) =
   let val (x,translations1) = m
       val (y,translations2) = f x
   in  (y,merge_translations translations1 translations2)
   end;

fun bindN_map (mf,mxs,f) =
   let fun bind_list mf [] = unitN []
         | bind_list mf (mx::mxs) =
          bindN (mf mx,fn y =>
          bindN (bind_list mf mxs,fn ys =>
          unitN (y :: ys)))
   in  bindN (bind_list mf mxs,f)
   end;

fun translate_vartype hname =
   let val cname = "t_" ^ substring (hname,1,size hname - 1)
   in  (cname,([(cname,hname)],[],[],[]))
   end;

fun translate_type_const hname =
   let val cname = "hol" ^ replace_symbols hname
   in  (cname,([],[(cname,hname)],[],[]))
   end;

fun translate_var v =
   let val (hname,_) = dest_var v
       val cname = translate_objectlanguage_name hname
   in  (cname,([],[],[(cname,hname)],[]))
   end;

fun translate_const c =
   let val (hname,_) = dest_const c
       val cname = translate_objectlanguage_name hname
   in  (cname,([],[],[],[(cname,hname)]))
   end;

fun translate_type ty =
   if (is_vartype ty)
   then bindN (translate_vartype (dest_vartype ty),unitN o C.Identifier)
   else (case (dest_type ty)
         of (con,[]) => bindN (translate_type_const con,unitN o C.Identifier)
          | ("fun",[fty,aty]) => bindN (translate_type fty,fn f =>
                                 bindN (translate_type aty,fn a =>
                                 unitN (C.Prop (f,"=>",a))))
          | (con,tys) => bindN (translate_type_const con,fn c =>
                         bindN_map (translate_type,tys,fn ts =>
                         unitN (C.Apply (c,ts)))));

fun translate_term tm =
   if (is_comb tm) then
      let val (f,args) = strip_comb tm
      in  if (is_const f)
          then bindN (translate_const f,fn f' =>
               bindN_map (translate_subterm,args,fn args' =>
               unitN (C.Apply (f',args'))))
          else bindN (translate_term f,fn f' =>
               bindN_map (translate_subterm,rev args,fn args' =>
               unitN (CLaReTPortable.fold (fn (x,f) => C.FunApp (f,x))
                         args' f')))
      end
   else if (is_abs tm) then
      let val (bvar,body) = dest_abs tm
          val (_,ty) = dest_var bvar
      in  bindN (translate_var bvar,fn v =>
          bindN (translate_type ty,fn ty' =>
          bindN (translate_subterm body,fn t =>
          unitN (C.Lambda (C.Inhabit (v,ty'),t)))))
      end
   else if (is_var tm) then bindN (translate_var tm,unitN o C.Identifier)
   else if (is_const tm) then bindN (translate_const tm,unitN o C.Identifier)
   else error "can't translate term"

and translate_subterm tm =
   if (type_of tm = bool)
   then translate_formula tm
   else translate_term tm

and translate_formula tm =
   if (is_forall tm) then
      let val (v,t) = dest_forall tm
          val (_,ty) = dest_var v
      in  bindN (translate_var v,fn v' =>
          bindN (translate_type ty,fn ty' =>
          bindN (translate_formula t,fn t' =>
          unitN (C.Prop (C.Inhabit (v',ty'),"=>",t')))))
      end
   else if (is_exists tm) then
      let val (v,t) = dest_exists tm
          val (_,ty) = dest_var v
      in  bindN (translate_var v,fn v' =>
          bindN (translate_type ty,fn ty' =>
          bindN (translate_formula t,fn t' =>
          unitN (C.Prop (C.Inhabit (v',ty'),"#",t')))))
      end
   else if (is_neg tm) then bindN (translate_formula (rand tm),fn f =>
                            unitN (C.Prop (f,"=>",C.Void)))
   else if (is_conj tm) then
      bindN (translate_formula (arg1 tm),fn a1 =>
      bindN (translate_formula (arg2 tm),fn a2 =>
      unitN (C.Prop (a1,"#",a2))))
   else if (is_disj tm) then
      bindN (translate_formula (arg1 tm),fn a1 =>
      bindN (translate_formula (arg2 tm),fn a2 =>
      unitN (C.Prop (a1,"\\",a2))))
   else if (is_imp tm) then
      bindN (translate_formula (arg1 tm),fn a1 =>
      bindN (translate_formula (arg2 tm),fn a2 =>
      unitN (C.Prop (a1,"=>",a2))))
   else if (is_eq tm) andalso (type_of (arg1 tm) = bool) then
      bindN (translate_formula (arg1 tm),fn a1 =>
      bindN (translate_formula (arg2 tm),fn a2 =>
      unitN (C.Prop (a1,"<=>",a2))))
   else if (is_eq tm) then
      bindN (translate_term (arg1 tm),fn a1 =>
      bindN (translate_term (arg2 tm),fn a2 =>
      bindN (translate_type (type_of (arg1 tm)),fn ty =>
      unitN (C.Eq (a1,a2,ty)))))
   else if (tm = T) then unitN C.True
   else if (tm = F) then unitN C.Void
   else translate_term tm;

fun translate_type_variable holtyvar =
   bindN (translate_vartype (dest_vartype holtyvar),fn v' =>
   unitN (C.Inhabit (v',C.Universe 1)));

fun bind_vartypes holtyvars clamty =
   let fun bind (v,cty) =
          bindN (translate_type_variable v,fn v' =>
          bindN (cty,fn cty' =>
          unitN (C.Prop (v',"=>",cty'))))
   in  CLaReTPortable.fold bind holtyvars clamty
   end;

fun translate_top_level_formula tm =
   bind_vartypes (type_vars_in_term tm) (translate_formula tm);

fun translate_goal (hyps,conc) =
   let val tyvars = rev_itlist (union o type_vars_in_term) (conc :: hyps) []
   in  bindN_map (translate_type_variable,tyvars,fn ts =>
       bindN_map (translate_formula,hyps,fn hs =>
       bindN (translate_formula conc,fn c =>
       unitN (C.Goal (ts @ hs,c)))))
   end;

end;

end; (* HOLtoClam *)
